# -*- coding: utf-8 -*-

"""
Configuration defaults and handling
"""

from six.moves import configparser 
import os.path
import tempfile


CONFIG_LOCATION = "~/.calibre-utils"


class Config(object):
    """
    Wrapper for configuration settings
    """

    # Command names for external programs

    calibredb = "calibredb"
    pdftotext = "pdftotext"
    catdoc = "catdoc"
    djvutxt = "djvutxt"
    archmage = "archmage"
    ebook_device = "ebook-device"

    # How many pages to scan looking for ISBN - in page-based docs
    guess_lead_pages = 10
    # How many lines to scan looking for ISBN - in non-paged docs
    guess_lead_lines = 10000

    # Where to cache calibre metadata. Created below as it involves creating directory
    cache_dir = None

    # Every that many files progress note is shown
    progress_report_every = 50

    def __init__(self, inifile=CONFIG_LOCATION):
        """
        Loads params from given config file, creates the file if missing,
        adds missing params to file if not all are present.
        """

        # .ini section naming
        commands_section = "commands"
        isbn_section = "isbn-search"
        cache_section = "cache"

        config = configparser.SafeConfigParser({})

        ini = os.path.abspath(os.path.expanduser(inifile))
        if os.path.exists(ini):
            config.read(ini)

        for section in [commands_section, isbn_section, cache_section]:
            if not config.has_section(section):
                config.add_section(section)

        config_changed = False

        for cmdopt in ["calibredb",
                       "ebook_device",
                       "pdftotext",
                       "catdoc",
                       "djvutxt",
                       "archmage"]:
            if config.has_option(commands_section, cmdopt):
                setattr(self, cmdopt, config.get(commands_section, cmdopt))
            else:
                config_changed = True
                config.set(commands_section, cmdopt, getattr(self, cmdopt))

        for cmdopt in ["guess_lead_pages", "guess_lead_lines"]:
            if config.has_option(isbn_section, cmdopt):
                setattr(self, cmdopt, config.getint(isbn_section, cmdopt))
            else:
                config_changed = True
                config.set(isbn_section, cmdopt, str(getattr(self, cmdopt)))

        if config.has_option(cache_section, "cache_dir"):
            self.cache_dir = config.get(cache_section, "cache_dir")
            if not os.path.isdir(self.cache_dir):
                try:
                    os.mkdir(self.cache_dir)
                except Exception:
                    raise Exception("""Cache directory {0} does not exist and can't be created.
Edit your config and fix it's location
(or just remove whole [cache] section to have it autogenerated).
Config location: {1}""".format(self.cache_dir, ini))
        else:
            config_changed = True
            self.cache_dir = tempfile.mkdtemp(prefix='calibre_utils_')
            config.set(cache_section, "cache_dir", self.cache_dir)

        if config_changed:
            config.write(open(ini, 'w'))


_CONFIG = None


def standard_config():
    """
    Return configuration singleton.

    It's a function (instead of global variable) to avoid parsing .ini
    if it is not used (for example - during tests).
    """
    global _CONFIG
    if not _CONFIG:
        _CONFIG = Config()
    return _CONFIG

# TODO: report missing tools and suggest how to install them
