#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""Middleware classes for johnny cache."""

import django
from django.core.exceptions import ImproperlyConfigured
from django.core.cache import get_cache
from django.core import signals
from django.middleware import transaction as trans_middleware
from django.db import transaction
from johnny import cache, settings

if settings.CACHE_BACKEND:
    cache_backend = get_cache(settings.CACHE_BACKEND)

    # Some caches -- python-memcached in particular -- need to do a cleanup at the
    # end of a request cycle. If the cache provides a close() method, wire it up
    # here.
    if hasattr(cache_backend, 'close'):
        signals.request_finished.connect(cache_backend.close)
else:
    from django.core.cache import cache as cache_backend

class QueryCacheMiddleware(object):
    """This middleware class monkey-patches django's ORM to maintain
    generational info on each table (model) and to automatically cache all
    querysets created via the ORM.  This should be the first middleware
    in your middleware stack."""
    __state = {} # alex martinelli's borg pattern
    def __init__(self):
        self.__dict__ = self.__state
        self.disabled = settings.DISABLE_QUERYSET_CACHE
        self.installed = getattr(self, 'installed', False)
        if not self.installed and not self.disabled:
            # when we install, lets refresh the blacklist, just in case johnny
            # was loaded before the setting exists somehow...
            cache.blacklist = settings.BLACKLIST
            self.query_cache_backend = cache.get_backend()(cache_backend)
            self.query_cache_backend.patch()
            self.installed = True

    def unpatch(self):
        self.query_cache_backend.unpatch()
        self.query_cache_backend.flush_query_cache()
        self.installed = False

class LocalStoreClearMiddleware(object):
    """This middleware clears the localstore cache in `johnny.cache.local`
    at the end of every request."""
    def process_exception(self, *args, **kwargs):
        cache.local.clear()
        raise

    def process_response(self, req, resp):
        cache.local.clear()
        return resp

class CommittingTransactionMiddleware(trans_middleware.TransactionMiddleware):
    """A version of the built in TransactionMiddleware that always commits its
    transactions, even if they aren't dirty."""
    def process_response(self, request, response):
        if transaction.is_managed():
            try: transaction.commit()
            except: pass
            transaction.leave_transaction_management()
        return response

