# Copyright (c) 2010 by Yaco Sistemas
#
# This file is part of Merengue.
#
# Merengue is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Merengue is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Merengue.  If not, see <http://www.gnu.org/licenses/>.

import traceback

from django.db import models
from django.utils.importlib import import_module
from django.utils.translation import ugettext_lazy as _

from south.modelsinspector import add_introspection_rules

from merengue.registry.dbfields import ConfigField
from merengue.registry.managers import RegisteredItemManager


class RegisteredItem(models.Model):
    class_name = models.CharField(max_length=100, db_index=True)
    module = models.CharField(max_length=200, db_index=True)
    category = models.CharField(max_length=100, db_index=True)
    active = models.BooleanField(default=False)
    broken = models.BooleanField(default=False, editable=False)
    order = models.IntegerField(_("Order"), blank=True, null=True)
    traceback = models.TextField(_("Error traceback"), default='', editable=False,
                                 help_text=_("Error traceback on broken item"))
    config = ConfigField(_('Configuration'))

    objects = RegisteredItemManager()

    class Meta:
        verbose_name = _('registered item')
        verbose_name_plural = _('registered items')
        ordering = ('order', )

    def __unicode__(self):
        return self.class_name

    def set_default_config(self, item_class):
        if not self.config:
            self.config = {}
        for param in item_class.config_params:
            if param.has_default():
                self.config[param.name] = param.default
        self.save()

    def get_registry_item(self):
        module = import_module(self.module)
        item_class = getattr(module, self.class_name)
        return item_class(self)

    def has_config(self):
        return bool(self.get_registry_item().config_params)

    def get_config(self):
        parent_instance = getattr(self, 'registereditem_ptr', None)
        if parent_instance is not None:
            # because a inheritance JSONField problem
            return parent_instance.config
        else:
            return self.config

    def activate(self, commit=True):
        if not self.active:
            self.active = True
            if commit:
                self.save()

    def deactivate(self, commit=True):
        if self.active:
            self.active = False
            if commit:
                self.save()

    def set_traceback(self, exc_type, exc_value, tb):
        formatted_exception = []
        formatted_exception.append('Exception Type: %s' % exc_type)
        formatted_exception.append('Exception Value: %s' % exc_value)
        formatted_exception.append('Traceback:')
        traceback_frames = traceback.extract_tb(tb)
        formatted_exception += traceback.format_list(traceback_frames)
        self.traceback = '<br/>'.join(formatted_exception)


# ----- adding south rules to help introspection -----

rules = [
  (
    (ConfigField, ),
    [],
    {},
  ),
]

add_introspection_rules(rules, ["^merengue\.registry"])
