# -*- coding: utf-8 -*-
# :Progetto:  PatchDB -- Generic SQL script execution context
# :Creato:    sab 31 mag 2014 13:00:48 CEST
# :Autore:    Lele Gaifax <lele@metapensiero.it>
# :Licenza:   GNU General Public License version 3 or later
#

from . import logger
from . import ExecutionContext, ExecutionError


class SqlContext(ExecutionContext):
    """
    Generic SQL execution context.

    This is still somewhat abstract, subclasses must reimplement
    at least :method:`makeConnection()` and :method:`setupContext()`.
    """

    language_name = 'sql'

    def __init__(self, **args):
        """Initialize the instance.

        Open the DB connection and execute the setup, if needed.
        """

        ExecutionContext.__init__(self)

        self.makeConnection(**args)
        self.setupContext()

        self._patches = None

    def __getitem__(self, patchid):
        """
        Get the applied revision of a given `patchid`, or None.
        """

        return self.patches.get(patchid)

    def __setitem__(self, patchid, revision):
        """
        Cache the given `revision` as the last applied version of `patchid`.
        """

        self.patches[patchid] = revision

    @property
    def patches(self):
        """
        Extract the applied info from the database, returning a
        dictionary mapping a patch id to its revision.
        """

        if self._patches is None:
            cursor = self.connection.cursor()
            cursor.execute("SELECT patchid, revision FROM patchdb")
            patches = cursor.fetchall()
            cursor.close()

            self._patches = dict((patchid.decode('utf-8'), revision)
                                 for patchid, revision in patches)

        return self._patches

    def apply(self, patch, options):
        """
        Try to execute the given `patch` script, which may be
        composed by one or more SQL statements separated by two
        consecutive semicomma ``;;`` on a line by their own::

          CREATE DOMAIN integer_t INTEGER
          ;;
          CREATE DOMAIN string_t VARCHAR(20)

        If everything goes well, update the persistent status of
        the given `patch`, storing its `revision` in the ``patchdb``
        table in the database.
        """

        from re import split, MULTILINE
        from sys import stderr

        cursor = self.connection.cursor()
        stmts = split(r'^\s*;;\s*$', patch.script, flags=MULTILINE)

        last_good_point = None
        current_line = 1
        for i, stmt in enumerate(stmts):
            if stmt:
                stmt = self.prepareStatement(stmt)
                if not stmt:
                    continue

                logger.debug(u"Executing '%s ...'" % stmt[:50])

                try:
                    cursor.execute(stmt.encode('utf-8'))
                    current_line += stmt.count('\n')
                    last_good_point = i+1
                    self.savePoint(last_good_point)
                    if not options.verbose:
                        stderr.write('.')
                except Exception as e:
                    errmsg, syntaxerror, nonexistingobj = self.classifyError(e)

                    if nonexistingobj and stmt.lstrip()[:5].lower() == u'drop ':
                        onerror = u'ignore'
                    else:
                        onerror = patch.onerror

                    if last_good_point:
                        self.rollbackPoint(last_good_point)
                    else:
                        self.connection.rollback()

                    if len(stmts)>1:
                        details = u"statement at line %d of " % current_line
                    else:
                        details = u""

                    if onerror == u'abort' or syntaxerror:
                        logger.critical(u"Execution of %s%s generated an"
                                        u" error: %s", details, patch, errmsg)
                        logger.debug(u"Statement: %s", stmt)
                        raise ExecutionError(u"Execution of %s%s generated an"
                                             u" error: %s" %
                                             (details, patch, errmsg))
                    elif onerror == u'ignore':
                        logger.info(u"Ignoring error generated by %s%s: %s",
                                    details, patch, errmsg)
                        if not options.verbose:
                            stderr.write('I')
                    elif onerror == u'skip':
                        logger.info(u"Skipping succeding statements due to"
                                    u" error executing %s%s: %s",
                                    details, patch, errmsg)
                        if not options.verbose:
                            stderr.write('S')
                        break

        self.applied(patch)

    def prepareStatement(self, statement):
        """Possibly adjust the given `statement` before execution.

        This implementation simply returns `statement.strip()`.
        Subclasses may apply arbitrary transformations to it, or return
        ``None`` to discard its execution.
        """

        return statement.strip()

    def classifyError(self, exc):
        """Determine the kind of error given its exception.

        Return a tuple (message, syntaxerror, nonexistingobj).
        """

        raise NotImplementedError(u'Subclass responsibility')

    def _recordAppliedInfo(self, pid, rev):
        """Persist the knowledge on the database."""

        cursor = self.connection.cursor()

        pid = pid.encode('utf-8')
        cursor.execute("SELECT revision"
                       " FROM patchdb"
                       " WHERE patchid = %s", (pid,))
        rec = cursor.fetchone()
        if rec is None:
            logger.debug(u'Inserting "%s@%s" into the database', pid, rev)
            cursor.execute("INSERT INTO patchdb (patchid, revision, applied)"
                           " VALUES (%s, %s, CURRENT_TIMESTAMP)", (pid, rev))
        else:
            logger.debug(u'Updating "%s@%s" in the database', pid, rev)
            cursor.execute("UPDATE patchdb"
                           " SET revision = %s, applied = CURRENT_TIMESTAMP"
                           " WHERE patchid = %s", (rev, pid))

        self[pid] = rev

    def applied(self, patch):
        """Register the given `patch` as *applied*.

        Update the persistent knowledge about the given `patch`,
        storing it's revision on the database. The same is done
        on all the patches this script may have upgraded.
        """

        if patch.brings:
            for pid,rev in patch.brings:
                self._recordAppliedInfo(pid, rev)
        self._recordAppliedInfo(patch.patchid, patch.revision)
        self.connection.commit()

    def makeConnection(self, **args):
        """Open the connection with the database."""

        raise NotImplementedError(u'Subclass responsibility')

    def setupContext(self):
        """Possibly create the tables used for persistent knowledge."""

        raise NotImplementedError(u'Subclass responsibility')

    def savePoint(self, point):
        """Possibly commit the work up to this point."""

    def rollbackPoint(self, point):
        """Possibly rollback to point."""
