
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""HACK : MySQLdb export facility for CA Track Nationals

"""

import threading
import Queue
import logging
import MySQLdb

from metarace import strops

# connection string:
# [user[:pass]@]host[:port][/db]
# defaults:
DB_USER = 'root'
DB_PASSWD = 'velodrome'
DB_HOST = 'localhost'
DB_PORT = 3306
DB_DATABASE = 'Cycling'

# dispatch thread queue commands
TCMDS = ('EXIT', 'PORT', 'MSG', 'COMMIT')

def parse_portstr(portstr=''):
    """Return a reasonable best guess user/pass/host/port/database."""
    user = DB_USER
    passwd = DB_PASSWD
    host = DB_HOST
    port = DB_PORT
    db = DB_DATABASE
    if len(portstr) > 0:
        # LHS: user:pass
        upart = ''
        if '@' in portstr:
            (upart, sep, portstr) = portstr.partition('@')
        if len(upart) > 0:
            (user, sep, passwd) = upart.partition(':')

        # RHS: host:port/db
        if '/' in portstr:
            # sets Database
            (portstr, sep, db) = portstr.partition('/')
        if ':' in portstr:
            # sets host and port
            (host, sep, pstr) = portstr.partition(':')
            if pstr.isdigit():	# but ensure ok to use
                port = int(pstr)
        else:
            # host resolved by elimination
            host = portstr
    return (user, passwd, host, port, db)
    
class dbexport(threading.Thread):
    """MySQLdb exporter

    """

    def __init__(self, port=None):
        """Constructor."""
        threading.Thread.__init__(self) 
        self.name = 'dbexport'
        self.port = None
        self.queue = Queue.Queue()
        self.log = logging.getLogger('dbexport')
        self.log.setLevel(logging.DEBUG)
        self.running = False
        if port is not None:
            self.setport(port)

    def commit(self):
        """Queue a 'COMMIT' transaction on the server."""
        self.queue.put_nowait(('COMMIT', ))

    def execute(self, sqlcmd, args=[()]):	# default is one exec no args
        """Execute a multiple sql query as provided, ignoring the return."""
        self.queue.put_nowait(('MSG', sqlcmd, args))

    def exit(self, msg=None):
        """Request thread termination."""
        self.running = False
        self.queue.put_nowait(('EXIT', msg))

    def wait(self):             # NOTE: Do not call from cmd thread
        """Suspend calling thread until cqueue is empty."""
        self.queue.join()

    def setport(self, port=None):
        """Dump command queue content and (re)connect

        Specify hostname and port for TCP connection:

            hostname:16372

        """
        try:
            while True:
                self.queue.get_nowait()
                self.queue.task_done()
        except Queue.Empty:
            pass 
        self.queue.put_nowait(('PORT', port))

    def connected(self):
        """Return true if SCB connected."""
        return self.port is not None

    def __procsql(self, cmd, args):
        """Send the sql to the db as a batch."""
        c = self.port.cursor()
        c.executemany(cmd, args)
        c.close()
        return None

    def __connect(self, portstr=''):
        """Connect to the db if possible, but allow error to raise."""
        (u, p, h, t, d) = parse_portstr(portstr) 
        self.log.info('Connecting to: {0}:{1}@{2}:{3}/{4}'.format(u,p,h,t,d))
        return MySQLdb.connect(host=h, port=t,
                               passwd=p, user=u, db=d)

    def run(self):
        """Called via threading.Thread.start()."""
        self.running = True
        self.log.debug('Starting')
        while self.running:
            m = self.queue.get()
            self.queue.task_done()
            try:
                if m[0] == 'MSG' and self.port is not None:
                    self.log.debug('Sending command: ' + repr(m[1]))
                    self.__procsql(m[1], m[2])
                elif m[0] == 'COMMIT' and self.port is not None:
                    self.log.debug('Calling COMMIT on db.')
                    self.port.commit()
                elif m[0] == 'EXIT':
                    self.log.debug('Request to close : ' + str(m[1]))
                    self.running = False
                elif m[0] == 'PORT':
                    if self.port is not None:
                        self.port.close()
                        self.port = None	# should get __del__ here
                    if m[1] is not None and m[1] != '' and m[1] != 'NULL':
                        self.log.debug('Re-Connect port: ' + str(m[1]))
                        self.port = self.__connect(str(m[1]))

                    else:
                        self.log.debug('Not connected.')

            except MySQLdb.Error as e:
                self.log.error('DB Error: ' + str(type(e)) + str(e))
                if self.port is not None:
                    self.port.close()
                self.port = None
            except Exception as e:
                self.log.error('Exception: ' + str(type(e)) + str(e))
        if self.port is not None:
            self.port.close()
            self.port = None
        self.log.info('Exiting')

if __name__ == "__main__":
    """Simple 'Hello World' example with logging."""
    import time
    h = logging.StreamHandler()
    h.setLevel(logging.DEBUG)
    s = dbexport('localhost')
    s.log.addHandler(h)
    s.start()

    evt = '21a'
    print ('deleting all records with event id = ' + repr(evt))
    s.execute('DELETE FROM Starters WHERE EventID=%s', [(evt,)])
    time.sleep(1)
    print ('Adding a few entries.')
    #s.execute('INSERT INTO Starters (EventID, SortOrder, RiderNumber, FirstName, Surname, State, InfoA, InfoB, Rank) VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)', [(evt, 0, '12', 'First', 'LAST', 'VIC', '', '', ''),
    #(evt, 1, '11', 'Second', 'GYUY', 'NSW', '', '', '')])
    time.sleep(1)
    s.exit('hello done.')		# signal thread to end
    s.join()				# wait for thread to terminate
