
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from xml.sax.saxutils import escape, quoteattr
import sys

"""HTML output lib.

Cheap and nasty functional primitives for HTML output. Each primitive
outputs a single string. No checking is performed on the structure of
the document produced. All elements take a named parameter 'attrs'
which is a dict of key/value attributes. Non-empty elements take a
parameter 'clist' which is a list of other constructed elements.

Example for an empty element:

    hr(attrs={'id':'thehr'})

returns the string: '<hr id="thehr" />'

Example for an element with content:

    a(['link text'], attrs={'href':'#target'})

returns the string: '<a href="#target">\nlink text\n</a>'

Example paragraph:

    p(['Check the',
       a(['website'], attrs={'href':'#website'}),
       'for more.'])

Returns: '<p>\nCheck the\n<a href="#website">\nwebsite\n</a>\nfor more.\n</p>'

"""

def html(headlist=[], bodylist=[]):
    """Emit HTML document."""
    return '\n'.join([
      preamble(),
      '<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">',
      head(headlist),
      body(bodylist),
      '</html>'])

def preamble():
    """Emit XHTML preamble."""
    return """<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE html 
     PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN"
     "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">"""

def attrlist(attrs):
    """Convert attr dict into properly escaped attrlist."""
    alist = []
    for a in attrs:
        alist.append(a.lower() + '=' + quoteattr(attrs[a]))
    if len(alist) > 0:
        return ' ' + ' '.join(alist) 
    else:
        return ''

def escapetext(text=''):
    """Return escaped copy of text."""
    return escape(text, {'"':'&quot;'})

def comment(commentstr=''):
    """Insert comment."""
    return '<!-- ' + commentstr.replace('--','') + ' -->'

# Declare all the empty types
for empty in ['meta', 'link', 'base', 'param',
              'hr', 'br', 'img', 'input', 'col']:
    def emptyfunc(attrs={}, elem=empty):
        return '<' + elem + attrlist(attrs) + ' />'
    setattr(sys.modules[__name__], empty, emptyfunc) 

# Declare all the non-empties
for nonempty in ['head', 'body', 'title', 'div', 'style', 'script',
                 'object', 'p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
                 'ul', 'ol', 'li', 'dl', 'dt', 'dd', 'address',
                 'pre', 'blockquote', 'a', 'span', 'em', 'strong',
                 'dfn', 'code', 'samp', 'kbd', 'var', 'cite',
                 'abbr', 'acronym', 'q', 'sub', 'sup', 'tt',
                 'i', 'big', 'small', 'label', 'form', 'select',
                 'optgroup', 'option', 'textarea', 'fieldset',
                 'legend', 'button', 'table', 'caption',
                 'thead', 'tfoot', 'tbody', 'colgroup',
                 'tr', 'th', 'td', ]:
    def nonemptyfunc(clist=[], attrs={}, elem=nonempty):
        if type(clist) is str:
            clist = [clist]
        return('<' + elem + attrlist(attrs) + '>'
                + '\n'.join(clist) + '</' + elem + '>')
    setattr(sys.modules[__name__], nonempty, nonemptyfunc) 


# test run
if __name__ == "__main__":
    print(
        html([title('Test Title'), meta({'name':'Author', 'content':'Test'}),
            link({'type':'text/css', 'href':'style.css', 'rel':'stylesheet'}),
            meta({'http-equiv':'Content-Style-Type', 'content':'text/css'})],
             [h1('Document Heading'),
              comment('inline comment'),
              p(em('Text under heading.')),
              h2('Subheading'),
              p(['The paragraph of text under the subheading, and with some',
                 tt('0h12:34.56'),
                 'time text.'])])
    )
