
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Basic string filtering, truncation and padding."""

import re
import metarace
import os


INTEGER_TRANS = '\
        \
        \
        \
        \
        \
     -  \
01234567\
89      \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        '
"""Bib translation table for digit strings."""

PLACELIST_TRANS = '\
        \
        \
        \
        \
        \
     -  \
01234567\
89      \
 ABCDEFG\
HIJKLMNO\
PQRSTUVW\
XYZ     \
 abcdefg\
hijklmno\
pqrstuvw\
xyz     \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        '
"""Bib translation table for place listings."""

PLACESERLIST_TRANS = '\
        \
        \
        \
        \
        \
     -. \
01234567\
89      \
 ABCDEFG\
HIJKLMNO\
PQRSTUVW\
XYZ     \
 abcdefg\
hijklmno\
pqrstuvw\
xyz     \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        '
"""Bib translation table for place listings with bib.ser strings."""

BIBLIST_TRANS = '\
        \
        \
        \
        \
        \
        \
01234567\
89      \
 ABCDEFG\
HIJKLMNO\
PQRSTUVW\
XYZ     \
 abcdefg\
hijklmno\
pqrstuvw\
xyz     \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        '
"""Bib translation table for parsing bib lists."""

BIBSERLIST_TRANS = '\
        \
        \
        \
        \
        \
      . \
01234567\
89      \
 ABCDEFG\
HIJKLMNO\
PQRSTUVW\
XYZ     \
 abcdefg\
hijklmno\
pqrstuvw\
xyz     \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        '
"""Bib.ser translation table for parsing bib.ser lists."""

PRINT_TRANS = '\
        \
        \
        \
        \
 !"#$%&\'\
()*+,-./\
01234567\
89:;<=>?\
@ABCDEFG\
HIJKLMNO\
PQRSTUVW\
XYZ[\\]^_\
`abcdefg\
hijklmno\
pqrstuvw\
xyz{|}~ \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        \
        '
"""Basic printing ASCII character table."""

def img_filename(path, basename=''):
    """Return the 'best' filename for the given basename image."""
    ret = None
    srchs = ['svg', 'eps', 'pdf', 'png', 'jpg', 'bmp']
    basename = basename.rstrip('.')
    for xtn in srchs:
        checka = os.path.join(path, basename + '.' + xtn)
        if os.path.isfile(checka):
            ret = checka
            break
    return ret

DNFCODEMAP = { 'hd': 0,
               'dsq': 1,
               'dnf': 2,
               'dns': 3,
               '': 2}

def cmp_dnf(x, y):
    """Comparison func for two dnf codes."""
    if x not in DNFCODEMAP:
        x = ''
    if y not in DNFCODEMAP:
        y = ''
    return cmp(DNFCODEMAP[x], DNFCODEMAP[y])
    
def riderno_key(bib):
    """Return a comparison key for sorting rider number strings."""
    return bibstr_key(bib)

def bibstr_key(bibstr=''):
    """Return a comparison key for sorting rider bib.ser strings."""
    (bib, ser) = bibstr2bibser(bibstr)
    bval = 0
    if bib.isdigit():
        bval = int(bib)
    else:
        bval = id(bib)
    sval = 0
    if ser != '':
        sval = ord(ser[0])<<12
    return sval | (bval&0xfff)

def promptstr(prompt='', value=''):
    """Prefix a non-empty string with a prompt, or return empty."""
    ret = ''
    if value:
        ret = prompt + ' ' + value
    return ret

def listsplit(liststr=''):
    """Return a split and stripped list."""
    ret = []
    for e in liststr.split(','):
        ret.append(e.strip())
    return ret

def heatsplit(heatstr):
    """Return a failsafe heat/lane pair for the supplied heat string."""
    hv = heatstr.split('.')
    while len(hv) < 2:
        hv.append('0')
    return(riderno_key(hv[0]), riderno_key(hv[1]))
    
def fitname(first, last, width, trunc=False):
    """Return a 'nicely' truncated name field for display.

    Attempts to modify name to fit in width as follows:

    1: 'First Lastone-Lasttwo'    - simple concat
    2: 'First Lasttwo'            - ditch hypenated name
    3: 'F. Lasttwo'               - abbrev first name
    4: 'F Lasttwo'                - get 1 xtra char omit period
    5: 'F. Lasttwo'               - give up and return name for truncation

    If optional param trunc is set and field would be longer than
    width, truncate and replace the last 3 chars with elipsis '...'
    Unless only two char longer than field - then just chop final chars

    """
    ret = ''
    fstr = str(first).strip().title()
    lstr = str(last).strip().upper()
    trystr = (fstr + ' ' + lstr).strip()
    if len(trystr) > width:
        lstr = lstr.split('-')[-1].strip()
        trystr = fstr + ' ' + lstr
        if len(trystr) > width:
            if len(fstr) > 0:
                trystr = fstr[0] + '. ' + lstr
            else:
                trystr = lstr
            if len(trystr) == width + 1 and len(fstr) > 0:  # opportunistic
                trystr = fstr[0] + ' ' + lstr
    if trunc:
        ret = trystr[0:width]
        if width > 6:
            if len(trystr) > width+2:
                ret = trystr[0:(width - 3)] + '...'
    else:
        ret = trystr
    return ret

def num2ord(place):
    """Return ordinal for the given place."""
    omap = { '1' : 'st',
             '2' : 'nd',
             '3' : 'rd',
             '11' : 'th',
             '12' : 'th',
             '13' : 'th' }
    if place in omap:
        return place + omap[place]
    elif place.isdigit():
        if len(place) > 1 and place[-1] in omap:
            return place + omap[place[-1]]
        else:
            return place + 'th'
    else:
        return place

def mark2int(handicap):
    """Convert a handicap string into an integer number of metres."""
    handicap = handicap.strip().lower()
    ret = None				# not recognised as handicap
    if handicap != '':
        if handicap[0:3] == 'scr':		# 'scr{atch}'
            ret = 0
        else:				# try [number]m form
           handicap = handicap.translate(INTEGER_TRANS).strip()
           try:
               ret = int(handicap)
           except:
               pass
    return ret
       
def truncpad(srcline, length, align='l', elipsis=True):
    """Return srcline truncated and padded to length, aligned as requested."""
    ret = srcline[0:length]
    if length > 6:
        if len(srcline) > length+2 and elipsis:
            ret = srcline[0:(length - 3)] + '...'
    if align == 'l':
        ret = ret.ljust(length)
    elif align == 'r':
        ret = ret.rjust(length)
    else:
        ret = ret.center(length)
    return ret

def search_name(namestr):
    return namestr.translate(BIBLIST_TRANS).strip().lower()

def resname_bib(bib, first, last, club):
    """Return rider name formatted for results with bib (champs/live)."""
    ret = bib + ' ' + fitname(first, last, 64)
    if club is not None and club != '':
        ret += ' (' + club + ')'
    return ret

def resname(first, last, club):
    """Return rider name formatted for results."""
    ret = fitname(first, last, 64)
    if club is not None and club != '':
        ret += ' (' + club + ')'
    return ret

def listname(first, last=None, club=None):
    """Return a rider name summary field for non-edit lists."""
    ret = fitname(first, last, 32)
    if club:
        ret += ' (' + club + ')'
    return ret

def reformat_bibserlist(bibserstr):
    """Filter and return a bib.ser start list."""
    return ' '.join(bibserstr.translate(BIBSERLIST_TRANS).split())

def reformat_bibserplacelist(placestr):
    """Filter and return a canonically formatted bib.ser place list."""
    if placestr.find('-') < 0:		# This is the 'normal' case!
        return reformat_bibserlist(placestr)
    # otherwise, do the hard substitutions...
    # TODO: allow the '=' token to indicate RFPLACES ok 
    placestr = placestr.translate(PLACESERLIST_TRANS).strip()
    placestr = re.sub(r'\s*\-\s*', r'-', placestr)	# remove surrounds
    placestr = re.sub(r'\-+', r'-', placestr)		# combine dupes
    return ' '.join(placestr.strip('-').split())

def reformat_biblist(bibstr):
    """Filter and return a canonically formatted start list."""
    return ' '.join(bibstr.translate(BIBLIST_TRANS).split())

def reformat_riderlist(riderstr, rdb=None, series=''):
    """Filter, search and return a list of matching riders for entry."""
    ret = ''
    ##riderstr = riderstr.translate(PLACELIST_TRANS).lower()
    riderstr = riderstr.lower()

    # special case: 'all' -> return all riders from the sepcified series.
    if riderstr.strip() == 'all':
        for cat in rdb.listcats(series):
            ret += ' ' + rdb.biblistfromcat(cat, series)
            riderstr = ''
    
    # pass 1: search for categories
    if rdb is not None:
        for cat in sorted(rdb.listcats(series), key=len, reverse=True):
            if len(cat) > 0 and riderstr.find(cat.lower()) >= 0:
                ret += ' ' + rdb.biblistfromcat(cat, series)
                riderstr = riderstr.replace(cat.lower(), '')

    # pass 2: append riders and expand any series if possible
    riderstr = reformat_placelist(riderstr)
    for nr in riderstr.split():
        if nr.find('-') >= 0:
            # try for a range...
            l = None
            n = None
            for r in nr.split('-'):
                if l is not None:
                    if l.isdigit() and r.isdigit():
                        start = int(l)
                        end = int(r)
                        if start < end:
                            c = start
                            while c < end:
                                ret += ' ' + str(c)
                                c += 1
                        else:
                            ret += ' ' + l	# give up on last val
                    else:
                        # one or both not ints
                        ret += ' ' + l
                else:
                    pass
                l = r
            if l is not None: # catch final value
                ret += ' ' + l
        else:
            ret += ' ' + nr
    # pass 3: reorder and join for return
    rvec = list(set(ret.split()))
    rvec.sort(key=riderno_key)
    return ' '.join(rvec)

def placeset(spec=''):
    """Convert a place spec into a set of place ints."""
    ret = ''
    spec = reformat_placelist(spec)
    # pass 1: expand ranges
    for nr in spec.split():
        if spec.find('-') >= 0:
            # try for a range...
            l = None
            n = None
            for r in nr.split('-'):
                if l is not None:
                    if l.isdigit() and r.isdigit():
                        start = int(l)
                        end = int(r)
                        if start < end:
                            c = start
                            while c < end:
                                ret += ' ' + str(c)
                                c += 1
                        else:
                            ret += ' ' + l	# give up on last val
                    else:
                        # one or both not ints
                        ret += ' ' + l
                else:
                    pass
                l = r
            if l is not None: # catch final value
                ret += ' ' + l
        else:
            ret += ' ' + nr
    # pass 2: filter out non-numbers
    rset = set()
    for i in ret.split():
        if i.isdigit():
            rset.add(int(i))
    return rset

def reformat_placelist(placestr):
    """Filter and return a canonically formatted place list."""
    if placestr.find('-') < 0:		# This is the 'normal' case!
        return reformat_biblist(placestr)
    # otherwise, do the hard substitutions...
    placestr = placestr.translate(PLACELIST_TRANS).strip()
    placestr = re.sub(r'\s*\-\s*', r'-', placestr)	# remove surrounds
    placestr = re.sub(r'\-+', r'-', placestr)		# combine dupes
    return ' '.join(placestr.strip('-').split())

def confopt_bool(confstr):
    """Check and return a boolean option from config."""
    if confstr.lower() in ['yes', 'true', '1']:
        return True
    else:
        return False

def plural(count=0):
    """Return plural extension for provided count."""
    ret = 's'
    if count == 1:
        ret = ''
    return ret
    
def confopt_float(confstr, default=None):
    """Check and return a floating point number."""
    ret = default
    try:
        ret = float(confstr)
    except ValueError:
        pass
    return ret

def confopt_distunits(confstr):
    """Check and return a valid unit from metres or laps."""
    if confstr.lower() == 'laps':
        return 'laps'
    else:
        return 'metres' 

def confopt_posint(confstr, default=None):
    """Check and return a valid positive integer."""
    ret = default
    if confstr.isdigit():
        ret = int(confstr)
    return ret

def confopt_dist(confstr, default=None):
    """Check and return a valid distance unit."""
    ret = default
    if confstr.isdigit():
        ret = int(confstr)
    return ret

def confopt_chan(confstr, default=None):
    """Check and return a valid timing channel id string."""
    from metarace import timy 	# for channel info
				# BAD BAD BAD: strops should have no
				# external dependencies
    ret = timy.chan2id(default)
    ival = timy.chan2id(confstr)
    if ival != timy.CHAN_UNKNOWN:
        ret = ival
    return ret

def confopt_pair(confstr, value, default=None):
    """Return value or the default."""
    ret = default
    if confstr.lower() == value.lower():
        ret = value
    return ret

def confopt_list(confstr, list=[], default=None):
    """Return an element from list or default."""
    ret = default
    for elem in list:
        if confstr.lower() == elem.lower():
            ret = elem
            break
    return ret

def bibstr2bibser(bibstr=''):
    """Split a bib.series string and return bib and series."""
    a = bibstr.strip().split('.')
    ret_bib = ''
    ret_ser = ''
    if len(a) > 0:
        ret_bib = a[0]
    if len(a) > 1:
        ret_ser = a[1]
    return (ret_bib, ret_ser)

def bibser2bibstr(bib='', ser=''):
    """Return a valid bib.series string."""
    ret = bib
    if ser != '':
        ret += '.' + ser
    return ret

def titlesplit(src='', linelen=22):
    """Split a string on word boundaries to try and fit into 3 fixed lines."""
    ret = ['', '', '']
    words = src.split()
    wlen = len(words)
    if wlen > 0:
        line = 0
        ret[line] = words.pop(0)
        for word in words:
            pos = len(ret[line])
            if pos + len(word) > linelen:
                # new line
                line += 1
                if line > 2:
                    break
                ret[line] = word
            else:
                ret[line] += ' ' + word

    return ret

class countback(object):
    __hash__ = None
    """Simple dict wrapper for countback store/compare."""
    def __init__(self):
        self.__store = {}

    def maxplace(self):
        """Return maximum non-zero place."""
        ret = 0
        if len(self.__store) > 0:
            ret = max(self.__store.keys())
        return ret
    def __str__(self):
        ret = []
        for i in range(1,self.maxplace()+1):
            if i in self.__store and self.__store[i] != 0:
                ret.append(str(self.__store[i]))
            else:
                ret.append('-')
        return ','.join(ret)
    def __len__(self):
        return len(self.__store.len)
    def __getitem__(self, key):
        """Use a default value id, but don't save it."""
        if key in self.__store:
            return self.__store[key]
        else:
            return 0
    def __setitem__(self, key, value):
        self.__store[key] = value
    def __delitem__(self, key):
        del(self.__store[key])
    def __iter__(self):
        return self.__store.iterkeys()
    def iterkeys(self):
        return self.__store.iterkeys()
    def __contains__(self, item):
        return item in self.__store
    def __lt__(self, other):
        if type(other) is not countback: return NotImplemented
        ret = False # assume all same
        for i in range(1,max(self.maxplace(), other.maxplace())+1):
            a = self[i]
            b = other[i]
            if a != b:
                ret = a < b
                break
        return ret
    def __le__(self, other):
        if type(other) is not countback: return NotImplemented
        ret = True # assume all same
        for i in range(1,max(self.maxplace(), other.maxplace())+1):
            a = self[i]
            b = other[i]
            if a != b:
                ret = a < b
                break
        return ret
    def __eq__(self, other):
        if type(other) is not countback: return NotImplemented
        ret = True
        for i in range(1,max(self.maxplace(), other.maxplace())+1):
            if self[i] != other[i]:
                ret = False
                break
        return ret
    def __ne__(self, other):
        if type(other) is not countback: return NotImplemented
        ret = False
        for i in range(1,max(self.maxplace(), other.maxplace())+1):
            if self[i] != other[i]:
                ret = True
                break
        return ret
    def __gt__(self, other):
        if type(other) is not countback: return NotImplemented
        ret = False # assume all same
        for i in range(1,max(self.maxplace(), other.maxplace())+1):
            a = self[i]
            b = other[i]
            if a != b:
                ret = a > b
                break
        return ret
    def __ge__(self, other):
        if type(other) is not countback: return NotImplemented
        ret = True # assume all same
        for i in range(1,max(self.maxplace(), other.maxplace())+1):
            a = self[i]
            b = other[i]
            if a != b:
                ret = a > b
                break
        return ret

