
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tag Heuer/Chronelec Decoder Interface

This module provides a thread object which interfaces with a
Tag Heuer/Chronelec Elite Decoder.


  Messages are returned as TOD objects:

	index: 		power:batt	eg 99:0 or '' for trig/sts
	chan:  		BOX	rfid on box channel
			STA	rfid on sta channel
			MAN	manual trigger
			STS	status, refid gets the noise/level info
	timeval:	time of day of passing/trig/status
	refid:		transponder id with leading zeros stripped


  Sent to mainloop via glib.idle_add of the provided callback

"""
import threading
import Queue
import serial
import logging
import glib

from metarace import tod

# System default timy serial port
DEFPORT = '/dev/ttyUSB0'

# Serial baudrate
THBC_BAUD = 19200

# Photofinish threshold - ~20cm based on tests at DISC
THBCPHOTOTHRESH = tod.tod('0.02')

# THbC protocol messages
ESCAPE = chr(0x1b)
HELOCMD = 'MR1'
STARTCMD = ESCAPE + chr(0x07)
STOPCMD = ESCAPE + chr(0x13) + chr(0x5c)
REPEATCMD = ESCAPE + chr(0x12)
ACKCMD = ESCAPE + chr(0x11)
STATCMD = ESCAPE + chr(0x05)
QUECMD = ESCAPE + chr(0x10)	# fetch configuration
SETCMD = ESCAPE + chr(0x08)	# set configuration
IPCMD = ESCAPE + chr(0x09)	# set IP configuration
STALVL = ESCAPE + chr(0x1e)
BOXLVL = ESCAPE + chr(0x1f)

NACK = chr(0x07)
LF = chr(0x0a)
SETTIME = ESCAPE + chr(0x48)
STATSTART = '['
PASSSTART = '<'

# thread queue commands -> private to timy thread
TCMDS = ('EXIT', 'PORT', 'MSG', 'TRIG', 'SYNC', 'REPL')

RFID_LOG_LEVEL = 16     # lower so not in status and on-screen logger.
logging.addLevelName(RFID_LOG_LEVEL, 'RFID')

adder = lambda sum, ch: sum + ord(ch)

def thbc_sum(msgstr=''):
    return '{0:04d}'.format(reduce(adder, msgstr, 0))

class thbc(threading.Thread):
    """Tag Heuer Elite thread object class."""
    def __init__(self, port=None, name='thbc'):
        """Construct thread object.

        Named parameters:

          port -- serial port
          name -- text identifier for use in log messages

        """
        threading.Thread.__init__(self) 
        self.name = name

        self.port = None
        self.error = False
        self.errstr = ''
        self.cqueue = Queue.Queue()	# command queue
        self.log = logging.getLogger(self.name)
        self.log.setLevel(logging.DEBUG)
        self.__cksumerr = 0
        self.__rdbuf = ''
        self.setcb()
        if port is not None:
            self.setport(port)

    def photothresh(self):
        """Return the relevant photo finish threshold."""
        return THBCPHOTOTHRESH		# allow override perhaps?

    def __defcallback(self, evt=None):
        """Default callback is a tod log entry."""
        self.log.debug(str(evt))
        return False

    def getcb(self):
        """Return the current callback function."""
        return self.__cb

    def setcb(self, func=None):
        """Set or clear the event callback."""
        # if func is not callable, gtk mainloop will catch the error
        if func is not None:
            self.__cb = func
        else:
            self.__cb = self.__defcallback

    def write(self, msg=None):
        """Queue a raw command string to attached decoder."""
        self.cqueue.put_nowait(('MSG', msg))

    def exit(self, msg=None):
        """Request thread termination."""
        self.running = False
        self.cqueue.put_nowait(('EXIT', msg)) # "Prod" command thread

    def setport(self, device=None):
        """Request (re)opening port as specified.

        Device may be a port number or a device identifier string.
        For information on port numbers or strings, see the
        documentation for serial.Serial().

        Call setport with no argument, None, or an empty string
        to close an open port or to run the timy thread with no
        external device.

        """
        self.cqueue.put_nowait(('PORT', device))

    def sync(self):
        """Roughly synchronise Decoder to host PC clock."""
        self.cqueue.put_nowait(('SYNC', None))

    def sane(self):
        """Send a prod packet, no other config."""
        self.write(HELOCMD)

    def trig(self, timeval='now', index='FAKE', chan='MAN', refid='0'):
        """Create a fake timing event.

	   Generate a new tod object to mimic a message as requested
           and pipe it to the command thread. Default tod is the
	   'now' time in the calling thread.

        """
        t = tod.tod(timeval, index, chan, refid.lstrip('0'))
        self.cqueue.put_nowait(('TRIG', t))

    def start_session(self):
        """Send a depart command to decoder."""
        self.write(STARTCMD)

    def stop_session(self):
        """Send a stop command to decoder."""
        self.write(STOPCMD)

    def status(self):
        """Request status message from decoder."""
        self.write(STATCMD)

    def setlvl(self, box=u'10', sta=u'10'):
        """Set the read level on box and sta channels."""
        # TODO: verify opts
        self.write(BOXLVL + box.encode('ascii', 'ignore')[0:2])
        self.write(STALVL + sta.encode('ascii', 'ignore')[0:2])
        
    def replay(self, filename=''):
        """Read passings from file and process."""
        self.cqueue.put_nowait(('REPL', filename))

    def wait(self):
        """Suspend calling thread until the command queue is empty."""
        self.cqueue.join()

    def __set_time_cmd(self, t):
        """Return a set time command string for the provided time of day."""
        s = int(t.timeval)
        st = chr(s%60)
        mt = chr((s//60)%60)
        ht = chr(s//3600)
        return SETTIME + ht + mt + st + chr(0x74)

    def __parse_message(self, msg, ack=True):
        """Return tod object from timing msg or None."""
        ret = None
        if len(msg) > 4:
            if msg[0] == PASSSTART:	# RFID message
                idx = msg.find('>')
                if idx == 37:		# Valid length
                    data = msg[1:33]
                    msum = msg[33:37]
                    tsum = thbc_sum(data)
                    if tsum == msum:	# Valid 'sum'
                        pvec = data.split()
                        istr = pvec[3] + ':' + pvec[5]
                        rstr = pvec[1].lstrip('0')
                        if pvec[5] == '3': # LOW BATTERY ALERT
                            self.log.warn('Low battery on id: ' + repr(rstr))
                        ret = tod.tod(pvec[2], index=istr, chan=pvec[0],
                                      refid=rstr)
                        self.log.log(RFID_LOG_LEVEL, msg.strip())
                        if ack:
                            self.port.write(ACKCMD)	# Acknowledge if ok
                        self.__cksumerr = 0
                    else:
                        self.log.info('Invalid checksum: ' 
                                      + repr(tsum) + ' != ' + repr(msum)
                                      + ' :: ' + repr(msg))
                        self.__cksumerr += 1
                        if self.__cksumerr > 3:
                            # assume error on decoder, so acknowledge and
                            # continue with log
                            # NOTE: This path is triggered when serial comms
                            # fail and a tag read happens before a manual trig
                            self.log.error('Erroneous message from decoder.')
                            if ack:
                                self.port.write(ACKCMD)	# Acknowledge if ok
                else:
                    self.log.info('Invalid message: ' + repr(msg))
            elif msg[0] == STATSTART:	# Status message
                data = msg[1:22]
                pvec = data.split()
                if len(pvec) == 5:
                    # Note: this path is not immune to error in stream
                    # but in the case of exception from tod construct
                    # it will be collected by the thread 'main loop'
                    #rstr = ':'.join(pvec[1:])
                    #ret = tod.tod(pvec[0].rstrip('"'), index='', chan='STS',
                                      #refid=rstr)

                    ## ! Choice: leave status logging via RFID
                    ## OR provide an alternate callback structure for
                    ## status... probably better and allows for battery
                    ## alerts
                    #self.log.log(RFID_LOG_LEVEL, msg.strip())
                    self.log.info(msg.strip())
                else:
                    self.log.info('Invalid status: ' + repr(msg))
            else:
                self.log.log(RFID_LOG_LEVEL, repr(msg))
        else:        
            self.log.info('Short message: ' + repr(msg))
        return ret

    def __read(self):
        """Read messages from the decoder until a timeout condition."""
        ch = self.port.read(1)
        while ch != '':
            if ch == NACK:
                # decoder has a passing to report
                self.port.write(REPEATCMD)
            elif ch == LF:
                # Newline ends the current 'message'
                self.__rdbuf += ch	# include trailing newline
                t = self.__parse_message(self.__rdbuf)
                if t is not None:
                    glib.idle_add(self.__cb, t)
                self.__rdbuf = ''
            else:
                self.__rdbuf += ch
            ch = self.port.read(1)

    def __readline(self, l):
        """Try to extract passing information from lines in a file."""
        t = self.__parse_message(l, False)
        if t is not None:
            glib.idle_add(self.__cb, t)

    def run(self):
        """Called via threading.Thread.start()."""
        running = True
        self.log.debug('Starting')
        while running:
            try:
                # Read phase
                if self.port is not None:
                    self.__read()
                    m = self.cqueue.get_nowait()
                else:
                    # when no read port avail, block on read of command queue
                    m = self.cqueue.get()
                self.cqueue.task_done()
                
                # Write phase
                if type(m) is tuple and type(m[0]) is str and m[0] in TCMDS:
                    if m[0] == 'MSG' and not self.error:
                        cmd = m[1] ##+ '\r\n'
                        self.log.debug('Sending rawmsg ' + repr(cmd))
                        self.port.write(cmd)
                    elif m[0] == 'TRIG':
                        if type(m[1]) is tod.tod:
                            self.log.log(RFID_LOG_LEVEL, str(m[1]))
                            glib.idle_add(self.__cb, m[1])
                    elif m[0] == 'SYNC':
                        t = tod.tod('now')
                        self.port.write(self.__set_time_cmd(t))
                        self.log.debug('Set time on decoder: ' + t.meridian())
                    elif m[0] == 'REPL':
                        self.log.info('Replay passings from: ' + repr(m[1]))
                        with open(m[1], 'rb') as f:
                            for l in f:
                                self.__readline(l)
                        self.log.info('Replay complete.')
                    elif m[0] == 'EXIT':
                        self.log.debug('Request to close : ' + str(m[1]))
                        running = False	# This may already be set
                    elif m[0] == 'PORT':
                        if self.port is not None:
                            self.port.close()
                            self.port = None
                        if m[1] is not None and m[1] != '' and m[1] != 'NULL':
                            self.log.debug('Re-Connect port : ' + str(m[1]))
                            self.port = serial.Serial(m[1], THBC_BAUD,
                                                  rtscts=False, timeout=0.2)
                            self.error = False
                        else:
                            self.log.debug('Not connected.')
                            self.error = True
                    else:
                        pass
                else:
                    self.log.warn('Unknown message: ' + repr(m))
            except Queue.Empty:
                pass
            except serial.SerialException as e:
                if self.port is not None:
                    self.port.close()
                    self.port = None
                self.errstr = "Serial port error."
                self.error = True
                self.log.error('Closed serial port: ' + str(type(e)) + str(e))
            except Exception as e:
                self.log.error('Exception: ' + str(type(e)) + str(e))
                #self.errstr = str(e)
                #self.error = True
        self.setcb()	# make sure callback in unrefed
        self.log.info('Exiting')

def printtag(t):
    print(t.refid)
    return False

def showstatus(data=None):
    data.status()
    return True

if __name__ == "__main__":
    import metarace
    import gtk
    import time
    import random
    metarace.init()
    t = thbc(DEFPORT)
    lh = logging.StreamHandler()
    lh.setLevel(logging.DEBUG)
    lh.setFormatter(logging.Formatter(
                      "%(asctime)s %(levelname)s:%(name)s: %(message)s"))
    t.log.addHandler(lh)
    try:
        t.start()
        t.sane()
        time.sleep(1.0)
        t.start_session()
        t.sync()
        t.setcb(printtag)
        #glib.timeout_add_seconds(2, showstatus, t)
        gtk.main()
    except:
        t.stop_session()
        t.wait()
        t.exit('Exception')
        t.join()
        raise
