
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

## TODO: expand subprocess interaction to deliver more logging back
##       to metarace calling process

"""rsync mirror in a thread

This module provides a thread object that runs an rsync mirror
operation in a separate thread. Thread completion is flagged with
a callback operation.

Executes rsync over ssh:

  rsync -av -e ssh --rsync-path=bin/rsync [srcdir] [host]:[path][dstdir]

Create a new mirror object with the required parameters then call
the start method to spawn the rsync process in a separate thread.

Example:

   mo = rsync.mirror(complete, 'export', 'test', 'rsync')
   mo.start()
   
Will mirror the contents of 'export' to the remote server in 'html/site/test'
using the command 'rsync'. Once complete, the function 'complete' will be
queued in the gtk mainloop.

mirror inherits from threading.thread, so all the normal thread
control methods apply.


"""

import threading
import subprocess
import logging
import glib
import os

RSYNC_HOST = u'metarace'	# user/ip/key should be configured in .ssh
RSYNC_PATH = u'html/site'	# default remote path root
RSYNC_CMD = u'rsync'

class mirror(threading.Thread):
    """Mirror thread object class."""
    def __init__(self, callback=None, localpath=u'.',
                       remotepath=None, mirrorcmd=None):
        """Construct mirror thread object.

        """
        threading.Thread.__init__(self) 
        self.name = u'mirror'
        self.log = logging.getLogger(self.name)
        self.log.setLevel(logging.DEBUG)
        self.__cb = None
        if callback is not None:
            self.__cb = callback
        self.__localpath = localpath
        self.__remotepath = u''
        if remotepath is not None:
            self.__remotepath = remotepath
        self.__mirrorcmd = RSYNC_CMD
        if mirrorcmd:
            self.__mirrorcmd = mirrorcmd

    def set_cb(self, func=None):
        """Set or clear the event callback."""
        # if func is not callable, gtk mainloop will catch the error
        if func is not None:
            self.__cb = func
        else:
            self.__cb = None

    def run(self):
        """Called via threading.Thread.start()."""
        running = True
        self.log.debug(u'Starting')
        ret = None
        dstpath = RSYNC_PATH
        if self.__remotepath:
            dstpath = os.path.join(dstpath, self.__remotepath)
        try:
            ret = subprocess.check_call([self.__mirrorcmd,
                                           u'-av', u'-e', u'ssh',
                                           u'--rsync-path=bin/rsync', 
                                           self.__localpath,
                                           RSYNC_HOST + u':' + dstpath])
            # this should be closer to subprocess to lod stderr and stdout
            # from rsync, for logging in metarace
        except Exception as e:
            self.log.error(u'Error: ' + unicode(type(e)) + unicode(e))
        if self.__cb:
            glib.idle_add(self.__cb, ret)
        self.log.info(u'Complete: returned with: ' + repr(ret))

if __name__ == "__main__":
    import metarace
    def cb(code):
        print(u'sync returned: ' + repr(code))

    metarace.init()
    lh = logging.StreamHandler()
    lh.setLevel(logging.DEBUG)
    lh.setFormatter(logging.Formatter(
                      "%(asctime)s %(levelname)s:%(name)s: %(message)s"))
    m = mirror(cb, u'testing')
    m.log.addHandler(lh)
    m.start()
    m.join()
