
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tag Heuer/Chronelec Decoder Interface

This module provides a thread object which interfaces with a
Tag Heuer/Chronelec Elite Decoder over serial or UDP.


  Messages are returned as TOD objects:

	index: 		power:batt	eg 99:0 or '' for trig/sts
	chan:  		BOX	rfid on box channel
			STA	rfid on sta channel
			MAN	manual trigger
			STS	status, refid gets the noise/level info
	timeval:	time of day of passing/trig/status
	refid:		transponder id with leading zeros stripped


  Sent to mainloop via glib.idle_add of the provided callback


Tag Heuer Protime Elite Decoder / Tag Heuer by Chronelec - V3 Protocol
----------------------------------------------------------------------

On transponder read, cell or manual trigger, decoder sends PASSING
message.  Host must reply with ACK to get next passing. If no
acknowledge sent, passing is repeated periodically.

Commands
--------

Get Status:
	Request running time, noise and level status from decoder
	command:	ESC + 0x05
	response:	[STATUS]

Start:
	Start decoder
	command:	ESC + 0x07
	response:	[DEPART] or none if decoder already started

Set Config:
	Update decoder configuration
	command:	ESC + 0x08 + 0x08 + [CONFIG] + [CRC] + '>'
	response:	none

Set IP Config:
	Update decoder IP configuration. Note: Decoder must be stopped
	before issuing this command (why?)
	command:	ESC + 0x09 + 0x09 + [IPCONFIG] + [CRC] + '>'
	response:	XPORT specific (TBC)

Get Config:
	Fetch current decoder configuration & identification
	command:	ESC + 0x10
	response:	[DECODERCONF]

Acknowledge:
	Acknowledge last passing sent by decoder/flag ready for next passing
	command:	ESC + 0x11
	response:	none or [PASSING]

Repeat:
	Repeat first unacknowledged passing, else last acknowledged passing
	command:	ESC + 0x12
	response:	[PASSING]

Stop:
	Stop decoder
	command:	ESC + 0x13 + '\'
	response:	[STOP] (even if already stopped)

Set Time:
	Update decoder time of day - also sets running time if decoder
	started and config option "Running time to time of decoder" set
	command:	ESC + 0x48 + [SETTIME] + 't'
	response:	none

Set Date:
	Update decoder date
	command:	ESC + 0x0a + 0x0a + [SETDATE] + [CRC] + '>'
	response:	none

Set STA Level:
	Set detection level on STA channel
	command:	ESC + 0x1e + [SETLEVEL]
	response:	none

Set BOX Level:
	Set Detection level on BOX channel
	command:	ESC + 0x1f + [SETLEVEL]
	response:	none

Stat BXX:
	Request status on remote decoder with id specified in B
	command:	ESC + 0x49 + [B]
	response:	(TBC)

BXX Level:
	Increment all detection levels by 0x10 (Note 2)
	command:	ESC + 0x4e + 0x2b
	response:	none
	

Messages
--------

PASSING:	'<' + ' ' + CHAN + ' ' + REFID + ' ' + PASSTIME + ' ' 
                           + POWER + ' ' + PASSCOUNT + ' ' 
                           + BATTERY + ' ' + CHARSUM + '>' + [NEWLINE]

CHAN:		'MAN'|'BOX'|'STA'...
REFID:		'000000'->'999999'  six digits, left zero pad 
PASSTIME:	'hh:mm'ss"dcm'	left zero pad
POWER:		'00'->'99'	passing power
COUNT:		'00'->'01'	count of times in loop (question?)
BATTERY:	'0'|'1'|'2'|'3'	0/high -> 3/low
CHARSUM:	'0000'->'8192'	decimal sum of bytes from offset 1-32

STATUS:		'[' + 'hh:mm'ss"' + ' ' + 'NS' + ' ' + 'NB'
                                  + ' ' + 'LS' + ' ' + 'LB' + ']' + [NEWLINE]
                        hh: hour eg 03
                        mm: minute eg 59
                        ss: second eg 31
			Noise: NS (STA) and NB (BOX) '00' -> '99'
			Levels: LS (STA) and LB (BOX) '00' -> '99'
			Note: time reported is running time, and will
			be 00:00'00" when decoder is stopped.

DEPART:		'DEPART_' + [DATESTR] + '__' + [TODSTR] + [NEWLINE]

STOP:		'STOP_' + [DATESTR] + '__' + [TODSTR] + [NEWLINE]

DATESTR:	'YYYY-MM-DD'
			YYYY: year eg 2012
			MM: month eg 02
			DD: day of month eg 12

TODSTR:		'hh:mm:ss'
			hh: hour eg 03
			mm: minute eg 59
			ss: second eg 31

ESC:		0x1b
NACK:		0x07
CRC:		CRC-16/MCRF4XX on bytes following ESC (Note 1)
B:		'1'|'2'|'3'...'7'	 (0x30 + id)
SETLEVEL:	level as two ascii chars eg: 45 => '4' + '5' or 0x34 + 0x35
SETTIME:	h + m + s	
			eg:	21h03:45	=>	0x15 0x03 0x2d
SETDATE:	D + M + Y	(Y == year - 2000)
			eg:	23/05/12	=>	0x17 0x05 0x0c
DECODERCONF:	'+' + '+' + '+' + [CONFIG] + [LEVELS] + [IPCONFIG]
				+ [IDENT] + [VERSION] + '>' + [NEWLINE]

NEWLINE:	CR + LF
CR:		0x0d
LF:		0x0a

LEVELS (2 bytes): 
0	STA level 30 => 0x30
1	BOX level "

IDENT (4 bytes):
0-3	'0129' => 0x00 + 0x01 + 0x02 + 0x09

VERSION (4 bytes): (TBC)
0	0x13	?
1-3	'201'	version string?

CONFIG (27 bytes):
offset	option			values
0	Time of day 		0x00=>runtime, 0x01=>timeofday
1	GPS Sync		0x00=>off, 0x01=>on
2	Time Zone Hour		0x10=>10h, 0x09=>9h
3	Time Zone Min		0x00=>:00, 0x30=>:30
4	Distant 232/485 select	0x00=>232, 0x01=>485	(check)
5	Distant Fibre Optic	0x00=>no, 0x01=>yes
6	Print pass on serial	0x00=>no, 0x01=>yes
7	Detect maximum		0x00=>no, 0x01=>yes
8	Protocol		0x00=>Cv3,0x01=>AMB,0x02=>Cv2
9	Generate sync		0x00=>no, 0x01=>yes
10	Sync interval min	0x01=>1min, ... , 0x99=>99min
11	Sync ToD on CELL	0x00=>off, 0x01=>on
12	Sync Time hours		0x00=>0h, ... , 0x23=>23h (Question Function?)
13	Sync Time min		0x00=>:00, ... , 0x59=>:59
14	Language		0x00=>Eng, 0x01=>Fr
15,16	STA Tone		0x12,0x34=>1234Hz 0x00,0x00=>no tone
17,18	BOX Tone		"
19,20	MAN Tone		"
21,22	CEL Tone		"
23,24	BXX Tone		"
25,26	Spare?			0x20 + 0x20

IPCONFIG (16 bytes):

0-3	IP Address, net order eg: 192.168.95.252 => 0xc0 + 0xa8 + 0x5f + 0xfc
4-7	Netmask, "
8-11	Gateway, "
12-15	Remote host, "

NOTES:

	1.	CRC is computed with the following parameters:
		width:		crc-16
		poly:		0x1021
		init:		0xffff
		reflect-in:	yes
		reflect-out:	yes
		xor-out:	0x0000
		alias:		CRC-16/MCRF4XX

	2.	Detection level appears to be stored or manipulated
		as byte, but displayed as decimal equivalent of hex string.
		When incrementing with command BXX Level, STA is wrapped to
		zero when STA level is > 0x99. BOX level will increment 
		> 0x90 all the way to 0xff as long as STA is < 0xa0. 
		Side effects of this have not been tested.

"""
import threading
import Queue
import serial
import socket
import logging
import glib

from metarace import tod
from metarace import crc_algorithms

# System default timy serial port
DEFPORT = '/dev/ttyUSB0'

# Serial baudrate
THBC_BAUD = 19200

# UDP Port number for ethernet connection
THBC_UDP_PORT = 2008

# Photofinish threshold - ~20cm based on tests at DISC,
# howevever, activator period is ~20ms... more testing required!
THBCPHOTOTHRESH = tod.tod('0.02')

# THbC protocol messages
ESCAPE = chr(0x1b)
HELOCMD = 'MR1'
STOPCMD = ESCAPE + chr(0x13) + chr(0x5c)
REPEATCMD = ESCAPE + chr(0x12)
ACKCMD = ESCAPE + chr(0x11)

STATCMD = ESCAPE + chr(0x05)	# fetch status
CHKCMD = ESCAPE + chr(0x06)	# UNKNOWN
STARTCMD = ESCAPE + chr(0x07)	# start decoder
SETCMD = ESCAPE + chr(0x08)	# set configuration
IPCMD = ESCAPE + chr(0x09)	# set IP configuration
QUECMD = ESCAPE + chr(0x10)	# fetch configuration

STALVL = ESCAPE + chr(0x1e)
BOXLVL = ESCAPE + chr(0x1f)

NACK = chr(0x07)
LF = chr(0x0a)
SETTIME = ESCAPE + chr(0x48)
STATSTART = '['
PASSSTART = '<'

# thread queue commands -> private to timy thread
TCMDS = ('EXIT', 'PORT', 'MSG', 'TRIG', 'SYNC', 'REPL')

RFID_LOG_LEVEL = 16     # lower so not in status and on-screen logger.
logging.addLevelName(RFID_LOG_LEVEL, 'RFID')

# decoder config consts
CONFIG_TOD = 0
CONFIG_GPS = 1
CONFIG_TZ = 2
CONFIG_485 = 4
CONFIG_FIBRE = 5
CONFIG_PRINT = 6
CONFIG_MAX = 7
CONFIG_PROT = 8
CONFIG_PULSE = 9
CONFIG_PULSEINT = 10
CONFIG_CELLSYNC = 11
CONFIG_CELLTOD = 12
CONFIG_LANG = 14
CONFIG_FLAGS = {
	CONFIG_TOD: u'Time of Day',
	CONFIG_GPS: u'GPS Sync',
	CONFIG_TZ: u'Timezone',
	CONFIG_485: u'Distant rs485',
	CONFIG_FIBRE: u'Distant Fibre',
	CONFIG_PRINT: u'Serial Print',
	CONFIG_MAX: u'Detect Max',
	CONFIG_PROT: u'Protocol',
	CONFIG_PULSE: u'Sync Pulse',
	CONFIG_PULSEINT: u'Sync Interval',
	CONFIG_CELLSYNC: u'CELL Sync',
	CONFIG_CELLTOD: u'CELL Sync ToD',
	CONFIG_LANG: u'Lang Fr'
}

# Initialise crc algorithm for CRC-16/MCRF4XX
crc_alg = crc_algorithms.Crc(width=16, poly=0x1021,
                              reflect_in=True, xor_in=0xffff,
                              reflect_out=True, xor_out=0x0000)

# Cheap and nasty byte addition
adder = lambda sum, ch: sum + ord(ch)

def thbc_sum(msgstr=''):
    """Return sum of character values as decimal string."""
    return '{0:04d}'.format(reduce(adder, msgstr, 0))

def thbc_crc(msgstr='123456789'):
    """Return CRC-16/MCRF4XX on input string."""
    return crc_alg.table_driven(msgstr)

class dgram(object):
    """UDP port object."""
    def __init__(self, host, port):
        self.host = host
        self.port = port
        self.s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.s.settimeout(0.2)
        self.s.bind(('', self.port))
        self.buf = ''	# these are not decoded!

    def read(self, sz=1):
        ret = None
        # buf empty?
        if len(self.buf) == 0:
            nb, addr = self.s.recvfrom(4096)	# timeout raises exception
            if addr[0] == self.host:
                self.buf += nb
        if len(self.buf) > 0:
            ret = self.buf[0]
            self.buf = self.buf[1:]
        return ret

    def write(self, buf=''):
        return self.s.sendto(buf, (self.host, self.port))

    def close(self):
        return self.s.close()

class thbc(threading.Thread):
    """Tag Heuer Elite thread object class."""
    def __init__(self, port=None, name='thbc'):
        """Construct thread object.

        Named parameters:

          port -- serial port
          name -- text identifier for use in log messages

        """
        threading.Thread.__init__(self) 
        self.name = name

        self.port = None
        self.error = False
        self.errstr = ''
        self.unitno = u''		# fetched on reload
        self.decoderconfig = {}		# fetched on reload
        self.cqueue = Queue.Queue()	# command queue
        self.log = logging.getLogger(self.name)
        self.log.setLevel(logging.DEBUG)
        self.__cksumerr = 0
        self.__rdbuf = ''
        self.setcb()
        if port is not None:
            self.setport(port)

    def photothresh(self):
        """Return the relevant photo finish threshold."""
        return THBCPHOTOTHRESH		# allow override perhaps?

    def __defcallback(self, evt=None):
        """Default callback is a tod log entry."""
        self.log.debug(str(evt))
        return False

    def getcb(self):
        """Return the current callback function."""
        return self.__cb

    def setcb(self, func=None):
        """Set or clear the event callback."""
        # if func is not callable, gtk mainloop will catch the error
        if func is not None:
            self.__cb = func
        else:
            self.__cb = self.__defcallback

    def write(self, msg=None):
        """Queue a raw command string to attached decoder."""
        self.cqueue.put_nowait(('MSG', msg))

    def exit(self, msg=None):
        """Request thread termination."""
        self.running = False
        self.cqueue.put_nowait(('EXIT', msg)) # "Prod" command thread

    def setport(self, device=None):
        """Request (re)opening port as specified.

        Device may be a port number or a device identifier string.
        For information on port numbers or strings, see the
        documentation for serial.Serial().

        Call setport with no argument, None, or an empty string
        to close an open port or to run the thread with no
        external device.

        """
        self.cqueue.put_nowait(('PORT', device))

    def sync(self):
        """Roughly synchronise Decoder to host PC clock."""
        self.cqueue.put_nowait(('SYNC', None))

    def sane(self):
        """No other config."""
        pass

    def trig(self, timeval='now', index='FAKE', chan='MAN', refid='0'):
        """Create a fake timing event.

	   Generate a new tod object to mimic a message as requested
           and pipe it to the command thread. Default tod is the
	   'now' time in the calling thread.

        """
        t = tod.tod(timeval, index, chan, refid.lstrip('0'))
        self.cqueue.put_nowait(('TRIG', t))

    def start_session(self):
        """Send a depart command to decoder."""
        self.write(STARTCMD)

    def stop_session(self):
        """Send a stop command to decoder."""
        self.write(STOPCMD)

    def status(self):
        """Request status message from decoder."""
        self.write(STATCMD)

    def get_config(self):
        """Request decoder configuration."""
        self.write(QUECMD)

    def v3_cmd(self, cmdstr=''):
        """Pack and send a v3 command"""
        crc = thbc_crc(cmdstr)
        crcstr = chr(crc>>8) + chr(crc&0xff)
        self.write(ESCAPE + cmdstr + crcstr + '>')

    def set_date(self, timestruct=None):
        """Set the date on the decoder."""
        if timestruct is None:
            timestruct = time.localtime()
        self.log.debug(u'Set date on decoder: '
                         + time.strftime('%Y-%m-%d',timestruct))
        cmd = chr(0x0a) + chr(0x0a)
        cmd += chr(0xff&timestruct[2])		# day
        cmd += chr(0xff&timestruct[1])		# day
        cmd += chr(0xff&(timestruct[0]-2000))	# year, after 2000
        self.v3_cmd(cmd)

    def setlvl(self, box=u'10', sta=u'10'):
        """Set the read level on box and sta channels."""
        # TODO: verify opts
        self.write(BOXLVL + box.encode('ascii', 'ignore')[0:2])
        self.write(STALVL + sta.encode('ascii', 'ignore')[0:2])
        
    def replay(self, filename=''):
        """Read passings from file and process."""
        self.cqueue.put_nowait(('REPL', filename))

    def wait(self):
        """Suspend calling thread until the command queue is empty."""
        self.cqueue.join()

    def __set_time_cmd(self, t):
        """Return a set time command string for the provided time of day."""
        s = int(t.timeval)
        st = chr(s%60)
        mt = chr((s//60)%60)
        ht = chr(s//3600)
        return SETTIME + ht + mt + st + chr(0x74)

    def __parse_config(self, msg):
        # decoder configuration message.
        self.decoderconfig = {}
        showflags = []
        for flag in sorted(CONFIG_FLAGS):	# import all
            if flag in [ CONFIG_TZ, CONFIG_PROT,
                         CONFIG_PULSEINT, CONFIG_CELLTOD]:
                pass	# TODO: Skip it for now
            else:
                if ord(msg[flag]):	# Time of day
                    self.decoderconfig[flag] = True
                    showflags.append(CONFIG_FLAGS[flag])
        self.unitno = u''
        for c in msg[43:47]:
            self.unitno += unichr(ord(c)+ord('0'))
        stalvl = hex(ord(msg[25]))	# ? question this
        boxlvl = hex(ord(msg[26]))

        self.log.info(u'Decoder ID: ' + self.unitno 
                               + '\tFlags: ' + u','.join(showflags))
        self.log.debug(u'Decoder Levels: STA=' + stalvl
                               + u', BOX=' + boxlvl)

    def __parse_message(self, msg, ack=True):
        """Return tod object from timing msg or None."""
        ret = None
        if len(msg) > 4:
            if msg[0] == PASSSTART:	# RFID message
                idx = msg.find('>')
                if idx == 37:		# Valid length
                    data = msg[1:33]
                    msum = msg[33:37]
                    tsum = thbc_sum(data)
                    if tsum == msum:	# Valid 'sum'
                        pvec = data.split()
                        istr = pvec[3] + ':' + pvec[5]
                        rstr = pvec[1].lstrip('0')
                        if pvec[5] == '3': # LOW BATTERY ALERT
                            self.log.warn('Low battery on id: ' + repr(rstr))
                        ret = tod.tod(pvec[2], index=istr, chan=pvec[0],
                                      refid=rstr)
                        self.log.log(RFID_LOG_LEVEL, msg.strip())
                        if ack:
                            self.port.write(ACKCMD)	# Acknowledge if ok
                        self.__cksumerr = 0
                    else:
                        self.log.info('Invalid checksum: ' 
                                      + repr(tsum) + ' != ' + repr(msum)
                                      + ' :: ' + repr(msg))
                        self.__cksumerr += 1
                        if self.__cksumerr > 3:
                            # assume error on decoder, so acknowledge and
                            # continue with log
                            # NOTE: This path is triggered when serial comms
                            # fail and a tag read happens before a manual trig
                            self.log.error('Erroneous message from decoder.')
                            if ack:
                                self.port.write(ACKCMD)	# Acknowledge if ok
                else:
                    self.log.info('Invalid message: ' + repr(msg))
            elif msg[0] == STATSTART:	# Status message
                data = msg[1:22]
                pvec = data.split()
                if len(pvec) == 5:
                    # Note: this path is not immune to error in stream
                    # but in the case of exception from tod construct
                    # it will be collected by the thread 'main loop'
                    #rstr = ':'.join(pvec[1:])
                    #ret = tod.tod(pvec[0].rstrip('"'), index='', chan='STS',
                                      #refid=rstr)

                    ## ! Choice: leave status logging via RFID
                    ## OR provide an alternate callback structure for
                    ## status... probably better and allows for battery
                    ## alerts
                    #self.log.log(RFID_LOG_LEVEL, msg.strip())
                    self.log.info(msg.strip())
                else:
                    self.log.info('Invalid status: ' + repr(msg))
            elif '+++' == msg[0:3] and len(msg) > 53:
                self.__parse_config(msg[3:])
            else:
                self.log.log(RFID_LOG_LEVEL, repr(msg))
        else:        
            self.log.info('Short message: ' + repr(msg))
        return ret

    def __read(self):
        """Read messages from the decoder until a timeout condition."""
        ch = self.port.read(1)
        while ch != '':
            if ch == NACK:
                # decoder has a passing to report
                self.port.write(REPEATCMD)
            elif ch == LF:
                # Newline ends the current 'message'
                self.__rdbuf += ch	# include trailing newline
                t = self.__parse_message(self.__rdbuf.lstrip('\0'))
                if t is not None:
                    glib.idle_add(self.__cb, t)
                self.__rdbuf = ''
            else:
                self.__rdbuf += ch
            ch = self.port.read(1)

    def __readline(self, l):
        """Try to extract passing information from lines in a file."""
        t = self.__parse_message(l, False)
        if t is not None:
            glib.idle_add(self.__cb, t)

    def __mkport(self, pstr):
        """Try to guess port type."""
        ret = None
        if u'/' not in pstr and u'.' in pstr:	# hack
            self.log.debug(u'Attempting UDP connection from ' + repr(pstr))
            ret = dgram(pstr, THBC_UDP_PORT)
        else:
            # assume device file
            ret = serial.Serial(pstr, THBC_BAUD,
                                      rtscts=False, timeout=0.2)
        return ret

    def run(self):
        """Called via threading.Thread.start()."""
        running = True
        self.log.debug('Starting')
        while running:
            try:
                # Read phase
                if self.port is not None:
                    try:
                        self.__read()
                    except socket.timeout:
                        pass
                    m = self.cqueue.get_nowait()
                else:
                    # when no read port avail, block on read of command queue
                    m = self.cqueue.get()
                self.cqueue.task_done()
                
                # Write phase
                if type(m) is tuple and type(m[0]) is str and m[0] in TCMDS:
                    if m[0] == 'MSG' and self.port and not self.error:
                        cmd = m[1] ##+ '\r\n'
                        self.log.debug('Sending rawmsg ' + repr(cmd))
                        self.port.write(cmd)
                    elif m[0] == 'TRIG':
                        if type(m[1]) is tod.tod:
                            self.log.log(RFID_LOG_LEVEL, str(m[1]))
                            glib.idle_add(self.__cb, m[1])
                    elif m[0] == 'SYNC':
                        t = tod.tod('now')
                        # DANGER: Busy wait may interfere with caller
                        while t-t.truncate(0) > tod.tod('0.02'):
                            t = tod.tod('now')
                        self.port.write(self.__set_time_cmd(t))
                        self.log.debug('Set time on decoder: ' + t.meridian())
                    elif m[0] == 'REPL':
                        self.log.info('Replay passings from: ' + repr(m[1]))
                        with open(m[1], 'rb') as f:
                            for l in f:
                                self.__readline(l)
                        self.log.info('Replay complete.')
                    elif m[0] == 'EXIT':
                        self.log.debug('Request to close : ' + str(m[1]))
                        running = False	# This may already be set
                    elif m[0] == 'PORT':
                        if self.port is not None:
                            self.port.close()
                            self.port = None
                        if m[1] is not None and m[1] != '' and m[1] != 'NULL':
                            self.log.debug('Re-Connect port : ' + str(m[1]))
                            self.port = self.__mkport(m[1])
                            self.error = False
                            self.unitno = u''
                            self.get_config()	# re-identify decoder
                        else:
                            self.log.debug('Not connected.')
                            self.error = True
                    else:
                        pass
                else:
                    self.log.warn(u'Unknown message: ' + repr(m))
            except Queue.Empty:
                pass
            except (serial.SerialException, socket.error) as e:
                if self.port is not None:
                    self.port.close()
                    self.port = None
                self.errstr = 'Port error.'
                self.error = True
                self.log.error('Closed port: ' + str(type(e)) + str(e))
            except Exception as e:
                self.log.error('Exception: ' + str(type(e)) + str(e))
                #self.errstr = str(e)
                #self.error = True
        self.setcb()	# make sure callback is unrefed
        self.log.info('Exiting')

def printtag(t):
    print(t.refid + u'\t' + t.rawtime(2))
    return False

def showstatus(data=None):
    data.status()
    return True

if __name__ == "__main__":
    import metarace
    import gtk
    import time
    import random
    metarace.init()
    t = thbc()
    lh = logging.StreamHandler()
    lh.setLevel(logging.DEBUG)
    lh.setFormatter(logging.Formatter(
                      "%(asctime)s %(levelname)s:%(name)s: %(message)s"))
    t.log.addHandler(lh)
    try:
        t.start()
        t.setport(u'192.168.95.252')
        t.sane()
        t.set_date()
        t.sync()
        t.setcb(printtag)
        glib.timeout_add_seconds(5, showstatus, t)
        t.start_session()
        gtk.main()
    except:
        t.stop_session()
        t.wait()
        t.exit('Exception')
        t.join()
        raise
