
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""UNT4 Protocol Wrapper.

Pack and unpack UNT4 protocol messages for Omega Galactica DHI.

Note: Normal program code will not use this library directly
      except to access the pre-defined overlay messages. All
      scoreboard communication should go via the sender or
      scbwin classes.

Messages are manipulated in unicode - it is up to the transport
layer to encode/decode as appropriate.

"""

# mode 1 constants
SOH = 0x01
STX = 0x02
EOT = 0x04
HOME= 0x08
CR  = 0x0d
LF  = 0x0a
ERL = 0x0b
ERP = 0x0c
DLE = 0x10
DC2 = 0x12
DC3 = 0x13
DC4 = 0x14
FS = 0x1c
GS = 0x1d
RS = 0x1e
US = 0x1f


ENCMAP = {
   unichr(SOH):'<O>',
   unichr(STX):'<T>',
   unichr(EOT):'<E>',
   unichr(HOME):'<H>',
   unichr(CR):'<R>',
   unichr(LF):'<A>',
   unichr(ERL):'<L>',
   unichr(ERP):'<P>',
   unichr(DLE):'<D>',
   unichr(DC2):'<2>',
   unichr(DC3):'<3>',
   unichr(DC4):'<4>',
   unichr(FS):'<F>',
   unichr(GS):'<G>',
   unichr(RS):'<R>',
   unichr(US):'<U>'}
  
def encode(ubuf=u''):
    """Encode the unt4 buffer for use with IRC."""
    for key in ENCMAP:
        ubuf = ubuf.replace(key, ENCMAP[key])
    return ubuf

def decode(ircbuf=u''):
    """Decode the irc buffer to unt4msg pack."""
    ircbuf = ircbuf.replace(u'<00>',u'')
    for key in ENCMAP:
        ircbuf = ircbuf.replace(ENCMAP[key], key)
    return ircbuf

# UNT4 Packet class
class unt4(object):
    """UNT4 Packet Class."""
    def __init__(self, unt4str=None, 
                   prefix=None, header=u'', 
                   erp=False, erl=False, 
                   xx=None, yy=None, text=u''):
        """Constructor.

        Parameters:

          unt4str -- packed unt4 string, overrides other params
          prefix -- prefix byte <DC2>, <DC3>, etc
          header -- header string eg 'R_F$'
          erp -- true for general clearing <ERP>
          erl -- true for <ERL>
          xx -- packet's column offset 0-99
          yy -- packet's row offset 0-99
          text -- packet content string

        """
        self.prefix = prefix    # <DC2>, <DC3>, etc
        self.header = header    # ident text string eg 'R_F$'
        self.erp = erp          # true for general clearing <ERP>
        self.erl = erl          # true for <ERL>
        self.xx = xx            # input column 0-99
        self.yy = yy            # input row 0-99
        self.text = text        # text string - assume encoded already?
        if unt4str is not None:
            self.unpack(unt4str)

    def unpack(self, unt4str=u''):
        """Unpack the UNT4 string into this object."""
        if len(unt4str) > 2 and unt4str[0] is unichr(SOH) \
                            and unt4str[-1] is unichr(EOT):
            self.prefix = None
            newhead = u''
            newtext = u''
            self.erl = False
            self.erp = False
            head = True		# All text before STX is considered header
            stx = False
            dle = False
            dlebuf = u''
            i = 1
            packlen = len(unt4str) - 1
            while i < packlen:
                och = ord(unt4str[i])
                if och == STX:
                    stx = True
                    head = False
                elif och == DLE and stx:
                    dle = True
                elif dle:
                    dlebuf += unt4str[i]
                    if len(dlebuf) == 4:
                        dle = False
                elif head:
                    if och in (DC2, DC3, DC4):
                        self.prefix = och   # assume pfx before head text
                    else:
                        newhead += unt4str[i]
                elif stx:
                    if och == ERL:
                        self.erl = True
                    elif och == ERP:
                        self.erp = True
                    else:
                        newtext += unt4str[i]
                i += 1
            if len(dlebuf) == 4:
                self.xx = int(dlebuf[:2])
                self.yy = int(dlebuf[2:])
            self.header = newhead
            self.text = newtext

    def pack(self):
        """Return UNT4 string packet."""
        head = u''
        text = u''
        if self.erp:	# overrides any other message content
            text = unichr(STX) + unichr(ERP)
        else:
            head = self.header
            if self.prefix is not None:
                head = unichr(self.prefix) + head
            if self.xx is not None and self.yy is not None:
                text += unichr(DLE) + u'{0:02d}{1:02d}'.format(self.xx, self.yy)
            if self.text:
                text += self.text
            if self.erl:
                text += unichr(ERL)
            if len(text) > 0:
                text = unichr(STX) + text
        return unichr(SOH) + head + text + unichr(EOT)
 
# 'Constant' messages
GENERAL_CLEARING = unt4(erp=True)
OVERLAY_ON = unt4(header=u'OVERLAY ON')
OVERLAY_OFF = unt4(header=u'OVERLAY OFF')
OVERLAY_1LINE = unt4(header=u'OVERLAY 00')
OVERLAY_2LINE = unt4(header=u'OVERLAY 01')
OVERLAY_3LINE = unt4(header=u'OVERLAY 02')
OVERLAY_4LINE = unt4(header=u'OVERLAY 03')
OVERLAY_R1P4 = unt4(header=u'OVERLAY 04')
OVERLAY_R2P2 = unt4(header=u'OVERLAY 05')
OVERLAY_R2P3 = unt4(header=u'OVERLAY 06')
OVERLAY_R2P4 = unt4(header=u'OVERLAY 07')
OVERLAY_T1P4 = unt4(header=u'OVERLAY 08')
OVERLAY_T1P5 = unt4(header=u'OVERLAY 09')
OVERLAY_24X5 = unt4(header=u'OVERLAY 10')
OVERLAY_24X6 = unt4(header=u'OVERLAY 11')
OVERLAY_CLOCK = unt4(header=u'OVERLAY 12')
OVERLAY_IMAGE = unt4(header=u'OVERLAY 13')
OVERLAY_BLANK = unt4(header=u'OVERLAY 14')

