
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#                                                                               
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from xml.sax.saxutils import escape, quoteattr
import sys

"""HTML output lib.

Cheap and nasty functional primitives for HTML output. Each primitive
outputs a single string. No checking is performed on the structure of
the document produced. All elements take a named parameter 'attrs'
which is a dict of key/value attributes. Non-empty elements take a
parameter 'clist' which is a list of other constructed elements.

Example for an empty element:

    hr(attrs={'id':'thehr'})

returns the string: '<hr id="thehr" />'

Example for an element with content:

    a(['link text'], attrs={'href':'#target'})

returns the string: '<a href="#target">\nlink text\n</a>'

Example paragraph:

    p(['Check the',
       a(['website'], attrs={'href':'#website'}),
       'for more.'])

Returns: '<p>\nCheck the\n<a href="#website">\nwebsite\n</a>\nfor more.\n</p>'

"""

def html(headlist=[], bodylist=[]):
    """Emit HTML document."""
    return u'\n'.join([
      preamble(),
      u'<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">',
      head(headlist),
      body(bodylist),
      u'</html>'])

def preamble():
    """Emit XHTML preamble."""
    return u"""<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE html 
     PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN"
     "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">"""

def attrlist(attrs):
    """Convert attr dict into properly escaped attrlist."""
    alist = []
    for a in attrs:
        alist.append(a.lower() + u'=' + quoteattr(attrs[a]))
    if len(alist) > 0:
        return u' ' + u' '.join(alist) 
    else:
        return u''

def escapetext(text=u''):
    """Return escaped copy of text."""
    return escape(text, {u'"':u'&quot;'})

def comment(commentstr=u''):
    """Insert comment."""
    return u'<!-- ' + commentstr.replace(u'--',u'') + u' -->'

# Declare all the empty types
for empty in [u'meta', u'link', u'base', u'param',
              u'hr', u'br', u'img', u'input', u'col']:
    def emptyfunc(attrs={}, elem=empty):
        return u'<' + elem + attrlist(attrs) + u' />'
    setattr(sys.modules[__name__], empty, emptyfunc) 

# Declare all the non-empties
for nonempty in [u'head', u'body', u'title', u'div', u'style', u'script',
                 u'object', u'p', u'h1', u'h2', u'h3', u'h4', u'h5', u'h6',
                 u'ul', u'ol', u'li', u'dl', u'dt', u'dd', u'address',
                 u'pre', u'blockquote', u'a', u'span', u'em', u'strong',
                 u'dfn', u'code', u'samp', u'kbd', u'var', u'cite',
                 u'abbr', u'acronym', u'q', u'sub', u'sup', u'tt',
                 u'i', u'big', u'small', u'label', u'form', u'select',
                 u'optgroup', u'option', u'textarea', u'fieldset',
                 u'legend', u'button', u'table', u'caption',
                 u'thead', u'tfoot', u'tbody', u'colgroup',
                 u'tr', u'th', u'td', ]:
    def nonemptyfunc(clist=[], attrs={}, elem=nonempty):
        if type(clist) in [str, unicode]:	# ERROR 2->3
            clist = [clist]
        return(u'<' + elem + attrlist(attrs) + u'>'
                + u'\n'.join(clist) + u'</' + elem + u'>')
    setattr(sys.modules[__name__], nonempty, nonemptyfunc) 


# test run
if __name__ == "__main__":
    # NOTE: all terminal strings require escaping before being sent to
    #       htlib. Unescaped strings below are technically in error
    print(
        html([title(u'Test Title'),
            meta({u'name':u'Author',
      u'content':u'Voyt\u011bch \u0160afa\u0159\u00edk \u0026 John Smith'}),
            link({u'type':u'text/css', u'href':u'style.css',
                     u'rel':u'stylesheet'}),
            meta({u'http-equiv':u'Content-Style-Type',
                    u'content':u'text/css'})],
             [h1(u'Document Heading'),
              comment(u'inline comment'),
              p(em(escapetext(u'Text \u0026 "under\u02ee heading.'))),
              h2(u'Subheading'),
              p([u'The paragraph of subheading\u2026 and with some',
                 tt(u'0h12:34.56'),
                 u'time text.'])]).encode('utf-8','replace')
    )
