
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#                                                                               
# This program is free software: you can redistribute it and/or modify          
# it under the terms of the GNU General Public License as published by          
# the Free Software Foundation, either version 3 of the License, or             
# (at your option) any later version.                                           
#                                                                               
# This program is distributed in the hope that it will be useful,               
# but WITHOUT ANY WARRANTY; without even the implied warranty of                
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 
# GNU General Public License for more details.                                  
#                                                                               
# You should have received a copy of the GNU General Public License             
# along with this program.  If not, see <http://www.gnu.org/licenses/>.         

"""Timing and data handling application for track meets at DISC."""

import pygtk
pygtk.require("2.0")

import gtk
import glib
import pango

import os
import sys
import logging

import metarace

from metarace import jsonconfig
from metarace import tod
from metarace import riderdb
from metarace import eventdb
from metarace import scbwin
from metarace import sender
from metarace import uscbsrv
from metarace import dbexport
from metarace import rsync
from metarace import timy
from metarace import gemini
from metarace import unt4
from metarace import strops
from metarace import loghandler
from metarace import race
from metarace import ps
from metarace import ittt
from metarace import f200
from metarace import flap
from metarace import sprnd
from metarace import omnium
from metarace import classification
from metarace import printing

LOGFILE = u'event.log'
LOGHANDLER_LEVEL = logging.DEBUG
DEFANNOUNCE_PORT = u''
CONFIGFILE = u'config.json'
TRACKMEET_ID = u'trackmeet_1.7'	# configuration versioning
EXPORTPATH = u'export'

def mkrace(meet, event, ui=True):
    """Return a race object of the correct type."""
    ret = None
    etype = event[u'type']
    if etype in [u'indiv tt',
                 u'indiv pursuit', u'pursuit race',
                 u'team pursuit', u'team pursuit race']:
        ret = ittt.ittt(meet, event, ui)
    elif etype in [u'points', u'madison']:
        ret = ps.ps(meet, event, ui)
    elif etype in [u'omnium', u'aggregate']:
        ret = omnium.omnium(meet, event, ui)
    elif etype == u'classification':
        ret = classification.classification(meet, event, ui)
    elif etype in [u'flying lap']:
        ret = flap.flap(meet, event, ui)
    elif etype in [u'flying 200']:
        ret = f200.f200(meet, event, ui)
    elif etype == u'sprint round':
        ret = sprnd.sprnd(meet, event, ui)
    else:
        ret = race.race(meet, event, ui)
    return ret

class trackmeet:
    """Track meet application class."""

    ## Meet Menu Callbacks
    def menu_meet_open_cb(self, menuitem, data=None):
        """Open a new meet."""
        self.close_event()

        dlg = gtk.FileChooserDialog(u'Open new track meet', self.window,
            gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER, (gtk.STOCK_CANCEL,
            gtk.RESPONSE_CANCEL, gtk.STOCK_OPEN, gtk.RESPONSE_OK))
        response = dlg.run()
        if response == gtk.RESPONSE_OK:
            self.configpath = dlg.get_filename()
            self.loadconfig()
            self.log.debug(u'Meet data loaded from'
                           + repr(self.configpath) + u'.')
        dlg.destroy()

    def get_event(self, evno, ui=False):
        """Return an event object for the given event number."""
        # NOTE: returned event will need to be destroyed
        ret = None
        eh = self.edb[evno]
        if eh is not None:
            ret = mkrace(self, eh, ui)
        return ret

    def menu_meet_save_cb(self, menuitem, data=None):
        """Save current meet data and open event."""
        self.saveconfig()

    def menu_meet_info_cb(self, menuitem, data=None):
        """Display meet information on scoreboard."""
        self.gemini.clear()
        self.clock.clicked()
        self.announce.gfx_overlay(0, self.graphicscb)

    def menu_meet_properties_cb(self, menuitem, data=None):
        """Edit meet properties."""
        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, u'trackmeet_props.ui'))
        dlg = b.get_object('properties')
        dlg.set_transient_for(self.window)

        # load meet meta
        tent = b.get_object('meet_title_entry')
        tent.set_text(self.titlestr)
        stent = b.get_object('meet_subtitle_entry')
        stent.set_text(self.subtitlestr)
        dent = b.get_object('meet_date_entry')
        dent.set_text(self.datestr)
        lent = b.get_object('meet_loc_entry')
        lent.set_text(self.locstr)
        cent = b.get_object('meet_comm_entry')
        cent.set_text(self.commstr)
        oent = b.get_object('meet_org_entry')
        oent.set_text(self.orgstr)
        lo = b.get_object('meet_logos_entry')
        lo.set_text(self.logos)

        # load data/result opts
        re = b.get_object('data_showevno')
        re.set_active(self.showevno)
        cm = b.get_object('data_clubmode')
        cm.set_active(self.clubmode)
        prov = b.get_object('data_provisional')
        prov.set_active(self.provisional)
        tln = b.get_object('tracklen_total')
        tln.set_value(self.tracklen_n)
        tld = b.get_object('tracklen_laps')
        tldl = b.get_object('tracklen_lap_label')
        tld.connect('value-changed',
                    self.tracklen_laps_value_changed_cb, tldl)
        tld.set_value(self.tracklen_d)

        # scb/timing ports
        spe = b.get_object('scb_port_entry')
        spe.set_text(self.scbport)
        upe = b.get_object('uscb_port_entry')
        upe.set_text(self.annport)
        spb = b.get_object('scb_port_dfl')
        spb.connect('clicked', self.set_default, spe, u'DISC')
        mte = b.get_object('timing_main_entry')
        mte.set_text(self.main_port)
        mtb = b.get_object('timing_main_dfl')
        mtb.connect('clicked', self.set_default, mte, timy.MAINPORT)
        bte = b.get_object('timing_backup_entry')
        bte.set_text(self.backup_port)
        btb = b.get_object('timing_backup_dfl')
        btb.connect('clicked', self.set_default, bte, timy.BACKUPPORT)

        # run dialog
        response = dlg.run()
        if response == 1:	# id 1 set in glade for "Apply"
            self.log.debug(u'Updating meet properties.')

            # update meet meta
            self.titlestr = tent.get_text().decode('utf-8','replace')
            self.subtitlestr = stent.get_text().decode('utf-8','replace')
            self.datestr = dent.get_text().decode('utf-8','replace')
            self.locstr = lent.get_text().decode('utf-8','replace')
            self.commstr = cent.get_text().decode('utf-8','replace')
            self.orgstr = oent.get_text().decode('utf-8','replace')
            self.logos = lo.get_text().decode('utf-8','replace')
            self.set_title()

            self.clubmode = cm.get_active()
            self.showevno = re.get_active()
            self.provisional = prov.get_active()
            self.tracklen_n = tln.get_value_as_int()
            self.tracklen_d = tld.get_value_as_int()
            nport = spe.get_text().decode('utf-8','replace')
            if nport != self.scbport:
                self.scbport = nport
                self.scb.setport(nport)
            nport = upe.get_text().decode('utf-8','replace')
            if nport != self.annport:
                self.annport = nport
                self.announce.set_portstr(self.annport)
            nport = mte.get_text().decode('utf-8','replace')
            if nport != self.main_port:
                self.main_port = nport
                self.main_timer.setport(nport)
            nport = bte.get_text().decode('utf-8','replace')
            if nport != self.backup_port:
                self.backup_port = nport
                self.backup_timer.setport(nport)
            self.log.debug(u'Properties updated.')
        else:
            self.log.debug(u'Edit properties cancelled.')
        dlg.destroy()

    def menu_meet_fullscreen_toggled_cb(self, button, data=None):
        """Update fullscreen window view."""
        if button.get_active():
            self.window.fullscreen()
        else:
            self.window.unfullscreen()

    def tracklen_laps_value_changed_cb(self, spin, lbl):
        """Laps changed in properties callback."""
        if int(spin.get_value()) > 1:
            lbl.set_text(u' laps = ')
        else:
            lbl.set_text(u' lap = ')

    def set_default(self, button, dest, val):
        """Update dest to default value val."""
        dest.set_text(val)

    def menu_meet_quit_cb(self, menuitem, data=None):
        """Quit the track meet application."""
        self.running = False
        self.window.destroy()

    def default_template(self):
        tfile = os.path.join(self.configpath, u'template.json')
        if not os.path.exists(tfile):
            tfile = None
        return tfile

    ## Report printing support
    def print_report(self, sections=[], subtitle=u'', docstr=u'',
                           prov=False, doprint=True, exportfile=None):
        """Print the pre-formatted sections in a standard report."""
        self.log.info(u'Printing report ' + repr(subtitle) + u'\u2026')

        tfile = self.default_template()
        rep = printing.printrep(template=tfile, path=self.configpath)
        rep.set_provisional(prov)
        rep.strings[u'title'] = self.titlestr
        # subtitle should probably be property of meet
        rep.strings[u'subtitle'] = (self.subtitlestr + u' ' + subtitle).strip()
        rep.strings[u'datestr'] = strops.promptstr(u'Date:', self.datestr)
        rep.strings[u'commstr'] = strops.promptstr(u'Chief Commissaire:',
                                                  self.commstr)
        rep.strings[u'orgstr'] = strops.promptstr(u'Organiser: ', self.orgstr)
        rep.strings[u'docstr'] = docstr
        rep.strings[u'diststr'] = self.locstr
        for sec in sections:
            rep.add_section(sec)

        # write out to files if exportfile set
        if exportfile:
            ofile = os.path.join(self.exportpath, exportfile+u'.pdf')
            with open(ofile, 'wb') as f:
                rep.output_pdf(f)
            ofile = os.path.join(self.exportpath, exportfile+u'.xls')
            with open(ofile, 'wb') as f:
                rep.output_xls(f)
            lb = u''
            lt = []
            if self.mirrorpath:
                lb = os.path.join(u'/site', self.mirrorpath, exportfile)
                lt = [u'pdf', u'xls']
            ofile = os.path.join(self.exportpath, exportfile+u'.txt')
            with open(ofile, 'wb') as f:
                rep.output_text(f, linkbase=lb, linktypes=lt)
        
        if not doprint:
            return False

        print_op = gtk.PrintOperation()
        print_op.set_print_settings(self.printprefs)
        print_op.set_default_page_setup(self.pageset)
        print_op.connect("begin_print", self.begin_print, rep)
        print_op.connect("draw_page", self.draw_print_page, rep)
        res = print_op.run(gtk.PRINT_OPERATION_ACTION_PRINT_DIALOG,
                               self.window)
        if res == gtk.PRINT_OPERATION_RESULT_APPLY:
            self.printprefs = print_op.get_print_settings()
            self.log.debug(u'Updated print preferences.')
        return False

    def begin_print(self,  operation, context, rep):
        """Set print pages and units."""
        rep.start_gtkprint(context.get_cairo_context())
        operation.set_n_pages(rep.get_pages())
        operation.set_unit("points")

    def draw_print_page(self, operation, context, page_nr, rep):
        """Draw to the nominated page."""
        rep.set_context(context.get_cairo_context())
        rep.draw_page(page_nr)

    def menu_meet_printprefs_activate_cb(self, menuitem=None, data=None):
        """Edit the printer properties."""
        dlg = gtk.PrintOperation()
        dlg.set_print_settings(self.printprefs)
        res = dlg.run(gtk.PRINT_OPERATION_ACTION_PRINT_DIALOG, self.window)
        if res == gtk.PRINT_OPERATION_RESULT_APPLY:
            self.printprefs = dlg.get_print_settings()
            self.log.info(u'Updated print preferences.')

    ## Race menu callbacks.
    def menu_race_startlist_activate_cb(self, menuitem, data=None):
        """Generate a startlist."""
        sections = []
        if self.curevent is not None:
            sections.extend(self.curevent.startlist_report())
        self.print_report(sections)

    def menu_race_result_activate_cb(self, menuitem, data=None):
        """Generate a result."""
        sections = []
        if self.curevent is not None:
            sections.extend(self.curevent.result_report())
        self.print_report(sections, u'Result')

    def menu_race_make_activate_cb(self, menuitem, data=None):
        """Create and open a new race of the chosen type."""
        event = self.edb.add_empty()
        event[u'type']=data
        # Backup an existing config
        oldconf = self.event_configfile(event[u'evid'])
        if os.path.isfile(oldconf):
            bakfile = oldconf + u'.old'
            self.log.info(u'Existing config saved to: ' + repr(bakfile))
            os.rename(oldconf, bakfile)
        self.open_event(event)
        self.menu_race_properties.activate()

    def menu_race_info_activate_cb(self, menuitem, data=None):
        """Show race information on scoreboard."""
        if self.curevent is not None:
            self.scbwin = None
            eh = self.curevent.event
            if self.showevno and eh[u'type'] not in [u'break', u'session']:
                self.scbwin = scbwin.scbclock(self.scb,
                                              u'Event ' + eh[u'evid'],
                                              eh[u'pref'], eh[u'info'])
            else:
                self.scbwin = scbwin.scbclock(self.scb,
                                              eh[u'pref'], eh[u'info'])
            self.scbwin.reset()
            self.curevent.delayed_announce()

    def menu_race_properties_activate_cb(self, menuitem, data=None):
        """Edit properties of open race if possible."""
        if self.curevent is not None:
            self.curevent.do_properties()

    def menu_race_run_activate_cb(self, menuitem=None, data=None):
        """Open currently selected event."""
        eh = self.edb.getselected()
        if eh is not None:
            self.open_event(eh)

    def event_row_activated_cb(self, view, path, col, data=None):
        """Respond to activate signal on event row."""
        self.menu_race_run_activate_cb()

    def menu_race_next_activate_cb(self, menuitem, data=None):
        """Open the next event on the program."""
        if self.curevent is not None:
            nh = self.edb.getnextrow(self.curevent.event)
            if nh is not None:
                self.open_event(nh)
            else:
                self.log.warn(u'No next event to open.')
        else:
            eh = self.edb.getselected()
            if eh is not None:
                self.open_event(eh)
            else:
                self.log.warn(u'No next event to open.')

    def menu_race_prev_activate_cb(self, menuitem, data=None):
        """Open the previous event on the program."""
        if self.curevent is not None:
            ph = self.edb.getprevrow(self.curevent.event)
            if ph is not None:
                self.open_event(ph)
            else:
                self.log.warn(u'No previous event to open.')
        else:
            eh = self.edb.getselected()
            if eh is not None:
                self.open_event(eh)
            else:
                self.log.warn(u'No previous event to open.')

    def menu_race_close_activate_cb(self, menuitem, data=None):
        """Close currently open event."""
        self.close_event()
    
    def menu_race_abort_activate_cb(self, menuitem, data=None):
        """Close currently open event without saving."""
        if self.curevent is not None:
            self.curevent.readonly = True
        self.close_event()

    def open_event(self, eventhdl=None):
        """Open provided event handle."""
        if eventhdl is not None:
            self.close_event()
            self.curevent = mkrace(self, eventhdl)
            self.curevent.loadconfig()
            self.race_box.add(self.curevent.frame)
            self.menu_race_info.set_sensitive(True)
            self.menu_race_close.set_sensitive(True)
            self.menu_race_abort.set_sensitive(True)
            starters = eventhdl[u'star']
            if starters is not None and starters != u'':
                if u'auto' in starters:
                    spec = starters.lower().replace(u'auto', u'').strip()
                    self.curevent.autospec += spec
                    self.log.info(u'Transferred autospec ' + repr(spec)
                                    + u' to event ' + self.curevent.evno)
                else:
                    self.addstarters(self.curevent, eventhdl, # xfer starters
                                     strops.reformat_biblist(starters))
                eventhdl[u'star'] = u''
            self.timer.setcb(self.curevent.timercb)
            self.menu_race_properties.set_sensitive(True)
            self.curevent.show()

    def addstarters(self, race, event, startlist):
        """Add each of the riders in startlist to the opened race."""
        starters = startlist.split()
        for st in starters:
            # check for category
            rlist = self.rdb.biblistfromcat(st, race.series)
            if len(rlist) > 0:
                for est in rlist.split():
                    race.addrider(est)
            else:                    
                race.addrider(st)

    def autostart_riders(self, race, autospec=u'', infocol=None):
        """Try to fetch the startlist from race result info."""
        # Dubious: infocol allows selection of seed info
        #          typical values:
        #                           1 -> timed event qualifiers
        #                           3 -> omnium/handicap
        # TODO: check default, maybe defer to None
        # TODO: IMPORTANT cache result gens for fast recall
        for egroup in autospec.split(u';'):
            self.log.debug(u'Autospec group: ' + repr(egroup))
            specvec = egroup.split(u':')
            if len(specvec) == 2:
                evno = specvec[0].strip()
                if evno not in self.autorecurse:
                    self.autorecurse.add(evno)
                    placeset = strops.placeset(specvec[1])
                    e = self.edb[evno]
                    if e is not None:
                        self.log.debug(u'Adding places ' + repr(placeset)
                                       + u' from event ' + evno)
                        h = mkrace(self, e, False)
                        h.loadconfig()
                        self.log.debug(u'Result gen returns: ')
                        for ri in h.result_gen():
                            self.log.debug(u'REgen: ' + repr(ri))
                            if type(ri[1]) is int and ri[1] in placeset:
                                seed = None
                                if infocol is not None and infocol < len(ri):
                                    seed = ri[infocol]
                                self.log.debug('Adding rider: ' + repr(ri[0]))
                                race.addrider(ri[0], seed)
                        h.destroy()
                    else:
                        self.log.warn(u'Autospec event number not found: '
                                        + repr(evno))
                    self.autorecurse.remove(evno)
                else:
                    self.log.debug(u'Ignoring loop in auto startlist: '
                                   + repr(evno))
            else:
                self.log.warn(u'Ignoring erroneous autospec group: '
                               + repr(egroup))

    def close_event(self):
        """Close the currently opened race."""
        if self.curevent is not None:
            self.timer.setcb()
            self.curevent.hide()
            self.race_box.remove(self.curevent.frame)
            self.curevent.destroy()
            self.curevent.event[u'dirt'] = True	# mark event exportable
            self.menu_race_properties.set_sensitive(False)
            self.menu_race_info.set_sensitive(False)
            self.menu_race_close.set_sensitive(False)
            self.menu_race_abort.set_sensitive(False)
            self.curevent = None

    def race_evno_change(self, old_no, new_no):
        """Handle a change in a race number."""
        oldconf = self.event_configfile(old_no)
        if os.path.isfile(oldconf):
            newconf = self.event_configfile(new_no)
            if os.path.isfile(newconf):
                os.rename(newconf, newconf + u'.old')
            os.rename(oldconf, newconf)
        self.log.info(u'Race ' + repr(old_no) + u' changed to ' + repr(new_no))

    ## Data menu callbacks.
    def menu_data_rego_activate_cb(self, menuitem, data=None):
        """Open rider registration dialog."""
        self.log.warn(u'TODO :: Rider registration dlg...')

    def menu_data_import_activate_cb(self, menuitem, data=None):
        """Open rider import dialog."""
        self.log.warn(u'TODO :: Rider import dlg...')

    def menu_data_result_activate_cb(self, menuitem, data=None):
        """Export final result."""
        provisional = self.provisional	# may be overridden below
        sections = []
        for e in self.edb:
            r = mkrace(self, e, False)
            if e[u'resu']:    # include in result
                if r.evtype in [u'break', u'session']:
                    sec = printing.section()
                    sec.heading = u' '.join([e[u'pref'], e[u'info']]).strip()
                    sec.subheading = u'\t'.join([strops.lapstring(e[u'laps']),
                                            e[u'dist'], e[u'prog']]).strip()
                    sections.append(sec)
                else:
                    r.loadconfig()
                    if r.onestart:	# in progress or done...
                        report = r.result_report()
                    else:
                        report = r.startlist_report()
                    if len(report) > 0:
                        sections.extend(report)
            r.destroy()	# is this valid??

        filebase = os.path.basename(self.configpath)
        if filebase in [u'', u'.']:
            filebase = u'result'
        else:
            filebase += u'_result'

        self.print_report(sections, u'Results', prov=provisional,
                          doprint=False,
                          exportfile=filebase.translate(strops.WEBFILE_UTRANS))

    def printprogram(self, prov=True):
        # TODO:: Jnrs, just map in xtra infos
        tmap = {}
        rlist = {}
        rmap = {}
        catmap = {}
        for r in self.rdb.model:
            bib = r[0]
            club = r[3]
            cat = r[4]
            abbrev = r[6]
            if len(club) > 4:
                abbrev = r[3]
                club = r[6]
            name = strops.resname(r[1], r[2], club)
            if cat not in rlist:
                rlist[cat] = []
            rlist[cat].append([u' ', bib, name, None, None, None])
            if club not in tmap and abbrev:
                tmap[club] = abbrev
            rmap[bib] = name
            catmap[bib] = cat
        tlist = []
        for t in sorted(tmap):
            tlist.append([t, None, tmap[t], None, None, None])

        tfile = self.default_template()
        r = printing.printrep(template=tfile, path=self.configpath)
        subtitlestr = 'Program of Events'
        if self.subtitlestr:
            subtitlestr = self.subtitlestr + ' - ' + subtitlestr
        r.strings['title'] = self.titlestr
        r.strings['subtitle'] = subtitlestr
        r.strings['datestr'] = strops.promptstr('Date:', self.datestr)
        r.strings['commstr'] = strops.promptstr('Chief Commissaire:',
                                                  self.commstr)
        r.strings['orgstr'] = strops.promptstr('Organiser: ', self.orgstr)
        r.strings['docstr'] = '' # What should go here?
        r.strings['diststr'] = self.locstr

        r.set_provisional(prov)

## !!! NO this is already in the event db !!!
        # load in extra event meta from eventsrc.csv
        evmap = {}
        emfile = os.path.join(self.configpath, 'eventsrc.csv')
        if os.path.isfile(emfile):
            self.log.info('Reading extended race info from ' + repr(emfile))
            with open(emfile, 'rb') as f:
                cr = csv.reader(f)
                for ri in cr:
                    evno = None
                    evname = None
                    xtra1 = ''
                    xtra2 = ''
                    lenstr = ''
                    timestr = None
                    holders = 0
                    showcats = False
                    phase = None
                    record = None
                    heatlist = []
                    if len(ri) > 2:
                        evno = ri[0]
                        evname = ri[1] + ' ' + ri[2]
                    if evno and evno != 'Num':
                        head = 'Event ' + evno + ' ' + evname
                        if len(ri) > 4 and ri[4]:
                            etype = ri[4]
                            if etype in ['keirin', 'sprint']:
                                timestr = '200m:'
                            elif etype == 'omnium':
                                timestr = ''
                            else:
                                timestr = 'Time:'
                        if len(ri) > 9 and ri[9]:
                            lenstr = ri[9] + ' Lap'
                            try:
                                if int(float(ri[9])) > 1:
                                    lenstr += 's'
                            except:
                                pass	# really should check numeric err
                        if len(ri) > 10 and ri[10]:
                            xtra1 = ri[10]
                        if len(ri) > 11 and ri[11]:
                            xtra2 = ri[11]
                        if len(ri) > 12 and ri[12]:
                            holders = int(ri[12])
                        if len(ri) > 13 and ri[13]:
                            showcats = strops.confopt_bool(ri[13])
                        if len(ri) > 14 and ri[14]:
                            phase = ri[14]
                        if len(ri) > 15 and ri[15]:
                            record = ri[15]
                        if len(ri) > 16 and ri[16]:
                            heatlist = strops.listsplit(ri[16])
                        evmap[evno] = [head, lenstr, xtra1, xtra2,
                                       timestr, holders, showcats,
                                       phase, record, heatlist]

        for e in self.edb:
            h = mkrace(self, e, False)
            h.loadconfig()
            sec = None
            gapcount = 0
            showcats = False
            if h.evtype in ['classification', 'break']:
                h.destroy()
                continue	# skip events that can't be in program?
            if h.evtype in ['flying 200', 'flying lap',
                                     'indiv tt', 'indiv pursuit',
                                     'pursuit race', 'team pursuit',
                                      'team pursuit race']:
                sec = printing.dual_ittt_startlist()
                if h.evno in evmap:
                    if 'race' not in h.evtype:
                        sec.showheats = True
                    rmeta = evmap[h.evno]
                    sec.heading = rmeta[0]
                    sec.subheading = '\t'.join([rmeta[1], rmeta[3],
                                            rmeta[2]]).strip()
                    if rmeta[5] > 0:	# placeholders - write in gaps
                        gapcount = rmeta[5]
                    showcats = rmeta[6]
                    if rmeta[8]:
                        sec.set_record(rmeta[8])
                else:
                    sec.heading = self.racenamecat(e, 64).strip()
                if h.evtype in ['flying 200', 'flying lap']:
                    sec.set_single()
                sec.lines = h.get_heats(gapcount)
                # hack in the cats 
                if showcats: # how to show cat?
                    pass
            elif h.evtype == 'sprint round':	# sudden-death sprint round
                sec = printing.sprintround()
                # TEMP: just use placeholders
                if h.evno in evmap:
                    rhold = [None, None, None, None, None, None]
                    rmeta = evmap[h.evno]
                    sec.heading = rmeta[0]
                    sec.subheading = '\t'.join([rmeta[1], rmeta[3],
                                            rmeta[2]]).strip()
                    for heat in rmeta[9]:
                        sec.lines.append([heat, rhold, rhold, None])
            elif h.evtype == 'sprint final':	# best of 3 sprint final
                sec = printing.sprintfinal()
                # TEMP: just use placeholders
                if h.evno in evmap:
                    rhold = [None, None, None, None, None, None]
                    rmeta = evmap[h.evno]
                    sec.heading = rmeta[0]
                    sec.subheading = '\t'.join([rmeta[1], rmeta[3],
                                            rmeta[2]]).split()
                    for heat in rmeta[9]:
                        sec.lines.append([heat, rhold, rhold, None])
            else:
                sec = printing.twocol_startlist()
                if h.evno in evmap:
                    rmeta = evmap[h.evno]
                    showcats = rmeta[6]

                for res in h.result_gen():
                    bib = res[0]
                    rank = res[1]
                    if type(rank) is int:
                        rank = unicode(rank) + u'.'
                    name = None
                    info = res[3]
                    if res[0] in rmap:
                        name = rmap[res[0]]
                        if showcats and res[0] in catmap:
                            name += ' ' + catmap[res[0]]
                    sec.lines.append([rank, bib, name, info, None, None])
                if h.evno in evmap:
                    rmeta = evmap[h.evno]
                    sec.heading = rmeta[0]
                    sec.lenstr = rmeta[1]
                    sec.xtra1 = rmeta[2]
                    sec.xtra2 = rmeta[3]
                    sec.timestr = rmeta[4]

                    if rmeta[5] > 0:	# placeholders - write in gaps
                        sec.lines = []
                        while len(sec.lines) < rmeta[5]:
                            sec.lines.append([None, None, None, None, None, None])
                else:
                    sec.heading = self.racenamecat(e, 64).strip()
            r.add_section(sec)
            h.destroy()
                
        # Add team abbrevs
        s = printing.twocol_startlist()
        s.heading = 'Abbreviations'
        s.lines = tlist
        if len(s.lines) > 0:
            r.add_section(s)

        ofile = os.path.join(self.configpath, u'event_program.pdf')
        with open(ofile, 'wb') as f:
            r.output_pdf(f)
            self.log.info(u'Exported pdf program to ' + repr(ofile))
        ofile = os.path.join(self.configpath, 'event_program.txt')
        with open(ofile, 'wb') as f:
            r.output_text(f)
            self.log.info(u'Exported text program to ' + repr(ofile))
        ofile = os.path.join(self.configpath, 'event_program.xls')
        with open(ofile, 'wb') as f:
            r.output_xls(f)
            self.log.info(u'Exported xls program to ' + repr(ofile))

    def menu_data_program_activate_cb(self, menuitem, data=None):
        """Export race program."""
        try:
            self.printprogram(self.provisional)
        except Exception as e:
            self.log.error(u'Error writing report: ' + unicode(e))
            # temp -> for debug
            raise

    def menu_data_update_activate_cb(self, menuitem, data=None):
        """Update event and rider tables in external database."""
        self.log.info(u'Exporting data:')
        try:
            ## 'update' riders
            #delset = []
            #addset = []
            ## scan rider list
            #for r in self.rdb:
                #series = r[riderdb.COL_SERIES]
                #if series == u'':  # HACK: don't export teams at CATN
                    #rno = r[riderdb.COL_BIB]
                    #first = r[riderdb.COL_FIRST]
                    #last = r[riderdb.COL_LAST]
                    #club = r[riderdb.COL_CLUB]
                    #delset.append( (rno, ) )
                    #addset.append( (rno, first, last, club) )
            #self.db.execute(u'DELETE FROM Competitors WHERE RiderNumber=%s',
                            #delset)
            #self.db.execute(u'INSERT INTO Competitors (RiderNumber, FirstName, Surname, State) VALUES (%s, %s, %s, %s)', addset)
            #self.db.commit()
            #self.log.info(u'Wrote ' + unicode(len(addset)) + u' competitors.')
#
            ## 'update' events
            #delset = []
            #addset = []
            #evcount = 0
            ## scan event list
            #for e in self.edb:
                #session = e[u'sess']
                #series = e[u'seri']
                #evno = e[u'evid']
                #cat = e[u'pref']
                #info = e[u'info']
                #delset.append( (evno, ) )
                #addset.append( (evno, session, evcount, cat, info) )
                #evcount += 1
            #self.db.execute(u'DELETE FROM Events WHERE EventID=%s',
                            #delset)
            #self.db.execute(u'INSERT INTO Events (EventID, SessionID, SortOrder, Category, Title) VALUES (%s, %s, %s, %s, %s)', addset)
            #self.db.commit()
            #self.log.info(u'Wrote ' + unicode(len(addset)) + u' events.')
            #self.log.info(u'Export complete.')

            # build index of events report
            if self.mirrorpath:
                tfile = self.default_template()
                orep = printing.printrep(template=tfile,
                                         path=self.configpath)
                orep.strings[u'title'] = self.titlestr
                orep.strings[u'subtitle'] = self.subtitlestr
                orep.strings[u'datestr'] = strops.promptstr(u'Date:',
                                                         self.datestr)
                orep.strings[u'commstr'] = strops.promptstr(
                                           'Chief Commissaire:',
                                              self.commstr)
                orep.strings['orgstr'] = strops.promptstr('Organiser: ',
                                              self.orgstr)
                orep.strings[u'diststr'] = self.locstr
                orep.set_provisional(self.provisional)	# ! TODO 

                sec = printing.event_index()
                sec.heading = 'Index of Events'
                #sec.subheading = Date?
                for eh in self.edb:
                    if eh[u'inde']:	# include in index?
                        evno = eh[u'evid']
                        if eh[u'type'] in [u'break', u'session']:
                            evno = None
                        referno = evno
                        if eh[u'refe']:	# overwrite ref no, even on specials
                            referno = eh[u'refe']
                        linkfile = None
                        if referno: 
                            linkfile = u'event_' + unicode(referno).translate(
                                                   strops.WEBFILE_UTRANS)
                        descr = u' '.join([eh[u'pref'], eh[u'info']]).strip()
                        extra = None	# STATUS INFO -> progress?
                        sec.lines.append([evno, None,
                                          descr, extra, linkfile])
                orep.add_section(sec)
                basename = u'program'
                ofile = os.path.join(self.exportpath, basename + u'.txt')
                with open(ofile, 'wb') as f:
                    orep.output_text(f)	# with links -> not yet
        except Exception as e:
            self.log.error(u'Error exporting event data: ' + unicode(e))
        glib.idle_add(self.mirror_start)

    def mirror_start(self):
        """Create a new rsync mirror thread unless in progress."""
        if self.mirrorpath and self.mirror is None:
            self.mirror = rsync.mirror(
                localpath = os.path.join(self.exportpath,u''),
                remotepath = self.mirrorpath,
                mirrorcmd = self.mirrorcmd)
            self.mirror.start()
        return False	# for idle_add

    def menu_data_export_activate_cb(self, menuitem, data=None):
        """Export race data."""
        try:
            self.log.debug('Exporting race info.')

            # make a rider detail map
            rmap = {u'':{}}
            for r in self.rdb:	# yeilds rows in unicode
                series = r[riderdb.COL_SERIES]
                if series not in rmap:
                    rmap[series] = {}
                rno = r[riderdb.COL_BIB]
                first = r[riderdb.COL_FIRST]
                last = r[riderdb.COL_LAST]
                club = r[riderdb.COL_CLUB]
                rmap[series][rno] = [first, last, club]

            # determine 'dirty' events 	## TODO !!
            dmap = {}
            dord = []
            for e in self.edb:	# note - this is the only traversal
                series = e[u'seri']
                if series not in rmap:
                    rmap[series] = {}
                evno = e[u'evid']
                etype = e[u'type']
                prefix = e[u'pref']
                info = e[u'info']
                export = e[u'resu']
                key = evno	# no need to concat series, evno is unique
                dirty = e[u'dirt']
                if not dirty:   # check for any dependencies
                    for dev in e[u'depe'].split():
                        if dev in dmap:
                            dirty = True
                            break
                if dirty:
                    dord.append(key)	# maintains ordering
                    dmap[key] = [e, evno, etype, series, prefix, info, export]
            self.log.debug(u'Marked ' + unicode(len(dord)) + u' events dirty.')
  
            for k in dmap:	# only output dirty events
                # turn key into read-only event handle
                e = dmap[k][0]
                evno = dmap[k][1]
                etype = dmap[k][2]
                series = dmap[k][3]
                evstr = (dmap[k][4] + u' ' + dmap[k][5]).strip()
                doexport = dmap[k][6]
                e[u'dirt']=False
                r = mkrace(self, e, False)
                r.loadconfig()

                # extract result
                #rset = []
                #rescount = 0
                #if r.onestart:		# skip this for unstarted?
                    #for result in r.result_gen():
                        #bib = result[0]
                        #first = u''
                        #last = u''
                        #club = u''
                        #if bib in rmap[series]:
                            #rider = rmap[series][bib]
                            #first = rider[0]
                            #last = rider[1]
                            #club = rider[2]
                        #rank = u''
                        #if result[1] is not None:
                            #rank = unicode(result[1])
                        #infoa = u''
                        #if result[2] is not None:
                            #if type(result[2]) is tod.tod:
                                #infoa = result[2].rawtime(2)
                            #else:
                                ##infoa = unicode(result[2])
                        #infob = u''
                        #if result[3] is not None:
                            #if type(result[3]) is tod.tod:
                                #infob = result[3].rawtime(2)
                            #else:
                                #infob = str(result[3])
                        ### NO RESULT FOR NO RANK?
                        #if rank:
                            #rset.append( (evno, rescount,
                               #bib, first, last, club, infoa, infob, rank) )
                        #rescount += 1
                #self.db.execute('DELETE FROM Results WHERE EventID=%s',
                                #[(evno, )])
                #self.db.execute('INSERT INTO Results (EventID, SortOrder, RiderNumber, FirstName, Surname, State, InfoA, InfoB, Rank) VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)', rset)
                #self.db.commit()

                # starters
                #stset = []
                #stcount = 0
                #startrep = r.startlist_report()	# trigger rider model reorder
                #for result in r.result_gen():
                    #bib = result[0]
                    #first = u''
                    #last = u''
                    #club = u''
                    #if bib in rmap[series]:
                        #rider = rmap[series][bib]
                        #first = rider[0]
                        #last = rider[1]
                        #club = rider[2]
                    #rank = u''
                    #infoa = u''
                    #infob = u''
                    #if etype in [u'handicap', u'sprint'] and result[3] is not None:
                        #if type(result[3]) is tod.tod:
                            #infob = result[3].rawtime(2)
                        #else:
                            #infob = str(result[3])
                    #stset.append( (evno, stcount,
                              #bib, first, last, club, infoa, infob, rank) )
                    #stcount += 1
                #self.db.execute('DELETE FROM Starters WHERE EventID=%s',
                                #[(evno, )] )
                #self.db.execute('INSERT INTO Starters (EventID, SortOrder, RiderNumber, FirstName, Surname, State, InfoA, InfoB, Rank) VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)', stset)
                #self.db.commit()
                #self.log.debug(u'Dirty event ' 
                               #+ repr(evno) + u' export complete.')

                if self.mirrorpath and doexport:
                    tfile = self.default_template()
                    orep = printing.printrep(template=tfile,
                                             path=self.configpath)
                    orep.strings[u'title'] = self.titlestr
                    # subtitle should probably be property of meet
                    orep.strings[u'subtitle'] = evstr
                    orep.strings[u'datestr'] = strops.promptstr(u'Date:',
                                                             self.datestr)
                    #orep.strings['commstr'] = strops.promptstr('Chief Commissaire:',
                                                  #self.commstr)
                    #orep.strings['orgstr'] = strops.promptstr('Organiser: ', self.orgstr)
                    orep.strings[u'diststr'] = self.locstr
                    orep.strings[u'docstr'] = evstr
                    if etype in [u'classification']:
                        orep.strings[u'docstr'] += u' Classification'
                    orep.set_provisional(self.provisional)	# ! TODO 
                    orep.indexlink = 'program'	# url to program of events
                    # update files and trigger mirror
                    if r.onestart:	# output result
                        outsec = r.result_report()
                        for sec in outsec:
                            orep.add_section(sec)
                    else:		# startlist
                        outsec = r.startlist_report()
                        for sec in outsec:
                            orep.add_section(sec)
                    basename = u'event_' + unicode(evno).translate(
                                                   strops.WEBFILE_UTRANS)
                    ofile = os.path.join(self.exportpath, basename + u'.txt')
                    with open(ofile, 'wb') as f:
                        orep.output_text(f)	# with links -> not yet
                r.destroy()
            glib.idle_add(self.mirror_start)
            self.log.info(u'Race info export.')
        except Exception as e:
            self.log.error(u'Error exporting results: ' + unicode(e))
            raise	# temporary DEBUG

    ## SCB menu callbacks
    def menu_scb_enable_toggled_cb(self, button, data=None):
        """Update scoreboard enable setting."""
        if button.get_active():
            self.scb.set_ignore(False)
            self.scb.setport(self.scbport)
            self.announce.set_portstr(self.annport)
            if self.scbwin is not None:
                self.scbwin.reset()
        else:
            self.scb.set_ignore(True)

    def menu_scb_clock_cb(self, menuitem, data=None):
        """Select timer scoreboard overlay."""
        self.gemini.clear()
        self.scbwin = None
        self.scb.setoverlay(unt4.OVERLAY_CLOCK)
        self.log.debug(u'Selected scoreboard timer overlay.')

    def menu_scb_logo_activate_cb(self, menuitem, data=None):
        """Select logo and display overlay."""
        self.gemini.clear()
        self.scbwin = scbwin.logoanim(self.scb, self.logos)
        self.scbwin.reset()
        self.log.debug(u'Running scoreboard logo anim.')

    def menu_scb_blank_cb(self, menuitem, data=None):
        """Select blank scoreboard overlay."""
        self.gemini.clear()
        self.scbwin = None
        self.scb.setoverlay(unt4.OVERLAY_BLANK)
        self.log.debug(u'Selected scoreboard blank overlay.')

    def menu_scb_test_cb(self, menuitem, data=None):
        """Run the scoreboard test pattern."""
        self.scbwin = scbwin.scbtest(self.scb)
        self.scbwin.reset()
        self.log.debug(u'Running scoreboard test pattern.')

    def menu_scb_connect_activate_cb(self, menuitem, data=None):
        """Force a reconnect to scoreboard."""
        self.scb.setport(self.scbport)
        self.announce.set_portstr(self.annport)
        self.log.debug(u'Re-connect scoreboard.')
        if self.gemport != u'':
            self.gemini.setport(self.gemport)
        if self.dbport != u'':
            self.db.setport(self.dbport)

    ## Timing menu callbacks
    def menu_timing_subtract_activate_cb(self, menuitem, data=None):
        """Run the time of day subtraction dialog."""
        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, u'tod_subtract.ui'))
        ste = b.get_object('timing_start_entry')
        fte = b.get_object('timing_finish_entry')
        nte = b.get_object('timing_net_entry')
        b.get_object('timing_start_now').connect('clicked',
                                                 self.entry_set_now, ste)
        b.get_object('timing_finish_now').connect('clicked',
                                                 self.entry_set_now, fte)
        ste.connect('activate', self.menu_timing_recalc, ste, fte, nte)
        fte.connect('activate', self.menu_timing_recalc, ste, fte, nte)
        dlg = b.get_object('timing')
        dlg.set_transient_for(self.window)
        dlg.run()
        dlg.destroy()

    def entry_set_now(self, button, entry=None):
        """Enter the 'now' time in the provided entry."""
        entry.set_text(tod.tod(u'now').timestr())
        entry.activate()

    def menu_timing_recalc(self, entry, ste, fte, nte):
        """Update the net time entry for the supplied start and finish."""
        st = tod.str2tod(ste.get_text().decode('utf-8','replace'))
        ft = tod.str2tod(fte.get_text().decode('utf-8','replace'))
        if st is not None and ft is not None:
            ste.set_text(st.timestr())
            fte.set_text(ft.timestr())
            nte.set_text((ft - st).timestr())

    def menu_timing_main_toggled_cb(self, button, data=None):
        """Update the selected primary timer."""
        if button.get_active():
            self.log.info(u'Selected main timer as race time source')
            self.timer = self.main_timer
        else:
            self.log.info(u'Selected backup timer as race time source')
            self.timer = self.backup_timer

    def menu_timing_clear_activate_cb(self, menuitem, data=None):
        """Clear memory in attached timing devices."""
        self.main_timer.clrmem()
        self.backup_timer.clrmem()
        self.log.info(u'Clear attached timer memories')

    def menu_timing_dump_activate_cb(self, menuitem, data=None):
        """Request memory dump from attached timy."""
        self.timer.dumpall()
        self.log.info(u'Dump active timer memory.')

    def menu_timing_reconnect_activate_cb(self, menuitem, data=None):
        """Reconnect timers and initialise."""
        self.main_timer.setport(self.main_port)
        self.main_timer.sane()
        self.backup_timer.setport(self.backup_port)
        self.backup_timer.sane()
        self.log.info(u'Re-connect and initialise attached timers.')

    ## Help menu callbacks
    def menu_help_docs_cb(self, menuitem, data=None):
        """Display program help."""
        metarace.help_docs(self.window)

    def menu_help_about_cb(self, menuitem, data=None):
        """Display metarace about dialog."""
        metarace.about_dlg(self.window)
  
    ## Menu button callbacks
    def menu_clock_clicked_cb(self, button, data=None):
        """Handle click on menubar clock."""
        (line1, line2, line3) = strops.titlesplit(self.titlestr + u' ' + self.subtitlestr)
        self.scbwin = scbwin.scbclock(self.scb, line1, line2, line3)
        self.scbwin.reset()
        self.log.debug(u'Displaying meet info and clock on scoreboard.')

    ## Directory utilities
    # !! CONVERT to event_NN.json
    def event_configfile(self, evno):
        """Return a config filename for the given event no."""
        return os.path.join(self.configpath, u'event_'+unicode(evno)+u'.ini')

    ## Timer callbacks
    def menu_clock_timeout(self):
        """Update time of day on clock button."""
        if not self.running:
            return False
        else:
            tt = tod.tod(u'now')
            self.clock_label.set_text(tt.meridian())

            # check for completion in the rsync module
            if self.mirror is not None:
                if not self.mirror.is_alive():
                    self.mirror = None
            #if type(self.scbwin) is scbwin.scbclock:
                #self.gemini.ctick(tt)	# dubious
            #self.announce.postxt(0,72,tt)
        return True

    def timeout(self):
        """Update internal state and call into race timeout."""
        if not self.running:
            return False
        try:
            if self.curevent is not None:      # this is expected to
                self.curevent.timeout()        # collect any timer events
            if self.scbwin is not None:
                self.scbwin.update()
        except Exception as e:
            self.log.error(u'Timeout: ' + unicode(e)) # use a stack trace
        return True

    ## Timy utility methods.
    def printimp(self, printimps=True):
        """Enable or disable printing of timing impulses on Timy."""
        self.main_timer.printimp(printimps)
        self.backup_timer.printimp(printimps)

    def timer_log_straight(self, bib, msg, tod, prec=4):
        """Print a tod log entry on the Timy receipt."""
        self.timer.printline(u'{0:3} {1: >4}: '.format(bib[0:3],
                              unicode(msg)[0:4]) + tod.timestr(prec))

    def timer_log_msg(self, bib, msg):
        """Print the given msg entry on the Timy receipt."""
        self.timer.printline(u'{0:3} '.format(bib[0:3]) + unicode(msg)[0:20])

    def event_string(self, evno):
        """Switch to suppress event no in delayed announce screens."""
        ret = u''
        if self.showevno:
            ret = u'Event ' + unicode(evno)
        else:
            ret = u' '.join([self.titlestr, self.subtitlestr]).strip()
        return ret

    def racenamecat(self, event, slen=None):
        """Concatentate race info for display on scoreboard header line."""
        if slen is None:
            slen = metarace.SCB_LINELEN
        evno = u''
        srcev = event[u'evid']
        if self.showevno and event[u'type'] not in [u'break', u'session']:
            evno = u'Ev ' + srcev
        info = event[u'info']
        prefix = event[u'pref']
        ret = u' '.join([evno, prefix, info]).strip()
        if len(ret) > slen + 1:
            ret = u' '.join([evno, info]).strip()
        return strops.truncpad(ret, slen)

    def racename(self, event):
        """Return a full event identifier string."""
        evno = u''
        if self.showevno and event[u'type'] not in [u'break', u'session']:
            evno = u'Event ' + event[u'evid']
        info = event[u'info']
        prefix = event[u'pref']
        return u' '.join([evno, prefix, info]).strip()

    ## Announcer methods
    def ann_default(self):
        self.announce.setline(0, strops.truncpad(
             u' '.join([self.titlestr, self.subtitlestr,
                               self.datestr]).strip(), 70, u'c'))

    def ann_title(self, titlestr=u''):
        self.announce.setline(0, strops.truncpad(titlestr.strip(), 70, u'c'))

    ## Window methods
    def set_title(self, extra=u''):
        """Update window title from meet properties."""
        self.window.set_title(u'trackmeet :: ' 
               + u' '.join([self.titlestr, self.subtitlestr]).strip())
        self.ann_default()

    def meet_destroy_cb(self, window, msg=u''):
        """Handle destroy signal and exit application."""
        lastevent = None
        if self.curevent is not None:
            lastevent = self.curevent.evno
            self.close_event()
        if self.started:
            self.saveconfig(lastevent)
            self.log.info(u'Meet shutdown: ' + msg)
            self.shutdown(msg)
        self.log.removeHandler(self.sh)
        self.log.removeHandler(self.lh)
        if self.loghandler is not None:
            self.log.removeHandler(self.loghandler)
        self.running = False
        gtk.main_quit()
        print(u'Exit.')

    def key_event(self, widget, event):
        """Collect key events on main window and send to race."""
        if event.type == gtk.gdk.KEY_PRESS:
            key = gtk.gdk.keyval_name(event.keyval) or 'None'
            if event.state & gtk.gdk.CONTROL_MASK:
                key = key.lower()
                if key in ['0','1','2','3','4','5','6','7','8','9']:	#?uni
                    self.timer.trig(chan=key)
                    return True
            if self.curevent is not None:
                return self.curevent.key_event(widget, event)
        return False

    def shutdown(self, msg):
        """Cleanly shutdown threads and close application."""
        self.started = False
        self.db.exit(msg)
        self.gemini.exit(msg)
        self.scb.exit(msg)
        self.announce.exit(msg)
        self.main_timer.exit(msg)
        self.backup_timer.exit(msg)
        print (u'Waiting for workers to exit...')
        if self.mirror is not None:
            print(u'\tlive result mirror.')
            self.mirror.join()
            self.mirror = None
        print(u'\tdatabase export.')
        self.db.join()
        print(u'\tauxilliary scoreboard.')
        self.gemini.join()
        print(u'\tmain scoreboard.')
        self.scb.join()
        print(u'\tuscbsrv announce.')
        self.announce.join()
        print(u'\tmain timer.')
        self.main_timer.join()
        print(u'\tbackup timer.')
        self.backup_timer.join()

    def start(self):
        """Start the timer and scoreboard threads."""
        if not self.started:
            self.log.debug(u'Meet startup.')
            self.scb.start()
            self.announce.start()
            self.main_timer.start()
            self.backup_timer.start()
            self.gemini.start()
            self.db.start()
            self.started = True

    ## Track meet functions
    def delayed_export(self):
        """Queue an export on idle add."""
        self.exportpending = True
        glib.idle_add(self.exportcb)

    def save_curevent(self):
        """Backup and save current event."""
        ## NOTE: assumes curevent is defined, test externally

        # backup an existing config
        oldconf = self.event_configfile(self.curevent.event[u'evid'])
        if os.path.isfile(oldconf):
            os.rename(oldconf, oldconf + u'.1')	# one level of versioning?
        
        # call into event and save
        self.curevent.saveconfig()

        # mark event dirty in event db
        self.curevent.event[u'dirt'] = True

    def exportcb(self):
        """Save current event and update race info in external db."""
        if not self.exportpending:
            return False	# probably doubled up
        self.exportpending = False
        if self.curevent is not None and self.curevent.winopen:
            self.save_curevent()
        self.menu_data_export_activate_cb(None)
        return False # for idle add

    def saveconfig(self, lastevent=None):
        """Save current meet data to disk."""
        cw = jsonconfig.config()
        cw.add_section(u'meet')
        cw.set(u'meet', u'id', TRACKMEET_ID)
        if self.curevent is not None and self.curevent.winopen:
            self.save_curevent()
            cw.set(u'meet', u'curevent', self.curevent.evno)
        elif lastevent is not None:
            cw.set(u'meet', u'curevent', lastevent)
        cw.set(u'meet', u'maintimer', self.main_port)
        cw.set(u'meet', u'backuptimer', self.backup_port)
        cw.set(u'meet', u'gemini', self.gemport)
        cw.set(u'meet', u'dbhost', self.dbport)
        if self.timer is self.main_timer:
            cw.set(u'meet', u'racetimer', u'main')
        else:
            cw.set(u'meet', u'racetimer', u'backup')
        cw.set(u'meet', u'scbport', self.scbport)
        cw.set(u'meet', u'uscbport', self.annport)
        cw.set(u'meet', u'title', self.titlestr)
        cw.set(u'meet', u'subtitle', self.subtitlestr)
        cw.set(u'meet', u'date', self.datestr)
        cw.set(u'meet', u'location', self.locstr)
        cw.set(u'meet', u'organiser', self.orgstr)
        cw.set(u'meet', u'commissaire', self.commstr)
        cw.set(u'meet', u'logos', self.logos)
        cw.set(u'meet', u'mirrorpath', self.mirrorpath)
        cw.set(u'meet', u'mirrorcmd', self.mirrorcmd)
        cw.set(u'meet', u'clubmode', self.clubmode)
        cw.set(u'meet', u'showevno', self.showevno)
        cw.set(u'meet', u'provisional', self.provisional)
        cw.set(u'meet', u'tracklayout', self.tracklayout)
        cw.set(u'meet', u'tracklen_n', unicode(self.tracklen_n))  # poss val?
        cw.set(u'meet', u'tracklen_d', unicode(self.tracklen_d))
        cw.set(u'meet', u'docindex', unicode(self.docindex))
        cw.set(u'meet', u'graphicscb', self.graphicscb)
        cwfilename = os.path.join(self.configpath, CONFIGFILE)
        self.log.debug(u'Saving meet config to ' + repr(cwfilename))
        with open(cwfilename , 'wb') as f:
            cw.write(f)
        self.rdb.save(os.path.join(self.configpath, u'riders.csv'))
        self.edb.save(os.path.join(self.configpath, u'events.csv'))
        # save out print settings
        self.printprefs.to_file(os.path.join(self.configpath, u'print.prf'))

    def loadconfig(self):
        """Load meet config from disk."""
        cr = jsonconfig.config({u'meet':{u'maintimer':timy.MAINPORT,
                                        u'backuptimer':timy.BACKUPPORT,
                                        u'racetimer':u'main',
                                        u'scbport':u'',
                                        u'uscbport':DEFANNOUNCE_PORT,
                                        u'showevno':True,
					u'resultnos':True,
                                        u'clubmode':True,
                                        u'tracklen_n':u'250',
                                        u'tracklen_d':u'1',
                                        u'docindex':u'0',
                                        u'gemini':u'',
                                        u'dbhost':u'',
                                        u'title':u'',
                                        u'subtitle':u'',
                                        u'date':u'',
                                        u'location':u'',
                                        u'organiser':u'',
                                        u'commissaire':u'',
                                        u'logos':u'',
                                        u'curevent':u'',
                                        u'mirrorpath':u'',
                                        u'mirrorcmd':u'echo',
                                        u'graphicscb':u'',
                                        u'tracklayout':u'DISC',
                                        u'provisional':False,
                                        u'id':u''}})
        cr.add_section(u'meet')
        cwfilename = os.path.join(self.configpath, CONFIGFILE)

        # re-set main log file
        if self.loghandler is not None:
            self.log.removeHandler(self.loghandler)
            self.loghandler.close()
            self.loghandler = None
        self.loghandler = logging.FileHandler(
                             os.path.join(self.configpath, LOGFILE))
        self.loghandler.setLevel(LOGHANDLER_LEVEL)
        self.loghandler.setFormatter(logging.Formatter(
                       '%(asctime)s %(levelname)s:%(name)s: %(message)s'))
        self.log.addHandler(self.loghandler)

        # check for config file
        try:
            with open(cwfilename, 'rb') as f:
                cr.read(f)
        except Exception as e:
            self.log.error(u'Reading meet config: ' + unicode(e))

        # set main timer port
        nport = cr.get(u'meet', u'maintimer')
        if nport != self.main_port:
            self.main_port = nport
            self.main_timer.setport(nport)
            self.main_timer.sane()

        # set backup timer port
        nport = cr.get(u'meet', u'backuptimer')
        if nport != self.backup_port:
            self.backup_port = nport
            self.backup_timer.setport(nport)
            self.backup_timer.sane()

        # add gemini board if defined
        self.gemport = cr.get(u'meet', u'gemini')
        if self.gemport != u'':
            self.gemini.setport(self.gemport)
            
        # add database export if present
        self.dbport = cr.get(u'meet', u'dbhost')
        if self.dbport != u'':
            self.db.setport(self.dbport)

        # choose race timer
        if cr.get(u'meet', u'racetimer') == u'main':
            if self.timer is self.backup_timer:
                self.menubut_main.activate()
        else:
            if self.timer is self.main_timer:
                self.menubut_backup.activate()

        # choose scoreboard port
        nport = cr.get(u'meet', u'scbport')
        if self.scbport != nport:
            self.scbport = nport
            self.scb.setport(nport)
        self.annport = cr.get(u'meet', u'uscbport')
        self.announce.set_portstr(self.annport)
        self.announce.clrall()

        # set meet meta infos, and then copy into text entries
        self.titlestr = cr.get(u'meet', u'title')
        self.subtitlestr = cr.get(u'meet', u'subtitle')
        self.datestr = cr.get(u'meet', u'date')
        self.locstr = cr.get(u'meet', u'location')
        self.orgstr = cr.get(u'meet', u'organiser')
        self.commstr = cr.get(u'meet', u'commissaire')
        self.logos = cr.get(u'meet', u'logos')
        self.mirrorpath = cr.get(u'meet', u'mirrorpath')
        self.mirrorcmd = cr.get(u'meet', u'mirrorcmd')
        self.set_title()

        # result options (bool)
        self.clubmode = strops.confopt_bool(cr.get(u'meet', u'clubmode'))
        self.showevno = strops.confopt_bool(cr.get(u'meet', u'showevno'))
        self.provisional = strops.confopt_bool(cr.get(u'meet', u'provisional'))

        # track length
        n = cr.get(u'meet', u'tracklen_n')
        d = cr.get(u'meet', u'tracklen_d')
        setlen = False
        if n.isdigit() and d.isdigit():
            n = int(n)
            d = int(d)
            if n > 0 and n < 5500 and d > 0 and d < 10: # sanity check
                self.tracklen_n = n
                self.tracklen_d = d
                setlen = True
        if not setlen:
            self.log.warn(u'Ignoring invalid track length - default used.')

        # track sector lengths
        self.tracklayout = cr.get(u'meet',u'tracklayout')
        self.sectormap = timy.make_sectormap(self.tracklayout)

        # document id
        self.docindex = strops.confopt_posint(cr.get(u'meet', u'docindex'), 0)

        self.rdb.clear()
        self.edb.clear()
        self.rdb.load(os.path.join(self.configpath, u'riders.csv'))
        self.edb.load(os.path.join(self.configpath, u'events.csv'))

        cureventno = cr.get(u'meet', u'curevent')
        if cureventno and cureventno in self.edb:
            self.open_event(self.edb[cureventno])

        # restore printer preferences
        psfilename = os.path.join(self.configpath, u'print.prf')
        if os.path.isfile(psfilename):
            try:
                self.printprefs.load_file(psfilename)
            except Exception as e:
                self.log.warn(u'Error loading print preferences: ' 
                                 + unicode(e))

        # make sure export path exists
        if not os.path.exists(self.exportpath):
            os.mkdir(self.exportpath)
            self.log.info(u'Created export path: ' + repr(self.exportpath))

        # add grpahic scb if required...
        self.graphicscb = cr.get(u'meet', u'graphicscb')
        if self.graphicscb:
            glib.timeout_add_seconds(30, self.gfxscb_connect)

        # After load complete - check config and report. This ensures
        # an error message is left on top of status stack. This is not
        # always a hard fail and the user should be left to determine
        # an appropriate outcome.
        cid = cr.get(u'meet', u'id')
        if cid != TRACKMEET_ID:
            self.log.error(u'Meet configuration mismatch: '
                           + repr(cid) + u' != ' + repr(TRACKMEET_ID))

    def gfxscb_connect(self):
        self.log.info('Joining gfx scb channel: ' + repr(self.graphicscb))
        self.announce.add_channel(self.graphicscb)
        return False

    def rider_edit(self, bib, series=u'', col=-1, value=u''):
        dbr = self.rdb.getrider(bib, series)
        if dbr is None:	# Scarmble>!!? it's bad form.
            dbr = self.rdb.addempty(bib, series)
        if col == riderdb.COL_FIRST:
            self.rdb.editrider(ref=dbr, first=value)
        elif col == riderdb.COL_LAST:
            self.rdb.editrider(ref=dbr, last=value)
        elif col == riderdb.COL_CLUB:
            self.rdb.editrider(ref=dbr, club=value)
        else:
            self.log.debug(u'Attempt to edit other rider column: ' + repr(col))

    def get_clubmode(self):
        return self.clubmode

    def get_distance(self, count=None, units=u'metres'):
        """Convert race distance units to metres."""
        ret = None
        if count is not None:
            try:
                if units in [u'metres',u'meters']:
                    ret = int(count)
                elif units == u'laps':
                    ret = self.tracklen_n * int(count)
                    if self.tracklen_d != 1 and self.tracklen_d > 0:
                        ret //= self.tracklen_d
### !! Check this is correct in >=2.6 !!
            except (ValueError, TypeError, ArithmeticError), v:
                self.log.warn('Error computing race distance: ' + repr(v))
        return ret

    def __init__(self, configpath=None):
        """Meet constructor."""
        # logger and log handler
        self.log = logging.getLogger()
        self.log.setLevel(logging.DEBUG)
        self.loghandler = None	# set in loadconfig to meet dir

        # meet configuration path and options
        if configpath is None:
            configpath = u'.'	# None assumes 'current dir'
        self.configpath = configpath
        self.exportpath = os.path.join(configpath, EXPORTPATH)
        self.titlestr = u''
        self.subtitlestr = u''
        self.datestr = u''
        self.locstr = u''
        self.orgstr = u''
        self.commstr = u''
        self.clubmode = True
        self.showevno = True
        self.provisional = False
        self.tracklen_n = 250	# numerator
        self.tracklen_d = 1	# d3nominator
        self.sectormap = {}	# map of timing channels to sector lengths
        self.tracklayout = None	# track configuration key
        self.logos = u''		# string list of logo filenames
        self.docindex = 0	# HACK: use for session number
        self.lastexport = None	# timestamp of last db dump
        self.exportpending = False
        self.mirrorpath = u''	# default rsync mirror path
        self.mirrorcmd = u''	# default rsync mirror command

        # printer preferences
        paper = gtk.paper_size_new_custom('metarace-full',
                      'A4 for reports', 595, 842, gtk.UNIT_POINTS)
        self.printprefs = gtk.PrintSettings()
        self.pageset = gtk.PageSetup()
        self.pageset.set_orientation(gtk.PAGE_ORIENTATION_PORTRAIT)
        self.pageset.set_paper_size(paper)
        self.pageset.set_top_margin(0, gtk.UNIT_POINTS)
        self.pageset.set_bottom_margin(0, gtk.UNIT_POINTS)
        self.pageset.set_left_margin(0, gtk.UNIT_POINTS)
        self.pageset.set_right_margin(0, gtk.UNIT_POINTS)

        # hardware connections
        self.scb = sender.sender(u'NULL')
        self.announce = uscbsrv.uscbsrv(80)
        self.scbport = u'NULL'
        self.annport = u''
        self.main_timer = timy.timy(u'', name=u'main')
        self.main_port = u''
        self.backup_timer = timy.timy(u'', name=u'bkup')
        self.backup_port = u''
        self.timer = self.main_timer
        self.gemini = gemini.gemini(u'')	# hack for Perth GP
        self.gemport = u''
        self.db = dbexport.dbexport()	# hack for CA Track Nats-> to be incl
        self.dbport = u''
        self.mirror = None
        self.graphicscb = u''	# no graphic scb to connect to

        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, u'trackmeet.ui'))
        self.window = b.get_object('meet')
        self.window.connect('key-press-event', self.key_event)
        self.clock = b.get_object('menu_clock')
        self.clock_label = b.get_object('menu_clock_label')
        self.status = b.get_object('status')
        self.log_buffer = b.get_object('log_buffer')
        self.log_view = b.get_object('log_view')
        self.log_view.modify_font(pango.FontDescription("monospace 9"))
        self.log_scroll = b.get_object('log_box').get_vadjustment()
        self.context = self.status.get_context_id('metarace meet')
        self.menubut_main = b.get_object('menu_timing_main')
        self.menubut_backup = b.get_object('menu_timing_backup')
        self.menu_race_info = b.get_object('menu_race_info')
        self.menu_race_properties = b.get_object('menu_race_properties')
        self.menu_race_close = b.get_object('menu_race_close')
        self.menu_race_abort = b.get_object('menu_race_abort')
        self.race_box = b.get_object('race_box')
        self.new_race_pop = b.get_object('menu_race_new_types')
        b.connect_signals(self)

        # additional obs
        self.scbwin = None

        # run state
        self.running = True
        self.started = False
        self.curevent = None
        self.autorecurse = set()

        # format and connect status and log handlers
        f = logging.Formatter('%(levelname)s:%(name)s: %(message)s')
        self.sh = loghandler.statusHandler(self.status, self.context)
        self.sh.setLevel(logging.INFO)	# show info upon status bar
        self.sh.setFormatter(f)
        self.log.addHandler(self.sh)
        self.lh = loghandler.textViewHandler(self.log_buffer,
                      self.log_view, self.log_scroll)
        self.lh.setLevel(logging.INFO)	# show info up in log view
        self.lh.setFormatter(f)
        self.log.addHandler(self.lh)

        # get rider db and pack into scrolled pane
        self.rdb = riderdb.riderdb()
        b.get_object('rider_box').add(self.rdb.mkview())

        # get event db and pack into scrolled pane
        self.edb = eventdb.eventdb()
        b.get_object('event_box').add(self.edb.mkview())
        #self.edb.view.connect('row-activated', self.event_row_activated_cb)
        #self.edb.set_evno_change_cb(self.race_evno_change)

	# now, connect each of the race menu types if present in builder
        for etype in self.edb.racetypes:
            lookup = u'mkrace_' + etype.replace(u' ', u'_')
            mi = b.get_object(lookup)
            if mi is not None:
                mi.connect('activate', self.menu_race_make_activate_cb, etype)

        # start timers
        glib.timeout_add_seconds(1, self.menu_clock_timeout)
        glib.timeout_add(50, self.timeout)

def main():
    """Run the trackmeet application."""
    configpath = None
    # expand config on cmd line to realpath _before_ doing chdir
    if len(sys.argv) > 2:
        print(u'usage: trackmeet [configdir]\n')
        sys.exit(1)
    elif len(sys.argv) == 2:
        configpath = os.path.realpath(os.path.dirname(sys.argv[1]))

    metarace.init()
    app = trackmeet(configpath)
    app.loadconfig()
    app.window.show()
    app.start()
    try:
        gtk.main()
    except:
        app.shutdown(u'Exception from gtk.main()')
        raise

if __name__ == '__main__':
    main()
