
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Road Mass-start race module.

This module provides a class 'rms' which implements the race
interface and manages data, timing and scoreboard for generic
road race events:

 - Criterium
 - Kermesse
 - Road race

"""

import gtk
import glib
import gobject
import pango
import os
import logging
import csv
import ConfigParser
import threading

import metarace
from metarace import tod
from metarace import eventdb
from metarace import riderdb
from metarace import strops
from metarace import uiutil
from metarace import printing

# Model columns

# basic infos
COL_BIB = 0
COL_NAMESTR = 1
COL_CAT = 2
COL_COMMENT = 3
COL_INRACE = 4		# boolean in the race
COL_PLACE = 5		# Place assigned in result
COL_LAPS = 6		# Incremented if inrace and not finished

# timing infos
COL_RFTIME = 7		# one-off finish time by rfid
COL_CBUNCH = 8		# computed bunch time	-> derived from rftime
COL_MBUNCH = 9		# manual bunch time	-> manual overrive
COL_STOFT = 10		# start time 'offset' - only reported in result
COL_BONUS = 11
COL_PENALTY = 12
COL_RFSEEN = 13		# list of tods this rider 'seen' by rfid

# rider commands
RIDER_COMMANDS_ORD = [ 'add', 'del', 'que', 'dns', 'hd',
                   'dnf', 'dsq', 'com', 'nst', 'ret',
                   '', 'fin']	# then intermediates...
RIDER_COMMANDS = {'dns':'Did not start',
                   'hd':'Outside time limit',
                   'dnf':'Did not finish',
                   'dsq':'Disqualify',
                   'add':'Add starters',
                   'del':'Remove starters',
                   'que':'Query riders',
                   'fin':'Final places',
                   'com':'Add comment',
                   'nst':'Add new start',
                   'ret':'Return to race',
                   '':'',
                   }

RESERVED_SOURCES = ['fin',	# finished stage
                    'reg',	# registered to stage
                    'start']	# started stage

DNFCODES = ['hd', 'dsq', 'dnf', 'dns']

# timing keys
key_announce = 'F4'
key_armstart = 'F5'
key_armlap = 'F6'
key_markrider = 'F7'
key_promoterider = 'F8'
key_armfinish = 'F9'
key_raceover = 'F10'

# extended fn keys	(ctrl + key)
key_abort = 'F5'
key_confirm = 'F8'
key_undo = 'Z'

# config version string
EVENT_ID = 'roadrace-1.3'

def key_bib(x):
    """Sort on bib field of aux row."""
    return strops.riderno_key(x[1])

def sort_bib(x, y):
    """Rider bib sorter."""
    return cmp(strops.riderno_key(x[1]),
               strops.riderno_key(y[1]))

def sort_tally(x, y):
    """Points tally sort using countback struct."""
    if x[0] == y[0]:
        return cmp(y[3], x[3])
    else:
        return cmp(y[0], x[0])

class rms(object):
    """Road race handler."""

    def loadconfig(self):
        """Load event config from disk."""
        self.riders.clear()
        self.resettimer()
        self.cats = []

        cr = ConfigParser.ConfigParser({'start':'',
                                        'lstart':'',
                                        'id':EVENT_ID,
                                        'finish':'',
                                        'finished':'No',
                                        'places':'',
                                        'comment':'',
                                        'hidecols':'',
                                        'categories':'',
                                        'intermeds':'',
                                        'contests':'',
                                        'tallys':'',
                                        'nolaps':'False',
                                        'startlist':''})
        cr.add_section('event')
        cr.add_section('riders')
        # sections for commissaire awarded bonus/penalty
        cr.add_section('stagebonus')
        cr.add_section('stagepenalty')
        if os.path.isfile(self.configpath):
            self.log.debug('Attempting to read config from path='
                            + repr(self.configpath))
            cr.read(self.configpath)

        # load _result_ categories
        catlist = cr.get('event', 'categories')
        if catlist == 'AUTO':
            self.cats = self.meet.rdb.listcats()
            self.autocats = True
        else:
            self.autocats = False
            for cat in catlist.split(','):
                if cat != '':
                    cat = cat.upper()
                    self.cats.append(cat)
        self.cats.append('')	# always include one empty cat

        # restore intermediates
        for i in cr.get('event', 'intermeds').translate(
                            strops.BIBLIST_TRANS).split():
            if i in RESERVED_SOURCES:
                self.log.info('Ignoring reserved intermediate: ' + repr(i))
            else:
                crkey = 'intermed_' + i
                descr = ''
                places = ''
                if cr.has_option(crkey, 'descr'):
                    descr = cr.get(crkey, 'descr')
                if cr.has_option(crkey, 'places'):
                    places = strops.reformat_placelist(
                                 cr.get(crkey, 'places'))
                if i not in self.intermeds:
                    self.log.debug('Adding intermediate: '
                                    + repr(i) + ':' + descr
                                    + ':' + places)
                    self.intermeds.append(i)
                    self.intermap[i] = {'descr':descr, 'places':places}
                else:
                    self.log.info('Ignoring duplicate intermediate: '
                                   + repr(i))

        # load contest meta data
        tallyset = set()
        for i in cr.get('event', 'contests').translate(
                            strops.BIBLIST_TRANS).split():
            if i not in self.contests:
                self.contests.append(i)
                self.contestmap[i] = {}
                crkey = 'contest_' + i
                tally = ''
                if cr.has_option(crkey, 'tally'):
                    tally = cr.get(crkey, 'tally')
                    if tally:
                        tallyset.add(tally)
                self.contestmap[i]['tally'] = tally
                descr = i
                if cr.has_option(crkey, 'descr'):
                    descr = cr.get(crkey, 'descr')
                    if descr == '':
                        descr = i
                self.contestmap[i]['descr'] = descr
                labels = []
                if cr.has_option(crkey, 'labels'):
                    labels = cr.get(crkey, 'labels').split()
                self.contestmap[i]['labels'] = labels
                source = i
                if cr.has_option(crkey, 'source'):
                    source = cr.get(crkey, 'source')
                    if source == '':
                        source = i
                self.contestmap[i]['source'] = source
                bonuses = []
                if cr.has_option(crkey, 'bonuses'):
                    for bstr in cr.get(crkey, 'bonuses').split():
                        bt = tod.str2tod(bstr)
                        if bt is None:
                            self.log.info('Invalid bonus ' + repr(bstr)
                              + ' in contest ' + repr(i))
                            bt = tod.ZERO
                        bonuses.append(bt)
                self.contestmap[i]['bonuses'] = bonuses
                points = []
                if cr.has_option(crkey, 'points'):
                    pliststr = cr.get(crkey, 'points').strip()
                    if pliststr and tally == '': # no tally for these points!
                        self.log.error('No tally for points in contest: '
                                        + repr(i))
                        tallyset.add('')	# add empty placeholder
                    for pstr in pliststr.split():
                        pt = 0
                        try:
                            pt = int(pstr)
                        except:
                            self.log.info('Invalid points ' + repr(pstr)
                              + ' in contest ' + repr(i))
                        points.append(pt)
                self.contestmap[i]['points'] = points
                allsrc = False		# all riders in source get same pts
                if cr.has_option(crkey, 'all_source'):
                    allsrc = strops.confopt_bool(cr.get(crkey, 'all_source'))
                self.contestmap[i]['all_source'] = allsrc
            else:
                self.log.info('Ignoring duplicate contest: ' + repr(i))

            # check for invalid allsrc
            if self.contestmap[i]['all_source']:
                if (len(self.contestmap[i]['points']) > 1 
                    or len(self.contestmap[i]['bonuses']) > 1):
                    self.log.info('Ignoring extra points/bonus for '
                                  + 'all source contest ' + repr(i))
     
        # load points tally meta data
        tallylist = cr.get('event', 'tallys').translate(
                            strops.BIBLIST_TRANS).split()
        # append any 'missing' tallys from points data errors
        for i in tallyset:
            if i not in tallylist:
                self.log.debug('Adding missing tally to config: ' + repr(i))
                tallylist.append(i)
        # then scan for meta data
        for i in tallylist:
            if i not in self.tallys:
                self.tallys.append(i)
                self.tallymap[i] = {}
                self.points[i] = {}	 # redundant, but ok
                self.pointscb[i] = {}
                crkey = 'tally_' + i
                descr = ''
                if cr.has_option(crkey, 'descr'):
                    descr = cr.get(crkey, 'descr')
                self.tallymap[i]['descr'] = descr
                keepdnf = False
                if cr.has_option(crkey, 'keepdnf'):
                    keepdnf = strops.confopt_bool(cr.get(crkey, 'keepdnf'))
                self.tallymap[i]['keepdnf'] = keepdnf
            else:
                self.log.info('Ignoring duplicate points tally: ' + repr(i))


        starters = cr.get('event', 'startlist').split()
        dolaps = not strops.confopt_bool(cr.get('event', 'nolaps'))
        #if strops.confopt_bool(cr.get('event', 'resort')):
            #starters.sort(cmp=lambda x,y: cmp(int(x), int(y)))
        for r in starters:
            self.addrider(r)
            if cr.has_option('riders', r):
                nr = self.getrider(r)
                # bib = comment,in,laps,rftod,mbunch,rfseen...
                ril = csv.reader([cr.get('riders', r)]).next()
                lr = len(ril)
                if lr > 0:
                    nr[COL_COMMENT] = ril[0]
                if lr > 1:
                    nr[COL_INRACE] = strops.confopt_bool(ril[1])
                if lr > 2:
                    if ril[2].isdigit():
                        nr[COL_LAPS] = int(ril[2])
                    else:
                        nr[COL_LAPS] = 0
                if lr > 3:
                    nr[COL_RFTIME] = tod.str2tod(ril[3])
                if lr > 4:
                    nr[COL_MBUNCH] = tod.str2tod(ril[4])
                if lr > 5:
                    nr[COL_STOFT] = tod.str2tod(ril[5])
                if dolaps and lr > 6:
                    for i in range(6, lr):
                        laptod = tod.str2tod(ril[i])
                        if laptod is not None:
                            nr[COL_RFSEEN].append(laptod)
            # record any extra bonus/penalty to rider model
            if cr.has_option('stagebonus', r):
                nr[COL_BONUS] = tod.str2tod(cr.get('stagebonus', r))
            if cr.has_option('stagepenalty', r):
                nr[COL_PENALTY] = tod.str2tod(cr.get('stagepenalty', r))
        self.set_start(cr.get('event', 'start'), cr.get('event', 'lstart'))
        self.set_finish(cr.get('event', 'finish'))
        self.places = strops.reformat_placelist(cr.get('event', 'places'))
        self.comment = cr.get('event', 'comment').splitlines()
        if cr.get('event', 'finished') == 'Yes':
            self.set_finished()
        self.recalculate()

 

        for col in cr.get('event', 'hidecols').split():
            target = strops.confopt_posint(col)
            if target is not None:
                tc = self.view.get_column(target)
                if tc:
                    tc.set_visible(False)

        # After load complete - check config and report. This ensures
        # an error message is left on top of status stack. This is not
        # always a hard fail and the user should be left to determine
        # an appropriate outcome.
        eid = cr.get('event', 'id')
        if eid and eid != EVENT_ID:
            self.log.error('Event configuration mismatch: '
                           + repr(eid) + ' != ' + repr(EVENT_ID))
            self.readonly = True

    def get_ridercmdorder(self):
        """Return rider command list order."""
        ret = RIDER_COMMANDS_ORD[0:]
        for i in self.intermeds:
            ret.append(i)
        return ret

    def get_ridercmds(self):
        """Return a dict of rider bib commands for container ui."""
        ret = {}
        for k in RIDER_COMMANDS:
            ret[k] = RIDER_COMMANDS[k]
        for k in self.intermap:
            descr = k
            if self.intermap[k]['descr']:
                descr = self.intermap[k]['descr']
            ret[k] = descr
        return ret

    def get_startlist(self):
        """Return a list of all rider numbers 'registered' to event."""
        ret = []
        for r in self.riders:
            ret.append(r[COL_BIB])
        return ' '.join(ret)

    def get_starters(self):
        """Return a list of riders that 'started' the race."""
        ret = []
        for r in self.riders:
            if r[COL_COMMENT] != 'dns' or r[COL_INRACE]:
                ret.append(r[COL_BIB])
        return ' '.join(ret)

    def checkpoint_model(self):
        """Write the current rider model to an undo buffer."""
        self.undomod.clear()
        self.placeundo = self.places
        for r in self.riders:
            self.undomod.append(r)
        self.canundo = True

    def undo_riders(self):
        """Roll back rider model to last checkpoint."""
        if self.canundo:
            self.riders.clear()
            for r in self.undomod:
                self.riders.append(r)
            self.places = self.placeundo
            self.canundo = False
          
    def get_catlist(self):
        """Return the ordered list of categories as a string."""
        rvec = []
        for cat in self.cats:
            if cat != '':
                rvec.append(cat)
        return ','.join(rvec)

    def ridercat(self, cat):
        """Return a category from the result for the riders cat."""
        ret = ''        # default is the 'None' category - uncategorised
        checka = cat.upper()
        if checka in self.cats:
            ret = checka
        return ret

    def saveconfig(self):
        """Save event config to disk."""
        if self.readonly:
            self.log.error('Attempt to save readonly ob.')
            return
        cw = ConfigParser.ConfigParser()
        cw.add_section('event')
        if self.start is not None:
            cw.set('event', 'start', self.start.rawtime())
        if self.lstart is not None:
            cw.set('event', 'lstart', self.lstart.rawtime())
        if self.finish is not None:
            cw.set('event', 'finish', self.finish.rawtime())
        if self.timerstat == 'finished':
            cw.set('event', 'finished', 'Yes')
        else:
            cw.set('event', 'finished', 'No')
        cw.set('event', 'places', self.places)

        # save intermediate data
        cw.set('event', 'intermeds', ' '.join(self.intermeds))
        for i in self.intermeds:
            crkey = 'intermed_' + i
            cw.add_section(crkey)
            cw.set(crkey, 'descr', self.intermap[i]['descr'])
            cw.set(crkey, 'places', self.intermap[i]['places'])

        # save contest meta data
        cw.set('event', 'contests', ' '.join(self.contests))
        for i in self.contests:
            crkey = 'contest_' + i
            cw.add_section(crkey)
            cw.set(crkey, 'tally', self.contestmap[i]['tally'])
            cw.set(crkey, 'source', self.contestmap[i]['source'])
            cw.set(crkey, 'all_source', self.contestmap[i]['all_source'])
            blist = []
            for b in self.contestmap[i]['bonuses']:
                blist.append(b.rawtime(0))
            cw.set(crkey, 'bonuses', ' '.join(blist))
            plist = []
            for p in self.contestmap[i]['points']:
                plist.append(str(p))
            cw.set(crkey, 'points', ' '.join(plist))

        # save tally meta data
        cw.set('event', 'tallys', ' '.join(self.tallys))
        for i in self.tallys:
            crkey = 'tally_' + i
            cw.add_section(crkey)
            cw.set(crkey, 'descr', self.tallymap[i]['descr'])
            cw.set(crkey, 'keepdnf', self.tallymap[i]['keepdnf'])

        # save riders
        cw.set('event', 'startlist', self.get_startlist())    
        if self.autocats:
            cw.set('event', 'categories', 'AUTO')
        else:
            cw.set('event', 'categories', self.get_catlist())
        cw.set('event', 'comment', '\n'.join(self.comment))

        hides = []
        idx = 0
        for c in self.view.get_columns():
            if not c.get_visible():
                hides.append(str(idx))
            idx += 1
        if len(hides) > 0:
            cw.set('event', 'hidecols', ' '.join(hides))
            
        cw.add_section('riders')
        # sections for commissaire awarded bonus/penalty
        cw.add_section('stagebonus')
        cw.add_section('stagepenalty')
        for r in self.riders:
            bf = ''
            if r[COL_INRACE]:
                bf='True'
            rt = ''
            if r[COL_RFTIME] is not None:
                rt = r[COL_RFTIME].rawtime(2)
            mb = ''
            if r[COL_MBUNCH] is not None:
                mb = r[COL_MBUNCH].rawtime(0)
            sto = ''
            if r[COL_STOFT] is not None:
                sto = r[COL_STOFT].rawtime(2)
            # bib = comment,in,laps,rftod,mbunch,rfseen...
            slice = [r[COL_COMMENT], bf, r[COL_LAPS], rt, mb, sto]
            for t in r[COL_RFSEEN]:
                if t is not None:
                    slice.append(t.rawtime(2))
            cw.set('riders', r[COL_BIB],
                    ','.join(map(lambda i: str(i).replace(',', '\\,'), slice)))
            if r[COL_BONUS] is not None:
                cw.set('stagebonus', r[COL_BIB], r[COL_BONUS].rawtime(0))
            if r[COL_PENALTY] is not None:
                cw.set('stagepenalty', r[COL_BIB], r[COL_PENALTY].rawtime(0))
        cw.set('event', 'id', EVENT_ID)
        self.log.debug('Saving config to: ' + self.configpath)
        with open(self.configpath, 'wb') as f:
            cw.write(f)

    def show(self):
        """Show event container."""
        self.frame.show()

    def hide(self):
        """Hide event container."""
        self.frame.hide()

    def title_close_clicked_cb(self, button, entry=None):
        """Close and save the race."""
        self.meet.close_event()

    def set_titlestr(self, titlestr=None):
        """Update the title string label."""
        if titlestr is None or titlestr == '':
            titlestr = 'Mass Start Road Race'
        self.title_namestr.set_text(titlestr)

    def destroy(self):
        """Emit destroy signal to race handler."""
        self.context_menu.destroy()
        self.frame.destroy()

    def get_results(self):
        """Extract results in flat mode (not yet implemented)."""
        return []

    def points_report(self):
        """Return the points tally report."""
        ret = []
        aux = []
        cnt = 0
        for tally in self.tallys:
            sec = printing.section()
            sec.heading = self.tallymap[tally]['descr']
            sec.units = 'pt'
            tallytot = 0
            aux = []
            for bib in self.points[tally]:
                r = self.getrider(bib)
                tallytot += self.points[tally][bib]
                aux.append([self.points[tally][bib], strops.riderno_key(bib),
                           [None, r[COL_BIB], r[COL_NAMESTR],
                           strops.truncpad(str(self.pointscb[tally][bib]), 10,
                                                   elipsis=False),
                             None, str(self.points[tally][bib])],
                           self.pointscb[tally][bib]
                        ])
            aux.sort(sort_tally)
            for r in aux:
                sec.lines.append(r[2])
            sec.lines.append([None, None, None])
            sec.lines.append([None, None, 'Total Points: ' + str(tallytot)])
            ret.append(sec)
        
        if len(self.bonuses) > 0:
            sec = printing.section()
            sec.heading = 'Stage Bonuses'
            sec.units = 'sec'
            aux = []
            for bib in self.bonuses:
                r = self.getrider(bib)
                aux.append([self.bonuses[bib], strops.riderno_key(bib),
                       [None,r[COL_BIB],r[COL_NAMESTR],None,None,
                        str(int(self.bonuses[bib].truncate(0).timeval))],
                        0, 0, 0])
            aux.sort(sort_tally)
            for r in aux:
                sec.lines.append(r[2])
            ret.append(sec)            
        return ret

    def reorder_startlist(self):
        """Reorder riders for a startlist."""
        aux = []
        cnt = 0
        for r in self.riders:
            aux.append([cnt, r[COL_BIB]])
            cnt += 1
        if len(aux) > 1:
            aux.sort(key=key_bib)
            self.riders.reorder([a[0] for a in aux])
        return cnt

    def startlist_report(self):
        """Return a startlist report."""
        ret = []
        sec = printing.section()
        sec.heading = 'Startlist'
        cnt = self.reorder_startlist()
        for r in self.riders:
            ucicode = None
            dbr = self.meet.rdb.getrider(r[COL_BIB],self.series)
            if dbr is not None:
                ucicode = self.meet.rdb.getvalue(dbr,
                                         riderdb.COL_UCICODE)
            comment = None
            if r[COL_CAT] == 'U23':
                comment = '*' 
            if not r[COL_INRACE]:
                comment = r[COL_COMMENT]
                if comment == '':
                    comment = 'dnf'
            sec.lines.append([comment, r[COL_BIB], r[COL_NAMESTR],
                                       ucicode])
        ret.append(sec)
        if cnt > 1:
            sec = printing.section()
            sec.lines.append([None, None, 'Total riders: ' + str(cnt)])
            ret.append(sec)
        return ret

    def camera_report(self):
        """Return the judges (camera) report."""
        ret = []
        self.recalculate()	# fill places and bunch info
        pthresh = self.meet.timer.photothresh()
        totcount = 0
        dnscount = 0
        dnfcount = 0
        fincount = 0
        lcomment = ''
        if self.timerstat != 'idle':
            sec = printing.section()
            sec.colheader = ['','no','rider','lap','finish','rftime']
            first = True
            ft = None
            lt = None
            lrf = None
            for r in self.riders:
                totcount += 1
                marker = ' '
                es = ''
                bs = ''
                pset = False
                if r[COL_INRACE]:
                    comment = '___'
                    bt = self.vbunch(r[COL_CBUNCH], r[COL_MBUNCH])
                    if bt is not None:
                        fincount += 1
                        if r[COL_PLACE] != '':
                           comment = r[COL_PLACE] + '.'
                           pset = True

                        # format 'elapsed' rftime
                        if r[COL_RFTIME] is not None:
                            if not pset and lrf is not None:
                                if r[COL_RFTIME]-lrf < pthresh:
                                    comment = '[pf]__'
                                    prcom = sec.lines[-1][0]
                                    if prcom in ['___', '[pf]__']:
                                        sec.lines[-1][0] = '[pf]__'
                            if self.start is not None:
                                es =  (r[COL_RFTIME]-self.start).rawtime(2)
                            else:
                                es = r[COL_RFTIME].rawtime(2)
                            lrf = r[COL_RFTIME]
                        else:
                            lrf = None

                        # format 'finish' time
                        if ft is None:
                            ft = bt
                            bs = ft.rawtime(0)
                        else:
                            if bt > lt:
                                # New bunch
                                sec.lines.append([None, None, None])
                                bs = "+" + (bt - ft).rawtime(0)
                            else:
                                # Same time
                                pass
                        lt = bt
                    else:
                        if r[COL_COMMENT].strip() != '':
                            comment = r[COL_COMMENT].strip()

                    sec.lines.append([comment, r[riderdb.COL_BIB],
                                     r[COL_NAMESTR], str(r[COL_LAPS]),
                                     bs, es])
                else:
                    comment = r[COL_COMMENT]
                    if comment == '':
                        comment = 'dnf'
                    if comment != lcomment:
                        sec.lines.append([None, None, None])
                    lcomment = comment
                    if comment == 'dns':
                        dnscount += 1
                    else:
                        dnfcount += 1
                    sec.lines.append([comment, r[riderdb.COL_BIB],
                                     r[COL_NAMESTR], str(r[COL_LAPS]),
                                     None, None])
                first = False

            ret.append(sec)
            sec = printing.section()
            sec.lines.append([None,None,'Total riders: ' + str(totcount)])
            sec.lines.append([None,None,'Did not start: ' + str(dnscount)])
            sec.lines.append([None,None,'Did not finish: ' + str(dnfcount)])
            sec.lines.append([None,None,'Finishers: ' + str(fincount)])
            residual = totcount - (fincount + dnfcount + dnscount)
            if residual > 0:
                sec.lines.append([None,None,
                                  'Unaccounted for: ' + str(residual)])
            if len(sec.lines) > 0:
                ret.append(sec)
        else:
            # nothing to report...
            sec.append([None,None,'-- Not Started --'])
        return ret

    def catresult_report(self):
        """Return a categorised race result report."""
        self.recalculate()
        ret = []
        firstcat = True
        for cat in self.cats:
            if not firstcat:
                ret.append('')
            firstcat = False
            # Don't do this since # riders is not known in this hackmode
            #catstr = cat
            #if cat == '':
                #catstr = 'UNCATEGORISED'
            ret.append(cat)
            wt = None
            lt = None
            first = True
            lcomment = ''
            lp = None
            plcnt = 1
            for r in self.riders:
                rcat = self.ridercat(r[COL_CAT])
                if cat == rcat:
                    bstr = r[COL_BIB].rjust(3)
                    nstr = strops.truncpad(r[COL_NAMESTR], 47)
                    #cstr = strops.truncpad(r[COL_CAT], 8, align='l', elipsis=False)
                    pstr = '    '
                    tstr = '        '
                    dstr = '        '
                    if r[COL_INRACE]:
                        if r[COL_PLACE] != '':
                            if lp != r[COL_PLACE]:
                                lp = str(plcnt)
                        else:
                            lp = ''
                        plcnt += 1
                        pstr = '    '
                        if lp is not None and lp != '':
                            pstr = (lp + '.').ljust(4)
                        bt = self.vbunch(r[COL_CBUNCH], r[COL_MBUNCH])
                        if bt is not None:
                            tstr = bt.rawtime(0).rjust(8)
                            if bt != lt:
                                if not first:
                                    #ret.append('')	# new bunch
                                    dstr = ('+' + (bt - wt).rawtime(0)).rjust(8)
                            if wt is None:	# first finish time
                                wt = bt
                            first = False
                            # compute elapsed
                            et = bt
                            if r[COL_STOFT] is not None:	# apply a start offset
                                dofastest = True	# will need to report!
                                et = bt - r[COL_STOFT]
                                tstr = et.rawtime(0).rjust(8)
                            #if fastest is None or et < fastest:
                                #fastest = et
                                #fastestbib = r[COL_BIB]
                        lt = bt
                    else:
                        comment = r[COL_COMMENT]
                        if comment == '':
                            comment = 'dnf'	# override empty comment
                        if comment != lcomment:
                            ret.append('')
                        lcomment = comment
                        pstr = strops.truncpad(comment, 4)
                    ret.append(' '.join([pstr, bstr, nstr, tstr, dstr]))
                else:
                    # not in this category.
                    pass
             
        return ret

    def result_report(self):
        """Return a race result report."""
        self.recalculate()
        ret = []
        wt = None
        dofastest = False	# ftime for handicap races
        fastest = None
        fastestbib = None
        totcount = 0
        dnscount = 0
        dnfcount = 0
        fincount = 0
        lcomment = ''
        lt = None
        if self.timerstat != 'idle':
            sec = printing.section()
            
            first = True
            for r in self.riders:
                totcount += 1
                bstr = r[COL_BIB]
                nstr = r[COL_NAMESTR]
                cstr = r[COL_CAT]
                pstr = ''
                tstr = ''
                dstr = ''
                if r[COL_INRACE]:
                    if r[COL_PLACE] != '':
                        pstr = r[COL_PLACE] + '.'
                    bt = self.vbunch(r[COL_CBUNCH], r[COL_MBUNCH])
                    if bt is not None:
                        fincount += 1
                        tstr = bt.rawtime(0)
                        if bt != lt:
                            if not first:
                                sec.lines.append([None, None, None])# new bunch
                                dstr = '+' + (bt - wt).rawtime(0)
                        if wt is None:	# first finish time
                            wt = bt
                            first = False
                        # compute elapsed
                        et = bt
                        if r[COL_STOFT] is not None:	# apply a start offset
                            dofastest = True	# will need to report!
                            et = bt - r[COL_STOFT]
                            tstr = et.rawtime(0)
                        if fastest is None or et < fastest:
                            fastest = et
                            fastestbib = r[COL_BIB]
                    lt = bt
                else:
                    comment = r[COL_COMMENT]
                    if comment == '':
                        comment = 'dnf'
                    if comment != lcomment:
                        sec.lines.append([None, None, None])# new bunch
                    lcomment = comment
                    if comment == 'dns':
                        dnscount += 1
                    else:
                        dnfcount += 1
                    pstr = strops.truncpad(comment, 4)
                sec.lines.append([pstr, bstr, nstr, cstr, tstr, dstr])
            if wt is not None:
                sec.lines.append([None, None, None])
                sec.lines.append([None, None, 'Race time: ' + wt.rawtime(0)])
            if dofastest:
                ftr = self.getrider(fastestbib)
                fts = ''
                if ftr is not None:
                    fts = strops.truncpad(ftr[COL_NAMESTR] 
                                          + ' ' + ftr[COL_CAT], 44)
                sec.lines.append([None, None, 
                       'Fastest time: ' + fastest.rawtime(0)
                           + '  ' + fastestbib + '  ' + fts])
            sec.lines.append([None, None, None])
            sec.lines.append([None, None,
                             'Total riders: ' + str(totcount)])
            sec.lines.append([None, None, 
                             'Did not start: ' + str(dnscount)])
            sec.lines.append([None, None,
                             'Did not finish: ' + str(dnfcount)])
            sec.lines.append([None, None,
                             'Finishers: ' + str(fincount)])
            residual = totcount - (fincount + dnfcount + dnscount)
            if residual > 0:
                sec.lines.append([None, None,
                          'Unaccounted for: ' + str(residual)])
            #if len(self.comment) > 0:
                #ret.append('')
                #for cl in self.comment:
                    #ret.append('* ' + strops.truncpad(cl.strip(), 64))
            ret.append(sec)
        else:
            pass
            #ret.append('-- Not Started --')
        return ret

    def stat_but_clicked(self, button=None):
        """Deal with a status button click in the main container."""
        self.log.info('Stat button clicked.')

    def ctrl_change(self, acode='', entry=None):
        """Notify change in action combo."""
        if acode == 'fin':
            if entry is not None:
                entry.set_text(self.places)
        elif acode in self.intermeds:
            if entry is not None:
                entry.set_text(self.intermap[acode]['places'])
        else:
            if entry is not None:
                entry.set_text('')

    def race_ctrl(self, acode='', rlist=''):
        """Apply the selected action to the provided bib list."""
        self.checkpoint_model()
        if acode in self.intermeds:
            rlist = strops.reformat_placelist(rlist)
            if self.checkplaces(rlist, dnf=False):
                self.intermap[acode]['places'] = rlist
                self.recalculate()
                self.intsprint(acode, rlist)
                self.log.info('Intermediate ' + repr(acode) + ' == '
                               + repr(rlist))
            else:
                self.log.error('Intermediate ' + repr(acode) + ' not updated.')
                return False
        elif acode == 'fin':
            rlist = strops.reformat_placelist(rlist)
            if self.checkplaces(rlist):
                self.places = rlist
                self.recalculate()
                self.finsprint(rlist)
                return False
            else:
                self.log.error('Places not updated.')
                return False
        elif acode == 'dnf':
            self.dnfriders(strops.reformat_biblist(rlist))
            return True
        elif acode == 'dsq':
            self.dnfriders(strops.reformat_biblist(rlist), 'dsq')
            return True
        elif acode == 'hd':
            self.dnfriders(strops.reformat_biblist(rlist), 'hd')
            return True
        elif acode == 'dns':
            self.dnfriders(strops.reformat_biblist(rlist), 'dns')
            return True
        elif acode == 'ret':
            self.retriders(strops.reformat_biblist(rlist))
            return True
        elif acode == 'del':
            rlist = strops.reformat_riderlist(rlist,
                                              self.meet.rdb, self.series)
            for bib in rlist.split():
                self.delrider(bib)
            return True
        elif acode == 'add':
            rlist = strops.reformat_riderlist(rlist,
                                              self.meet.rdb, self.series)
            for bib in rlist.split():
                self.addrider(bib)
            return True
        elif acode == 'que':
            rlist = strops.reformat_biblist(rlist)
            if rlist != '':
                for bib in rlist.split():
                    self.query_rider(bib)
            return True
        elif acode == 'nst':
            self.new_start(rlist)
            return True
        elif acode == 'com':
            self.add_comment(rlist)
            return True
        else:
            self.log.error('Ignoring invalid action.')
        return False

    def add_comment(self, comment=''):
        """Append a race comment."""
        self.comment.append(comment.strip())
        self.log.info('Added race comment: ' + repr(comment))

    def query_rider(self, bib=None):
        """List info on selected rider in the scratchpad."""
        self.log.info('Query rider: ' + repr(bib))
        r = self.getrider(bib)
        if r is not None:
            ns = strops.truncpad(r[COL_NAMESTR] + ' ' + r[COL_CAT], 30)
            bs = ''
            bt = self.vbunch(r[COL_CBUNCH], r[COL_MBUNCH])
            if bt is not None:
                bs = bt.timestr(0)
            ps = r[COL_COMMENT]
            if r[COL_PLACE] != '':
                ps = strops.num2ord(r[COL_PLACE])
            self.log.info(' '.join([bib, ns, bs, ps]))
            lt = None
            if len(r[COL_RFSEEN]) > 0:
                for rft in r[COL_RFSEEN]:
                    nt = rft.truncate(0)
                    ns = rft.timestr(1)
                    ls = ''
                    if lt is not None:
                        ls = (nt - lt).timestr(0)
                    self.log.info(' '.join(['\t', ns, ls]))
                    lt = nt
            if r[COL_RFTIME] is not None:
                self.log.info(' '.join([' Finish:',
                                          r[COL_RFTIME].timestr(1)]))
        else:
            self.log.info(bib.ljust(4) + ' ' + 'Not in startlist.')

    def startlist_gen(self, cat=''):
        """Generator function to export a startlist."""
        mcat = self.ridercat(cat)
        self.reorder_startlist()
        for r in self.riders:
            if mcat == '' or mcat == self.ridercat(r[COL_CAT]):
                start = ''
                if r[COL_STOFT] is not None and r[COL_STOFT] != tod.ZERO:
                    start = r[COL_STOFT].rawtime(0)
                bib = r[COL_BIB]
                series = self.series
                name = r[COL_NAMESTR]
                cat = r[COL_CAT]
                firstxtra = ''
                lastxtra = ''
                clubxtra = ''
                dbr = self.meet.rdb.getrider(r[COL_BIB],self.series)
                if dbr is not None:
                    firstxtra = self.meet.rdb.getvalue(dbr,
                                         riderdb.COL_FIRST).capitalize()
                    lastxtra = self.meet.rdb.getvalue(dbr, 
                                         riderdb.COL_LAST).upper()
                    clubxtra = self.meet.rdb.getvalue(dbr, riderdb.COL_CLUB)
                yield [start, bib, series, name, cat,
                       firstxtra, lastxtra, clubxtra]

    def result_gen(self, cat=''):
        """Generator function to export a final result."""
        self.recalculate()	# fix up ordering of rows
        mcat = self.ridercat(cat)
        rcount = 0
        lrank = None
        lcrank = None
        for r in self.riders:
            if mcat == '' or mcat == self.ridercat(r[COL_CAT]):
                rcount += 1
                bib = r[COL_BIB]
                crank = None
                rank = None
                bonus = None
                ft = None
                if r[COL_INRACE]:
                    bt = self.vbunch(r[COL_CBUNCH], r[COL_MBUNCH])
                    ft = bt
                    if r[COL_STOFT] is not None:
                        ft = bt - r[COL_STOFT]

                if r[COL_PLACE].isdigit():
                    rank = int(r[COL_PLACE])
                    if rank != lrank: 
                        crank = rcount
                    else:
                        crank = lcrank
                    lcrank = crank
                    lrank = rank
                else:
                    crank = r[COL_COMMENT]
                # !! TODO: test
                bonus = None
                if bib in self.bonuses or r[COL_BONUS] is not None:
                    bonus = tod.ZERO
                    if bib in self.bonuses:
                        bonus += self.bonuses[bib]
                    if r[COL_BONUS] is not None:
                        bonus += r[COL_BONUS]
                penalty = None
                if r[COL_PENALTY] is not None:
                    penalty = r[COL_PENALTY]
                yield [crank, bib, ft, bonus, penalty]

    def clear_results(self):
        """Clear all data from event model."""
        self.resetplaces()
        self.places = ''
        self.log.debug('Cleared event result.')

    def getrider(self, bib):
        """Return reference to selected rider no."""
        ret = None
        for r in self.riders:
            if r[COL_BIB] == bib:
                ret = r
                break
        return ret

    def getiter(self, bib):
        """Return temporary iterator to model row."""
        i = self.riders.get_iter_first()
        while i is not None:
            if self.riders.get_value(i, COL_BIB) == bib:
                break
            i = self.riders.iter_next(i)
        return i

    def delrider(self, bib='', series=''):
        """Remove the specified rider from the model."""
        i = self.getiter(bib)
        if i is not None:
            self.riders.remove(i)

    def starttime(self, start=None, bib='', series=''):
        """Adjust start time for the rider."""
        if series == self.series:
            r = self.getrider(bib)
            if r is not None:
                r[COL_STOFT] = start

    def addrider(self, bib='', series=None):
        """Add specified rider to race model."""
        if series is not None and series != self.series:
            self.log.debug('Ignoring non-series starter: '
                            + repr(strops.bibser2bibstr(bib, series)))
            return None
        if bib == '' or self.getrider(bib) is None:
            nr = [bib, '', '', '', True, '', 0, None, None, None, None, 
                           None, None, []]
            dbr = self.meet.rdb.getrider(bib, self.series)
            if dbr is not None:
                nr[COL_NAMESTR] = strops.listname(
                      self.meet.rdb.getvalue(dbr, riderdb.COL_FIRST),
                      self.meet.rdb.getvalue(dbr, riderdb.COL_LAST),
                      self.meet.rdb.getvalue(dbr, riderdb.COL_CLUB))
                nr[COL_CAT] = self.meet.rdb.getvalue(dbr, riderdb.COL_CAT)
            return self.riders.append(nr)
        else:
            return None

    def resettimer(self):
        """Reset race timer."""
        self.set_finish()
        self.set_start()
        self.clear_results()
        self.timerstat = 'idle'
        uiutil.buttonchg(self.meet.stat_but, uiutil.bg_none, 'Idle')
        self.meet.stat_but.set_sensitive(True)
        self.set_elapsed()
        self.live_announce = True
        
    def armstart(self):
        """Process an armstart request."""
        if self.timerstat == 'idle':
            self.timerstat = 'armstart'
            uiutil.buttonchg(self.meet.stat_but,
                             uiutil.bg_armstart, 'Arm Start')
        elif self.timerstat == 'armstart':
            self.timerstat = 'idle'
            uiutil.buttonchg(self.meet.stat_but, uiutil.bg_none, 'Idle') 
        elif self.timerstat == 'ready':
            self.set_running()

    def armfinish(self):
        """Process an armfinish request."""
        if self.timerstat in ['ready', 'running', 'finished']:
            self.timerstat = 'armfinish'
            uiutil.buttonchg(self.meet.stat_but,
                             uiutil.bg_armfin, 'Arm Finish')
            self.meet.stat_but.set_sensitive(True)
        elif self.timerstat == 'armfinish':
            self.timerstat = 'running'
            uiutil.buttonchg(self.meet.stat_but,
                             uiutil.bg_none, 'Running')

    def last_rftime(self):
        """Find the last rider with a RFID finish time set."""
        ret = None
        for r in self.riders:
            if r[COL_RFTIME] is not None:
                ret = r[COL_BIB]
        return ret
        
    def key_event(self, widget, event):
        """Handle global key presses in event."""
        if event.type == gtk.gdk.KEY_PRESS:
            key = gtk.gdk.keyval_name(event.keyval) or 'None'
            if event.state & gtk.gdk.CONTROL_MASK:
                if key == key_abort:    # override ctrl+f5
                    self.resettimer()
                    return True
                elif key == key_confirm:  # do recalc with current model
                    lbib = self.last_rftime()
                    if lbib is not None:
                        self.fill_places_to(lbib) # fill all finished
                    return True
                elif key.upper() == key_undo:	# Undo model change if possible
                    self.undo_riders()
                    return True
            if key[0] == 'F':
                if key == key_armstart:
                    self.armstart()
                    return True
                elif key == key_announce:
                    if self.places:
                        self.finsprint(self.places)
                    else:
                        self.reannounce_lap()
                    return True
                elif key == key_armfinish:
                    self.armfinish()
                    return True
                elif key == key_raceover:
                    self.set_finished()
                    return True
                elif key == key_armlap:
                    #self.meet.scratch_clear()
                    if self.live_announce:
                        self.meet.announce_clear()
                        #self.meet.announce_title(self.title_namestr.get_text())
                        lapstr = self.lapentry.get_text()
                        if not lapstr:
                            lapstr = self.title_namestr.get_text()
                        self.meet.announce_title(lapstr)
                        self.meet.announce_gap()
                    self.last_scratch = self.scratch_start
                    self.scratch_start = None
                    self.scratch_tot = 0
                    self.scratch_map = {}
                    self.scratch_ord = []
                    return True
                elif key == key_markrider:
                    self.set_ridermark()
                    return True
                elif key == key_promoterider:
                    self.promoterider()
                    return True
        return False

    def set_ridermark(self):
        """Mark the current position in the result."""
        self.ridermark = None
        sel = self.view.get_selection().get_selected()
        if sel is not None:
            self.ridermark = sel[1]
        
    def promoterider(self):
        """Promote the selected rider to the current ridermark."""
        if self.ridermark is not None:
            sel = self.view.get_selection().get_selected()
            if sel is not None:
                i = sel[1]
                selbib = self.riders.get_value(i, COL_BIB)
                markbib = self.riders.get_value(self.ridermark, COL_BIB)
                self.log.debug('Promote rider ' + str(selbib) + ' to rider '
                                + str(markbib))
                # compare bunch times for both riders
                # if not same, clear the rftime and mbunch on the
                # promoted rider
                markbunch = self.vbunch(self.riders.get_value(self.ridermark,
                                                              COL_CBUNCH),
                                        self.riders.get_value(self.ridermark,
                                                              COL_MBUNCH))

                selbunch = self.vbunch(self.riders.get_value(i, COL_CBUNCH),
                                       self.riders.get_value(i, COL_MBUNCH))
                if selbunch != markbunch:
                    self.log.error('Selected bunch time not equal to marked bunch time - ignored.')
                    self.log.error('\tSelbunch: ' + selbunch.rawtime(0))
                    self.log.error('\tMarkbunch: ' + markbunch.rawtime(0))
                    # BUT ALLOW IT AND FORCE MANUAL REPAIR
                    # problem here is a late read of close riders - leading
                    # to a new bunch on the middle rider. After clearing out
                    # rftime, rider may re-cross the line! Need a flag for
                    # cleared but finished... perhaps 'False' ?
                    #self.log.info('Cleared incorrect finish time for rider '
                                  #+ self.riders.get_value(i, COL_BIB))
                    #self.riders.set_value(i, COL_RFTIME, None)
                    #self.riders.set_value(i, COL_MBUNCH, None)

                self.riders.move_before(i, self.ridermark)
                self.fill_places_to(self.riders.get_value(self.ridermark,
                                      COL_BIB))

    def dnfriders(self, biblist='', code='dnf'):
        """Remove each rider from the race with supplied code."""
        recalc = False
        for bib in biblist.split():
            r = self.getrider(bib)
            if r is not None:
                r[COL_INRACE] = False
                r[COL_COMMENT] = code
                recalc = True
                self.log.info('Rider ' + str(bib) 
                               + ' did not finish with code: ' + code)
            else:
                self.log.warn('Unregistered Rider ' + str(bib) + ' unchanged.')
        if recalc:
            self.recalculate()
        return False
  
    def retriders(self, biblist=''):
        """Return all listed riders to the race."""
        recalc = False
        for bib in biblist.split():
            r = self.getrider(bib)
            if r is not None:
                r[COL_INRACE] = True
                r[COL_COMMENT] = ''
                recalc = True
                self.log.info('Rider ' + str(bib) 
                               + ' returned to race.')
            else:
                self.log.warn('Unregistered Rider ' + str(bib) + ' unchanged.')
        if recalc:
            self.recalculate()
        return False
  
    def shutdown(self, win=None, msg='Race Sutdown'):
        """Close event."""
        self.log.debug('Event shutdown: ' + msg)
        if not self.readonly:
            self.saveconfig()
        self.meet.edb.editevent(self.event, winopen=False)
        self.winopen = False

    def starttrig(self, e):
        """Process a 'start' trigger signal."""
        if self.timerstat == 'armstart':
            self.set_start(e, tod.tod('now'))
        return False

    def rfidtrig(self, e):
        """Process rfid event."""
        if e.refid == '':	# got a trigger
            return self.starttrig(e)

        # else assume this is a passing
        r = self.meet.rdb.getrefid(e.refid)
        if r is None:
            self.log.info('Unknown tag: ' + e.refid + '@' + e.rawtime(2))
            return False

        bib = self.meet.rdb.getvalue(r, riderdb.COL_BIB)
        ser = self.meet.rdb.getvalue(r, riderdb.COL_SERIES)
        if ser != self.series:
            self.log.error('Ignored non-series rider: ' + bib + '.' + ser)
            return False

        # at this point should always have a valid source rider vector
        lr = self.getrider(bib)
        if lr is None:
            self.log.warn('Ignoring non starter: ' + bib
                          + ' @ ' + e.rawtime(2))
            return False

        if not lr[COL_INRACE]:
            self.log.warn('Withdrawn rider: ' + lr[COL_BIB]
                          + ' @ ' + e.rawtime(2))
            # but continue anyway just in case it was not correct?
        else:
            self.log.info('Saw: ' + bib + ' @ ' + e.rawtime(2))

        # check run state
        if self.timerstat in ['idle', 'ready', 'armstart']:
            return False

        # save RF ToD into 'passing' vector and log
        lr[COL_RFSEEN].append(e)

        if self.timerstat == 'armfinish':
            if self.finish is None:
                if self.live_announce:
                    self.meet.announce_title('Finish')
                self.set_finish(e)
                self.set_elapsed()
            if lr[COL_RFTIME] is None:
                lr[COL_LAPS] += 1
                lr[COL_RFTIME] = e
                self.__dorecalc = True	# cleared in timeout, from same thread
                if lr[COL_INRACE]:
                    if self.scratch_start is None:
                        self.scratch_start = e
                    self.announce_rider('', bib, lr[COL_NAMESTR],
                                    lr[COL_CAT], e)
            else:
                self.log.error('Duplicate finish rider = ' + bib
                                  + ' @ ' + str(e))
        elif self.timerstat in 'running':
            lr[COL_LAPS] += 1
            if lr[COL_INRACE]:
                if self.scratch_start is None:
                    self.scratch_start = e
                self.announce_rider('', bib, lr[COL_NAMESTR],
                                lr[COL_CAT], e)
        return False	# called from glib_idle_add

    def announce_rider(self, place, bib, namestr, cat, rftime):
        """Log a rider in the lap and emit to announce."""
        if bib not in self.scratch_map:
            self.scratch_map[bib] = rftime
            self.scratch_ord.append(bib)
        if self.live_announce:
            glib.idle_add(self.meet.announce_rider, [place,bib,namestr,
                                     cat,rftime.rawtime()])

    def finsprint(self, places):
        """Display a final sprint 'provisional' result."""

        self.live_announce = False
        self.meet.announce_clear()
        ## needs better annotation?
        #self.meet.announce_title(self.title_namestr.get_text()
        self.meet.announce_title('Provisional Result')
        placeset = set()
        idx = 0
        st = tod.tod('0')
        if self.start is not None:
            st = self.start
        # result is sent in final units, not absolutes
        self.meet.announce_start()
        wt = None
        lb = None
        for placegroup in places.split():
            curplace = idx + 1
            for bib in placegroup.split('-'):
                if bib not in placeset:
                    placeset.add(bib)
                    r = self.getrider(bib)
                    if r is not None:
                        ft = self.vbunch(r[COL_CBUNCH],
                                         r[COL_MBUNCH])
                        fs = ''
                        if ft is not None:
                            #if ft != lb:
                                #fs = ft.rawtime()
                            #else:
                                #if r[COL_RFTIME] is not None:
                                    #fs = (r[COL_RFTIME]-st).rawtime()
                                #else:
                                    #fs = ft.rawtime()
                            # temp -> just use the no-blob style to correct
                            fs = ft.rawtime()
                            if wt is None:
                                wt = ft
                            lb = ft
                        glib.idle_add(self.meet.announce_rider,
                                                 [r[COL_PLACE]+'.',
                                                 bib,
                                                 r[COL_NAMESTR],
                                                 r[COL_CAT], fs])
                    idx += 1
        if wt is not None:
            # set winner's time
            self.meet.announce_time(wt.rawtime(0))

    def intsprint(self, acode='', places=''):
        """Display an intermediate sprint 'provisional' result."""

        ## TODO : Fix offset time calcs - too many off by ones
        if acode not in self.intermeds:
            self.log.debug('Attempt to display non-existent intermediate: '
                              + repr(acode))
            return
        descr = acode
        if self.intermap[acode]['descr']:
            descr = self.intermap[acode]['descr']
        self.live_announce = False
        self.meet.announce_clear()
        self.meet.announce_title(descr)
        placeset = set()
        idx = 0
        for placegroup in places.split():
            curplace = idx + 1
            for bib in placegroup.split('-'):
                if bib not in placeset:
                    placeset.add(bib)
                    r = self.getrider(bib)
                    if r is not None:
                        glib.idle_add(self.meet.announce_rider,
                                                [str(curplace)+'.',
                                                 bib,
                                                 r[COL_NAMESTR],
                                                 r[COL_CAT], ''])
                    idx += 1
                else:
                    self.log.warn('Duplicate no. = ' + str(bib) + ' in places.')

        glib.timeout_add_seconds(15, self.reannounce_lap)

    def reannounce_lap(self):
        self.live_announce = False
        self.meet.announce_clear()
        if self.timerstat == 'armfinish':
            self.meet.announce_title('Finish')
        else:
            self.meet.announce_title(self.title_namestr.get_text())
        if self.last_scratch is not None:
            self.meet.announce_start(self.last_scratch)
        for bib in self.scratch_ord:
            r = self.getrider(bib)
            if r is not None:
                glib.idle_add(self.meet.announce_rider,
                                        ['',bib,r[COL_NAMESTR],r[COL_CAT],
                                         self.scratch_map[bib].rawtime()])
        self.live_announce = True
        return False

    def timeout(self):
        """Update elapsed time and recalculate if required."""
        if not self.winopen:
            return False
        #if self.finish is None and self.start is not None:
        if self.start is not None:
            self.set_elapsed()
        if self.__dorecalc:
            self.__dorecalc = False
            self.recalculate()
        return True

    def set_start(self, start='', lstart=''):
        """Set the start time."""
        if type(start) is tod.tod:
            self.start = start
        else:
            self.start = tod.str2tod(start)
        if type(lstart) is tod.tod:
            self.lstart = lstart
        else:
            self.lstart = tod.str2tod(lstart)
            if self.lstart is None:
                self.lstart = self.start
        if self.start is not None:
            self.meet.announce_start(self.start)
            self.last_scratch = self.start
            if self.finish is None:
                self.set_ready()

    def set_finish(self, finish=''):
        """Set the finish time."""
        if type(finish) is tod.tod:
            self.finish = finish
        else:
            self.finish = tod.str2tod(finish)
        if self.finish is None:
            if self.start is not None:
                self.set_ready()
        else:
            if self.start is None:
                self.set_start('0')

    def get_elapsed(self):
        """Hack mode - always returns time from start."""
        ret = None
        if self.lstart is not None and self.timerstat != 'finished':
            ret = (tod.tod('now') - self.lstart).truncate(0)
        return ret

    def set_elapsed(self):
        """Update the elapsed time field."""
        if self.start is not None and self.finish is not None:
            elap = (self.finish - self.start).truncate(0)
            self.time_lbl.set_text(elap.rawtime(0))
            # finish gap time
            if self.timerstat != 'finished':
                self.meet.announce_gap('+' + (tod.tod('now') - self.finish.truncate(0)).rawtime(0))
                if self.meet.distance is not None:
                    self.meet.announce_avg(elap.speedstr(1000.0*self.meet.distance))
            else:
                self.meet.announce_gap()
        elif self.start is not None:    # Note: uses 'local start' for RT
            self.time_lbl.set_text((tod.tod('now') - self.lstart).rawtime(0))
            # lap gap time
            if self.scratch_start is not None:
                self.meet.announce_gap('+' + (tod.tod('now') - self.scratch_start.truncate(0)).rawtime(0))
        elif self.timerstat == 'armstart':
            self.time_lbl.set_text(tod.tod(0).rawtime(0))
            self.meet.announce_gap()
        else:
            self.time_lbl.set_text('')
            self.meet.announce_gap()

        self.meet.announce_time(self.time_lbl.get_text())

    def set_ready(self):
        """Update event status to ready to go."""
        self.timerstat = 'ready'
        uiutil.buttonchg(self.meet.stat_but, uiutil.bg_armint, 'Ready')

    def set_running(self):
        """Update event status to running."""
        self.timerstat = 'running'
        uiutil.buttonchg(self.meet.stat_but, uiutil.bg_none, 'Running')

    def set_finished(self):
        """Update event status to finished."""
        self.timerstat = 'finished'
        uiutil.buttonchg(self.meet.stat_but, uiutil.bg_none, 'Finished')
        self.meet.stat_but.set_sensitive(False)
        if self.finish is None:
            self.set_finish(tod.tod('now'))
        self.set_elapsed()

    def title_place_xfer_clicked_cb(self, button, data=None):
        """Transfer current rider list order to place entry."""
        nplaces = ''
        lplace = None
        for r in self.riders:
            if r[COL_INRACE] and r[COL_PLACE] != '':
                if lplace == r[COL_PLACE] and r[COL_PLACE] != '':
                    nplaces += '-' + r[COL_BIB] # dead heat riders
                else:
                    nplaces += ' ' + r[COL_BIB]
                    lplace = r[COL_PLACE]
        self.places = strops.reformat_placelist(nplaces)
        self.meet.action_combo.set_active(5)
        self.meet.action_entry.set_text(self.places)
        
    def fill_places_to(self, bib=None):
        """Fill in finish places up to the nominated bib."""
        if self.places.find('-') > 0:
            self.log.warn('Fill places with dead heat not yet implemented.')
            return
        oplaces = self.places.split()	# only patch if no dead heats
        nplaces = []
        for r in self.riders:
            if r[COL_INRACE]:
                if r[COL_BIB] in oplaces:
                    oplaces.remove(r[COL_BIB])	# remove from old list
                nplaces.append(r[COL_BIB])      # add to new list
            else:
                if r[COL_BIB] in oplaces:
                    oplaces.remove(r[COL_BIB])	# strip out DNFed riders
            if r[COL_BIB] == bib:		# break after to get sel rider
                break
        nplaces.extend(oplaces)
        self.checkpoint_model()
        self.places = ' '.join(nplaces)
        self.recalculate()

    def info_time_edit_clicked_cb(self, button, data=None):
        """Run an edit times dialog to update race time."""
        st = ''
        if self.start is not None:
            st = self.start.rawtime(2)
        ft = ''
        if self.finish is not None:
            ft = self.finish.rawtime(2)
        ret = uiutil.edit_times_dlg(self.meet.window, stxt=st, ftxt=ft)
        if ret[0] == 1:
            self.set_start(ret[1])
            self.set_finish(ret[2])
            self.log.info('Adjusted race times.')

    def editcol_cb(self, cell, path, new_text, col):
        """Edit column callback."""
        new_text = new_text.strip()
        self.riders[path][col] = new_text

    def editlap_cb(self, cell, path, new_text, col):
        """Edit the lap field if valid."""
        new_text = new_text.strip()
        if new_text.isdigit():
            self.riders[path][col] = int(new_text)
        else:
            self.log.error('Invalid lap count.')

    def resetplaces(self):
        """Clear places off all riders."""
        for r in self.riders:
            r[COL_PLACE] = ''
        self.bonuses = {}	# bonuses are global to stage
        for c in self.tallys:	# points are grouped by tally
            self.points[c] = {}
            self.pointscb[c] = {}
            
    def sortrough(self, x, y):
        # aux cols: ind, bib, in, place, rftime, laps
        #             0    1   2      3       4     5
        if x[2] != y[2]:		# in the race?
            if x[2]:
                return -1
            else:
                return 1
        else:
            if x[3] != y[3]:		# places not same?
                if y[3] == '':
                    return -1
                elif x[3] == '':
                    return 1
                if int(x[3]) < int(y[3]):
                    return -1
                else:
                    return 1
            else:
                if x[4] == y[4]:	# same time?
                    if x[5] == y[5]:	# same laps?
                        return 0
                    else:
                        if x[5] > y[5]:
                            return -1
                        else:
                            return 1
                else:
                    if y[4] is None:
                        return -1
                    elif x[4] is None:
                        return 1
                    elif x[4] < y[4]:
                        return -1
                    else:
                        return 1
        return 0

    # do final sort on manual places then manual bunch entries
    def sortvbunch(self, x, y):
        # aux cols: ind, bib, in, place, vbunch, comment
        #             0    1   2      3       4        5
        if x[2] != y[2]:		# in the race?
            if x[2]:			# return bool comparison equiv
                return -1
            else:
                return 1
        else:
            if x[2]:			# in the race...
                if x[3] != y[3]:		# places not same?
                    if y[3] == '':
                        return -1
                    elif x[3] == '':
                        return 1
                    if int(x[3]) < int(y[3]):
                        return -1
                    else:
                        return 1
                else:
                    if x[4] == y[4]:	# same time?
                        return 0
                    else:
                        if y[4] is None:
                            return -1
                        elif x[4] is None:
                            return 1
                        elif x[4] < y[4]:
                            return -1
                        else:
                            return 1
            else:			# not in the race
                if x[5] != y[5]:
                    return strops.cmp_dnf(x[5], y[5]) # sort by code
                else:
                    return cmp(strops.riderno_key(x[1]), 
                               strops.riderno_key(y[1])) # sort on no
        return 0

    def vbunch(self, cbunch=None, mbunch=None):
        """Switch to return best choice bunch time."""
        ret = None
        if mbunch is not None:
            ret = mbunch
        elif cbunch is not None:
            ret = cbunch
        return ret

    def showstart_cb(self, col, cr, model, iter, data=None):
        """Draw start time offset in rider view."""
        st = model.get_value(iter, COL_STOFT)
        otxt = ''
        if st is not None:
            otxt = st.rawtime(0)
        cr.set_property('text', otxt)

    def showbunch_cb(self, col, cr, model, iter, data=None):
        """Update bunch time on rider view."""
        cb = model.get_value(iter, COL_CBUNCH)
        mb = model.get_value(iter, COL_MBUNCH)
        if mb is not None:
            cr.set_property('text', mb.rawtime(0))
            cr.set_property('style', pango.STYLE_OBLIQUE)
        else:
            cr.set_property('style', pango.STYLE_NORMAL)
            if cb is not None:
                cr.set_property('text', cb.rawtime(0))
            else:
                cr.set_property('text', '')

    def editstart_cb(self, cell, path, new_text, col=None):
        """Edit start time on rider view."""
        newst = tod.str2tod(new_text)
        if newst:
            newst = newst.truncate(0)
        self.riders[path][COL_STOFT] = newst

    def editbunch_cb(self, cell, path, new_text, col=None):
        """Edit bunch time on rider view."""
        # NOTE: This is the cascading bunch time editor, 
        new_text = new_text.strip()
        dorecalc = False
        if new_text == '':	# user request to clear RFTIME?
            self.riders[path][COL_RFTIME] = None
            self.riders[path][COL_MBUNCH] = None
            self.riders[path][COL_CBUNCH] = None
            dorecalc = True
        else:
            # get 'current bunch time'
            omb = self.vbunch(self.riders[path][COL_CBUNCH],
                              self.riders[path][COL_MBUNCH])

            # assign new bunch time
            nmb = tod.str2tod(new_text)
            if self.riders[path][COL_MBUNCH] != nmb:
                self.riders[path][COL_MBUNCH] = nmb
                dorecalc = True
            if nmb is not None:
                i = int(path)+1
                tl = len (self.riders)
                # until next rider has mbunch set OR place clear assign new bt
                while i < tl:
                    ivb = self.vbunch(self.riders[i][COL_CBUNCH], 
                                      self.riders[i][COL_MBUNCH])
                    if (self.riders[i][COL_PLACE] != ''
                          and (ivb is None
                              or ivb == omb)):
                        self.riders[i][COL_MBUNCH] = nmb
                        dorecalc = True
                    else:
                        break
                    i += 1
        if dorecalc:
            self.recalculate()
    
    def checkplaces(self, rlist='', dnf=True):
        """Check the proposed places against current race model."""
        ret = True
        placeset = set()
        for no in strops.reformat_biblist(rlist).split():
            if no != 'x':
                # repetition? - already in place set?
                if no in placeset:
                    self.log.error('Duplicate no in places: ' + repr(no))
                    ret = False
                placeset.add(no)
                # rider in the model?
                lr = self.getrider(no)
                if lr is None:
                    self.log.error('Non-starter in places: ' + repr(no))
                    ret = False
                else:
                    # rider still in the race?
                    if not lr[COL_INRACE]:
                        self.log.info('DNF/DNS rider in places: ' + repr(no))
                        if dnf:
                            ret = False
            else:
                # placeholder needs to be filled in later or left off
                self.log.info('Placeholder in places.')
        return ret

    def recalculate(self):
        """Recalculator, acquires lock and then continues."""
        if not self.recalclock.acquire(False):
            self.log.warn('Recalculate already in progress.')
            return None	# allow only one entry
        try:
            self.__recalc()
        finally:
            self.recalclock.release()

    def assign_finish(self):
        """Transfer finish line places into rider model."""
        placestr = self.places
        placeset = set()
        idx = 0
        for placegroup in placestr.split():
            curplace = idx + 1
            for bib in placegroup.split('-'):
                if bib not in placeset:
                    placeset.add(bib)
                    r = self.getrider(bib)
                    if r is None:
                        self.addrider(bib)
                        r = self.getrider(bib)
                    if r[COL_INRACE]:
                        idx += 1
                        r[COL_PLACE] = str(curplace)
                    else:
                        self.log.warn('DNF Rider ' + str(bib)
                                       + ' in finish places.')
                else:
                    self.log.warn('Duplicate no. = ' + str(bib) +
                                   ' in finish places.')

    def assign_places(self, contest):
        """Transfer points and bonuses into the named contest."""
        # fetch context meta infos
        src = self.contestmap[contest]['source']
        if src not in RESERVED_SOURCES and src not in self.intermeds:
            self.log.info('Invalid intermediate source: ' + repr(src)
                           + ' in contest: ' + repr(contest))
            return
        tally = self.contestmap[contest]['tally']
        bonuses = self.contestmap[contest]['bonuses']
        points = self.contestmap[contest]['points']
        allsrc = self.contestmap[contest]['all_source']
        allpts = 0
        allbonus = tod.ZERO
        if allsrc:
            if len(points) > 0:
                allpts = points[0]
            if len(bonuses) > 0:
                allbonus = bonuses[0]
        placestr = ''
        if src == 'fin':
            placestr = self.places
        elif src == 'reg':
            placestr = self.get_startlist()
        elif src == 'start':
            placestr = self.get_starters()
        else:
            placestr = self.intermap[src]['places']
        placeset = set()
        idx = 0
        for placegroup in placestr.split():
            curplace = idx + 1
            for bib in placegroup.split('-'):
                if bib not in placeset:
                    placeset.add(bib)
                    r = self.getrider(bib)
                    if r is None:
                        self.addrider(bib)
                        r = self.getrider(bib)
                    idx += 1
                    if allsrc:	# all listed places get same pts/bonus..
                        if allbonus is not tod.ZERO:
                            if bib in self.bonuses:
                                self.bonuses[bib] += allbonus 
                            else:
                                self.bonuses[bib] = allbonus
                        if allpts != 0:
                            if bib in self.points[tally]:
                                self.points[tally][bib] += allpts
                            else:
                                self.points[tally][bib] = allpts
                                self.pointscb[tally][bib] = strops.countback()
                            # No countback for all_source entries
                    else:	# points/bonus as per config
                        if len(bonuses) >= curplace:	# bonus is vector
                            if bib in self.bonuses:
                                self.bonuses[bib] += bonuses[curplace-1]
                            else:
                                self.bonuses[bib] = bonuses[curplace-1]
                        if tally:
                            if len(points) >= curplace: # points vector
                                if bib in self.points[tally]:
                                    self.points[tally][bib] += points[curplace-1]
                                else:
                                    self.points[tally][bib] = points[curplace-1]
                            if bib not in self.pointscb[tally]:
                                self.pointscb[tally][bib] = strops.countback()
                            self.pointscb[tally][bib][curplace] += 1
                else:
                    self.log.warn('Duplicate no. = ' + str(bib) + ' in '
                                    + repr(contest) + ' places.')

    def __recalc(self):
        """Internal 'protected' recalculate function."""
        self.log.debug('Recalculate model.')
        # pass one: clear off old places and bonuses
        self.resetplaces()

        # pass two: assign places
        for c in self.contests:
            self.assign_places(c)
        self.assign_finish()

        # pass three: do rough sort on in, place, rftime -> existing
        auxtbl = []
        idx = 0
        for r in self.riders:
            # aux cols: ind, bib, in, place, rftime, laps
            auxtbl.append([idx, r[COL_BIB], r[COL_INRACE], r[COL_PLACE],
                           r[COL_RFTIME], r[COL_LAPS]])
            idx += 1
        if len(auxtbl) > 1:
            auxtbl.sort(self.sortrough)
            self.riders.reorder([a[0] for a in auxtbl])

        # pass four: compute cbunch values on auto time gaps and manual inputs
        #            At this point all riders are assumed to be in finish order
        ft = None	# the finish or first bunch time
        lt = None	# the rftime of last competitor across line
        bt = None	# the 'current' bunch time
        if self.start is not None:
            for r in self.riders:
                if r[COL_INRACE]:
                    if r[COL_MBUNCH] is not None:
                        bt = r[COL_MBUNCH]	# override with manual bunch
                        r[COL_CBUNCH] = bt
                    elif r[COL_RFTIME] is not None:
                        # establish elapsed, but allow subsequent override
                        et = r[COL_RFTIME] - self.start
    
                        # establish bunch time
                        if ft is None:
                            ft = et.truncate(0)	# compute first time
                            bt = ft
                        else:
                            if et < lt or et - lt < tod.tod('1.12'): #NTG!
                                # same time
                                pass
                            else:
                                bt = et.truncate(0)

                        # assign and continue
                        r[COL_CBUNCH] = bt
                        lt = et
                    else:
                        # empty rftime with non-empty rank implies no time gap
                        if r[COL_PLACE] != '':
                            r[COL_CBUNCH] = bt	# use current bunch time
                        else: r[COL_CBUNCH] = None
                
        # pass five: resort on in,vbunch (todo -> check if place cmp reqd)
        #            at this point all riders will have valid bunch time
        auxtbl = []
        idx = 0
        for r in self.riders:
            # aux cols: ind, bib, in, place, vbunch
            auxtbl.append([idx, r[COL_BIB], r[COL_INRACE], r[COL_PLACE],
                           self.vbunch(r[COL_CBUNCH], r[COL_MBUNCH]),
                           r[COL_COMMENT]])
            idx += 1
        if len(auxtbl) > 1:
            auxtbl.sort(self.sortvbunch)
            self.riders.reorder([a[0] for a in auxtbl])
        return False	# allow idle add

    def new_start_trigger(self, rfid):
        """Collect a RFID trigger signal and apply it to the model."""
        if self.newstartdlg is not None and self.newstartent is not None:
            nt = tod.tod('now')
            et = tod.str2tod(self.newstartent.get_text())
            if et is not None:
                st = rfid - et
                lt = nt - et
                self.set_start(st, lt)
                self.newstartdlg.response(1)
                self.newstartdlg = None	# try to ignore the 'up' impulse
            else:
                self.log.warn('Invalid elapsed time: Start not updated.')
        return False

    def new_start_trig(self, button, entry=None):
        """Use the 'now' time to update start offset."""
        self.meet.timer.trig(refid='0')

    def verify_timent(self, entry, data=None):
        et = tod.str2tod(entry.get_text())
        if et is not None:
            entry.set_text(et.rawtime())
        else:
            self.log.info('Invalid elasped time.')

    def elapsed_dlg(self, addriders=''):
        """Run a 'new start' dialog."""
        if self.timerstat == 'armstart':
            self.log.error('Start is armed, unarm to add new start time.')
            return

        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, 'new_start.ui'))
        dlg = b.get_object('newstart')
        dlg.set_transient_for(self.meet.window)
        self.newstartdlg = dlg

        timent = b.get_object('time_entry')
        self.newstartent = timent
        timent.connect('activate', self.verify_timent)

        self.meet.timer.setcb(self.new_start_trigger)
        b.get_object('now_button').connect('clicked', self.new_start_trig)

        response = dlg.run()
        self.newstartdlg = None
        if response == 1:       # id 1 set in glade for "Apply"
            self.log.info('Start time updated: ' + self.start.rawtime(2))
        else:
            self.log.info('Set elapsed time cancelled.')
        self.meet.timer.setcb(self.rfidtrig)
        dlg.destroy()

    def time_context_menu(self, widget, event, data=None):
        """Popup menu for result list."""
        self.context_menu.popup(None, None, None, event.button,
                                event.time, selpath)

    def treerow_selected(self, treeview, path, view_column, data=None):
        """Select row, use to confirm places to."""
        if self.timerstat not in ['idle', 'armstart', 'armfinish']:
            self.fill_places_to(self.riders[path][COL_BIB])

    def treeview_button_press(self, treeview, event):
        """Set callback for mouse press on model view."""
        if event.button == 3:
            pathinfo = treeview.get_path_at_pos(int(event.x), int(event.y))
            if pathinfo is not None:
                path, col, cellx, celly = pathinfo
                treeview.grab_focus()
                treeview.set_cursor(path, col, 0)
                self.context_menu.popup(None, None, None,
                                        event.button, event.time)
                return True
        return False

    def rms_context_edit_activate_cb(self, menuitem, data=None):
        """Edit rider."""
        self.log.debug('rms_context_edit_activate_cb not implemented.')

    def rms_context_clear_activate_cb(self, menuitem, data=None):
        """Clear finish time for selected rider."""
        sel = self.view.get_selection().get_selected()
        if sel is not None:
            self.riders.set_value(sel[1], COL_RFTIME, None)
            self.riders.set_value(sel[1], COL_MBUNCH, None)
            self.recalculate()

    def rms_context_del_activate_cb(self, menuitem, data=None):
        """Remove selected rider from event."""
        sel = self.view.get_selection().get_selected()
        bib = None
        if sel is not None:
            bib = self.riders.get_value(sel[1], COL_BIB)
            self.delrider(bib)

    def rms_context_refinish_activate_cb(self, menuitem, data=None):
        """Try to automatically re-finish rider from last passing."""
        sel = self.view.get_selection().get_selected()
        if sel is not None:
            splits = self.riders.get_value(sel[1], COL_RFSEEN)
            if splits is not None and len(splits) > 0:
                self.riders.set_value(sel[1], COL_RFTIME, splits[-1])
                self.recalculate()

    def __init__(self, meet, event, ui=True):
        self.meet = meet
        self.event = event      # Note: now a treerowref
        self.evno = meet.edb.getvalue(event, eventdb.COL_EVNO)
        self.series = meet.edb.getvalue(event, eventdb.COL_SERIES)
        self.configpath = meet.event_configfile(self.evno)

        self.log = logging.getLogger('roadrace')
        self.log.setLevel(logging.DEBUG)
        self.log.debug('opening event: ' + str(self.evno))

        self.recalclock = threading.Lock()
        self.__dorecalc = False

        # race property attributes

        # race run time attributes
        self.readonly = not ui
        self.start = None
        self.lstart = None
        self.finish = None
        self.winopen = True
        self.timerstat = 'idle'
        self.places = ''
        self.comment = []
        self.ridermark = None
        self.cats = []
        self.autocats = False
        self.bonuses = {}
        self.points = {}
        self.pointscb = {}

        # intermediates
        self.intermeds = []	# sorted list of intermediate keys
        self.intermap = {}	# map of intermediate keys to results
        self.contests = []	# sorted list of contests
        self.contestmap = {}	# map of contest keys
        self.tallys = []	# sorted list of points tallys
        self.tallymap = {}	# map of tally keys

        # Scratch pad status variables - check if needed?
        self.last_scratch = None
        self.scratch_start = None
        self.scratch_last = None
        self.scratch_count = 0
        self.scratch_tot = 0
 
        # lap tracking
        self.scratch_map = {}
        self.scratch_ord = []
        self.live_announce = True

        # new start dialog
        self.newstartent = None
        self.newstartdlg = None

        self.riders = gtk.ListStore(gobject.TYPE_STRING, # BIB = 0
                                    gobject.TYPE_STRING, # NAMESTR = 1
                                    gobject.TYPE_STRING, # CAT = 2
                                    gobject.TYPE_STRING, # COMMENT = 3
                                    gobject.TYPE_BOOLEAN, # INRACE = 4
                                    gobject.TYPE_STRING,  # PLACE = 5
                                    gobject.TYPE_INT,  # LAP COUNT = 6
                                    gobject.TYPE_PYOBJECT, # RFTIME = 7
                                    gobject.TYPE_PYOBJECT, # CBUNCH = 8
                                    gobject.TYPE_PYOBJECT, # MBUNCH = 9
                                    gobject.TYPE_PYOBJECT, # STOFT = 10
                                    gobject.TYPE_PYOBJECT, # BONUS = 11
                                    gobject.TYPE_PYOBJECT, # PENALTY = 12
                                    gobject.TYPE_PYOBJECT) # RFSEEN = 13
        self.undomod = gtk.ListStore(gobject.TYPE_STRING, # BIB = 0
                                    gobject.TYPE_STRING, # NAMESTR = 1
                                    gobject.TYPE_STRING, # CAT = 2
                                    gobject.TYPE_STRING, # COMMENT = 3
                                    gobject.TYPE_BOOLEAN, # INRACE = 4
                                    gobject.TYPE_STRING,  # PLACE = 5
                                    gobject.TYPE_INT,  # LAP COUNT = 6
                                    gobject.TYPE_PYOBJECT, # RFTIME = 7
                                    gobject.TYPE_PYOBJECT, # CBUNCH = 8
                                    gobject.TYPE_PYOBJECT, # MBUNCH = 9
                                    gobject.TYPE_PYOBJECT, # STOFT = 10
                                    gobject.TYPE_PYOBJECT, # BONUS = 11
                                    gobject.TYPE_PYOBJECT, # PENALTY = 12
                                    gobject.TYPE_PYOBJECT) # RFSEEN = 13
        self.canundo = False
        self.placeundo = None

        # !! does this need a builder? perhaps make directly...
        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, 'rms.ui'))

        # !! destroy??
        self.frame = b.get_object('race_vbox')
        self.frame.connect('destroy', self.shutdown)

        # meta info pane
        self.title_namestr = b.get_object('title_namestr')
        self.set_titlestr()
        self.time_lbl = b.get_object('time_lbl')
        self.time_lbl.modify_font(pango.FontDescription("bold"))

        self.lapentry = b.get_object('lapentry')

        # results pane
        t = gtk.TreeView(self.riders)
        t.set_reorderable(True)
        t.set_rules_hint(True)
        t.show()
        self.view = t
        uiutil.mkviewcoltxt(t, 'No.', COL_BIB, calign=1.0)
        uiutil.mkviewcoltxt(t, 'Rider', COL_NAMESTR, expand=True,maxwidth=500)
        uiutil.mkviewcoltxt(t, 'Cat', COL_CAT)
        uiutil.mkviewcoltxt(t, 'Com', COL_COMMENT,
                                cb=self.editcol_cb)
        uiutil.mkviewcolbool(t, 'In', COL_INRACE, width=50)
				# too dangerous!
                                #cb=self.cr_inrace_toggled, width=50)
        uiutil.mkviewcoltxt(t, 'Laps', COL_LAPS, width=40, cb=self.editlap_cb)
        uiutil.mkviewcoltod(t, 'Start', cb=self.showstart_cb, width=50,
                                editcb=self.editstart_cb)
        uiutil.mkviewcoltod(t, 'Bunch', cb=self.showbunch_cb,
                                editcb=self.editbunch_cb,
                                width=50)
        uiutil.mkviewcoltxt(t, 'Place', COL_PLACE, calign=0.5, width=50)
        b.get_object('race_result_win').add(t)

        if ui:
            # connect signal handlers
            b.connect_signals(self)
            b = gtk.Builder()
            b.add_from_file(os.path.join(metarace.UI_PATH, 'rms_context.ui'))
            self.context_menu = b.get_object('rms_context')
            self.view.connect('button_press_event', self.treeview_button_press)
            #self.view.connect('row-activated', self.treerow_selected)
            b.connect_signals(self)
            self.meet.timer.setcb(self.rfidtrig)
