## TODO:	- merge autoclear announce into notepad
##		- unified announcer output
##		- multi-announce framework
##		- remote control via uscbsrv
##		- export triggers
##		- standardise notepad formats for laps, passings, etc
##		- begin uscbsrv bot with proper announce handshake
##		- ditch plaintext reports and prepare for new style
##		  sectioned output, line based with pagebreak rules
##		- save all RFID data for later analysis, or re-play
##		- apply consistent code, time, bonus, penalty, [points...]
##		- finalise export/import semantics and interfaces

# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Timing and data handling application wrapper for CSV road events."""

import pygtk
pygtk.require("2.0")

import gtk
import glib
import pango
import gobject

import os
import sys
import csv
import logging
import ConfigParser
import random

import metarace

from metarace import tod
from metarace import eventdb
from metarace import riderdb
from metarace import thbc
from metarace import wheeltime
from metarace import timy
from metarace import uscbsrv
from metarace import unt4
from metarace import strops
from metarace import loghandler
from metarace import uiutil
from metarace import printing
from metarace import rsync

LOGHANDLER_LEVEL = logging.DEBUG
ROADRACE_TYPES = {'irtt':'Road Time Trial',
                  'rms':'Road Race',
                  'trtt':'Team Road Time Trial',
                  'rhcp':'Handicap',
                  'sportif':'Sportif Ride',
                  'cross':'Cyclocross Race'}
CONFIGFILE = 'config.ini'
LOGFILE = 'event.log'
ROADMEET_ID = 'roadmeet_1.5'  # configuration versioning
DEFAULT_RFID_HANDLER = 'thbc'
RFID_HANDLERS = {'thbc':thbc.thbc,
                 'wheeltime':wheeltime.wheeltime}
# Bunches colourmap
COLOURMAP=[['#ffa0a0','red',1.0,0.1,0.1],
           ['#a0ffa0','green',0.1,1.0,0.1],
           ['#a0a0ff','blue',0.1,0.1,1.0],
           ['#f0b290','amber',0.9,0.6,0.1],
           ['#b290f0','violet',0.7,0.1,0.8],
           ['#f9ff10','yellow',0.9,1.0,0.1],
           ['#ff009b','pink',1.0,0.0,0.7],
           ['#00ffc3','cyan',0.1,1.0,0.8]]
COLOURMAPLEN=len(COLOURMAP)

def rfid_device(devstr=''):
    """Return a pair: (device, port) for the provided device string."""
    (a, b, c) = devstr.partition(':')
    devtype = DEFAULT_RFID_HANDLER
    if b:
        a = a.lower()
        if a in RFID_HANDLERS:
            devtype = a
        a = c	# shift port into a
    devport = a
    return((devtype, devport))

class fakemeet:
    """Road meet placeholder for external event manipulations."""
    def __init__(self, edb, rdb, path):
        self.edb = edb
        self.rdb = rdb
        self.configpath = path
        self.timer = thbc.thbc()
        self.alttimer = timy.timy()
        self.stat_but = gtk.Button()
        self.scb = uscbsrv.uscbsrv()
        self.title_str = ''
        self.date_str = ''
        self.organiser_str = ''
        self.commissaire_str = ''
        self.logo = ''
        self.distance = None
        self.docindex = 0
        self.bibs_in_results = True
    def announce_time(self, data=None):
        pass
    def announce_clear(self):
        pass
    def announce_start(self, data=None):
        pass
    def announce_rider(self, data=None):
        pass
    def announce_gap(self, data=None):
        pass
    def announce_avg(self, data=None):
        pass
    def loadconfig(self):
        """Load meet config from disk."""
        cr = ConfigParser.ConfigParser({
               'title':'',
               'date':'',
               'organiser':'',
               'commissaire':'',
               'logo':'',
               'distance':'',
               'docindex':'0',
               'resultnos':'Yes',
               'id':''
        })
        cr.add_section('meet')
        cwfilename = os.path.join(self.configpath, CONFIGFILE)
        # check for config file
        try:
            a = len(cr.read(cwfilename))
            if a == 0:
                self.log.warn('No config file - loading default values.')
        except e:
            self.log.error('Error reading meet config: ' + str(e))
        # set meet meta, and then copy into text entries
        self.title_str = cr.get('meet', 'title')
        self.date_str = cr.get('meet', 'date')
        self.organiser_str = cr.get('meet', 'organiser')
        self.commissaire_str = cr.get('meet', 'commissaire')
        self.logo = cr.get('meet', 'logo')
        self.distance = strops.confopt_float(cr.get('meet', 'distance'))
        self.docindex = strops.confopt_posint(cr.get('meet', 'docindex'), 0)
        self.bibs_in_results = strops.confopt_bool(cr.get('meet', 'resultnos'))
    def event_configfile(self, evno):
        """Return a config filename for the given event no."""
        return os.path.join(self.configpath, 'event_' + str(evno) + '.ini')

class roadmeet:
    """Road meet application class."""

    ## Meet Menu Callbacks

    def menu_meet_open_cb(self, menuitem, data=None):
        """Open a new meet."""
        if self.curevent is not None:
            self.close_event()

        dlg = gtk.FileChooserDialog('Open new road meet', self.window,
            gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER, (gtk.STOCK_CANCEL,
            gtk.RESPONSE_CANCEL, gtk.STOCK_OPEN, gtk.RESPONSE_OK))
        response = dlg.run()
        if response == gtk.RESPONSE_OK:
            self.configpath = dlg.get_filename()
            self.loadconfig()
            self.log.info('Meet data loaded from'
                           + repr(self.configpath) + '.')
        else:
            self.log.debug('Load new meet cancelled.')
        dlg.destroy()

    def menu_meet_save_cb(self, menuitem, data=None):
        """Save current all meet data to config."""
        self.saveconfig()
        self.log.info('Meet data saved to ' + repr(self.configpath) + '.')

    # replace with properties dialog class method
    # Q: call into curevent and fetch configurables ?
    def menu_meet_properties_cb(self, menuitem, data=None):
        """Edit meet properties."""
        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, 'roadmeet_props.ui'))
        dlg = b.get_object('properties')
        dlg.set_transient_for(self.window)

        # fill text entries
        t_ent = b.get_object('title_entry')
        t_ent.set_text(self.title_str)
        d_ent = b.get_object('date_entry')
        d_ent.set_text(self.date_str)
        o_ent = b.get_object('organiser_entry')
        o_ent.set_text(self.organiser_str)
        c_ent = b.get_object('commissaire_entry')
        c_ent.set_text(self.commissaire_str)
        l_ent = b.get_object('logo_entry')
        l_ent.set_text(self.logo)
        di_ent = b.get_object('distance_entry')
        if self.distance is not None:
            di_ent.set_text(str(self.distance))
        n_opt = b.get_object('resultno_opt')
        n_opt.set_active(self.bibs_in_results)
        
        upe = b.get_object('uscbsrv_host_entry')
        upe.set_text(self.uscbport)
        uce = b.get_object('uscbsrv_channel_entry')
        uce.set_text(self.uscbchan)
        upb = b.get_object('scb_port_dfl')
        upb.connect('clicked', 
                    lambda b: upe.set_text('uSCBsrv@localhost'))
        mte = b.get_object('timing_main_entry')
        mte.set_text(self.timer_port)
        mtb = b.get_object('timing_main_dfl')
        mtb.connect('clicked',
                    lambda b: mte.set_text('thbc:' + thbc.DEFPORT))
        ate = b.get_object('timing_alt_entry')
        ate.set_text(self.alttimer_port)
        atb = b.get_object('timing_alt_dfl')
        atb.connect('clicked',
                    lambda b: ate.set_text(timy.DEFPORT))
        response = dlg.run()
        if response == 1:	# id 1 set in glade for "Apply"
            self.log.debug('Updating meet properties.')
            self.title_str = t_ent.get_text()
            self.date_str = d_ent.get_text()
            self.organiser_str = o_ent.get_text()
            self.commissaire_str = c_ent.get_text()
            self.logo = l_ent.get_text()
            self.distance = strops.confopt_float(di_ent.get_text())
            self.bibs_in_results = n_opt.get_active()
            self.set_title()

            nchan = uce.get_text()
            nport = upe.get_text()
            if nport != self.uscbport or nchan != self.uscbchan:
                self.uscbport = nport
                self.uscbchan = nchan
                self.scb.set_portstr(portstr=self.uscbport,
                                     channel=self.uscbchan)
            self.set_timer(mte.get_text())
            nport = ate.get_text()
            if nport != self.alttimer_port:
                self.alttimer_port = nport
                self.alttimer.setport(nport)
            self.log.debug('Properties updated.')
        else:
            self.log.debug('Edit properties cancelled.')
        dlg.destroy()

    def menu_meet_fullscreen_toggled_cb(self, button, data=None):
        """Update fullscreen window view."""
        if button.get_active():
            self.window.fullscreen()
        else:
            self.window.unfullscreen()

    def print_report(self, sections=[], subtitle='', docstr=''):
        """Print the pre-formatted sections in a standard report."""
        self.log.info('Printing report ' + repr(subtitle) + '...')

        tfile = os.path.join(self.configpath, 'template.ini')
        if not os.path.exists(tfile):
            tfile = None
        rep = printing.printrep(template=tfile, path=self.configpath)
        #rep.strings['title'] = self.title_str
        # subtitle should probably be property of meet
        rep.strings['subtitle'] = self.title_str
        #rep.strings['subtitle'] = subtitle.strip()
        rep.strings['datestr'] = strops.promptstr('Date:', self.date_str)
        rep.strings['commstr'] = strops.promptstr('Chief Commissaire:',
                                                  self.commissaire_str)
        rep.strings['orgstr'] = strops.promptstr('Organiser:',
                                                  self.organiser_str)
        rep.strings['docstr'] = docstr
        if self.distance:
            rep.strings['diststr'] = strops.promptstr('Distance:',
                                                  str(self.distance) + ' km')
        else:
            rep.strings['diststr'] = ''

        for sec in sections:
            rep.add_section(sec)
        print_op = gtk.PrintOperation()
        print_op.set_print_settings(self.printprefs)
        print_op.set_default_page_setup(self.pageset)
        print_op.connect("begin_print", self.begin_print, rep)
        print_op.connect("draw_page", self.draw_print_page, rep)
        res = print_op.run(gtk.PRINT_OPERATION_ACTION_PREVIEW,
                               self.window)
        if res == gtk.PRINT_OPERATION_RESULT_APPLY:
            self.printprefs = print_op.get_print_settings()
            self.log.debug('Updated print preferences.')
        self.docindex += 1

        ofile = os.path.join(self.configpath, 'output.pdf')
        with open(ofile, 'wb') as f:
            rep.output_pdf(f)
        ofile = os.path.join(self.configpath, 'output.xls')
        with open(ofile, 'wb') as f:
            rep.output_xls(f)

        return False

    def begin_print(self,  operation, context, rep):
        """Set print pages and units."""
        rep.start_gtkprint(context.get_cairo_context())
        operation.set_use_full_page(True)
        operation.set_n_pages(rep.get_pages())
        operation.set_unit('points')

    def draw_print_page(self, operation, context, page_nr, rep):
        """Draw to the nominated page."""
        rep.set_context(context.get_cairo_context())
        rep.draw_page(page_nr)

    def menu_meet_printprefs_activate_cb(self, menuitem=None, data=None):
        """Edit the printer properties."""
        dlg = gtk.PrintOperation()
        dlg.set_print_settings(self.printprefs)
        res = dlg.run(gtk.PRINT_OPERATION_ACTION_PRINT_DIALOG, self.window)
        if res == gtk.PRINT_OPERATION_RESULT_APPLY:
            self.printprefs = dlg.get_print_settings()
            self.log.info('Updated print preferences.')

    def menu_meet_quit_cb(self, menuitem, data=None):
        """Quit the application."""
        self.running = False
        self.window.destroy()

    ## Race Menu Callbacks
    def menu_race_run_activate_cb(self, menuitem=None, data=None):
        """Open the event handler."""
        eh = self.edb.getevent() # only one event
        if eh is not None:
            self.open_event(eh)
            self.set_title()

    def menu_race_close_activate_cb(self, menuitem, data=None):
        """Close callback - disabled in roadrace."""
        self.close_event()
    
    def menu_race_abort_activate_cb(self, menuitem, data=None):
        """Close the currently open event without saving."""
        if self.curevent is not None:
            self.curevent.readonly = True
        self.close_event()

    def open_event(self, eventhdl=None):
        """Open provided event handle."""
        if eventhdl is not None:
            self.close_event()
            if self.etype == 'irtt':
                from metarace import irtt
                self.curevent = irtt.irtt(self, eventhdl, True)
            elif self.etype == 'trtt':
                from metarace import trtt
                self.curevent = trtt.trtt(self, eventhdl, True)
            elif self.etype == 'cross':
                from metarace import cross
                self.curevent = cross.cross(self, eventhdl, True)
            elif self.etype == 'sportif':
                from metarace import sportif
                self.curevent = sportif.sportif(self, eventhdl, True)
            else:	# default is fall back to road mass start 'rms'
                from metarace import rms
                self.curevent = rms.rms(self, eventhdl, True)
            
            assert(self.curevent is not None)
            self.curevent.loadconfig()
            self.race_box.add(self.curevent.frame)

            # re-populate the rider command model.
            cmdo = self.curevent.get_ridercmdorder()
            cmds = self.curevent.get_ridercmds()
            if cmds is not None:
                self.action_model.clear()
                for cmd in cmdo:
                    self.action_model.append([cmd, cmds[cmd]])
                self.action_combo.set_active(0)

            self.menu_race_close.set_sensitive(True)
            self.menu_race_abort.set_sensitive(True)
            starters = self.edb.getvalue(eventhdl, eventdb.COL_STARTERS)
            if starters is not None and starters != '':
                self.curevent.race_ctrl('add', starters)
                self.edb.editevent(eventhdl, starters='') # and clear
            self.curevent.show()

    def close_event(self):
        """Close the currently opened race."""
        if self.curevent is not None:
            self.curevent.hide()
            self.race_box.remove(self.curevent.frame)
            self.curevent.destroy()
            self.menu_race_close.set_sensitive(False)
            self.menu_race_abort.set_sensitive(False)
            self.curevent = None
            uiutil.buttonchg(self.stat_but, uiutil.bg_none, 'Closed')
            self.stat_but.set_sensitive(False)

    ## Reports menu callbacks.
    def menu_reports_startlist_activate_cb(self, menuitem, data=None):
        """Generate a startlist."""
        if self.curevent is not None:
            title = 'Startlist'
            self.print_report(self.curevent.startlist_report(), title, '')

    def menu_reports_camera_activate_cb(self, menuitem, data=None):
        """Generate the camera operator report."""
        if self.curevent is not None:
            title = 'Judges Report'
            self.print_report(self.curevent.camera_report(), title, '')

    def race_results_points_activate_cb(self, menuitem, data=None):
        """Generate the points tally report."""
        if self.curevent is not None:
            title = 'Points Tally'
            self.print_report(self.curevent.points_report(), title, '')

    def menu_reports_result_activate_cb(self, menuitem, data=None):
        """Generate the race result report."""
        if self.curevent is not None:
            title = 'Result'
            self.print_report(self.curevent.result_report(), title, '')

    def menu_reports_catresult_activate_cb(self, menuitem, data=None):
        """Generate the categorised race result report."""
        self.log.info('Cat result not implemented.')
        return False
        lines = ['- no data -']
        header = ''
        if self.curevent is not None:
            lines = self.curevent.catresult_report()
            header = self.curevent.result_header()
        title = 'Result'
        self.print_report(title, lines, header)

    ### TODO: if reports have options, this is the dialog, saved to meet
    def menu_reports_prefs_activate_cb(self, menuitem, data=None):
        """Run the report preferences dialog."""
        self.log.info('Report preferences not implemented.')

    ## Data menu callbacks.

    ## TODO: launch rego dlg with card swiper -> namebank hooks
    def menu_data_rego_activate_cb(self, menuitem, data=None):
        """Open rider registration dialog."""
        self.log.info('Rider registration dialog not implemented.')

    def menu_data_uscb_activate_cb(self, menuitem, data=None):
        """Request a re-connect to the uSCBsrv IRC connection."""
        if self.uscbport:
            self.log.info('Requesting re-connect to announcer: ' + 
                           str(self.uscbport) + str(self.uscbchan))
        else:
            self.log.info('Announcer not configured.')
            # but still call into uscbsrv...
        self.scb.set_portstr(portstr=self.uscbport,
                             channel=self.uscbchan,
                             force=True)

    def menu_import_riders_activate_cb(self, menuitem, data=None):
        """Add riders to database."""
        sfile = uiutil.loadcsvdlg('Select rider file to import', self.window)
        if sfile is not None:
            self.rdb.load(sfile)
        else:
            self.log.debug('Import riders cancelled.')

    def menu_import_chipfile_activate_cb(self, menuitem, data=None):
        """Import a transponder chipfile."""
        sfile = uiutil.loadcsvdlg('Select chipfile to import', self.window)
        if sfile is not None:
            self.rdb.load_chipfile(sfile)
        else:
            self.log.debug('Import chipfile cancelled.')

    # NOTE: Assumes snippet is suitably edited and then smashes in data
    def menu_import_replay_activate_cb(self, menuitem, data=None):
        """Replay an RFID logfile snippet."""
        if self.curevent is None:
            self.log.info('No event open.')
            return

        sfile = uiutil.loadcsvdlg('Select logfile to import',
                                  self.window)
        if sfile and os.path.isfile(sfile):
            self.timer.replay(sfile)
        else:
            self.log.debug('Replay RF data cancelled.')

    ## TODO: repair the start time import for all event types
    ##       fix the series ambiguity in all event types
    def menu_import_startlist_activate_cb(self, menuitem, data=None):
        """Import a startlist."""
        if self.curevent is None:
            self.log.info('No event open to add starters to.')
            return

        count = 0
        sfile = uiutil.loadcsvdlg('Select startlist file to import',
                                  self.window)
        if os.path.isfile(sfile):
            with open(sfile, 'rb') as f:
                cr = csv.reader(f)
                for r in cr:
                    if len(r) > 1 and r[1].isalnum() and r[1] != 'No.':
                        bib = r[1].strip().lower()
                        series = ''
                        if len(r) > 2:
                            series = r[2].strip()
                        self.curevent.addrider(bib, series)
                        start = tod.str2tod(r[0])
                        if start is not None:
                            self.curevent.starttime(start, bib, series)
                        count += 1
            self.log.info('Loaded ' + str(count) + ' riders from '
                           + repr(sfile))
        else:
            self.log.debug('Import startlist cancelled.')

    # no support yet
    def menu_export_rftimes_activate_cb(self, menuitem, data=None):
        """Export raw rf timing data."""
        self.log.info('Export of raw rf timing data not implemented.')

    def menu_export_riders_activate_cb(self, menuitem, data=None):
        """Export rider database."""
        sfile = uiutil.savecsvdlg('Select file to export riders to',
                                  self.window)
        if sfile is not None:
            self.rdb.save(sfile)
        else:
            self.log.debug('Export riders cancelled.')

    def menu_export_chipfile_activate_cb(self, menuitem, data=None):
        """Export transponder chipfile from rider model."""
        sfile = uiutil.savecsvdlg('Select file to export refids to',
                                   self.window)
        if sfile is not None:
            self.rdb.save_chipfile(sfile)
        else:
            self.log.debug('Export chipfile cancelled.')

    def menu_export_result_activate_cb(self, menuitem, data=None):
        """Export raw result to disk."""
        if self.curevent is None:
            self.log.info('No event open.')
            return
    
        rfilename = uiutil.savecsvdlg('Select file to save results to.',
                                       self.window,
                                       'results.csv')
        if rfilename is not None:
            with open(rfilename , 'wb') as f:
                cw = csv.writer(f)
                cw.writerow(['Rank', 'No.', 'Time', 'Bonus', 'Penalty'])
                for r in self.curevent.result_gen(''):
                    if r[2] is not None:
                        r[2] = str(r[2].timeval)
                    if r[3] is not None:
                        r[3] = r[3].rawtime(0)
                    if r[4] is not None:
                        r[4] = r[4].rawtime(0)
                    cw.writerow(r)
            self.log.info('Exported meet results to ' + repr(rfilename))

    def menu_export_startlist_activate_cb(self, menuitem, data=None):
        """Extract startlist from current event."""
        if self.curevent is None:
            self.log.info('No event open.')
            return
    
        rfilename = uiutil.savecsvdlg('Select file to save startlist to.',
                                       self.window,
                                       'startlist.csv')
        if rfilename is not None:
            with open(rfilename , 'wb') as f:
                cw = csv.writer(f)
                cw.writerow(['Start', 'No.', 'Series', 'Name', 'Cat'])
                for r in self.curevent.startlist_gen():
                    cw.writerow(r)
            self.log.info('Exported startlist to ' + repr(rfilename))
    
    def menu_data_results_cb(self, menuitem, data=None):
        """Create live result report and export to MR"""
        if self.curevent is None:
            return

        # 1: make report with meta from event and meet
        tfile = os.path.join(self.configpath, 'template.ini')
        if not os.path.exists(tfile):
            tfile = None
        rep = printing.printrep(template=tfile, path=self.configpath)
        rep.strings['title'] = ''	# use the subtitle only
        rep.strings['subtitle'] = self.title_str
        rep.strings['datestr'] = strops.promptstr('Date:', self.date_str)
        rep.strings['commstr'] = strops.promptstr('Chief Commissaire:',
                                                  self.commissaire_str)
        rep.strings['orgstr'] = strops.promptstr('Organiser:',
                                                  self.organiser_str)
        rep.strings['diststr'] = 'Location: Marysville, Victoria'
        rep.strings['docstr'] = ''
        #if self.distance:
            #rep.strings['diststr'] = strops.promptstr('Distance:',
                                                  #str(self.distance) + ' km')
        #else:
            #rep.strings['diststr'] = ''


        # 2: ensure expot path valid
        exportpath = os.path.join(self.configpath, 'export')
        if not os.path.exists(exportpath):
            os.mkdir(exportpath)
 
        # 3: set provisional status
        if self.curevent.timerstat != 'finished':
            rep.set_provisional(True)

        # 4: call into curevent to get result sections
        for sec in self.curevent.result_report():
            rep.add_section(sec)

        # 5: write out files
        filebase = os.path.basename(self.configpath)
        self.log.info('was: ' + repr(self.configpath) + ', is: ' + 
                       repr(filebase))
        if filebase == '':
            filebase = 'result'
            self.log.warn('Using default flename for export: result')

        ofile = os.path.join(exportpath, filebase+'.pdf')
        with open(ofile, 'wb') as f:
            rep.output_pdf(f)
        ofile = os.path.join(exportpath, filebase+'.xls')
        with open(ofile, 'wb') as f:
            rep.output_xls(f)
        lb = os.path.join('/site', self.mirrorpath, filebase)
        lt = ['pdf', 'xls']
        ofile = os.path.join(exportpath, filebase+'.txt')
        with open(ofile, 'wb') as f:
            rep.output_text(f, linkbase=lb, linktypes=lt)

        # 6: if not in progress, create and trigger rsync proc
        glib.idle_add(self.mirror_start)
        self.log.info('Race info export.')

    ## Directory utilities
    def event_configfile(self, evno):
        """Return a config filename for the given event no."""
        return os.path.join(self.configpath, 'event_' + str(evno) + '.ini')

    ## Timing menu callbacks
    def menu_timing_subtract_activate_cb(self, menuitem, data=None):
        """Run the time of day subtraction dialog."""
        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, 'tod_subtract.ui'))
        ste = b.get_object('timing_start_entry')
        ste.modify_font(pango.FontDescription("monospace"))
        fte = b.get_object('timing_finish_entry')
        fte.modify_font(pango.FontDescription("monospace"))
        nte = b.get_object('timing_net_entry')
        nte.modify_font(pango.FontDescription("monospace"))
        b.get_object('timing_start_now').connect('clicked',
                                                 self.entry_set_now, ste)
        b.get_object('timing_finish_now').connect('clicked',
                                                 self.entry_set_now, fte)
        ste.connect('activate', self.menu_timing_recalc, ste, fte, nte)
        fte.connect('activate', self.menu_timing_recalc, ste, fte, nte)
        dlg = b.get_object('timing')
        dlg.set_transient_for(self.window)
        dlg.run()
        dlg.destroy()

    def menu_timing_start_activate_cb(self, menuitem, data=None):
        """Manually set race elapsed time via RFU trigger."""
        if self.curevent is None:
            self.log.info('No event open to set elapsed time on.')
        else:
            self.curevent.elapsed_dlg()

    def entry_set_now(self, button, entry=None):
        """Enter the 'now' time in the provided entry."""
        entry.set_text(tod.tod('now').timestr())
        entry.activate()

    def menu_timing_recalc(self, entry, ste, fte, nte):
        """Update the net time entry for the supplied start and finish."""
        st = tod.str2tod(ste.get_text())
        ft = tod.str2tod(fte.get_text())
        if st is not None and ft is not None:
            ste.set_text(st.timestr())
            fte.set_text(ft.timestr())
            nte.set_text((ft - st).timestr())

    def menu_timing_clear_activate_cb(self, menuitem, data=None):
        """Clear memory in attached timing devices."""
        self.log.info('Clear timer memory disabled.')

    def menu_timing_sync_activate_cb(self, menuitem, data=None):
        """Roughly synchronise decoder."""
        self.timer.sync()
        self.log.info('Rough sync decoder clock.')

    def menu_timing_reconnect_activate_cb(self, menuitem, data=None):
        """Reconnect timer and re-start."""
        (cdev, cport) = rfid_device(self.timer_port)
        self.timer.setport(cport)	# force reconnect
        self.timer.stop_session()
        self.timer.sane()
        self.timer.start_session()
        self.alttimer.setport(self.alttimer_port)
        self.alttimer.sane()
        self.log.info('Re-connect/re-start attached timer.')

    ## Help menu callbacks
    def menu_help_docs_cb(self, menuitem, data=None):
        """Display program help."""
        metarace.help_docs(self.window)

    def menu_help_about_cb(self, menuitem, data=None):
        """Display metarace about dialog."""
        metarace.about_dlg(self.window)

    ## Race Control Elem callbacks
    def race_stat_but_clicked_cb(self, button, data=None):
        """Call through into event if open."""
        if self.curevent is not None:
            self.curevent.stat_but_clicked(button)

    def race_stat_entry_activate_cb(self, entry, data=None):
        """Pass the chosen action and bib list through to curevent,"""
        action = self.action_model.get_value(
                       self.action_combo.get_active_iter(), 0)
        if self.curevent is not None:
            if self.curevent.race_ctrl(action, self.action_entry.get_text()):
                self.action_entry.set_text('')
   
    def race_action_combo_changed_cb(self, combo, data=None):
    ## Menu button callbacks
        """Notify curevent of change in combo."""
        aiter = self.action_combo.get_active_iter()
        if self.curevent is not None and aiter is not None:
            action = self.action_model.get_value(aiter, 0)
            self.curevent.ctrl_change(action, self.action_entry)

    def menu_rfustat_clicked_cb(self, button, data=None):
        self.timer.status()
        self.alttimer.status()

    def menu_clock_clicked_cb(self, button, data=None):
        """Handle click on menubar clock."""
        self.log.info('PC ToD: ' + self.clock_label.get_text())

    ## 'Slow' Timer callback - this is the main event routine
    def timeout(self):
        """Update status buttons and time of day clock button."""
        if self.running:
            # update pc ToD label
            self.clock_label.set_text(tod.tod('now').meridian())

            # call into race timeout handler
            if self.curevent is not None:
                self.curevent.timeout()

            # check for completion in the rsync module
            if self.mirror is not None:
                if not self.mirror.is_alive():
                    self.mirror = None
        else:
            return False
        return True

    ## Window methods
    def set_title(self, extra=''):
        """Update window title from meet properties."""
        typepfx = ''
        title = self.title_str.strip()
        if self.etype in ROADRACE_TYPES:
            typepfx = ROADRACE_TYPES[self.etype] + ' :: '
        self.window.set_title(typepfx + title)
        if self.curevent is not None:
            self.curevent.set_titlestr(title)

    def meet_destroy_cb(self, window, msg=''):
        """Handle destroy signal and exit application."""
        self.close_event()
        self.log.removeHandler(self.sh)
        self.log.removeHandler(self.lh)
        self.window.hide()
        self.log.info('Meet destroyed. ' + msg)
        glib.idle_add(self.meet_destroy_handler)

    def meet_destroy_handler(self):
        if self.started:
            # Close threads and wait on them
            self.saveconfig()
            self.shutdown()
            self.alttimer.join()
            self.timer.join()
            self.scb.join()
        # close event and remove main log handler
        if self.loghandler is not None:
            self.log.removeHandler(self.loghandler)
        self.running = False
        # flag quit in main loop
        gtk.main_quit()
        return False

    def key_event(self, widget, event):
        """Collect key events on main window and send to race."""
        if event.type == gtk.gdk.KEY_PRESS:
            key = gtk.gdk.keyval_name(event.keyval) or 'None'
            if event.state & gtk.gdk.CONTROL_MASK:
                key = key.lower()
                if key in ['0','1']:
                    self.timer.trig(chan='MAN', refid=key)
                    return True
                elif key in ['2','3','4','5','6','7','8','9']:
                    self.alttimer.trig(chan=key)
                    return True
            if self.curevent is not None:
                return self.curevent.key_event(widget, event)
        return False

    def shutdown(self, msg=''):
        """Cleanly shutdown threads and close application."""
        if self.curevent is not None:
            self.curevent.destroy()	# necessary here?

        self.started = False

        self.window.hide()
        self.timer.exit(msg)
        self.alttimer.exit(msg)
        self.scb.exit(msg)
        print ('Waiting for workers to exit...')
        if self.mirror is not None:
            print('\tlive result mirror.')
            self.mirror.join()
            self.mirror = None
        print('\tmain timer.')
        self.timer.join()
        print('\talt timer.')
        self.alttimer.join()
        print('\tannouncer.')
        self.scb.join()

    def start(self):
        """Start the timer and rfu threads."""
        if not self.started:
            self.log.debug('Meet startup.')
            self.scb.start()
            self.timer.start()
            self.alttimer.start()
            self.started = True

    ## Roadmeet functions
    def saveconfig(self):
        """Save current meet data to disk."""
        if self.curevent is not None and self.curevent.winopen:
            self.curevent.saveconfig()
        cw = ConfigParser.ConfigParser()
        cw.add_section('meet')
        cw.set('meet', 'id', ROADMEET_ID)
        cw.set('meet', 'uscbport', self.uscbport)
        cw.set('meet', 'uscbchan', self.uscbchan)
        cw.set('meet', 'timer', self.timer_port)
        cw.set('meet', 'alttimer', self.alttimer_port)

        cw.set('meet', 'title', self.title_str)
        cw.set('meet', 'date', self.date_str)
        cw.set('meet', 'organiser', self.organiser_str)
        cw.set('meet', 'commissaire', self.commissaire_str)
        cw.set('meet', 'logo', self.logo)

        cw.set('meet', 'resultnos', self.bibs_in_results)
        cw.set('meet', 'distance', str(self.distance))
        cw.set('meet', 'docindex', str(self.docindex))
        cw.set('meet', 'loglevel', str(self.loglevel))
        cw.set('meet', 'mirrorpath', self.mirrorpath)

        cwfilename = os.path.join(self.configpath, CONFIGFILE)
        self.log.debug('Saving meet config to ' + repr(cwfilename))
        with open(cwfilename , 'wb') as f:
            cw.write(f)
        self.rdb.save(os.path.join(self.configpath, 'riders.csv'))
        self.edb.save(os.path.join(self.configpath, 'events.csv'))
        # save out print settings
        if self.printprefs is not None:
            self.printprefs.to_file(os.path.join(self.configpath, 'print.prf'))
        if self.pageset is not None:
            self.pageset.to_file(os.path.join(self.configpath, 'page.prf'))

    def set_timer(self, newdevice=''):
        """Re-set the main timer devices as specified."""
        # Step 1: is a change required?
        if newdevice != self.timer_port:
            (dev, nport) = rfid_device(newdevice)
            if type(self.timer) is RFID_HANDLERS[dev]:
                self.log.debug('timer thread same type - no change required.')
                self.timer.setport(nport)
            else:	# need to 'switch' to new handler
                wasalive = self.timer.is_alive()
                self.timer.exit('Switching devices.')
                self.timer = None	# unref
                self.timer = RFID_HANDLERS[dev](port=nport)
                if wasalive:
                    self.timer.start()
            # save change
            self.timer_port = newdevice 
        else:
            self.log.debug('set_timer - No change required.')

    def loadconfig(self):
        """Load meet config from disk."""
        cr = ConfigParser.ConfigParser({
               'title':'',
               'date':'',
               'organiser':'',
               'commissaire':'',
               'logo':'',
               'distance':'',
               'docindex':'0',
               'timer':'',
               'alttimer':'',
               'resultnos':'Yes',
               'uscbport':'',
               'uscbchan':'#announce',
               'uscbopt':'No',
               'mirrorpath':'',
               'loglevel':str(logging.INFO),
               'id':''
        })
        cr.add_section('meet')
        cwfilename = os.path.join(self.configpath, CONFIGFILE)

        # re-set log file
        if self.loghandler is not None:
            self.log.removeHandler(self.loghandler)
            self.loghandler.close()
            self.loghandler = None
        self.loghandler = logging.FileHandler(
                             os.path.join(self.configpath, LOGFILE))
        self.loghandler.setLevel(LOGHANDLER_LEVEL)
        self.loghandler.setFormatter(logging.Formatter(
                       '%(asctime)s %(levelname)s:%(name)s: %(message)s'))
        self.log.addHandler(self.loghandler)

        # check for config file
        try:
            a = len(cr.read(cwfilename))
            if a == 0:
                self.log.warn('No config file - loading default values.')
        except e:
            self.log.error('Error reading meet config: ' + str(e))

        # set uSCBsrv connection
        self.uscbchan = cr.get('meet', 'uscbchan')
        self.uscbport = cr.get('meet', 'uscbport')
        self.scb.set_portstr(portstr=self.uscbport,
                             channel=self.uscbchan)

        # set timer port (decoder)
        self.set_timer(cr.get('meet', 'timer'))

        # set alt timer port (timy)
        nport = cr.get('meet', 'alttimer')
        if nport != self.alttimer_port:
            self.alttimer_port = nport
            self.alttimer.setport(nport)
            self.alttimer.sane() # sane prod here is probably good idea

        # set meet meta, and then copy into text entries
        self.title_str = cr.get('meet', 'title')
        self.date_str = cr.get('meet', 'date')
        self.organiser_str = cr.get('meet', 'organiser')
        self.commissaire_str = cr.get('meet', 'commissaire')
        self.logo = cr.get('meet', 'logo')
        self.distance = strops.confopt_float(cr.get('meet', 'distance'))
        self.docindex = strops.confopt_posint(cr.get('meet', 'docindex'), 0)
        self.bibs_in_results = strops.confopt_bool(cr.get('meet', 'resultnos'))
        self.mirrorpath = cr.get('meet', 'mirrorpath')

        # configurable log level on UI (does not change log file)
        self.loglevel = strops.confopt_posint(cr.get('meet', 'loglevel'),
                                              logging.INFO)
        self.lh.setLevel(self.loglevel)

        # Re-Initialise rider and event databases
        self.rdb.clear()
        self.edb.clear()
        self.rdb.load(os.path.join(self.configpath, 'riders.csv'))
        self.edb.load(os.path.join(self.configpath, 'events.csv'))
        event = self.edb.getevent()
        if event is None:	# add a new event of the right type
            event = self.edb.editevent(num='00', etype=self.etype)
        else:
            self.etype = self.edb.getvalue(event, eventdb.COL_TYPE)
            self.log.debug('Existing event in db: ' + repr(self.etype))
        self.open_event(event) # always open on load if posible
        self.set_title()

        # restore printer and page preferences
        psfilename = os.path.join(self.configpath, 'print.prf')
        if os.path.isfile(psfilename):
            try:
                self.printprefs.load_file(psfilename)
            except:
                self.log.warn('Error loading print preferences.')

        # After load complete - check config and report. This ensures
        # an error message is left on top of status stack. This is not
        # always a hard fail and the user should be left to determine
        # an appropriate outcome.
        cid = cr.get('meet', 'id')
        if cid and cid != ROADMEET_ID:
            self.log.error('Meet configuration mismatch: '
                           + repr(cid) + ' != ' + repr(ROADMEET_ID))

    def get_distance(self):
        """Return race distance in km."""
        return self.distance

    def announce_clear(self):
        """Clear announce panels."""
        self.announce_model.clear()
        self.scb.clrall()

    def announce_title(self, msg):
        """Set the announcer title."""
        # no local announce title?
        self.scb.set_title(msg)

    def announce_start(self, starttod=tod.ZERO):
        """Set the announce start offset."""
        if starttod != None:
            self.an_cur_start = starttod
        self.scb.set_start(starttod)

    def announce_time(self, timestr=None):
        """Set the announcer time."""
        if timestr is None:
            timestr = tod.tod('now').rawtime(0)
        # no local announce time?
        self.scb.set_time(timestr)

    def announce_gap(self, timestr=''):
        """Set the announcer gap time."""
        self.scb.set_gap(timestr)

    def announce_avg(self, timestr=''):
        """Set the announcer avg."""
        self.scb.set_avg(timestr)

    def announce_rider(self, rvec):
        """Announce the supplied rider vector."""
        # announce locally
        nr = ['','','','','','','#000000',None] # error bar?
        if len(rvec) == 5:
            rftime = tod.str2tod(rvec[4])
            if rftime is not None:
                if len(self.announce_model) == 0:
                    # Case 1: Starting a new lap
                    self.an_cur_lap = (rftime-self.an_cur_start).truncate(0)
                    self.an_cur_split = rftime.truncate(0)
                    self.an_cur_bunchid = 0
                    self.an_cur_bunchcnt = 1
                    self.an_last_time = rftime
                    nr=[rvec[0],rvec[1],rvec[2],rvec[3],
                        self.an_cur_lap.rawtime(0),
                        self.an_cur_bunchcnt,
                        COLOURMAP[self.an_cur_bunchid][0],
                        rftime]
                elif (rftime < self.an_last_time
                      or rftime - self.an_last_time < tod.tod('1.12')):
                    # Case 2: Same bunch
                    self.an_last_time = rftime
                    self.an_cur_bunchcnt += 1
                    nr=[rvec[0],rvec[1],rvec[2],rvec[3],
                        (rftime-self.an_cur_start).rawtime(0),
                        self.an_cur_bunchcnt,
                        COLOURMAP[self.an_cur_bunchid][0],
                        rftime]
                else:
                    # Case 3: New bunch
                    self.announce_model.append(
                         ['','','','','','','#fefefe',None])
                    self.an_cur_bunchid = (self.an_cur_bunchid + 1)%COLOURMAPLEN
                    self.an_cur_bunchcnt = 1
                    self.an_last_time = rftime
                    nr=[rvec[0],rvec[1],rvec[2],rvec[3],
                        (rftime-self.an_cur_start).rawtime(0),
                        self.an_cur_bunchcnt,
                        COLOURMAP[self.an_cur_bunchid][0],
                        rftime]
            else:
                # Informative non-timeline record
                nr=[rvec[0],rvec[1],rvec[2],rvec[3],
                        rvec[4], '', '#fefefe',None]
            self.announce_model.append(nr)
        else:
            self.log.info('Ignored unknown rider announce vector')
        self.scb.add_rider(rvec)

        return False	# so can be idle-added

    def mirror_start(self):
        """Create a new rsync mirror thread unless in progress."""
        if self.mirrorpath and self.mirror is None:
            self.mirror = rsync.mirror(
                localpath = os.path.join(self.configpath,'export',''),
                remotepath = self.mirrorpath)
            self.mirror.start()
        return False    # for idle_add

    def __init__(self, configpath=None, etype=None):
        """Meet constructor."""
        # logger and log handler
        self.log = logging.getLogger()
        self.log.setLevel(logging.DEBUG)
        self.loghandler = None	# set in loadconfig to meet dir

        if etype not in ROADRACE_TYPES:
            etype = None	# fall back on default
        self.etype = etype

        # meet configuration path and options
        if configpath is None:
            configpath = '.'	# None assumes 'current dir'
        self.configpath = configpath
        self.title_str = ''
        self.date_str = ''
        self.organiser_str = ''
        self.commissaire_str = ''
        self.logo = ''
        self.distance = None
        self.docindex = 0
        self.bibs_in_results = True
        self.loglevel = logging.INFO	# UI log window

        # printer preferences
        paper = gtk.paper_size_new_custom('metarace-full',
                      'A4 for reports', 595, 842, gtk.UNIT_POINTS)
        self.printprefs = gtk.PrintSettings()
        self.pageset = gtk.PageSetup()
        self.pageset.set_orientation(gtk.PAGE_ORIENTATION_PORTRAIT)
        self.pageset.set_paper_size(paper)
        self.pageset.set_top_margin(0, gtk.UNIT_POINTS)
        self.pageset.set_bottom_margin(0, gtk.UNIT_POINTS)
        self.pageset.set_left_margin(0, gtk.UNIT_POINTS)
        self.pageset.set_right_margin(0, gtk.UNIT_POINTS)

        # hardware connections
        self.timer = thbc.thbc()	# default is Tag, may change later
        self.timer_port = ''
        self.alttimer = timy.timy()
        self.alttimer_port = ''
        self.uscbport = ''
        self.uscbchan = '#announce'
        self.scb = uscbsrv.uscbsrv()
        self.mirrorpath = ''    # default rsync mirror path
        self.mirror = None 

        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, 'roadmeet.ui'))
        self.window = b.get_object('meet')
        self.window.connect('key-press-event', self.key_event)
        self.clock = b.get_object('menu_clock')
        self.clock_label = b.get_object('menu_clock_label')
        self.menu_rfustat_img = b.get_object('menu_rfustat_img')
        self.status = b.get_object('status')
        self.log_buffer = b.get_object('log_buffer')
        self.log_view = b.get_object('log_view')
        self.log_view.modify_font(pango.FontDescription("monospace 12"))
        self.log_scroll = b.get_object('log_box').get_vadjustment()
        self.context = self.status.get_context_id('metarace meet')
        self.menu_race_close = b.get_object('menu_race_close')
        self.menu_race_abort = b.get_object('menu_race_abort')
        self.race_box = b.get_object('race_box')
        self.stat_but = b.get_object('race_stat_but')
        self.action_model = b.get_object('race_action_model')
        self.action_combo = b.get_object('race_action_combo')
        self.action_entry = b.get_object('race_action_entry')
        b.get_object('race_tab_img').set_from_file(metarace.SCB_LOGOFILE)
        b.get_object('race_stat_hbox').set_focus_chain([self.action_combo,
                                             self.action_entry,
                                             self.action_combo])

        # prepare local scratch pad
        self.an_cur_lap = tod.ZERO
        self.an_cur_split = tod.ZERO
        self.an_cur_bunchid = 0
        self.an_cur_bunchcnt = 0
        self.an_last_time = None
        self.an_cur_start = tod.ZERO
        self.announce_model = gtk.ListStore(gobject.TYPE_STRING,  # rank
                                    gobject.TYPE_STRING,  # no.
                                    gobject.TYPE_STRING,  # namestr
                                    gobject.TYPE_STRING,  # cat/com
                                    gobject.TYPE_STRING,  # timestr
                                    gobject.TYPE_STRING,  # bunchcnt
                                    gobject.TYPE_STRING,  # colour
                                    gobject.TYPE_PYOBJECT) # rftod
        t = gtk.TreeView(self.announce_model)
        t.set_reorderable(False)
        t.set_rules_hint(False)
        t.set_headers_visible(False)
        t.set_search_column(1)
        t.modify_font(pango.FontDescription('bold 20px'))
        uiutil.mkviewcoltxt(t, 'Rank', 0,width=60)
        uiutil.mkviewcoltxt(t, 'No.', 1,calign=1.0,width=60)
        uiutil.mkviewcoltxt(t, 'Rider', 2,expand=True,fixed=True)
        uiutil.mkviewcoltxt(t, 'Cat', 3,calign=0.0)
        uiutil.mkviewcoltxt(t, 'Time', 4,calign=1.0,width=100,
                                        fontdesc='monospace bold')
        uiutil.mkviewcoltxt(t, 'Bunch', 5,width=50,bgcol=6,calign=0.5)
        t.show()
        b.get_object('notepad_box').add(t)

        b.connect_signals(self)

        # run state
        self.running = True
        self.started = False
        self.curevent = None

        # format and connect status and log handlers
        f = logging.Formatter('%(levelname)s:%(name)s: %(message)s')
        self.sh = loghandler.statusHandler(self.status, self.context)
        self.sh.setLevel(logging.INFO)	# show info+ on status bar
        self.sh.setFormatter(f)
        self.log.addHandler(self.sh)
        self.lh = loghandler.textViewHandler(self.log_buffer,
                      self.log_view, self.log_scroll)
        self.lh.setLevel(self.loglevel)
        self.lh.setFormatter(f)
        self.log.addHandler(self.lh)

        # get rider db and pack into a dialog
        self.rdb = riderdb.riderdb()
        b.get_object('riders_box').add(self.rdb.mkview(cat=True,
                                                  series=False,refid=True))

        # select event page in notebook.
        b.get_object('meet_nb').set_current_page(1)

        # get event db -> loadconfig adds empty event if not already present
        self.edb = eventdb.eventdb([])

        # start timer
        glib.timeout_add_seconds(1, self.timeout)

def main(etype=None):
    """Run the road meet application."""
    configpath = None

    # expand configpath on cmd line to realpath _before_ doing chdir
    if len(sys.argv) > 2:
        print('usage: roadmeet [configdir]\n')
        sys.exit(1)
    elif len(sys.argv) == 2:
        rdir = sys.argv[1]
        if not os.path.isdir(rdir):
            rdir = os.path.dirname(rdir)
        configpath = os.path.realpath(rdir)

    metarace.init()
    if configpath is not None:
        os.chdir(configpath)
    app = roadmeet(configpath, etype)
    app.loadconfig()
    app.window.show()
    app.start()
    try:
        gtk.main()
    except:
        app.shutdown('Exception from gtk.main()')
        raise

if __name__ == '__main__':
    main()

