
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Text-only scoreboard for VGA videowall (Launceston)

This module implements a simple scoreboard that mimics the Omega
DHI at DISC. The output is intended to be displayed on a vieowall.

"""


import pygtk
pygtk.require("2.0")

import gtk
import glib
import pango
import pangocairo
import cairo
import random
import ConfigParser
import logging
import os
import csv

import metarace
from metarace import strops
from metarace import uscbsrv

# Globals
CONFIGFILE='text_scb.ini'
USCBSRV_HOST='textscb@localhost'
USCBSRV_CHANNEL='#tscb'
LOGFILE = 'text_scb.log'
#SCBFONT = 'Nimbus Sans L Condensed '
SCBFONT = 'TimesSquare '

overlays = [[0],
            [0,'-','','',1],
            [0,'-','',1,2],
            [0,'-','',1,2,3],
            [4,'-','',6,7,8,9],
            [4,5,'-','',6,7],
            [4,5,'-','',6,7,8],
            [4,5,'-','',6,7,8,9],
            [10,'-','',11,12,13,14,15],
            [10,'-',11,12,13,14,15,16,17],
            [10,11,12,13,14,15,16,17],
            [10,11,12,13,14,15,16,17],
            ['c'], ['i'], [], [], [], [], [], [], []]

# draw text centred
def text_cent(cr, pr, oh, ow, msg, ch, fn):
    """Position text centred at h."""
    if msg is not None:
        cr.save()
        l = pr.create_layout()
        l.set_font_description(pango.FontDescription(fn + str(ch)))
        l.set_text(msg)
        (tw,th) = l.get_pixel_size()
        cr.move_to(ow-(0.5 * tw), oh)
        pr.update_layout(l)
        pr.show_layout(l)
        cr.restore()

class text_scb(object):
    """IRTT Start Display application."""
 
    def doline(self, cr, h):
        cr.save()
        pxsz = 2
        if self.height > 0.1:
            pxsz = self.height/76
        cr.set_line_width(pxsz)
        cr.set_line_cap(cairo.LINE_CAP_ROUND)
        cr.set_dash([0, 2*pxsz])
        cr.move_to(0+pxsz, h+3*pxsz)
        cr.line_to(self.width - pxsz, h+3*pxsz)
        cr.stroke()
        cr.restore()
        return h+2*pxsz
        
    def show(self):
        self.window.show()

    def hide(self):
        self.window.show()

    def start(self):
        """Start threads."""
        if not self.started:
            self.scb.start()
            self.scb.set_pub_cb(self.msg_cb)
            self.started = True

    def postxt(self, line, pos, txt, eol=False):
        """Position string in txt on line in pos."""
        if eol:
            txt += ' '*32	# lazy
        if line in range(0,20): 
            if pos < 32:
                trunclen = 32 - pos
                txt = strops.truncpad(txt, trunclen, elipsis=False)
                ol = self.srctxt[line]
                self.srctxt[line] = ol[0:pos] + txt + ol[pos+trunclen:]
            self.draw_and_update()
                
    def shutdown(self):
        """Cleanly shutdown."""
        self.scb.exit()
        self.scb.join()
        self.started = False

    def window_destroy_cb(self, window):
        """Handle destroy signal."""
        if self.started:
            self.shutdown()
        self.running = False
        gtk.main_quit()
    
    def area_expose_event_cb(self, widget, event):
        """Update desired portion of drawing area."""
        x , y, width, height = event.area
        widget.window.draw_drawable(widget.get_style().fg_gc[gtk.STATE_NORMAL],
                                    self.area_src, x, y, x, y, width, height)
        return False

    def area_redraw(self):
        """Lazy full area redraw method."""
        cr = self.area_src.cairo_create()
        pr = pangocairo.CairoContext(cr)
        cr.identity_matrix()

        # bg filled
        cr.set_source_rgb(0.06,0.06,0.06)
        cr.paint()

        # test txt
        cr.set_source_rgb(1.0,1.0,0.9)
        ov = overlays[self.curov]
        oh = 0
        ch = 0.105 * self.height
        lh = 0.115 * self.height
        for line in ov:
            txt = ' '*24
            if line == '-':
                oh =self.doline(cr, oh)
            elif line == '':
                oh += lh
            elif type(line) is int:
                txt = self.srctxt[line][0:24]
                text_cent(cr, pr, oh, 0.5 * self.width, txt, ch, SCBFONT)
                oh += lh

    def area_configure_event_cb(self, widget, event):
        """Re-configure the drawing area and redraw the base image."""
        x, y, width, height = widget.get_allocation()
        ow = 0
        oh = 0
        if self.area_src is not None:
            ow, oh = self.area_src.get_size()
        if width > ow or height > oh:
            self.area_src = gtk.gdk.Pixmap(widget.window, width, height)
        self.width = width
        self.height = height
        self.area_redraw()
        self.area.queue_draw()
        return True

    def clearsrc(self):
        """Clear database."""
        for i in range(0, 20):
            self.srctxt[i] = ' '*32

    def clear(self):
        """Clear all elements."""
        self.clearsrc()
        self.draw_and_update()
        
    def loadconfig(self):
        """Load config from disk."""
        cr = ConfigParser.ConfigParser({'uscbsrv':USCBSRV_HOST,
                                        'channel':USCBSRV_CHANNEL,
                                        'fullscreen':'No',
                                       })
        cr.add_section('text_scb')

        # check for config file
        try:
            a = len(cr.read(CONFIGFILE))
            if a == 0:
                self.log.info('No config file found - loading default values.')
        except Exception as e:
            self.log.error('Error reading config: ' + str(e))

        if strops.confopt_bool(cr.get('text_scb', 'fullscreen')):
            self.window.fullscreen()

        # set sender port
        nhost = cr.get('text_scb', 'uscbsrv')
        nchannel = cr.get('text_scb', 'channel')
        self.scb.set_portstr(nhost, nchannel)

    def draw_and_update(self, data=None):
        """Redraw in main loop, not in timeout."""
        self.area_redraw()
        self.area.queue_draw()
        return False

    def delayed_cursor(self):
        """Remove the mouse cursor from the text area."""
        pixmap = gtk.gdk.Pixmap(None, 1, 1, 1)
        color = gtk.gdk.Color()
        cursor = gtk.gdk.Cursor(pixmap, pixmap, color, color, 0, 0)
        self.area.get_window().set_cursor(cursor)
        return False

    def msg_cb(self, msg, data=None):
        """Handle a public message from the channel."""
        if msg.erp or msg.header == '0040100096':
            self.clear()
        if msg.header != '':
            if msg.header[0:7] == 'OVERLAY' and msg.header[8:10].isdigit():
                self.set_overlay(int(msg.header[8:10]))
        elif msg.xx is not None and msg.yy is not None:
            txttoadd = msg.text.replace('\n', '')
            self.postxt(msg.yy, msg.xx, txttoadd, msg.erl)
        else:
            pass
        return False	# idle added
 
    def remote_msg(self, msg):
        """Log a message to the uscbsrv."""
        self.log.debug(msg)
        self.scb.add_rider([msg], 'message')

    def set_overlay(self, newov=0):
        if newov >= 0 and newov < 16:
            self.curov = newov
            self.draw_and_update()

    def __init__(self):
        # logger and handler
        self.log = logging.getLogger()
        self.log.setLevel(logging.DEBUG)
        self.loghandler = logging.FileHandler(LOGFILE)
        self.loghandler.setLevel(logging.DEBUG)
        self.loghandler.setFormatter(logging.Formatter(
                       '%(asctime)s %(levelname)s:%(name)s: %(message)s'))
        self.log.addHandler(self.loghandler)
        self.log.debug('Text SCB - Init.')

        # require one timy and one uscbsrv
        self.scb = uscbsrv.uscbsrv()
        self.started = False
        self.running = True

        # variables
        self.width = 0
        self.height = 0
        self.srctxt = []
        self.curov = 3
        for i in range(0,20):
            self.srctxt.append(' '*32)

        self.window = gtk.Window()
        self.window.set_title('Text SCB')
        self.window.connect('destroy', self.window_destroy_cb)

        self.area_src = None
        self.area = gtk.DrawingArea()
        self.area.connect('configure_event', self.area_configure_event_cb)
        self.area.connect('expose_event', self.area_expose_event_cb)
        self.area.set_size_request(400,240)
        self.area.show()
        self.window.add(self.area)
        #glib.timeout_add_seconds(1, self.timeout)
        glib.timeout_add_seconds(5, self.delayed_cursor)

def main():
    """Run the application."""
    metarace.init()
    app = text_scb()
    app.loadconfig()
    app.show()
    app.start()
    try:
        gtk.main()
    except:
        app.shutdown()
        raise

if __name__ == '__main__':
    main()

