
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""uSCBsrv/IRC server class.

This module provides a thread object which maintains a persistent
uSCBsrv server connection to the configured irc server. Announce
messages are broadcast to the announcer.

Live announce messages are stored in a Queue object and written out
to the irc server using blocking I/O.

TODO: IRC connect and error handling is VERY messy - some more
      thought is required to cleanly handle all init and error
      conditions to avoid spurious reconnects and hang states

	- add callback methods for handling pub/priv msgs and
	  channel traffic
"""

import threading
import Queue
import logging
import socket
import random
import time
import glib

from metarace import unt4
from metarace import tod
from metarace import strops

# Global Defaults
USCBSRV_HOST=''		# default is "not present"
USCBSRV_PORT=6667
USCBSRV_CHANNEL='#announce'
USCBSRV_SRVNICK='uSCBsrv'

# dispatch thread queue commands
TCMDS = ('EXIT', 'PORT', 'MSG')

# client initiated ping interval - helps to break a network error
PING_INTERVAL = tod.tod('30')

def parse_portstr(portstr=''):
    """Read a port string and split into defaults."""
    port = USCBSRV_PORT
    host = ''
    nick = USCBSRV_SRVNICK

    # strip off nickname
    ar = portstr.rsplit('@', 1)
    if len(ar) == 2:
        nick = ar[0][0:9]	# limit nick to 9 char
        portstr = ar[1]
    else:
        portstr = ar[0]

    # read host:port
    ar = portstr.split(':')
    if len(ar) > 1:
        host = ar[0]
        if ar[1].isdigit():
            port = int(ar[1])
    else:
        host = ar[0]

    return (host, port, nick)

class uscbsrv(threading.Thread):
    """uSCBsrv server thread.

       methods are grouped as follows:

	- metarace methods called by gtk main thread for 
	  manipulating the live announce stream, includes
          old-style DHI postxt and setline methods

	- uSCBsrv protocol methods called by uSCBsrv client
          communications - todo

	- irc protocol methods called by the lower level irclib


       If irclib is not present, this module reverts to a disconnected
       'black hole' sender.

    """

    ### GTK main thread methods

    def clrall(self):
        """Clear the live announce screen."""
        self.sendmsg(unt4.GENERAL_CLEARING)

    def clrline(self, line):
        """Clear the specified line in DHI database."""
        self.sendmsg(unt4.unt4(xx=0,yy=int(line),erl=True))

    def set_title(self, line):
        """Update the announcer's title line."""
        self.sendmsg(unt4.unt4(header='title',text=line))

    def set_time(self, tstr):
        """Update the announcer's time."""
        self.sendmsg(unt4.unt4(header='time',text=tstr))

    def set_start(self, stod):
        """Update the announcer's relative start time."""
        self.sendmsg(unt4.unt4(header='start',text=stod.rawtime()))

    def set_gap(self, tstr):
        """Update the announcer's gap time (if relevant)."""
        self.sendmsg(unt4.unt4(header='gap',text=tstr))

    def set_avg(self, tstr):
        """Update the announcer's average speed."""
        self.sendmsg(unt4.unt4(header='average',text=tstr))

    def add_rider(self, rvec, header_txt='rider'):
        """Send a rider vector to the announcer."""
        self.sendmsg(unt4.unt4(header=header_txt,text=chr(unt4.US).join(rvec)))

    def gfx_overlay(self, newov, chan=u'#tscb'):
        """Update graphic channel overlay."""
        self.queue.put_nowait(('MSG', 
                   unt4.unt4(header=u'overlay', text=unicode(newov)).pack(),
                       chan))

    def gfx_clear(self, chan=u'#tscb'):
        """Update graphic channel overlay."""
        self.queue.put_nowait(('MSG', unt4.GENERAL_CLEARING, chan))

    def gfx_set_title(self, title, chan=u'#tscb'):
        """Update graphic channel title."""
        self.queue.put_nowait(('MSG', 
                    unt4.unt4(header=u'set_title', text=title).pack(),
                       chan))

    def gfx_add_row(self, rvec, chan=u'#tscb'):
        """Update graphic channel title."""
        ovec = []
        for c in rvec:	# replace nulls and empties
            nc = u''
            if c:	# but assume strings?
                nc = c
            ovec.append(nc)
        self.queue.put_nowait(('MSG', 
                       unt4.unt4(header=u'add_row', 
                                 text=chr(unt4.US).join(ovec)).pack(),
                       chan))

    def setline(self, line, msg):
        """Set the specified DHI database line to msg."""
        msg = msg[0:self.linelen].ljust(self.linelen)
        msg = msg + ' ' * (self.linelen - len(msg))
        self.sendmsg(unt4.unt4(xx=0,yy=int(line),text=msg))

    def linefill(self, line, char='_'):
        """Use char to fill the specified line."""
        msg = char * self.linelen
        self.sendmsg(unt4.unt4(xx=0,yy=int(line),text=msg))

    def postxt(self, line, oft, msg):
        """Position msg at oft on line in DHI database."""
        assert oft >= 0, 'Offset should be >= 0'
        if oft < self.linelen:
            msg = msg[0:(self.linelen-oft)]
            self.sendmsg(unt4.unt4(xx=int(oft),yy=int(line),text=msg))

    def setoverlay(self, newov):
        """Request overlay newov to be displayed on the scoreboard."""
        if self.curov != newov:
            self.sendmsg(newov)
            self.curov = newov

    def sendmsg(self, unt4msg=None):
        """Pack and send a unt4 message to the live announce stream."""
        self.queue.put_nowait(('MSG', unt4msg.pack()))

    def write(self, msg=None):
        """Send the provided raw text msg to the live announce stream."""
        self.queue.put_nowait(('MSG', msg))

    def exit(self, msg=None):
        """Request thread termination."""
        self.running = False
        self.queue.put_nowait(('EXIT', msg))

    def wait(self):             # NOTE: Do not call from cmd thread
        """Suspend calling thread until cqueue is empty."""
        self.queue.join()

    def add_channel(self, addchan=''):
        """Request additional channel."""	# HACK -> Perth GP
        self.queue.put_nowait(('ADDCHAN', addchan))

    def rejoin_channel(self, newchan=''):
        """Request another attempt to join channel."""
        self.queue.put_nowait(('CHAN', newchan))

    ### uSCBsrv protocol methods

    # TODO

    ### IRC protocol methods

    def irc_event_cb(self, c, e):
        """Debug method to collect all IRC events."""
        self.log.debug(str(e.eventtype()) + ' :: '
                         + str(e.source()) + '->' + str(e.target()) + ' :: '
                         + '/'.join(map(str, e.arguments())))

    def server_join_cb(self, c, e):
        """Register server join."""
        self.log.debug('Connected to server')
        self.queue.put_nowait(('CHAN', ''))

    def channel_join_cb(self, c, e):
        """Register channel join."""
        su = self.il.nm_to_n(e.source()).lower()
        tg = e.target().lower()
        if su == self.srvnick.lower() and tg == self.channel:
            self.chanstatus = True
            self.connect_pending = False # flags queue processing ok
            self.log.debug('Joined channel ' + str(e.target()))
            self.dumbcnt = 0
        else:
            self.log.debug('Joined additional channel ' + str(e.target()))

    def channel_part_cb(self, c, e):
        """Register channel part."""
        tg = e.target().lower()
        if (len(e.arguments()) > 0 and tg == self.channel
            and e.arguments()[0].lower() == self.srvnick.lower()):
            self.chanstatus = False
            self.log.debug('Left channel ' + str(e.target()))

    def set_pub_cb(self, cb=None):
        """Set the public message callback function."""
        self.pub_cb = cb

    def pubmsg_cb(self, c, e):
        """Handle message in channel."""
        su = self.il.nm_to_n(e.source()).lower()
        tg = e.target().lower()
        if tg == self.channel:
            self.log.debug('Public message from ' + su)
            if self.pub_cb is not None:
                self.pub_rdbuf += unt4.decode(''.join(e.arguments()))
                idx = self.pub_rdbuf.find(chr(unt4.EOT))
                if idx >= 0:
                    msgtxt = self.pub_rdbuf[0:idx+1]
                    self.pub_rdbuf = self.pub_rdbuf[idx+1:]
                    glib.idle_add(self.pub_cb, unt4.unt4(unt4str=msgtxt.decode('utf-8','replace')))
                    self._pacing(0.0)
        
    def privmsg_cb(self, c, e):
        """Handle private message."""
        su = self.il.nm_to_n(e.source()).lower()
        self.log.debug('Private message from ' + su + ' :: '
                        + ''.join(e.arguments()))
        # TODO

    ### uSCBsrv internals

    def __init__(self, linelen=32):
        """Constructor."""
        threading.Thread.__init__(self) 
        self.running = False
        self._curpace = 0.1
        self.il = None
        self.localsrv = False	# if set, try to oper
        self.pub_rdbuf = ''
        self.pub_cb = None
        self.priv_rdbuf = ''
        self.priv_cb = None

        self.log = logging.getLogger('uscbsrv')
        self.log.setLevel(logging.DEBUG)

        try:
            import irclib
            # CHECK: 16.2.9. "all import attempts must be completed
            # before the interpreter starts shutting itself down."
            self.ih = irclib.IRC()
            self.il = irclib
        except ImportError:
            self.log.warn('irclib not present: Announcer will not function.')
            self.ih = fakeirc()
        self.ic = self.ih.server()

        self.np = tod.tod('now') + PING_INTERVAL

        self.name = 'uSCBsrv'
        self.rdbuf = ''
        self.wrbuf = ''
        self.chanstatus = False
        self.host = USCBSRV_HOST
        self.port = USCBSRV_PORT
        self.channel = USCBSRV_CHANNEL
        self.srvnick = USCBSRV_SRVNICK
        self.doreconnect = False
        self.connect_pending = False
        self.dumbcnt = 0

        self.curov = None
        self.linelen = linelen

        self.queue = Queue.Queue()

    def set_portstr(self, portstr='', channel='#announce', force=False):
        """Set irc connection by a port string."""
        if channel == '' or channel[0] != '#':
            self.log.error('Invalid announce channel specified: '
                           + repr(channel) + ', using #announce.')
        (host, port, nick) = parse_portstr(portstr)
        self.set_port(host, port, channel, nick, reconnect=force)

    def set_port(self, host=None, port=None, channel=None,
                       srvnick=None, reconnect=False):
        """Request change in irc connection."""
        if host is not None and host != self.host:
            self.host = host
            if self.host == '' and self.ic.is_connected():
                self.ic.disconnect()
            else:
                reconnect = True
        if port is not None and port != self.port:
            self.port = port
            reconnect = True
        if channel is not None and channel != self.channel:
            self.channel = channel
            reconnect = True
        if srvnick is not None:
            self.srvnick = srvnick
            #self.srvnick = srvnick.lower()
            reconnect = True
        if reconnect:
            if self.host == 'localhost' or self.host == '127.0.0.1':
                self.localsrv = True	# assume oper for same machine
            try:		# dump queue before reconnect
                while True:	# QUERY: is this necessary?
                    self.queue.get_nowait()
                    self.queue.task_done()
            except Queue.Empty:
                pass 
            self.queue.put_nowait(('PORT', ''))

    def connected(self):
        """Return true if connected and in channel."""
        return self.ic.is_connected() and self.chanstatus

    def _addchan(self, adchan):
        self.ic.join(adchan)

    def _rechan(self):
        if self.localsrv:
            self.ic.oper('metairc', 'ogfPOHYkaw')
        self.ic.join(self.channel)
        if self.localsrv:
            self.ic.send_raw('OPME ' + self.channel)
        #self.ic.mode(self.channel, '+tn')
        #self.ic.topic(self.channel, 'uSCBsrv Live Result Feed')

    def _reconnect(self):
        if not self.connect_pending:
            self.log.debug('Connecting to '
                            + self.host + ':' + str(self.port))
            self.connect_pending = True
            self.ic.connect(self.host, self.port,
                            self.srvnick,'','metarace node')

    def _pacing(self, delay=None):
        """Adjust internal pacing delay."""
        if delay is None:
            self._curpace += 0.01
            if self._curpace > 0.1:
                self._curpace = 0.1
        else:
            self._curpace = delay
        return self._curpace

    def run(self):
        """Called via threading.Thread.start()."""
        self.running = True
        self.log.debug('Starting')
        self.ic.add_global_handler('pubmsg', self.pubmsg_cb, -10)
        self.ic.add_global_handler('privmsg', self.privmsg_cb, -10)
        self.ic.add_global_handler('welcome', self.server_join_cb, -10)
        self.ic.add_global_handler('join', self.channel_join_cb, -10)
        self.ic.add_global_handler('part', self.channel_part_cb, -10)
        self.ic.add_global_handler('kick', self.channel_part_cb, -10)
        #self.ic.add_global_handler('all_events', self.irc_event_cb, 0)
        while self.running:
            try:
                self.ih.process_once(0.01)
                if self.host != '':
                    # irc process phase
                    if not self.connected() or self.doreconnect:
                        self.doreconnect = False
                        if not self.connect_pending:
                            self.chanstatus = False
                            self._reconnect()    

                    # keepalive ping
                    now = tod.tod('now')
                    if now > self.np:
                        self.ic.ctcp('PING', self.srvnick,
                                     str(int(time.time())))
                        self.np = now + PING_INTERVAL
                else:
                    time.sleep(1.0)

                # queue process phase - queue empty exception breaks loop
                while True:
                    m = self.queue.get_nowait()
                    self.queue.task_done()
                    if m[0] == 'MSG' and self.host != '':
                        ## TODO : split message > 450 ?
                        chan = self.channel
                        if len(m) > 2 and m[2]:
                            chan = m[2]
                        self.ic.privmsg(chan, unt4.encode(m[1]).encode('utf-8'))
                    elif m[0] == 'CHAN':
                        if m[1] != '':
                            self.channel = m[1]
                        self._rechan()
                    elif m[0] == 'ADDCHAN':
                        if m[1] != '':
                            self._addchan(m[1])
                    elif m[0] == 'EXIT':
                        self.log.debug('Request to close : ' + str(m[1]))
                        self.running = False
                    elif m[0] == 'PORT':
                        if not self.connect_pending:
                            self.doreconnect = True
                    self._pacing(0.0)

            except Queue.Empty:
                time.sleep(self._pacing())	# pacing
            except Exception as e:
                self.log.error('Exception: ' + str(type(e)) + str(e))
                self.connect_pending = False
                self.dumbcnt += 1
                if self.dumbcnt > 2:
                    self.host = ''
                    self.log.debug('Not connected.')
                time.sleep(2.0)
        self.log.info('Exiting')

class fakeirc(object):
    """Relacement dummy class for when irclib is not present."""
    def server(self):
        return self

    def process_once(self, delay=None):
        pass

    def is_connected(self):
        return False

    def disconnect(self):
        pass

    def connect(self, host=None, port=None, nick=None, data=None):
        """Fake an IOError to effectively shut down object."""
        raise IOError('IRC library not present.')

    def close(self, data=None):
        pass

    def oper(self, user=None, pword=None, data=None):
        pass

    def join(self, chan=None, data=None):
        pass

    def mode(self, chan=None, mode=None, data=None):
        pass

    def topic(self, chan=None, topic=None, data=None):
        pass

    def add_global_handler(self, sig=None, cb=None, arg=None, data=None):
        pass

    def ctcp(self, cmd=None, nick=None, ts=None, data=None):
        pass

    def privmsg(self, chan=None, msg=None, data=None):
        pass
    
if __name__ == '__main__':
    h = logging.StreamHandler()
    h.setLevel(logging.DEBUG)
    ann = uscbsrv()
    ann.log.addHandler(h)
    ann.start()
    count = 0
    while not ann.connected():
        print ('waiting for connect...')
        time.sleep(1)
    try:
        ann.clrall()
        ann.set_title('uSCBsrv library test.')
        ann.set_split(tod.tod('now'))
        ann.add_rider(['1.','2','Three FOUR (Five)', 'SIX', '43:01.64'])
        ann.add_rider([])
        ann.add_rider(['1.','2','Three FOUR (Five)', 'SIX', '43:03.64'])
        ann.add_rider(['1.','2','Three FOUR (Five)', 'SIX', '43:04.64'])
        ann.add_rider(['6.','2','Three FOUR (Five)', 'SIX', ''])
        ann.add_rider(['7.','2','Three FOUR (Five)', 'SIX', ''])
        ann.add_rider(['1.','2','Three FOUR (Five)', 'SIX', '43:05.64'])
        ann.add_rider(['1.','2','Three FOUR (Five)', 'SIX', '43:07.64'])
        while True:
           time.sleep(1)
           ann.set_split(tod.tod('now'))
           count += 1
           if count == 40:
               ann.set_port(host='')
           if count == 80:
               ann.set_port(host='192.168.95.16')
    except:
        ann.clrall()
        ann.wait()
        ann.exit()
        ann.join()
        raise
    
    ann.clrall()
    ann.wait()
    ann.exit()
    ann.join()
