
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Comm2 Race Crontrol Application"""

import pygtk
pygtk.require("2.0")

import gtk
import glib
import pango
import gobject

import os
import sys
import logging
import serial
import random

import metarace

from metarace import jsonconfig
from metarace import tod
from metarace import telegraph
from metarace import unt4
from metarace import strops
from metarace import loghandler
from metarace import uiutil
from metarace import gpsclient
from metarace import riderdb

LOGHANDLER_LEVEL = logging.DEBUG
CONFIGFILE = u'comm2.json'
LOGFILE = u'comm2.log'
APP_ID = u'comm2_1.0'  # configuration versioning
FONTFACE = u'Nimbus Sans L Bold Condensed'

# race configurations
CONFIGS = {
 0: [u'neutral', u'comm2', u'-10'],
 1: [u'break', u'comm2', u'5'],
 2: [u'chase', u'comm2', u'5'],
 3: [u'lead', u'moto1', u'1'],
 4: [u'manual', u'comm2', u'10']
 
}

class comm2:
    """Comm2 race control."""

    def quit_cb(self, menuitem, data=None):
        """Quit the application."""
        self.running = False
        self.window.destroy()

    def timeout(self):
        """Update status."""
        # 1: Terminate?
        if not self.running:
            return False
        # 2: Process?
        # watchdog telegraph and gps

        return True	# re-run later

    ## Window methods
    def config_combo_changed_cb(self, combo, data=None):
        """Handle configuration change."""
        i = combo.get_active()
        if i in CONFIGS:
            conf = CONFIGS[i][0]
            source = CONFIGS[i][1]
            corr = CONFIGS[i][2]
            self.remote.send_cmd(u'configuration', conf)
            self.remote.send_cmd(u'positionsrc', source)
            self.correction.set_text(corr)
            self.correction.activate()
            self.log.info(u'comm2: New config: ' + repr(conf))
        else:
            self.remote.send_cmd(u'configuration', u'disabled')
            self.remote.send_cmd(u'positionsrc', None)

    def correction_button_clicked_cb(self, button, data=None):
        """Handle correction button."""
        pass

    def gap_button_clicked_cb(self, entry, data=None):
        """Handle gap button."""
        pass

    def gap_entry_activate_cb(self, entry, data=None):
        """Update manual time gap."""
        et = entry.get_text().decode('utf-8')
        if et:
            nt = tod.str2tod(entry.get_text().decode('utf-8'))
            if nt is not None:
                nts = nt.rawtime(0)
                entry.set_text(nts)
                entry.set_position(-1)
                self.remote.send_cmd(u'timegap', nts)
                self.log.info(u'comm2: Sent timegap ' + nts)
            else:
                self.log.info('Ignored invalid time gap.')
        else:
            entry.set_text(u'')
            self.remote.send_cmd(u'timegap', None)

    def position_button_clicked_cb(self, button, data=None):
        """Return to position automatic."""
        pass

    def position_entry_activate_cb(self, entry, data=None):
        """Manual override position."""
        et = entry.get_text().decode('utf-8').translate(
                                     strops.NUMERIC_UTRANS)
        if et:
            pv = strops.confopt_float(et)
            if pv > 0.0 and pv < 300.0: # arbitrary limits! should be config
                pt = u'{0:0.1f}'.format(pv)
                entry.set_text(pt+u'km')
                entry.set_position(-1)
                self.remote.send_cmd(u'position', pt)
                self.log.info(u'comm2: Manual position override ' + pt)
            else:
                self.log.info(u'Ignored invalid distance.')
        else:
            self.remote.send_cmd(u'position', None)
            entry.set_text(u'')

    def correction_entry_activate_cb(self, entry, data=None):
        """Handle new position/time correction."""
        orig = entry.get_text().decode('utf-8').translate(
                                 strops.NUMERIC_UTRANS)
        if orig:
            et = strops.confopt_float(orig)
            ets = u'{0:+0.0f}'.format(et)
            entry.set_text(ets + u's')
            entry.set_position(-1)
            self.remote.send_cmd(u'poscorrection', ets)
            self.log.info(u'comm2: Position correction ' + ets)
        else:
            entry.set_text(u'')
            self.remote.send_cmd(u'poscorrection', None)

    def gps_button_clicked_cb(self, button, data=None):
        """Handle click on GPS status button."""
        self.log.debug(u'GPS button click.')
        
    def break_button_clicked_cb(self, button, data=None):
        """Clear break entry and report."""
        self.riders.clear()
        self.breakriders.set_text(u'')
        self.remote.send_cmd(u'breakriders', None)

    def populate_rlist(self, riderlist):
        """Scan riderlist and add all onto view."""
        ret = []
        count = 0
        for r in riderlist.split():
            if r in self.startlist:
                count += 1
                ret.append(r)
                nr = [r, u'', u'', unicode(count), '#a0ffa0']
                if r in self.ridermap:
                    nr[1] = self.ridermap[r][u'namestr']
                else:
                    self.log.info(u'Unknown rider in break: ' + repr(r))
                self.riders.append(nr)
            else:
                self.log.info(u'Ignored non-starter: ' + repr(r))
        return u' '.join(ret)

    def overridebreak(self, breaklist):
        """Override break content from transponder passing."""
        rvec = breaklist.split(unichr(unt4.US))
        if len(rvec) == 2:
            loopid = rvec[0]
            newrset = rvec[1]
            self.log.info(u'Override riders in break from ' + loopid)
            self.breakriders.set_text(newrset)
            self.breakriders.activate()
        else:
            pass

    def break_entry_activate_cb(self, entry, data=None):
        """Reformat and send break info to telegraph."""
        dat = entry.get_text().decode('utf-8')
        rlist = None
        self.riders.clear()
        if dat:
            rlist = strops.reformat_riderlist(dat)
            rlist = self.populate_rlist(rlist)
            entry.set_text(rlist)
            entry.set_position(-1)
            self.log.info(u'comm2: Updated riders in break ' + rlist)
        # allow clear by empty
        self.remote.send_cmd(u'breakriders', rlist)
            
    def message_entry_activate_cb(self, entry, data=None):
        """Send a chat message to telegraph."""
        msg = entry.get_text().decode('utf-8')
        entry.set_text(u'')
        if msg:
            self.log.info(u'comm2: ' + msg)
            self.remote.send_cmd(u'chat', msg)

    def app_set_title(self, extra=u''):
        """Update window title from meet properties."""
        title = u'Comm2'
        if self.title:
            title += u': ' + self.title
        self.window.set_title(title)

    def window_destroy_cb(self, window, msg=u''):
        """Handle destroy signal and exit application."""
        if self.dosave:
            self.saveconfig()
        self.log.removeHandler(self.lh)
        self.window.hide()
        glib.idle_add(self.app_destroy_handler)

    def app_destroy_handler(self):
        if self.started:
            self.shutdown()	# threads are joined in shutdown
        # close event and remove main log handler
        if self.loghandler is not None:
            self.log.removeHandler(self.loghandler)
        self.running = False
        # flag quit in main loop
        gtk.main_quit()
        return False

    def shutdown(self, msg=''):
        """Cleanly shutdown threads and close application."""
        self.started = False
        self.window.hide()	# usually already called
        self.remote.exit(msg)
        self.gps.exit(msg)
        print (u'Waiting for remote to exit...')
        self.remote.join()
        print (u'Waiting for gps to exit...')
        self.gps.join()

    def start(self):
        """Start the worker threads."""
        if not self.started:
            self.log.debug(u'App startup.')
            self.remote.start()
            self.gps.start()
            self.started = True

    ## UI callbacks
    def gpsfixchange(self, newmode):
        """Register change in GPS fix."""
        modestr = u'No Fix.'
        if newmode == 2:
            modestr = u'2D Fix.'
            self.gpsicon.set_from_stock(gtk.STOCK_YES,
                                        gtk.ICON_SIZE_MENU)
        elif newmode == 3:
            modestr = u'3D Fix.'
            self.gpsicon.set_from_stock(gtk.STOCK_YES,
                                        gtk.ICON_SIZE_MENU)
        else:
            self.gpsicon.set_from_stock(gtk.STOCK_DIALOG_WARNING,
                                        gtk.ICON_SIZE_MENU)
        self.fix = newmode
        self.gpsbutton.set_tooltip_text(modestr)
        self.log.info(u'GPS change state: ' + modestr)

    def gps_cb(self, mode, msg):
        """Relay GPS fix to telegraph and update fix icon."""
        if mode != self.fix:
            self.gpsfixchange(mode)
        if self.fixemit % 5 == 0:
            self.remote.sendmsg(msg)	# avoid flood on irc chan
        self.fixemit += 1
        return False

    def load_startlist(self, starters):
        """Load a new startlist."""
        self.startlist = []
        for r in starters.split():
            self.startlist.append(r)

    def remote_cb(self, cmd, nick, chan):
        """Handle unt message from remote (in main loop)."""
        if self.remoteuser and self.remoteuser.lower() != nick.lower():
            #self.log.debug(u'Ignoring command from ' + repr(nick))
            return False

        #self.log.debug(u'Command from ' + repr(nick) + u': '
                       #+ repr(cmd.header) + u'::' + repr(cmd.text))
        if cmd.header == u'start':
            self.elapstart = tod.str2tod(cmd.text)
        elif cmd.header == u'chat':
            self.log.info(nick + ': ' + cmd.text)
        elif cmd.header == u'startlist':
            self.load_startlist(cmd.text)
            self.log.info(u'Loaded startlist: ' + unicode(len(self.startlist))
                             + u' riders.')
        elif cmd.header == u'breakoverride':
            self.overridebreak(cmd.text)
        elif cmd.header == u'title':
            self.title.set_text(cmd.text)
        elif cmd.header == u'distance':
            self.distent.set_text(cmd.text + u'km')
        ## add remote control methods here
        return False

    def saveconfig(self):
        """Save current config to disk."""
        cw = jsonconfig.config()
        cw.add_section(u'comm2')
        cw.set(u'comm2', u'id', APP_ID)
        cw.set(u'comm2', u'remoteport', self.remoteport)
        cw.set(u'comm2', u'remotechan', self.remotechan)
        cw.set(u'comm2', u'remoteuser', self.remoteuser)

        cwfilename = os.path.join(self.configpath, CONFIGFILE)
        self.log.debug(u'Saving app config to ' + repr(cwfilename))
        with open(cwfilename , 'wb') as f:
            cw.write(f)

    def loadconfig(self):
        """Load app config from disk."""
        cr = jsonconfig.config({u'comm2':{
               u'id':'',
               u'remoteport':u'',
               u'remotechan':u'',
               u'remoteuser':u''
              }})
        cr.add_section(u'comm2')
        cwfilename = metarace.default_file(CONFIGFILE)
        # read in sysdefaults before checking for config file
        cr.merge(metarace.sysconf, u'comm2')

        # re-set log file
        if self.loghandler is not None:
            self.log.removeHandler(self.loghandler)
            self.loghandler.close()
            self.loghandler = None
        self.loghandler = logging.FileHandler(
                             os.path.join(self.configpath, LOGFILE))
        self.loghandler.setLevel(LOGHANDLER_LEVEL)
        self.loghandler.setFormatter(logging.Formatter(
                       '%(asctime)s %(levelname)s:%(name)s: %(message)s'))
        self.log.addHandler(self.loghandler)

        # check for config file
        try:
            with open(cwfilename, 'rb') as f:
                cr.read(f)
        except Exception as e:
            self.log.error(u'Reading app config: ' + repr(e))

        # set uSCBsrv connection
        self.remoteuser = cr.get(u'comm2', u'remoteuser')
        self.remotechan = cr.get(u'comm2', u'remotechan')
        self.remoteport = cr.get(u'comm2', u'remoteport')
        self.remote.set_portstr(portstr=self.remoteport,
                             channel=self.remotechan)
        if self.remoteuser:
            self.log.info(u'Enabled remote control by: '
                          + repr(self.remoteuser))
        else:
            self.log.info(u'Promiscuous remote control enabled.')

        # configurable log level on UI (does not change log file)
        #self.loglevel = strops.confopt_posint(cr.get(u'comm2', u'loglevel'),
                                              #logging.INFO)
        #self.lh.setLevel(self.loglevel)

        # After load complete - check config and report. This ensures
        # an error message is left on top of status stack. This is not
        # always a hard fail and the user should be left to determine
        # an appropriate outcome.
        cid = cr.get(u'comm2', u'id')
        if cid and cid != APP_ID:
            self.log.error(u'Meet configuration mismatch: '
                           + repr(cid) + u' != ' + repr(APP_ID))

        # load riders from riders.csv
        rdfilename = metarace.default_file(u'riders.csv')
        rdb = riderdb.riderdb()
        rdb.load(rdfilename)
        self.ridermap = rdb.mkridermap()

    def __init__(self, configpath=None):
        """App constructor."""
        # logger and log handler
        self.log = logging.getLogger()
        self.log.setLevel(logging.DEBUG)
        self.loghandler = None	# set in loadconfig to meet dir

        # meet configuration path and options
        if configpath is None:
            configpath = u'.'	# None assumes 'current dir'
        self.configpath = configpath
        self.loglevel = logging.INFO	# UI log window
        self.dosave = False
        self.ridermap = {}
        self.startlist = []

        # hardware connections
        self.remote = telegraph.telegraph()
        self.remoteuser = u''		# match against remote nick
        self.remoteport = u''		# only connect if requested
        self.remotechan = u'#announce'
        self.remote.set_pub_cb(self.remote_cb)
        self.gps = gpsclient.gpsclient(name=u'comm2')
        self.gps.setcb(self.gps_cb)
        self.fix = -1
        self.fixemit = 0
   
        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, u'comm2.ui'))
        self.window = b.get_object('window')
        self.window.maximize()
        self.title = b.get_object('title_lbl')
        self.log_buffer = b.get_object('log_buffer')
        self.log_view = b.get_object('log_view')
        #self.log_view.modify_font(pango.FontDescription("monospace 9"))
        self.log_scroll = b.get_object('log_box').get_vadjustment()
        self.gpsbutton = b.get_object('gps_button')
        self.gpsicon = b.get_object('gps_image')
        self.breakriders = b.get_object('break_entry')
        self.correction = b.get_object('correction_entry')
        self.distent = b.get_object('position_entry')

        # add model/view for breakriders
        self.riders = gtk.ListStore(gobject.TYPE_STRING,  # no.
                                    gobject.TYPE_STRING,  # namestr
                                    gobject.TYPE_STRING,  # timestr (reserved)
                                    gobject.TYPE_STRING,  # bunchcnt
                                    gobject.TYPE_STRING)  # bunccol
        t = gtk.TreeView(self.riders)
        self.view = t
        t.set_reorderable(False)
        t.set_rules_hint(True)
        t.set_headers_visible(False)
        t.modify_font(pango.FontDescription(FONTFACE))
        uiutil.mkviewcoltxt(t, 'No.', 0,calign=1.0,width=45)
        uiutil.mkviewcoltxt(t, 'Rider', 1,expand=True,fixed=True)
        #uiutil.mkviewcoltxt(t, '', 2,calign=1.0)
        uiutil.mkviewcoltxt(t, 'Cnt', 3,width=50,bgcol=4,calign=0.5)
        t.show()
        ts = b.get_object('announce_box')
        ts.add(t)


        b.connect_signals(self)

        # run state
        self.running = True
        self.started = False

        # format and connect status handlers
        #f = logging.Formatter()
        self.lh = loghandler.textViewHandler(self.log_buffer,
                      self.log_view, self.log_scroll)

        self.lh.setLevel(logging.INFO)	# show info+ on status bar
        #self.lh.setFormatter(f)
        self.log.addHandler(self.lh)

        # start gps timer
        glib.timeout_add_seconds(5, self.timeout)

def main():
    """Run the road meet application."""
    configpath = None

    # expand configpath on cmd line to realpath _before_ doing chdir
    if len(sys.argv) > 2:
        print(u'usage: comm2 [configdir]\n')
        sys.exit(1)
    elif len(sys.argv) == 2:
        rdir = sys.argv[1]
        if not os.path.isdir(rdir):
            rdir = os.path.dirname(rdir)
        configpath = os.path.realpath(rdir)

    metarace.init()
    if configpath is not None:
        os.chdir(configpath)
    app = comm2(configpath)
    app.loadconfig()
    app.window.show()
    app.start()
    try:
        metarace.mainloop()
    except:
        app.shutdown(u'Exception from main loop.')
        raise

if __name__ == '__main__':
    main()

