
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Relay timer channel."""

import pygtk
pygtk.require("2.0")

import gtk
import glib
import pango
import gobject

import os
import sys
import logging
import socket
import random

import metarace

from metarace import jsonconfig
from metarace import tod
from metarace import telegraph
from metarace import unt4
from metarace import strops
from metarace import loghandler
from metarace import uiutil

LOGHANDLER_LEVEL = logging.DEBUG
CONFIGFILE = u'relay.json'
LOGFILE = u'relay.log'
APP_ID = u'relay_1.0'  # configuration versioning
ENCODING = 'iso8859-15'
SERIAL_BAUD = 9600
DEFAULTPORT = 2152
DEFAULTADDR = u'localhost'

class relay:
    """Timer channel relay application."""

    def quit_cb(self, menuitem, data=None):
        """Quit the application."""
        self.running = False
        self.window.destroy()

    def uscb_activate_cb(self, menuitem, data=None):
        """Request a re-connect to the uSCBsrv IRC connection."""
        if self.uscbport:
            self.log.info(u'Requesting re-connect to announcer: ' + 
                           repr(self.uscbport) + repr(self.uscbchan))
        else:
            self.log.info(u'Announcer not configured.')
            # but still call into uscbsrv...
        self.scb.set_portstr(portstr=self.uscbport,
                             channel=self.uscbchan,
                             force=True)
            
    ## Help menu callbacks
    def menu_help_docs_cb(self, menuitem, data=None):
        """Display program help."""
        metarace.help_docs(self.window)

    def menu_help_about_cb(self, menuitem, data=None):
        """Display metarace about dialog."""
        metarace.about_dlg(self.window)

    ## Window methods
    def app_set_title(self, extra=u''):
        """Update window title from meet properties."""
        title = u'Timer Frame Relay'
        if self.title:
            title += u': ' + self.title
        self.window.set_title(title)

    def app_destroy_cb(self, window, msg=u''):
        """Handle destroy signal and exit application."""
        self.log.removeHandler(self.sh)
        self.window.hide()
        self.log.info(u'App destroyed. ' + msg)
        glib.idle_add(self.app_destroy_handler)

    def app_destroy_handler(self):
        if self.started:
            self.shutdown()	# threads are joined in shutdown
        # close event and remove main log handler
        if self.loghandler is not None:
            self.log.removeHandler(self.loghandler)
        self.running = False
        # flag quit in main loop
        gtk.main_quit()
        return False

    def shutdown(self, msg=''):
        """Cleanly shutdown threads and close application."""
        self.started = False

        self.window.hide()	# usually already called
        self.remote.exit(msg)
        self.close_display()
        print (u'Waiting for remote to exit...')
        self.remote.join()

    def start(self):
        """Start the timer and rfu threads."""
        if not self.started:
            self.log.debug(u'App startup.')
            self.remote.start()
            self.started = True

    ## UI callbacks

    def bmp2dimg(self, filename):
        """Try and load the supplied image as dimg array."""
        try:
            pb = gtk.gdk.pixbuf_new_from_file(filename)
            w = pb.get_width()
            h = pb.get_height()
            if w != DISPLAYW or h != DISPLAYH:
                self.log.info(u'Source dimension: '
                               + repr(w) + u'x' + repr(h)
                               + u', scaling image.')
                pb = pb.scale_simple(DISPLAYW, DISPLAYH, gtk.gdk.INTERP_HYPER)
            w = pb.get_width()
            h = pb.get_height()
            self.log.debug(u'Read image: ' + repr(w) + u'x' + repr(h))
            px = pb.get_pixels()
            pxcount = 3	# RGB
            if pb.get_has_alpha():
                pxcount = 4	# RGBA
            self.log.debug(u'Total pixels: ' + repr(len(px)/pxcount))
            data = bytearray()
            pcount = 0
            acc = 0x00			# pixel accumulator
            for i in range(0,16):	# each row
                rof = i * 192 * pxcount
                for j in range(0,192):	# each col
                    pof = rof + j * pxcount
                    mval = 0
                    for k in range(0,3):
                        mval += ord(px[pof+k])
                    acc <<= 1
                    if mval > 384:
                        acc |= 0x01
                    pcount += 1
                    if pcount%8 == 0:
                        data.append(acc)
                        acc = 0x00
            self.log.debug(u'Data packed mono: '
                            + repr(len(data)) + u' bytes')
            self.logo = data
        except Exception as e:
            self.log.error(u'Loading logo file: ' + repr(e))

    ## App functions

    def reset(self):
        """Reset run state."""
        pass

    def serialwrite(self, cmd):
        """Output command blocking."""
        try:
            if self.scb:
                #self.scb.write(cmd)
                self.scb.sendto((cmd+u'\n').encode(ENCODING, u'ignore'),
                                self.sendto)
        except Exception as e:
            self.log.error(u'Writing to scoreboard: ' + repr(e))

    def parse_reply(self, rbuf=''):
        """Ignore reply."""
        self.log.debug(u'Read: ' + repr(rbuf))

    def serialread(self):
        """Read from serial blocking."""
        try:
            if self.scb:
                self.parse_reply(self.scb.read(1024))
        except Exception as e:
            self.log.error(u'Reading from port: ' + repr(e))

    def remote_cb(self, cmd, nick, chan):
        """Handle unt message from remote (in main loop)."""
        if self.remoteuser and self.remoteuser.lower() != nick.lower():
            self.log.debug(u'Ignoring command from ' + repr(nick))
            return False
        #self.log.debug(u'Command from ' + repr(nick) + u': '
                       #+ repr(cmd.header) + u'::' + repr(cmd.text))
        self.serialwrite(cmd.pack())
        return False

    def close_display(self):
        """Close port."""
        if self.scb is not None:
            self.log.info(u'Closing port.')
            self.scb.close()	# serial port close
            self.scb = None		# release handle

    def setport(self, portstr):
        """Set the addr/port tuple."""
        addr = DEFAULTADDR
        port = DEFAULTPORT
        avec = portstr.split(u':', 1)
        if len(avec) > 1:
            try:
                port = int(avec[1])
            except:
                pass
        if len(avec) > 0:
            addr = avec[0]
        self.applbl.set_text(u'Relay to: ' + addr + u':' + unicode(port))
        self.sendto = (addr,port)
            
    def reconnect_display(self):
        """Re-connect to display serial port."""
        self.close_display()
        self.log.info('Connecting port: ' + repr(self.port))
        self.setport(self.port)
        try:
            self.scb = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            self.scb.settimeout(0.2)
        except Exception as e:
            self.log.error(u'Opening serial port: ' + repr(e))
            self.scb = None

    def loadconfig(self):
        """Load app config from disk."""
        cr = jsonconfig.config({u'relay':{
               u'id':'',
               u'port':u'',
               u'remoteport':u'',
               u'remotechan':u'',
               u'remoteuser':u'',
               u'loglevel':unicode(logging.INFO)
              }})
        cr.add_section(u'relay')
        cwfilename = metarace.default_file(CONFIGFILE)
        # read in sysdefaults before checking for config file
        cr.merge(metarace.sysconf, u'relay')

        # re-set log file
        if self.loghandler is not None:
            self.log.removeHandler(self.loghandler)
            self.loghandler.close()
            self.loghandler = None
        self.loghandler = logging.FileHandler(
                             os.path.join(self.configpath, LOGFILE))
        self.loghandler.setLevel(LOGHANDLER_LEVEL)
        self.loghandler.setFormatter(logging.Formatter(
                       '%(asctime)s %(levelname)s:%(name)s: %(message)s'))
        self.log.addHandler(self.loghandler)

        # check for config file
        try:
            with open(cwfilename, 'rb') as f:
                cr.read(f)
        except Exception as e:
            self.log.error(u'Reading app config: ' + repr(e))

        # set uSCBsrv connection
        self.remoteuser = cr.get(u'relay', u'remoteuser')
        self.remotechan = cr.get(u'relay', u'remotechan')
        self.remoteport = cr.get(u'relay', u'remoteport')
        self.remote.set_portstr(portstr=self.remoteport,
                                channel=self.remotechan)
        if self.remoteuser:
            self.log.info(u'Enabled remote control by: '
                          + repr(self.remoteuser))
        else:
            self.log.info(u'Promiscuous remote control enabled.')

        # set display serial port
        self.port = cr.get(u'relay', u'port')
        self.reconnect_display()

        # configurable log level on UI (does not change log file)
        self.loglevel = strops.confopt_posint(cr.get(u'relay', u'loglevel'),
                                              logging.INFO)
        self.sh.setLevel(self.loglevel)

        # After load complete - check config and report. This ensures
        # an error message is left on top of status stack. This is not
        # always a hard fail and the user should be left to determine
        # an appropriate outcome.
        cid = cr.get(u'relay', u'id')
        if cid and cid != APP_ID:
            self.log.error(u'Meet configuration mismatch: '
                           + repr(cid) + u' != ' + repr(APP_ID))

    def __init__(self, configpath=None):
        """App constructor."""
        # logger and log handler
        self.log = logging.getLogger()
        self.log.setLevel(logging.DEBUG)
        self.loghandler = None	# set in loadconfig to meet dir

        # meet configuration path and options
        if configpath is None:
            configpath = u'.'	# None assumes 'current dir'
        self.configpath = configpath
        self.loglevel = logging.DEBUG	# UI log window

        # hardware connections
        self.remote = telegraph.telegraph()
        self.remoteuser = u''		# match against remote nick
        self.remoteport = u''		# only connect if requested
        self.remotechan = u'#announce'
        self.remote.set_pub_cb(self.remote_cb)
        self.port = u''			# re-set in loadconfig
        self.scb = None
        self.sendto = (DEFAULTADDR, DEFAULTPORT)
        self.obuf = []			# current output buffer
        self.lbuf = []			# previous output
        for j in range(0,8):
            self.obuf.append(u''.ljust(8))
            self.lbuf.append(u'01234567')
   
        b = gtk.Builder()
        b.add_from_file(os.path.join(metarace.UI_PATH, u'relay.ui'))
        self.window = b.get_object('appwin')
        self.status = b.get_object('status')
        self.context = self.status.get_context_id('metarace relay')
        self.applbl = b.get_object('applbl')
        b.connect_signals(self)

        # run state
        self.running = True
        self.started = False

        # format and connect status handlers
        f = logging.Formatter('%(levelname)s:%(name)s: %(message)s')
        self.sh = loghandler.statusHandler(self.status, self.context)
        self.sh.setLevel(logging.DEBUG)	# show info+ on status bar
        self.sh.setFormatter(f)
        self.log.addHandler(self.sh)

def main():
    """Run the road meet application."""
    configpath = None

    # expand configpath on cmd line to realpath _before_ doing chdir
    if len(sys.argv) > 2:
        print(u'usage: relay [configdir]\n')
        sys.exit(1)
    elif len(sys.argv) == 2:
        rdir = sys.argv[1]
        if not os.path.isdir(rdir):
            rdir = os.path.dirname(rdir)
        configpath = os.path.realpath(rdir)

    metarace.init()
    if configpath is not None:
        os.chdir(configpath)
    app = relay(configpath)
    app.loadconfig()
    app.window.show()
    app.start()
    try:
        metarace.mainloop()
    except:
        app.shutdown(u'Exception from main loop.')
        raise

if __name__ == '__main__':
    main()

