
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Graphic Overlay Hackup."""

import pygtk
pygtk.require("2.0")

import gtk
import glib
import pango
import pangocairo
import cairo
import random
import logging
import os
import csv
import rsvg

import metarace
from metarace import jsonconfig
from metarace import strops
from metarace import telegraph
from metarace import printing
from metarace import unt4
from metarace import tod
from metarace import ucsv

# Globals
CONFIGFILE = u'gfx_panel.json'
LOGFILE = u'gfx_panel.log'
MONOFONT = u'unifont'	# fixed-width font for text panels
STDFONT = u'Nimbus Sans L Bold Italic Condensed' # proportional font for gfx
NOHDR = [u'Start', u'start', u'time', u'Time', u'']
DEFAULT_WIDTH = 544
DEFAULT_HEIGHT = 306
# http://graphics.stanford.edu/courses/cs248-98-fall/Final/q1.html
ARC_TO_BEZIER = 0.55228475

# colours !@#
DARK_R = 0.15765
DARK_G = 0.14588
DARK_B = 0.15373
LIGHT_R = 0.85
LIGHT_G = 0.85
LIGHT_B = 0.85

# Userkeys (roadtt mode)
KEY_START = 'tab'
KEY_STANDINGS = 'space'
KEY_FINISH = 'return'
KEY_CLEAR = 'backspace'

def scaleup(c, s):
    """Scale colour value roughly up toward 1."""
    return 1.0-(s - s*c)

def roundedrec(cr,x,y,w,h,radius_x=8,radius_y=8):
    """Draw a rounded rectangle."""

    #from mono moonlight aka mono silverlight
    #test limits (without using multiplications)
    # http://graphics.stanford.edu/courses/cs248-98-fall/Final/q1.html
    ARC_TO_BEZIER = 0.55228475
    if radius_x > w - radius_x:
        radius_x = w / 2
    if radius_y > h - radius_y:
        radius_y = h / 2

    #approximate (quite close) the arc using a bezier curve
    c1 = ARC_TO_BEZIER * radius_x
    c2 = ARC_TO_BEZIER * radius_y
    cr.new_path();
    cr.move_to ( x + radius_x, y)
    cr.rel_line_to ( w - 2 * radius_x, 0.0)
    cr.rel_curve_to ( c1, 0.0, radius_x, c2, radius_x, radius_y)
    cr.rel_line_to ( 0, h - 2 * radius_y)
    cr.rel_curve_to ( 0.0, c2, c1 - radius_x, radius_y, -radius_x, radius_y)
    cr.rel_line_to ( -w + 2 * radius_x, 0)
    cr.rel_curve_to ( -c1, 0, -radius_x, -c2, -radius_x, -radius_y)
    cr.rel_line_to (0, -h + 2 * radius_y)
    cr.rel_curve_to (0.0, -c2, radius_x - c1, -radius_y, radius_x, -radius_y)
    cr.close_path ()

def draw_text(cr, pr, oh, x1, x2, msg, align=0, invert=False,
                                    font=None, colour=False):
    if msg is not None:
        cr.save()
        cr.set_line_cap(cairo.LINE_CAP_ROUND)
        cr.set_line_join(cairo.LINE_JOIN_ROUND)
        cr.set_line_width(4.0)
        maxw = x2 - x1
        l = pr.create_layout()
        l.set_font_description(font)
        l.set_text(msg)
        (tw,th) = l.get_pixel_size()
        oft = 0.0
        if align != 0 and tw < maxw:
            oft = align * (maxw - tw)	# else squish
        cr.move_to(x1+oft, oh)	# move before applying conditional scale
        if tw > maxw:
            cr.scale(float(maxw)/float(tw),1.0)
        pr.update_layout(l)

        # fill 
        if invert:
            cr.set_source_rgb(1.0,1.0,1.0)
        elif colour:
            cr.set_source_rgb(0.992616,0.86275,0.0078431)
        else:
            cr.set_source_rgb(0.0,0.0,0.0)

        pr.show_layout(l)
        cr.restore()

def tod2key(tod=None):
    """Return a key from the supplied time of day."""
    ret = None
    if tod is not None:
        ret = int(tod.truncate(0).timeval)
    return ret

class gfx_panel(object):
    """Graphical Panel."""
 
    def show(self):
        self.window.show()

    def hide(self):
        self.window.show()

    def start(self):
        """Start threads."""
        if not self.started:
            self.scb.start()
            self.scb.set_pub_cb(self.msg_cb)
            self.started = True

    def shutdown(self):
        """Cleanly shutdown."""
        self.scb.exit()
        self.scb.join()
        self.started = False

    def window_destroy_cb(self, window):
        """Handle destroy signal."""
        if self.started:
            self.shutdown()
        self.running = False
        gtk.main_quit()
    
    def area_expose_event_cb(self, widget, event):
        """Update desired portion of drawing area."""
        x , y, width, height = event.area
        widget.window.draw_drawable(widget.get_style().fg_gc[gtk.STATE_NORMAL],
                                    self.area_src, x, y, x, y, width, height)
        return False

    def general_clearing(self):
        """Handle a GC request and update dbrows accordingly."""
        nr = u'X' * self.monocols
        for i in range(0, self.monorows):
            self.dbrows[i] = nr

    def draw_monotext(self, cr, pr):
        """Draw the plain text scoreboard."""
        ## currently fixed to 32x8 for consistency
        lh = self.ph/self.monorows
        nt = u' '*self.monocols
        for i in range(0,self.monorows):
            if i in self.dbrows:
                nt = self.dbrows[i]
            curof = lh * i + self.monofontoffset
            draw_text(cr, pr, curof, 0.0, self.pw, nt,
                          font=self.monofontdesc, invert=True)
        # watermark
        
    def draw_image(self, cr, pr):
        """Draw an all-image overlay."""
        ov = self.overlays[self.curov]
        cr.set_source_surface(ov[u'image'])
        cr.paint()
        
    def draw_align(self, cr, pr):
        """Draw the alignment screen."""
        # re-set BG to solid white
        cr.set_source_rgb(1.0,1.0,1.0)
        cr.paint()

        # place alignment marks
        cr.set_source_rgb(0.1,0.1,0.1)
        cr.rectangle(0.0,0.0,self.pw,self.ph)
        cr.fill()

        cr.set_source_rgb(1.0,1.0,1.0)
        cr.set_line_width(0.75)
        lx = 0.2*self.pw
        ly = 0.2*self.ph
        bx = 0.8*self.pw
        by = 0.8*self.ph
        llx = 0.02*self.pw
        lly = 0.02*self.ph
        bbx = 0.98*self.pw
        bby = 0.98*self.ph

        # TL 10% crop
        cr.move_to(llx,ly)
        cr.line_to(llx,lly)
        cr.line_to(lx,lly)
        # BL 10% crop
        cr.move_to(llx,by)
        cr.line_to(llx,bby)
        cr.line_to(lx,bby)
        # TR 10% crop
        cr.move_to(bbx,ly)
        cr.line_to(bbx,lly)
        cr.line_to(bx,lly)
        # BR 10% crop
        cr.move_to(bbx,by)
        cr.line_to(bbx,bby)
        cr.line_to(bx,bby)
        # TL diagonal
        cr.move_to(0.0,0.0)
        cr.line_to(lx, ly)
        # BL diagonal
        cr.move_to(0.0,self.ph)
        cr.line_to(lx,by)
        # BR diagonal
        cr.move_to(self.pw, self.ph)
        cr.line_to(bx,by)
        # TR diagonal
        cr.move_to(self.pw, 0.0)
        cr.line_to(bx,ly)
        cr.stroke()
         
        # Reference circle
        cr.set_source_rgb(0.5,0.5,1.0)
        xp = 0.3*self.pw
        yp = 0.25*self.ph
        rad = 0.15*self.ph
        cr.move_to(xp, yp)
        cr.arc(xp, yp, rad, 0, 6.3)
        cr.fill()
        cr.set_source_rgb(0.0,0.0,0.5)
        cr.move_to(xp-rad, yp)
        cr.line_to(xp+rad, yp)
        cr.move_to(xp, yp-rad)
        cr.line_to(xp, yp+rad)
        cr.stroke()
        
        # halfmarks
        cr.set_source_rgb(1.0,0.0,0.0)
        cr.set_dash([10.0,7.5])
        cr.move_to(0.0,0.5*self.ph)
        cr.line_to(self.pw,0.5*self.ph)
        cr.stroke()
        cr.set_source_rgb(0.0,1.0,0.0)
        cr.move_to(0.5*self.pw,0.0)
        cr.line_to(0.5*self.pw,self.ph)
        cr.stroke()

        # font desc
        cr.set_source_rgb(1.0,1.0,1.0)
        draw_text(cr, pr, 0.51*self.ph,
                          0.50*self.pw, 0.99*self.pw,
                          u'First LAST 1h23:45.67',
                          font=self.stdfontdesc, invert=True)

        cr.move_to(0.50*self.pw, 0.70*self.ph)
        cr.line_to(self.pw, 0.70*self.ph)
        nh = 0.70*self.ph + (self.ph/8.0)
        cr.move_to(0.50*self.pw, nh)
        cr.line_to(self.pw, nh)
        cr.stroke()
 
        draw_text(cr, pr, 0.70*self.ph+self.monofontoffset,
                          0.50*self.pw, 0.99*self.pw,
                          u'0123456789ABCDEF',
                          font=self.monofontdesc, invert=True)

        # draw pixel patch
        cr.save()
        cr.translate(0.5*self.pw-120,0.5*self.ph+20)
        cr.rectangle(0.0,0.0,100.0,100.0)
        cr.clip()
        cr.set_source_surface(self.overlays[u'align'][u'patch'])
        cr.paint()
        cr.restore()

    def draw_clock(self, cr, pr):
        """Draw the analog facility clock with start pips."""
        pass

    def title_pane(self, cr, pr):
        pass

    def draw_gfx(self, cr, pr):
        """Draw the whole graphical panel."""
        ov = self.overlays[u'gfx']
        self.title_pane(cr, pr)
        # logos
        ml = printing.image_elem(self.growrad,self.growrad,
                                 self.growmlb,1.85*self.growh,
                                 0.0, 0.5,
                                 ov[u'mainlogo'])
        ml.draw(cr, pr)
        sl = printing.image_elem(self.growmrt,self.growrad,
                                 self.pw-self.growrad,1.85*self.growh,
                                 1.0, 0.5,
                                 ov[u'mainlogo'])
        sl.draw(cr, pr)
        if self.title:
            draw_text(cr, pr, self.growth + self.stdfontoffset,
                          self.growmlt, self.growmrb,
                          self.title, align = 0.0,
                          font=self.stdfontdesc, colour=True)
        if self.subtitle:
            draw_text(cr, pr, self.growh + self.stdfontoffset,
                          self.growmlt, self.growmrb,
                          self.subtitle, align = 0.0,
                          font=self.stdfontdesc, colour=True)
            pass
        
        rowh = 2*self.growof
        self.left_panel(cr, pr, rowh)
        draw_text(cr, pr, rowh + self.stdfontoffset,
                          self.growll+self.growrad,
                          self.growll+self.growlbw,
                          u'dnf',
                          align = 0.5,
                          font=self.stdfontdesc, invert=True)
        self.mid_panel(cr, pr, rowh)
        draw_text(cr, pr, rowh + self.stdfontoffset,
                          self.growmlt, self.growmrb,
                          u'16 Anthony GIACOPPO (VIC)',
                          align = 0.0,
                          font=self.stdfontdesc)
        self.right_panel(cr, pr, rowh)
        draw_text(cr, pr, rowh + self.stdfontoffset,
                          self.growrr-self.growrtw,
                          self.growrr-2*self.growrad,
                          u'4:23.32',
                          align = 1.0,
                          font=self.stdfontdesc, invert=True)

    def area_redraw(self):
        """Lazy full area redraw method."""

        # reset
        cr = self.area_src.cairo_create()
        pr = pangocairo.CairoContext(cr)
        cr.identity_matrix()

        # adjust geometry
        cr.translate(self.xoft, self.yoft)
        cr.scale(self.width/(self.xscale*self.pw),
                 self.height/(self.yscale*self.ph))

        # clear
        cr.set_source_rgb(0.0,0.0,0.0)
        cr.paint()

        # draw overlay
        if self.curov in self.overlays:
            if self.curov == u'align':
                self.draw_align(cr, pr)
            elif self.curov == u'gfx':
                self.draw_gfx(cr, pr)
            elif self.curov == u'0':
                self.draw_clock(cr, pr)
            elif self.curov == u'1':
                self.draw_monotext(cr, pr)
            elif self.curov in [u'2', u'3']:
                self.draw_image(cr, pr)
            else:
                self.log.error(u'missing handler for overlay '
                                 + repr(self.curov))
        return

        ov = self.overlays[self.curov]
        if self.curov == 1:	# this is standings (raw) mode
            # 'expose' background rows
            cr.save()
            cr.rectangle(0,0, self.vpw,ov['clipstart']
                           + len(self.rows) * ov['rowheight'])
            cr.clip()
            cr.set_source_surface(ov['surface'])
            cr.paint()
            cr.restore()

            # Draw title if available
            if self.title:
                draw_text(cr, pr, ov['title_h'], ov['title_x1'],
                                  ov['title_x2'], self.title)

            curof = ov['rowstart']
            count = 0
            ft = None
            for row in self.rows:
                if row[0]:
                    draw_text(cr, pr, curof,70,150,row[0],align=0.5,
                              invert=True) 
                if row[1]:
                    draw_text(cr, pr, curof,156,200,row[1],align=1.0)
                if row[2]:
                    draw_text(cr, pr, curof,208,562,row[2])
                if row[3]:
                    tstr = row[3]
                    nt = tod.str2tod(row[3])
                    if nt:
                        if ft is None:
                            ft = nt
                            tstr = nt.rawtime(0)
                        else:
                            tstr = u'+'+(nt-ft).rawtime(0)
                    draw_text(cr, pr, curof,590,730,tstr,align=1.0,
                                       invert=True)
                curof += ov['rowheight']
                count += 1
                if count >= ov['maxrows']:
                    break
        elif self.curov == 2:	# Start line
            if ov['riderno']:
                curof = ov['rowstart']
                cr.set_source_surface(ov['surface'])
                cr.paint()
                draw_text(cr, pr, curof,192,268,ov['label'],align=0.5,
                              invert=True) 
                draw_text(cr, pr, curof,280,330,ov['riderno'],align=1.0)
                if ov['ridername']:
                    draw_text(cr, pr, curof,340,680,ov['ridername'])
                if ov['countdown']:
                    draw_text(cr, pr, curof,764,808,ov['countdown'],align=0.5,
                                       invert=True)

                # draw bulbs
                cr.set_source_rgb(0.1, 0.0, 0.0)
                cr.move_to(750.0, 514.0)
                cr.arc(750.0, 514.0, 10.5, 0, 6.3)
                cr.fill()
                if ov['redbulb']:
                    cr.set_source_rgb(1.0, 0.0, 0.0)
                    cr.arc(750.0, 514.0, 9.5, 0, 6.3)
                    cr.fill()
                cr.set_source_rgb(0.0, 0.1, 0.0)
                cr.move_to(824.0, 514.0)
                cr.arc(824.0, 514.0, 10.5, 0, 6.3)
                cr.fill()
                if ov['greenbulb']:
                    cr.set_source_rgb(0.0, 1.0, 0.0)
                    cr.arc(824.0, 514.0, 9.5, 0, 6.3)
                    cr.fill()
        elif self.curov == 3:	# Finish line
            if ov['riderno']:
                curof = ov['rowstart']
                cr.set_source_surface(ov['surface'])
                cr.paint()
                draw_text(cr, pr, curof,192,268,ov['rank'],align=0.5,
                              invert=True) 
                draw_text(cr, pr, curof,280,330,ov['riderno'],align=1.0)
                if ov['ridername']:
                    draw_text(cr, pr, curof,340,680,ov['ridername'])
                if ov['elapsed']:
                    draw_text(cr, pr, curof,720,850,ov['elapsed'],align=1.0,
                                       invert=True)
        
    def number_box(self, cr, hof):
        cr.move_to(255.0,hof-5.0)
        cr.rectangle(255.0,hof-5.0,90.0,80.0)
        cr.set_line_cap(cairo.LINE_CAP_ROUND)
        cr.set_line_join(cairo.LINE_JOIN_ROUND)
        cr.set_source_rgb(0.95,0.75,0.75)
        cr.fill()
        cr.rectangle(257.0,hof-3.0,90.0,80.0)
        cr.set_source_rgb(0.5,0.3,0.3)
        cr.fill()
        cr.rectangle(256.0,hof-4.0,90.0,80.0)
        cr.set_source_rgb(0.9,0.2,0.2)
        cr.fill()
        

    def area_configure_event_cb(self, widget, event):
        """Re-configure the drawing area and redraw the base image."""
        x, y, width, height = widget.get_allocation()
        ow = 0
        oh = 0
        if self.area_src is not None:
            ow, oh = self.area_src.get_size()
        if width > ow or height > oh:
            self.area_src = gtk.gdk.Pixmap(widget.window, width, height)
        self.width = float(width)
        self.height = float(height)
        self.area_redraw()
        self.area.queue_draw()
        return True

    def clear(self):
        """Clear all elements."""
        self.title = u''
        self.subtitle = u''
        self.rows = []
        self.draw_and_update()
        
    def set_title(self, tstr=''):
        """Draw title and update."""
        self.title = tstr
        self.draw_and_update()

    def set_subtitle(self, tstr=''):
        """Draw subtitle and update."""
        self.subtitle = tstr
        self.draw_and_update()

    def add_row(self, msg=''):
        """Split row and then append to self.rows"""
        sr = msg.split(chr(unt4.US))
        if len(sr) > 4:
            rank = sr[0]
            if rank and rank.isdigit():
                rank += u'.'
            self.rows.append([rank, sr[1], sr[2], sr[4]])
        self.doredraw = True

    def loadconfig(self):
        """Load config from disk."""
        cr = jsonconfig.config({u'gfx_panel':{
                                    u'fullscreen':False,
                                    u'remoteport':u'',
                                    u'remoteuser':u'',
                                    u'remotechan':u'#agora',
                                    u'monofontsize':40,
                                    u'monofont':MONOFONT,
                                    u'monofontoffset':-4.0,
                                    u'stdfontoffset':-4.0,
                                    u'monorows':8,
                                    u'monocols':32,
                                    u'stdfontsize':20,
                                    u'stdfont':STDFONT,
                                    u'curov':u'blank'
                                 }})
        cr.add_section(u'gfx_panel')
        cwfilename = metarace.default_file(CONFIGFILE)
        cr.merge(metarace.sysconf, u'gfx_panel')

        # check for config file
        try:
            with open(cwfilename, 'rb') as f:
                cr.read(f)
        except Exception as e:
            self.log.error(u'Reading app config: ' + repr(e))

        if strops.confopt_bool(cr.get(u'gfx_panel', u'fullscreen')):
            self.window.fullscreen()

        self.remoteport = cr.get(u'gfx_panel', u'remoteport')
        self.remotechan = cr.get(u'gfx_panel', u'remotechan')
        self.remoteuser = cr.get(u'gfx_panel', u'remoteuser')
        self.scb.set_portstr(portstr=self.remoteport,
                             channel=self.remotechan)
        if self.remoteuser:
            self.log.info(u'Enabled remote control by: '
                          + repr(self.remoteuser))
        else:
            self.log.info(u'Promiscuous remote control enabled.')

        self.stdfont = cr.get(u'gfx_panel',u'stdfont')
        self.monofont = cr.get(u'gfx_panel',u'monofont')
        self.monofontoffset = strops.confopt_float(cr.get(u'gfx_panel',
                                                      u'monofontoffset'))
        self.monofontsize = strops.confopt_float(cr.get(u'gfx_panel',
                                                      u'monofontsize'))
        self.stdfontoffset = strops.confopt_float(cr.get(u'gfx_panel',
                                                      u'stdfontoffset'))
        self.stdfontsize = strops.confopt_float(cr.get(u'gfx_panel',
                                                      u'stdfontsize'))
        self.set_monofontsize(self.monofontsize)
        self.set_stdfontsize(self.stdfontsize)

        self.monorows = strops.confopt_posint(cr.get(u'gfx_panel',
                                  u'monorows'), 8)
        self.monocols = strops.confopt_posint(cr.get(u'gfx_panel',
                                  u'monocols'), 32)
        self.curov = cr.get(u'gfx_panel', u'curov')

        if cr.has_option(u'gfx_panel', u'geometry'):
            self.set_geometry(unichr(unt4.US).join(cr.get(u'gfx_panel', u'geometry')))

        self.general_clearing()

        # try to load start line infos
        try:
            rlist = []
            with open(metarace.default_file(u'startlist.csv'),'rb') as f:
                cr = ucsv.UnicodeReader(f)
                for r in cr:
                    key = None
                    st = None
                    bib = u''
                    series = u''
                    name = u''
                    next = None
                    # load rider info
                    # start, no, series, name, cat
                    if len(r) > 0 and r[0] not in NOHDR: # time & no provided
                        st = tod.str2tod(r[0])
                        if len(r) > 1:  # got bib
                            bib = r[1]
                        if len(r) > 2:  # got series
                            series = r[2]
                        if len(r) > 3:  # got name
                            name = r[3]
                        if st is not None:
                            # enough data to add a starter
                            key = tod2key(st)
                            nr = [st, bib, series, name, next]
                            self.ridermap[key] = nr
                            rlist.append(key)
            # sort startlist and build list linkages
            curoft = tod2key(tod.tod(u'now'))
            self.currider = None
            rlist.sort()
            prev = None
            for r in rlist:
                if prev is not None:
                    self.ridermap[prev][4] = r  # prev -> next
                prev = r
                if self.currider is None and r > curoft:
                    self.currider = r
                    rvec  = self.ridermap[r]
                    stxt = tod.tod(r).meridian()
                    sno = rvec[1]
                    sname = rvec[3]
                    self.log.info(u'Setting first rider to: '
                          + u','.join([sno, sname]) + u' @ ' + stxt)
            # last link will be None
        except Exception as e:
            # always an error - there must be startlist to continue
            self.log.error(u'Error loading from startlist: '
                             + unicode(e))

    def draw_and_update(self, data=None):
        """Redraw in main loop, not in timeout."""
        self.area_redraw()
        self.area.queue_draw()
        return False

    def timeout(self, data=None):
        """Handle timeout."""

        # 1: Terminate?
        if not self.running:
            return False

        # 2: Process?
        try:
            ntime = tod.tod(u'now')
            ntod = ntime.truncate(0)
            if ntime >= self.nc.truncate(1):
                self.tod = ntod
                self.nc += tod.ONE
                self.process_timeout()
            else:
                self.log.debug(u'Timeout called early: ' + ntime.rawtime())
                # no need to advance, desired timeout not yet reached
        except Exception as e:
            self.log.error(u'Timeout: ' + unicode(e))

        # 3: Re-Schedule
        tt = tod.tod(u'now')+tod.tod(u'0.01')
        while self.nc < tt:     # ensure interval is positive
            if tod.MAX - tt < tod.ONE:
                self.log.debug(u'Midnight rollover.')
                break
            self.log.debug(u'May have missed an interval, catching up.')
            self.nc += tod.ONE  # 0.01 allows for processing delay
        ival = int(1000.0 * float((self.nc - tod.tod(u'now')).timeval))
        glib.timeout_add(ival, self.timeout)

        # 4: I/O Connection
        if not self.scb.connected():
            self.failcount += 1
            if self.failcount > self.failthresh:
                self.scb.set_portstr(force=True)
                self.failcount = 0
            self.log.debug(u'Telegraph connection failed, count = '
                           + repr(self.failcount))
        else:
            self.failcount = 0

        # 4: Return False
        return False    # must return False

    def setbulb(self, ov, newstate=u'off'):
        """Write bulb state consistently."""
        if newstate == u'red':
            ov['redbulb'] = True
            ov['greenbulb'] = False
        elif newstate == u'green':
            ov['redbulb'] = False
            ov['greenbulb'] = True
        else:
            ov['redbulb'] = False
            ov['greenbulb'] = False

    def left_panel(self, cr, pr, h):
        """Draw the left panel poly."""
        pat = cairo.LinearGradient(0.0, h, 0.0, h+self.growh)
        cor = DARK_R
        cog = DARK_G
        cob = DARK_B
        pat.add_color_stop_rgb(0.0,scaleup(cor, 0.4),
                                    scaleup(cog, 0.4),
                                    scaleup(cob, 0.4))
        pat.add_color_stop_rgb(0.25,cor,cog,cob)
        pat.add_color_stop_rgb(0.85,0.8*cor,0.8*cog,0.8*cob)
        pat.add_color_stop_rgb(1.0,0.5*cor,0.5*cog,0.5*cob)
        cx = ARC_TO_BEZIER * self.growrad
        cr.new_path();
        cr.move_to (self.growll + self.growrad, h)
        cr.rel_line_to (self.growltw - self.growrad, 0.0)
        cr.rel_line_to (self.growlbw - self.growltw, self.growh)
        cr.rel_line_to (-self.growlbw + self.growrad, 0.0)
        cr.rel_curve_to (-cx, 0, -self.growrad, -cx,
                         -self.growrad, -self.growrad)
        cr.rel_line_to (0, -self.growh + 2 * self.growrad)
        cr.rel_curve_to (0.0, -cx, self.growrad - cx,
                        -self.growrad, self.growrad, -self.growrad)
        cr.close_path ()

        #roundedrec(c, 32.0, h, 400.0, self.growh, self.growrad, self.growrad)
        cr.set_source(pat)
        cr.fill()

    def mid_panel(self, cr, pr, h):
        """Draw the mid panel poly."""
        pat = cairo.LinearGradient(0.0, h, 0.0, h+self.growh)
        cor = LIGHT_R
        cog = LIGHT_G
        cob = LIGHT_B
        pat.add_color_stop_rgb(0.0,scaleup(cor, 0.4),
                                    scaleup(cog, 0.4),
                                    scaleup(cob, 0.4))
        pat.add_color_stop_rgb(0.25,cor,cog,cob)
        pat.add_color_stop_rgb(0.85,0.8*cor,0.8*cog,0.8*cob)
        pat.add_color_stop_rgb(1.0,0.5*cor,0.5*cog,0.5*cob)
        cr.new_path();
        cr.move_to (self.growmlt, h)
        cr.line_to (self.growmrt, h)
        cr.line_to (self.growmrb, h+self.growh)
        cr.line_to (self.growmlb, h+self.growh)
        cr.close_path ()

        #roundedrec(c, 32.0, h, 400.0, self.growh, self.growrad, self.growrad)
        cr.set_source(pat)
        cr.fill()
        pass

    def right_panel(self, cr, pr, h):
        """Draw the right panel poly."""
        pat = cairo.LinearGradient(0.0, h, 0.0, h+self.growh)
        cor = DARK_R
        cog = DARK_G
        cob = DARK_B
        pat.add_color_stop_rgb(0.0,scaleup(cor, 0.4),
                                    scaleup(cog, 0.4),
                                    scaleup(cob, 0.4))
        pat.add_color_stop_rgb(0.25,cor,cog,cob)
        pat.add_color_stop_rgb(0.85,0.8*cor,0.8*cog,0.8*cob)
        pat.add_color_stop_rgb(1.0,0.5*cor,0.5*cog,0.5*cob)
        cx = ARC_TO_BEZIER * self.growrad
        cr.new_path();
        cr.move_to (self.growrr - self.growrad, h)
        cr.rel_curve_to (cx, 0.0, self.growrad, cx,
                             self.growrad, self.growrad)
        cr.rel_line_to (0, self.growh - 2 * self.growrad)
        cr.rel_curve_to (0.0, cx, cx - self.growrad,
                            self.growrad, -self.growrad, self.growrad)
        cr.rel_line_to (-self.growrbw + self.growrad, 0.0)
        cr.rel_line_to (self.growrbw - self.growrtw, -self.growh)
        #cr.rel_line_to (self.growrbw - self.growrad, 0.0)
        #cr.rel_curve_to (cx, 0, self.growrad, cx,
                         #self.growrad, self.growrad)
        #cr.rel_curve_to (0.0, cx, -self.growrad + cx,
                        #self.growrad, -self.growrad, self.growrad)
        cr.close_path ()

        #roundedrec(c, 32.0, h, 400.0, self.growh, self.growrad, self.growrad)
        cr.set_source(pat)
        cr.fill()
        pass

    def process_timeout(self):
        """Process countdown, redraw display if required."""
        curoft = tod2key(self.tod)
        if self.currider is not None:
            ov = self.overlays[2]
            cdn = self.currider - curoft
            if cdn == 50 or cdn == 24:
                #loadriderintooverlay
                ov['riderno'] = self.ridermap[self.currider][1]
                ov['ridername'] = self.ridermap[self.currider][3]
                self.setbulb(ov, u'red')
            if cdn == 0:
                self.setbulb(ov, u'green')
                ov['countdown'] = u'0'
            if cdn == -5:       # load sets minimum gap-> ~25sec
                ov['riderno'] = None
                ov['ridername'] = None
                self.setbulb(ov)
                self.currider = self.ridermap[self.currider][4]
            if cdn >= 0 and cdn <= 30:
                if ov['redbulb']:
                    ov['countdown'] = unicode(cdn)
            else:
                if cdn < 0 and cdn > -5:
                    ov['countdown'] = u'+' + unicode(-1*cdn)
                else:
                    ov['countdown'] = None
        else:
            self.riderstr = u'[finished]'
            # no more riders or error in init.
        if self.doredraw or self.curov == 2:	# always redraw in start mode
            self.doredraw = False
            self.draw_and_update()
        return False	# superfluous

    def delayed_cursor(self):
        """Remove the mouse cursor from the text area."""
        pixmap = gtk.gdk.Pixmap(None, 1, 1, 1)
        color = gtk.gdk.Color()
        cursor = gtk.gdk.Cursor(pixmap, pixmap, color, color, 0, 0)
        self.area.get_window().set_cursor(cursor)
        return False

    def finish_command(self, msgtxt=''):
        """Update finish line data from standard finish message."""
        fvec = msgtxt.split(unichr(unt4.US))
        ov = self.overlays[3]
        if len(fvec) > 4:
            rank = fvec[0]
            if rank and rank.isdigit():
                rank = rank + u'.'
            ov['rank']=rank
            ov['riderno']=fvec[1]
            ov['ridername']=fvec[2]
            ov['elapsed']=fvec[4]
            if u'.' not in ov['elapsed']:
                ov['elapsed'] += u'   '	# add decimal padding
            self.draw_and_update()

    def set_monofontsize(self, msgtxt=''):
        self.monofontsize = strops.confopt_float(msgtxt, self.monofontsize)
        fnsz = u' ' + unicode(self.monofontsize) + u'px'
        self.monofontdesc = pango.FontDescription(self.monofont + fnsz)

    def set_stdfontsize(self, msgtxt=''):
        self.stdfontsize = strops.confopt_float(msgtxt, self.stdfontsize)
        fnsz = u' ' + unicode(self.stdfontsize) + u'px'
        self.stdfontdesc = pango.FontDescription(self.stdfont + fnsz)

    def set_geometry(self, msgtxt=''):
        """Update geometry."""
        fvec = msgtxt.split(unichr(unt4.US))
        if len(fvec) == 6:
            self.pw = strops.confopt_float(fvec[0], self.pw)
            self.ph = strops.confopt_float(fvec[1], self.ph)
            self.xoft = strops.confopt_float(fvec[2], self.xoft)
            self.yoft = strops.confopt_float(fvec[3], self.yoft)
            self.xscale = strops.confopt_float(fvec[4], self.xscale)
            self.yscale = strops.confopt_float(fvec[5], self.xscale)
            self.growof = self.ph/9.0	# divide into 8 slots
            self.growh = self.growof * 14.0/15.0	# ~56/60px
            self.growrad = self.growof - self.growh	# remainder on rad
            self.gstart = 2.0 * self.growof
            self.growll = 0.02 * self.pw	# start 2% in
            self.growltw = 1.6 * self.growh	# aim for 2*h
            self.growlbw = 1.4 * self.growh	# aim for 2*h
            self.growrr = 0.98 * self.pw	# start 2% in
            self.growrtw = 3.0 * self.growh	# aim for 4*h
            self.growrbw = 3.2 * self.growh	# aim for 4*h
            self.growmlt = self.growll + self.growltw + 0.25 # allow 1/4 gap
            self.growmlb = self.growll + self.growlbw + 0.25 # allow 1/4 gap
            self.growmrt = self.growrr - self.growrtw - 0.25
            self.growmrb = self.growrr - self.growrbw - 0.25
            self.growth = self.growrad

    def positioned_text(self, msg):
        nr = u''
        if msg.yy in self.dbrows:
            nr = self.dbrows[msg.yy]
        sp = u''
        md = msg.text
        ep = u''
        if msg.xx > 0:
            sp = nr[0:msg.xx]
        edx = msg.xx + len(md)
        if edx < self.monocols:
            ep = nr[edx:]
        if msg.erl:	# space pad to EOL
            ep = u' ' * (len(md) - edx)
        self.dbrows[msg.yy] = strops.truncpad(sp + md + ep,
                                       self.monocols, elipsis=False)
      
    def msg_cb(self, msg, nick=None, chan=None):
        """Handle a public message from the channel."""
        if msg.erp or msg.header == '0040100096':
            self.general_clearing()
            self.draw_and_update()
        elif msg.yy is not None:
            self.positioned_text(msg)
            self.draw_and_update()
        elif msg.header != '':
            command = msg.header.lower()
            if command == u'title':
                self.set_title(msg.text)
            elif command == u'subtitle':
                self.set_subtitle(msg.text)
            elif command == u'rider':
                self.add_row(msg.text)
            elif command == u'finish':
                self.finish_command(msg.text)
            elif command == u'overlay':	# for remote control
                self.set_overlay(msg.text)
            elif command == u'geometry': # reset panel geometry
                self.set_geometry(msg.text)
                self.draw_and_update()
            elif command == u'monofontsize': # update font size
                self.set_monofontsize(msg.text)
                self.draw_and_update()
            elif command == u'stdfontsize': # update font size
                self.set_stdfontsize(msg.text)
                self.draw_and_update()
            #else:
                #self.log.info(u'Ignoring unknown command: ' + repr(command))
        #else:
            #self.log.info(u'Ignoring unknown message type.')
        return False	# idle added
 
    def remote_msg(self, msg):
        """Log a message to the uscbsrv."""
        self.log.debug(msg)
        self.scb.add_rider([msg], 'message')

    def set_overlay(self, newov=None):
        if newov is not None and newov in self.overlays:
            self.curov = newov
        else:
            self.curov = u'blank'
        self.draw_and_update()	# pull out?

    def key_event(self, widget, event):
        """Collect key events on main window."""
        if event.type == gtk.gdk.KEY_PRESS:
            key = gtk.gdk.keyval_name(event.keyval) or 'None'
            key = key.lower()
            if key == KEY_START:
                self.set_overlay(u'2')
            elif key == KEY_STANDINGS:
                #self.standings_mode()
                self.set_overlay(u'1')
            elif key == KEY_FINISH:
                #self.finish_mode()
                self.set_overlay(u'3')
            elif key == KEY_CLEAR:
                self.set_overlay(u'0')
            return True	# 'handle' all keys

    def __init__(self):
        # logger and handler
        self.log = logging.getLogger()
        self.log.setLevel(logging.DEBUG)
        self.loghandler = logging.FileHandler(LOGFILE)
        self.loghandler.setLevel(logging.DEBUG)
        self.loghandler.setFormatter(logging.Formatter(
                       '%(asctime)s %(levelname)s:%(name)s: %(message)s'))
        self.log.addHandler(self.loghandler)
        self.log.debug('Init')

        # require one timy and one uscbsrv
        self.scb = telegraph.telegraph()
        self.remoteport = None
        self.remotechan = None
        self.remoteuser = None

        self.started = False
        self.running = True
        self.doredraw = False
        self.failcount = 0
        self.failthresh = 45    # connect timeout ~45sec
        self.tod = tod.tod(u'now').truncate(0)
        self.nc = self.tod + tod.tod(u'1.22') # set interval a little off mark

        # variables
        self.title = u''	# current title string
        self.subtitle = u''	# current title string
        self.countdown = None
        self.riderstr = None
        self.bulb = None
        self.currider = None
        self.ridermap = {}
        self.monorows = 8	# size of monospace text panel
        self.monocols = 32	# size of monospace text panel
        self.dbrows = {}
        self.curov = 0		# start on blank screen - the 'none' overlay
        self.overlays = { u'align': {		# alignment screen
                           u'patch':cairo.ImageSurface.create_from_png(
                                metarace.default_file('patch.png'))
                          },
                          u'0': {		# SCB Clock
                          },
                          u'1': {		# SCB Preformat Text
                          },
                          u'2': {		# SCB Image 1
                           u'image':cairo.ImageSurface.create_from_png(
                                metarace.default_file('overlay_image_2.png'))
                          },
                          u'3': {		# SCB Image 2
                           u'image':cairo.ImageSurface.create_from_png(
                                metarace.default_file('overlay_image_3.png'))
                          },
                          u'gfx': {
                           u'subtitle':None,
                           u'mainlogo':rsvg.Handle(metarace.default_file(u'gfxmainlogo.svg')),
                           u'rows': []
                          }
                        }
                         #{'clipstart':56.0,	# standings from announcer
                          #'rowheight':56.0,
                          #'rowstart':60.0,
                          #'maxrows':8,
                          #'title_h':6.0,
                          #'title_x1':160.0,
                          #'title_x2':686.0,
                          #'surface':cairo.ImageSurface.create_from_png(
                                                #metarace.default_file('frame_fullpage.png'))
                          #},
                         #{'clipstart':608.5,	# not relevant?
                          #'redbulb':False,
                          #'greenbulb':False,
                          #'countdown':None,
                          #'riderno':None,
                          #'ridername':None,
                          #'label':'Start:',
                          #'rowheight':56.0,
                          #'rowstart':490.0,
                          #'maxrows':1,
                          #'title_h':60.0,	# not used in lower
                          #'title_x1':1200.0,
                          #'title_x2':1600.0,
                          #'surface':cairo.ImageSurface.create_from_png(
                                 #metarace.default_file('frame_lower.png'))
                          #},
                         #{'clipstart':608.5,	# not relevant?
                          #'rank':None,
                          #'riderno':None,
                          #'ridername':None,
                          #'elapsed':None,
                          #'rowheight':56.0,
                          #'rowstart':490.0,
                          #'maxrows':1,
                          #'title_h':60.0,	# not used in lower
                          #'title_x1':1200.0,
                          #'title_x2':1600.0,
                          #'surface':cairo.ImageSurface.create_from_png(
                                 #metarace.default_file('frame_lower.png'))
                          #},
                         #]	# list of configured overlays

        # Geometry
        self.width = float(DEFAULT_WIDTH)	# device w
        self.height = float(DEFAULT_HEIGHT)	# device h
        self.pw = self.width			# panel width
        self.ph = self.height			# panel height
        self.xscale = 1.0			# panel pre-scale x
        self.yscale = 1.0			# panel pre-scale y
        self.xoft = 0.0				# panel pre-translate x
        self.yoft = 0.0				# panel pre-translate y
        self.stdfontoffset = -4			# font vertical oft
        self.stdfontsize = 20			# font size in panel units
        self.monofontoffset = -4		# font vertical oft
        self.monofontsize = 34			# font size in panel units
        self.monofont = MONOFONT
        self.monofontdesc = None
        self.stdfont = STDFONT
        self.stdfontdesc = None
        ## TODO: rotation

        # Prepare UI
        self.window = gtk.Window()
        self.window.set_title('Metarace LoGFX')
        self.window.connect('destroy', self.window_destroy_cb)
        self.window.connect('key-press-event', self.key_event)
        self.area_src = None	# off-screen drawable
        self.area = gtk.DrawingArea()
        self.area.connect('configure_event', self.area_configure_event_cb)
        self.area.connect('expose_event', self.area_expose_event_cb)
        self.area.set_size_request(DEFAULT_WIDTH, DEFAULT_HEIGHT)
        self.area.show()
        self.window.add(self.area)
        self.log.debug(u'Starting clock intervals at: ' + self.nc.rawtime(3))
        glib.timeout_add(2000, self.timeout)
        glib.timeout_add_seconds(5, self.delayed_cursor)

def main():
    """Run the application."""
    metarace.init()
    app = gfx_panel()
    app.loadconfig()
    app.show()
    app.start()
    try:
        gtk.main()
    except:
        app.shutdown()
        raise

if __name__ == '__main__':
    main()

