
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""GPSd thread with callback to main thread."""

import threading
import Queue
import logging
import gps	# this is the gpsd library
import glib

import metarace
from metarace import unt4

# thread queue commands -> private to timy thread
TCMDS = ('EXIT')

class gpsclient(threading.Thread):
    """gpsd client thread object class."""
    def __init__(self, port=None, name='gpsclient'):
        """Construct thread object.

        Named parameters:

          port -- reserved gps port identifier?
          name -- text identifier for use in log messages

        """
        threading.Thread.__init__(self) 
        self.name = name

        self.port = None	 # not used but retain for expand
        self.error = False
        self.errstr = ''
        self.fix = False
        self.cqueue = Queue.Queue()	# command queue
        self.log = logging.getLogger(self.name)
        self.log.setLevel(logging.DEBUG)
        self.setcb()

    def __defcallback(self, evt=None):
        """Default callback is a tod log entry."""
        self.log.debug(repr(evt.pack()))
        return False

    def getcb(self):
        """Return the current callback function."""
        return self.__cb

    def setcb(self, func=None):
        """Set or clear the event callback."""
        # if func is not callable, gtk mainloop will catch the error
        if func is not None:
            self.__cb = func
        else:
            self.__cb = self.__defcallback

    def exit(self, msg=None):
        """Request thread termination."""
        self.running = False
        self.cqueue.put_nowait(('EXIT', msg)) # "Prod" command thread

    def sane(self):
        """Request sanity check."""
        pass

    def wait(self):
        """Suspend calling thread until the command queue is empty."""
        self.cqueue.join()

    def run(self):
        """Called via threading.Thread.start()."""
        self.running = True
        self.log.debug('Starting')
        session = gps.gps(mode=gps.WATCH_ENABLE|gps.WATCH_JSON)
        while self.running:
            # Command Queue Read phase
            try:
                m = self.cqueue.get_nowait()
                self.cqueue.task_done()
                # Handle thread commands	 (TODO)
                if type(m) is tuple and type(m[0]) is str and m[0] in TCMDS:
                    if m[0] == 'EXIT':
                        self.log.debug('Request to close : ' + str(m[1]))
                        running = False	# This may already be set
                    else:
                        pass
                else:
                    self.log.warn(u'Unknown message: ' + repr(m))
            except Queue.Empty:
                pass
            # GPS Read phase (blocking read?)
            try:
                report = session.next()	# timeout?
                if report:
                    #self.log.debug(u'GPS Report: ' + repr(report))
                    #self.log.debug(repr(session.fix))
                    #self.log.debug(u'Got class == ' + repr(report['class']))
                    if report['class'] == 'TPV':
                        rvec = [self.name, 
                            unicode(session.fix.mode),
                            unicode(session.fix.latitude),
                            unicode(session.fix.longitude),
                            unicode(session.fix.altitude),
                            unicode(session.fix.speed),
                            unicode(session.fix.time)]
                        rep = unt4.unt4(header=u'gps',
                                    text=unichr(unt4.US).join(rvec))
                        self.__cb(session.fix.mode, rep)
                        if session.fix.mode > 1:
                            self.log.debug(u'Mode ' + unicode(session.fix.mode)
                                                + u' Fix.')
                            self.fix = True
                        else:
                            self.log.debug(u'No Fix')
                            self.fix = False
            except StopIteration:
                self.running = False
                self.log.debug('GPSd terminated.')
                # should attempt to re-connect later
            except Exception as e:
                self.log.error('Exception: ' + str(type(e)) + str(e))
        self.setcb()	# make sure callback is unrefed
        self.log.info('Exiting')


if __name__ == "__main__":
    import metarace
    import gtk
    import time
    import random
    import json
    metarace.init()
    g = gpsclient()
    lh = logging.StreamHandler()
    #lh = logging.FileHandler(u'batlog')
    lh.setLevel(logging.DEBUG)
    lh.setFormatter(logging.Formatter(
                      "%(asctime)s %(levelname)s:%(name)s: %(message)s"))
    g.log.addHandler(lh)
    try:
        g.start()
        metarace.mainloop()
    except:
        g.exit('Exception')
        g.join()
        raise
