
# Metarace : Cycle Race Abstractions
# Copyright (C) 2012  Nathan Fraser
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

## TODO : Rename and fix for use as a general purpose graphical display,
##        requires paging support and some sort of transition

"""Text-only scoreboard for VGA videowall (Launceston)

This module implements a simple scoreboard that mimics the Omega
DHI at DISC. The output is intended to be displayed on a vieowall.

"""


import pygtk
pygtk.require("2.0")

import gtk
import glib
import pango
import pangocairo
import cairo
import random
import ConfigParser
import logging
import os
import csv

import metarace
from metarace import strops
from metarace import uscbsrv
from metarace import unt4

# Globals
CONFIGFILE='text_scb.ini'
USCBSRV_HOST='textscb@localhost'
USCBSRV_CHANNEL='#tscb'
LOGFILE = 'text_scb.log'
SCBFONT = pango.FontDescription('Nimbus Sans L Bold 38px')

def draw_text(cr, pr, oh, x1, x2, msg, align=0, invert=False):
    if msg is not None:
        cr.save()
        cr.set_line_cap(cairo.LINE_CAP_ROUND)
        cr.set_line_join(cairo.LINE_JOIN_ROUND)
        cr.set_line_width(4.0)
        maxw = x2 - x1
        l = pr.create_layout()
        l.set_font_description(SCBFONT)
        l.set_text(msg)
        (tw,th) = l.get_pixel_size()
        oft = 0.0
        if align != 0 and tw < maxw:
            oft = align * (maxw - tw)	# else squish
        cr.move_to(x1+oft, oh)	# move before applying conditional scale
        if tw > maxw:
            cr.scale(float(maxw)/float(tw),1.0)
        pr.update_layout(l)
        pr.layout_path(l)	# drop path into context

        # outline
        if invert:
            cr.set_source_rgb(0.2,0.2,0.2)
        else:
            cr.set_source_rgb(1.0,1.0,1.0)
        cr.stroke_preserve()

        # fill 
        if invert:
            cr.set_source_rgb(1.0,1.0,1.0)
        else:
            cr.set_source_rgb(0.2,0.2,0.2)
        cr.fill()

        cr.restore()

class text_scb(object):
    """IRTT Start Display application."""
 
    def show(self):
        self.window.show()

    def hide(self):
        self.window.show()

    def start(self):
        """Start threads."""
        if not self.started:
            self.scb.start()
            self.scb.set_pub_cb(self.msg_cb)
            self.started = True

    def shutdown(self):
        """Cleanly shutdown."""
        self.scb.exit()
        self.scb.join()
        self.started = False

    def window_destroy_cb(self, window):
        """Handle destroy signal."""
        if self.started:
            self.shutdown()
        self.running = False
        gtk.main_quit()
    
    def area_expose_event_cb(self, widget, event):
        """Update desired portion of drawing area."""
        x , y, width, height = event.area
        widget.window.draw_drawable(widget.get_style().fg_gc[gtk.STATE_NORMAL],
                                    self.area_src, x, y, x, y, width, height)
        return False

    def area_redraw(self):
        """Lazy full area redraw method."""
        cr = self.area_src.cairo_create()
        pr = pangocairo.CairoContext(cr)
        cr.identity_matrix()

        # pre-scale, 16:9 to 4:3 (assumes 1024x768 for 1366x768 screen)
        # TODO: determine scale factor in reconfig display area
        #cr.scale(1024.0/1366.0, 1.0)

        # bg filled
        cr.set_source_rgb(0.0,0.0,0.0)
        cr.paint()

        if self.curov == 0:
            return	# the 'none' overlay is a blank screen

        ov = self.overlays[self.curov]

        # 'expose' background rows
        cr.save()
        cr.rectangle(0,0, 1366,ov['clipstart']
                       + len(self.rows) * ov['rowheight'])
        cr.clip()
        cr.set_source_surface(ov['surface'])
        cr.paint()
        cr.restore()

        # Draw title if available
        if self.title:
            draw_text(cr, pr, ov['title_h'], ov['title_x1'],
                              ov['title_x2'], self.title, invert=True)

        curof = ov['rowstart']
        count = 0
        for row in self.rows:
            if row[0]:
                draw_text(cr, pr, curof-4.0,260,310,row[0],align=0.5,
                                   invert=True) 
            if row[1]:
                draw_text(cr, pr, curof-4.0,328,948,row[1], invert=True)
            if row[2]:
                draw_text(cr, pr, curof-4.0,952,1095,row[2],align=1.0,
                                   invert=True)
            curof += ov['rowheight']
            count += 1
            if count >= ov['maxrows']:
                break
        # test drawing
        #self.number_box(cr, curof)
        #name_box(curof)
        #draw_text(cr, pr, curof-4.0,260,310,'dnf',align=0.5,
                               #invert=True) 
        #draw_text(cr, pr, curof-4.0,328,948,'Example NAMEFACE (CLB)',
                               #invert=True)
        #draw_text(cr, pr, curof-4.0,952,1095,'1:23.45',align=1.0,
                               #invert=True)
        
    def number_box(self, cr, hof):
        cr.move_to(255.0,hof-5.0)
        cr.rectangle(255.0,hof-5.0,90.0,80.0)
        cr.set_line_cap(cairo.LINE_CAP_ROUND)
        cr.set_line_join(cairo.LINE_JOIN_ROUND)
        cr.set_source_rgb(0.95,0.75,0.75)
        cr.fill()
        cr.rectangle(257.0,hof-3.0,90.0,80.0)
        cr.set_source_rgb(0.5,0.3,0.3)
        cr.fill()
        cr.rectangle(256.0,hof-4.0,90.0,80.0)
        cr.set_source_rgb(0.9,0.2,0.2)
        cr.fill()
        

    def area_configure_event_cb(self, widget, event):
        """Re-configure the drawing area and redraw the base image."""
        x, y, width, height = widget.get_allocation()
        ow = 0
        oh = 0
        if self.area_src is not None:
            ow, oh = self.area_src.get_size()
        if width > ow or height > oh:
            self.area_src = gtk.gdk.Pixmap(widget.window, width, height)
        self.width = width
        self.height = height
        self.area_redraw()
        self.area.queue_draw()
        return True

    def clear(self):
        """Clear all elements."""
        self.title = u''
        self.rows = []
        self.draw_and_update()
        
    def set_title(self, tstr=''):
        """Draw title and update."""
        self.title = tstr
        self.draw_and_update()

    def add_row(self, msg=''):
        """Split row and then append to self.rows"""
        sr = msg.split(chr(unt4.US))
        if len(sr) == 3:
            self.rows.append([sr[0], sr[1], sr[2]])
        self.draw_and_update()

    def loadconfig(self):
        """Load config from disk."""
        cr = ConfigParser.ConfigParser({'uscbsrv':USCBSRV_HOST,
                                        'channel':USCBSRV_CHANNEL,
                                        'fullscreen':'No',
                                       })
        cr.add_section('text_scb')

        # check for config file
        try:
            a = len(cr.read(CONFIGFILE))
            if a == 0:
                self.log.info('No config file found - loading default values.')
        except Exception as e:
            self.log.error('Error reading config: ' + str(e))

        if strops.confopt_bool(cr.get('text_scb', 'fullscreen')):
            self.window.fullscreen()

        # set sender port
        nhost = cr.get('text_scb', 'uscbsrv')
        nchannel = cr.get('text_scb', 'channel')
        self.scb.set_portstr(nhost, nchannel)

    def draw_and_update(self, data=None):
        """Redraw in main loop, not in timeout."""
        self.area_redraw()
        self.area.queue_draw()
        return False

    def delayed_cursor(self):
        """Remove the mouse cursor from the text area."""
        pixmap = gtk.gdk.Pixmap(None, 1, 1, 1)
        color = gtk.gdk.Color()
        cursor = gtk.gdk.Cursor(pixmap, pixmap, color, color, 0, 0)
        self.area.get_window().set_cursor(cursor)
        return False

    def msg_cb(self, msg, data=None):
        """Handle a public message from the channel."""
        if msg.erp or msg.header == '0040100096':
            self.clear()
        if msg.header != '':
            command = msg.header.lower()
            if command == u'set_title':
                self.set_title(msg.text)
            elif command == u'add_row':
                self.add_row(msg.text)
            elif command == u'overlay':
                self.set_overlay(int(msg.text))
            else:
                self.log.info(u'Ignoring unknown command: ' + repr(command))
        else:
            self.log.info(u'Ignoring unknown message type.')
        return False	# idle added
 
    def remote_msg(self, msg):
        """Log a message to the uscbsrv."""
        self.log.debug(msg)
        self.scb.add_rider([msg], 'message')

    def set_overlay(self, newov=0):
        if newov >= 0 and newov < len(self.overlays):
            self.title = u''
            self.rows = []
            self.curov = newov
            self.log.debug('Set overlay to: ' + repr(self.curov))
            self.draw_and_update()

    def __init__(self):
        # logger and handler
        self.log = logging.getLogger()
        self.log.setLevel(logging.DEBUG)
        self.loghandler = logging.FileHandler(LOGFILE)
        self.loghandler.setLevel(logging.DEBUG)
        self.loghandler.setFormatter(logging.Formatter(
                       '%(asctime)s %(levelname)s:%(name)s: %(message)s'))
        self.log.addHandler(self.loghandler)
        self.log.debug('Text SCB - Init.')

        # require one timy and one uscbsrv
        self.scb = uscbsrv.uscbsrv()
        self.started = False
        self.running = True

        # variables
        self.width = 0
        self.height = 0
        self.title = u''	# current title string
        self.rows = []	# rows of 3 cols
        self.curov = 0		# start on blank screen - the 'none' overlay
        self.overlays = [None,
                         {'clipstart':164.50,	# 10 line scb
                          'rowheight':55.5,
                          'rowstart':172.0,
                          'rowtext':-4.0,
                          'maxrows':10,
                          'title_h':60.0,
                          'title_x1':400.0,
                          'title_x2':1120.0,
                          'surface':cairo.ImageSurface.create_from_png(
                                                'overlay_bg_01.png')
                          },
                         {'clipstart':608.5,	# 2 line (sprint)
                          'rowheight':55.5,
                          'rowstart':616.0,
                          'rowtext':-4.0,
                          'maxrows':2,
                          'title_h':532.0,
                          'title_x1':400.0,
                          'title_x2':1120.0,
                          'surface':cairo.ImageSurface.create_from_png(
                                                'overlay_bg_02.png')
                          },
                         ]	# list of configured overlays

        self.window = gtk.Window()
        self.window.set_title('Metarace LoGFX SCB')
        self.window.connect('destroy', self.window_destroy_cb)

        self.area_src = None
        self.area = gtk.DrawingArea()
        self.area.connect('configure_event', self.area_configure_event_cb)
        self.area.connect('expose_event', self.area_expose_event_cb)
        self.area.set_size_request(1024,768)	# CHECK?
        self.area.show()
        self.window.add(self.area)
        glib.timeout_add_seconds(5, self.delayed_cursor)

def main():
    """Run the application."""
    metarace.init()
    app = text_scb()
    app.loadconfig()
    app.show()
    app.start()
    try:
        metarace.mainloop()
    except:
        app.shutdown()
        raise

if __name__ == '__main__':
    main()

