from __future__ import with_statement
from threading import local
from types import FunctionType
from contextlib import contextmanager
from functools import wraps

from mext.classhacks import classy



__all__ = [
    'State', 'Service', 'ServiceStub', 'StateSlice',
    'Setting', #'Action', 'Resource',
    'no_state',
    'InputConflict', 'DynamicRuleError', 'ScopeError',
]

#from peak.util.proxies import CallbackWrapper, CallbackProxy
#def stateval_proxy(key):
#    return CallbackProxy(lambda: State[key])


#@@ State layers, Global services. http://tinyurl.com/yb5mzup



_in_place = """__iadd__ __isub__ __imul__ __idiv__ __itruediv__ __ifloordiv__
__imod__ __ipow__ __ilshift__ __irshift__ __iand__ __ixor__ __ior__""".split()

_dont_delegate_global = set("""
    __service__ __default__
    __class_new__ __class_init__ __class_call__
    __name__ __module__ __return__ __slots__
    get
    __init__ __metaclass__ __doc__
    __call__ __new__ __class__""".strip().split()
)
_dont_delegate_global.update(_in_place)


def _no_in_place(self, *args):
    raise TypeError("In-place operators cannot be performed on a service class")

_std_attrs = dict.fromkeys(_in_place, _no_in_place)


def redirect_attribute(cls, name):
    setattr(type(cls), name, property(
        lambda s: getattr(s.get(), name),
        lambda s, v: setattr(s.get(), name, v),
        lambda s: delattr(s.get(), name),
    ))


class _ClassDelegate(classy):
    """
        Type whose attributes/methods are delegated to ``cls.get()``
    """
    __slots__ = ()
    get = None  # dummy

    @classmethod
    def __class_init__(cls, name, bases, cdict, supr):
        meta = type(cls)
        if getattr(meta, '__for_class__', None) is not cls:
            cls.__class__ = meta = type(meta)(
                cls.__name__+'Class', (meta,),
                dict(_std_attrs, __module__=cls.__module__, __for_class__=cls)
            )
            # XXX activate_attrs(meta)?

        supr()(cls, name, bases, cdict, supr)

        if cls.get is None:
            if bases != (classy,): # cls is not *the* _ClassDelegate
                raise TypeError("%s should define .get()" % name)
            return

        _delegate = cdict.pop('_delegate_attrs', ())
        _dont_delegate = set(cdict.pop('_dont_delegate', ()))
        _dont_delegate.update(_dont_delegate_global)

        for k in _delegate:
            assert k not in _dont_delegate
            redirect_attribute(cls, k)

        for k, v in cdict.items():
            if (isinstance(k, basestring)
                and not isinstance(v, (classmethod, staticmethod))
                and k not in _dont_delegate
            ):
                redirect_attribute(cls, k)



class InputConflict(Exception):
    """
        Attempt to set a rule that causes a visible conflict in the state
    """

class DynamicRuleError(Exception):
    """
        A fallback or wildcard rule attempted to access dynamic state
    """

class ScopeError(Exception):
    """
        A problem with scoping occurred
    """








thread_state = local()



class RootState(object):
    __enter__ = __exit__ = swap = _swap = __setitem__ = None
    def __getitem__(self, key):
        return key.__default__()




class State(_ClassDelegate):
    """
        A thread's current configuration and state
    """
    _dont_delegate = ('root',)
    root = RootState()
    parent = None

    def __init__(self, parent=None):
        if parent is None:
            parent = self.root
        self.parent = parent # the state to query for missing values
        self.outer_state = None # the state to set when exiting
        self.values = {}
        self.frozen = False

    @staticmethod
    def get():
        try:
            return thread_state.state
        except AttributeError:
            r = thread_state.state = State()
            r.outer_state = True # mark as entered
            return r

    def child(self):
        return State(self)

    def clone(self):
        self.frozen = True
        r = State(self.parent)
        r.values = self.values
        r.frozen = True
        return r

    def swap(self):
        prev_state = State.get()
        if prev_state is _disabled_state: #@@
            raise DynamicRuleError("Don't try to circumvent disabled state")
        thread_state.state = self
        return prev_state


    def __enter__(self):
        if self.outer_state is not None:
            raise ScopeError("Can't re-enter a currently entered state")
        try:
            # inline the most common case
            self.outer_state = thread_state.state
        except AttributeError:
            self.outer_state = self.get()
        thread_state.state = self
        return self

    def __exit__(self, *exc_info):
        if not self.outer_state:
            raise ScopeError("State already exited or hasn't been entered yet")
        elif thread_state.state is not self:
            #if exc_info: import traceback; traceback.print_tb(exc_info[2])
            raise ScopeError("Can't exit a non-current state")
        thread_state.state = self.outer_state
        self.outer_state = None



    def __setitem__(self, key, value):
        if self.frozen:
            raise InputConflict("This state is frozen")
        old = self.values.setdefault(key, value)
        if value != old:
            raise InputConflict(key, old, value)

    def __getitem__(self, key):
        try:
            return self.values[key]
        except KeyError:
            return self.values.setdefault(key, self.parent[key])


    def activate(self, __svc, *args, **kw):
        r = self[__svc.__service__] = __svc(*args, **kw)
        return r

    def wrap_cb(self, __func): # mode='enter'/'swap'/'child'
        #@wraps
        def wrapped(*args, **kw):
            with self.child():
                return __func(*args, **kw)
        if isinstance(__func, FunctionType):
            wrapped_cb = wraps(__func)(wrapped)
        return wrapped



class _DisabledState(object):
    #_swap = State._swap.im_func
    def __getattr__(self, name):
        raise DynamicRuleError("default rule or exit function tried to read dynamic state")

_disabled_state = _DisabledState()

class no_state(object):
    """
        x = Setting(None)
        with no_state():
            assert_raises(DynamicRuleError, x.get)
    """
    def __enter__(self):
        self.prev_state = State.get()
        thread_state.state = _disabled_state

    def __exit__(self, *exc_info):
        thread_state.state = self.prev_state







@classmethod
def no_default(cls):
    # this will never get called because this service will not be used as a key
    # its __service__ attr will be used instead
    raise RuntimeError("%s is an override service, it has no default" % cls)


class Service(_ClassDelegate):
    """
        A replaceable, thread-local singleton
    """
    __slots__ = ()

    @classmethod
    def __class_call__(__cls, *args, **kw):
        with no_state():
            return super(Service, __cls).__class_call__(*args, **kw)

    @classmethod
    def __class_init__(cls, name, bases, cdict, supr):
        if '__service__' in cdict:
            cls.__default__ = no_default
        else:
            cls.__service__ = cls # circular reference, but it's ok
        supr()(cls, name, bases, cdict, supr)

    @classmethod
    def get(cls):
        return State[cls.__service__]

    @classmethod
    def __default__(cls):
        return cls()

    @classmethod
    def activate(__cls, *args, **kw):
        r = State[__cls.__service__] = __cls(*args, **kw)
        return r

    @classmethod
    @contextmanager
    def new(__cls, *args, **kw):
        with State.child():
            yield __cls.activate(*args, **kw)


class ServiceStub(Service): #@@ tests
    @classmethod
    def __default__(cls):
        raise RuntimeError("%s service doesn't have a default instance"
            % cls.__name__
        )



_no_default = object()

class Setting(object):
    __slots__ = ('default')
    def __init__(self, default=_no_default):
        self.default = default

    def __default__(self):
        if self.default is _no_default:
            raise RuntimeError("This Setting has no default")
        return self.default

    def get(self):
        return State[self]

    def set(self, value):
        State[self] = value

    value = property(get, set)

    def __repr__(self):
        return "<%s(%r) at 0x%x>" % (type(self).__name__, self.value, id(self))





class StateSlice(object):
    def __init__(self):
        self.services = {}
        self.frozen = False

    def __setitem__(self, key, value):
        if self.frozen:
            raise RuntimeError("This StateSlice is alrady frozen", self)
        self.services[key] = value

    def add(self, service):
        self[service.__service__] = service

    def enable(self, state=State):
        for key, value in self.services.items():
            state[key] = value

    def __enter__(self):
        self.frozen = True
        state = State.child().__enter__()
        self.enable(state)
        return state

    def __exit__(self, *args):
        State.get().__exit__(*args)


    def wrap_cb(self, cb):
        def wrapped_cb(*args, **kw):
            with self:
                return cb(*args, **kw)
        if isinstance(cb, FunctionType):
            wrapped_cb = wraps(cb)(wrapped_cb)
        return wrapped_cb
