#!/usr/bin/env python
#coding: utf-8

from sys import exit as kill
from sys import argv
from sys import stderr

class NoOptionElementError(BaseException):
    """optionを構成する値がないときに発生"""
    pass

class NoArgumentsError(BaseException):
    """optionがフックされたら起動する関数に引数が何もなかった場合発生"""
    pass

class DuplicationError(BaseException):
    """option定義時に指定したhookが以前定義したhookと重複している場合発生"""
    pass

def OutOfParserError(msg):
    error_context = "Error: %s\n" % msg
    stderr.write(error_context)

def parser(version=None,description=None):
    return MinimumParser(version, description)

class MinimumParser:
    def __init__(self, version, description):
        self.app_version = version
        self.description = description
        self.options = []

    def add_option(self, hooks, description, func, arg_types=None):
        """オプションのフック、説明、関数を引数に指定してオプション定義"""
        argcount, argument_names = self.__get_func_info(func)

        inspect_flags = {"hooks":None, "arg_types":None}
        # Trueを返す以外はすべてエラーを発生させるのでflagをFalseにする必要はない
        inspect_flags["hooks"] = self.__check_hooks(hooks)
        # 同上
        inspect_flags["arg_types"] = self.__check_type_dict(arg_types, func)

        option_obj = Option(hooks, description, argcount, argument_names, arg_types, func)
        self.options.append( option_obj )
        return option_obj

    def option(self, *hooks, **kwargs):
        """オプションを追加するデコレータ"""

        # get description
        if kwargs.has_key("description"):
            option_description = kwargs["description"]
        else:
            raise NoOptionElementError("option description not found.")

        # get arg types
        if kwargs.has_key("argument_types"):
            option_argument_types = kwargs["argument_types"]
        else:
            option_argument_types = None

        def function_wrapper(func):
            self.add_option(hooks, option_description, func, option_argument_types)

        return function_wrapper

    def parse(self, target=argv):
        """コマンドラインオプションをパース。target引数でパースするものを指定できます"""
        self.prog_name = target[0]

        # ヘルプとバージョンをprintするoptionを追加
        self.__add_usage_opt()
        if self.app_version:
            self.__add_version_opt()

        if len(target) < 2:
            self.call_usage(self.prog_name)
            kill(1)

        hook_in = target[1]

        option_obj = self.__get_option_func(hook_in)

        option_arguments_values = target[2:]
        option_arguments = dict(zip(option_obj.func_args, option_arguments_values))

        if len(option_arguments) != 0:
            option_obj.run(option_arguments)
        else:
            option_obj.run()

    def show_version(self):
        """print version"""
        version_content = "%s var.%s" % (self.prog_name, self.app_version)
        print version_content

    def call_usage(self, prog_name):
        """show usage help msg"""
        usage_content = ""
        if self.app_version:
            usage_content += "%s var.%s" % (prog_name, self.app_version)
        if self.description:
            usage_content += "  - "+self.description+"\n"

        usage_content += "\n"
        usage_content += "$ %s [OPTIONS]\n" % prog_name
        usage_content += "-"*30+"\n"
        usage_content += "OPTIONS :\n"

        for option_obj in self.options:
            if len(option_obj.func_args) == 0:
                usage_content += " "*3+", ".join(option_obj.hooks)+"\n"
                usage_content += " "*6+option_obj.description+"\n\n"
            else:
                usage_content += " "*3+", ".join(option_obj.hooks)+" [["+"], [".join(option_obj.func_args)+"]]\n"
                usage_content += " "*6+option_obj.description+"\n\n"

        print usage_content

    def __check_hooks(self, hooks):
        """hooksのサイズが0かどうか、hooksが重複してないかどうか"""
        # size
        if len(hooks) == 0:
            raise NoOptionElementError("Option's hooks does not found!")

        # duplication same tuples
        no_duplicate=[]
        for hook in hooks:
            if hook not in no_duplicate:
                no_duplicate.append(hook)

        if len(hooks) != len(no_duplicate):
            raise DuplicationError("Hooks are duplicate!")

        # duplication before added
        all_hook_tuples = [obj.hooks for obj in self.options]
        if len(all_hook_tuples) == 0:
            return True

        for hook_tuple in all_hook_tuples:
            for hook in hook_tuple:
                if hook in hooks:
                    raise DuplicationError("'%s' hook is duplicate." % hook)

        return True

    def __check_type_dict(self, dictionary, function):
        """引数に指定された辞書のkeyがoptionの関数の引数にあったものでなければFalseを返す"""
        # check intance type
        if dictionary == None:
            return True

        if isinstance(dictionary,dict) == False:
            raise TypeError("the 'argument_types' option required dictionary.")

        argcount, func_args = self.__get_func_info(function)
        for key in dictionary.keys():
            if key not in func_args:
                raise KeyError("'%s' this argument does not found." % key)

        return True

    def __get_option_func(self, hook_in):
        """hook_inが追加したoptionのhooksに存在するかどうか"""
        find_result = False
        for option_obj in self.options:
            if hook_in in option_obj.hooks:
                find_result = True
                break

        if find_result == True:
            return option_obj
        else:
            self.call_usage(self.prog_name)
            kill(1)

    def __get_func_info(self, function):
        """指定したfunctionの引数の数と引数名を返す"""
        argcount = function.func_code.co_argcount
        varnames = function.func_code.co_varnames
        return argcount, varnames[:argcount]

    def __add_usage_opt(self):
        """usage help msgをprintするためのオプションを追加"""
        self.add_option(("-h","--help","help"), "Show this usage message.", (lambda: self.call_usage(self.prog_name)))

    def __add_version_opt(self):
        """versionをprintするためのオプションを追加"""
        self.add_option(("-v","--version","version"), "Show this application version.", (lambda: self.show_version()))

class Option(object):
    """コマンドラインオプションを定義
        option_hooks        - オプションをフックするための文字列のタプル ex): ("-m", "--message","msg")
        option_desc         - オプションの説明
        func_argcount       - オプションがフックされたら起動する関数の引数の数
        func_args           - オプションがフックされたら起動する関数の引数の名前のタプル
        func_argument_types - オプションがフックされたら起動する関数の引数の型の辞書
        option_func         - オプションがフックされたら起動する関数
    """
    def __init__(self, option_hooks, option_desc, func_argcount, func_args, func_argument_types, option_func):
        self.hooks = option_hooks
        self.description   = option_desc
        self.func_argcount = func_argcount
        self.func_args     = func_args
        self.func          = option_func
        self.func_argument_types = func_argument_types

    def run(self, option_arguments=None):
        """関数を起動させる。
        option_argumentsで関数の引数と値をdictionaryでセットする。
        関数に引数が無い場合はNoneで良い"""
        if len(self.func_args) == 0 and option_arguments == None:
            self.func()

        elif len(self.func_args) != 0 and option_arguments != None:
            if self.__check_argument(option_arguments):
                option_arguments = self.__change_type(option_arguments)
                self.func(**option_arguments)

        elif len(self.func_args) != 0 and option_arguments == None:
            OutOfParserError("Not found option function's argument!")

    def __check_argument(self, kwargs):
        """関数に渡されようとしている引数と型をチェック"""
        if len(kwargs) == 0:
            return True

        if len(kwargs) != self.func_argcount:
            OutOfParserError("option takes exactly %d arguments" % len(self.func_args))
            return False

        for kw_name, kw_value in kwargs.iteritems():
            for arg_name, typeof in self.func_argument_types.iteritems():
                if arg_name == kw_name:
                    try:
                        kwargs[arg_name] = typeof(kw_value)
                    except TypeError:
                        OutOfParserError("'%s' this argument required '%s' type!" % (arg_name, typeof))

        return True

    def __change_type(self, kwargs):
        """関数が起動する際与える引数の型を変換させる"""
        for kw_name, kw_value in kwargs.iteritems():
            for arg_name, typeof in self.func_argument_types.iteritems():
                if arg_name == kw_name:
                    kwargs[arg_name] = typeof(kw_value)

        return kwargs

    def __repr__(self):
        return "Option(hooks=(%s), description='%s', func_argcount=%d, func_args=(%s), func=%s)" % (
                ",".join(self.hooks),
                self.description,
                self.func_argcount,
                ",".join(self.func_args),
                self.func
                )
        def __str__(self):
            return self.__repr__()
