import sys
import signal
import uuid
import yaml
import importlib
import logging
import tornado.process
from optparse import OptionParser
from tornado import netutil
from tornado.httpserver import HTTPServer


class Object(object):
    """
    Classes that inherit from this, work on the principle that each class 
    constructor is called once.

    When the object is initialized, all parameters are passed to their names.
    ::
        
        import minsol
        
        
        class Child(minsol.Object):
            def init(arg1, arg2=None, **other):
                self.arg1 = arg1
                self.arg2 = arg2


        ff = Child(arg1="val1", arg2="val2")

    """

    def __init__(self, **kwargs):
        if hasattr(self, "__initialized__"):
            return

        self.__initialized__ = True

        _mro = list(self.__class__.__mro__)
        _mro.reverse()

        for _class in _mro:
            if _class == Object \
            or not issubclass(_class, Object):
                continue

            try:
                _class.init(self, **kwargs)

            except Exception as error:
                sys.stderr.write("Error init class %s\n" % str(_class))
                sys.stderr.flush()

                raise error

    def init(self, **other):
        pass


class UUID(uuid.UUID):
    """
    ``jsonpickle`` lib can't serialize ``uuid.UUID``, 
    because ``__setattr__`` raise ``TypeError``
    
    In this class ``__setattr__`` works normaly.
    """
    def __setattr__(self, name, value):
        object.__setattr__(self, name, value)

    @staticmethod
    def gen4():
        """
        :return: generated ``minsol.UUID``
        """
        _uuid = uuid.uuid4()
        return UUID(int=_uuid.int)


class AsyncLoader(Object):
    def init(self, **kwargs):
        self.tasks = []

    def on_load(self):
        self.counter -= 1

        if not self.counter:
            self.callback()

    def load(self, callback):
        if not self.tasks:
            callback()
            return

        self.callback = callback
        self.counter = len(self.tasks)

        for task in self.tasks:
            task(self.on_load)


class Service(Object):
    """
    :ivar config: parsed ``yaml`` config file, ``dict``
    """

    # : Required command line arguments
    required_options = ["config"]

    def init(self, **kwargs):
        signal.signal(signal.SIGINT, self._on_sigint)
        signal.signal(signal.SIGTERM, self._on_sigterm)

        self.option_parser = OptionParser()

        self.config = {}

    def _on_sigint(self, signum, frame):
        self.stop()

    def _on_sigterm(self, signum, frame):
        self.stop()

    def init_option_parser(self):
        """
        Example::

            def init_option_parser(self):
                # rewrite standard option parser, if needed
                self.option_parser = OptionParser(...)

                # init standards
                minsol.Service.init_option_parser(self)

                self.option_parser.add_option...

        """
        self.option_parser.add_option(
            "--pidfile", type="string", dest="pidfile",
            help="Path to pidfile")

        self.option_parser.add_option(
            "--config", type="string", dest="config",
            help="Path to YAML config file")

    def init_config(self):
        """
        Config defaults. Example::

            def init_config(self):
                minsol.Service.init_config(self)

                self.config.update(dict(
                    host="localhost",
                    port=1234
                ))

        """
        pass

    def parse_options(self):
        self.option_kwargs, self.option_args = self.option_parser.parse_args()

        for r_option in self.required_options:
            if r_option not in self.option_kwargs.__dict__:
                raise Error("Option '%s' required" % r_option)

    def run(self):
        """
        The entry point to start the service.
        """

        self.init_config()
        self.init_option_parser()

        self.parse_options()

        config_path = self.option_kwargs.config
        if config_path:
            config_file = open(config_path, "r")
            config = yaml.load(config_file)
            config_file.close()

            self.config.update(config)

        logging.root.setLevel(logging.INFO)

        if "logfile" in self.config.keys():
            handler = logging.handlers.RotatingFileHandler(
                self.config["logfile"])

            logging.root.addHandler(handler)

    def stop(self):
        """
        The entry point to stop the service.
        """
        pass


class MultiProcService(Service):
    def init_config(self):
        Service.init_config(self)

        self.config.update({
            "num_processes":-1
        })

    def fork(self):
        tornado.process.fork_processes(self.config["num_processes"])


class SocketService(MultiProcService):
    def init_sockets(self):
        raise NotImplementedError

    def init_socket_server(self, io_loop=None, ssl_options=None):
        raise NotImplementedError

    def run(self, io_loop=None):
        """
        must after super run::
        
            from tornado.ioloop import IOLoop
            IOLoop.instance().start()
        
        """
        MultiProcService.run(self)

        self.init_sockets()
        self.init_socket_server(io_loop=io_loop)

        self.socket_server.start(num_processes=self.config["num_processes"])
        self.socket_server.add_sockets(self.sockets)

    def stop(self):
        from tornado.ioloop import IOLoop

        self.socket_server.stop()
        IOLoop.instance().stop()


class TCPSocketService(SocketService):
    def init_sockets(self):
        self.sockets = netutil.bind_sockets(
            self.config["port"], self.config["address"])


class UNIXSocketService(SocketService):
    def init_sockets(self):
        self.sockets = [netutil.bind_unix_socket(self.config["socket_path"])]


class HTTPService(TCPSocketService):

    def init(self, application, **kwargs):
        self.application = application

    def init_socket_server(self, io_loop=None):
        self.socket_server = HTTPServer(self.application, io_loop=io_loop)


class Error(Exception):
    def __init__(self, message):
        Exception.__init__(self)

        self.message = message

    def __str__(self):
        return self.get_message()

    def __repr__(self):
        return self.get_message()

    def get_message(self):
        return self.message

