#!/usr/bin/env python3

"""
Name    : latexUse
Version : 2012.09
Author  : Christophe BAL
Mail    : projetmbc@gmail.com

This script gives some usefull functions in relation with the powerfull LaTeX
langage for typing scientific documents.

See the documentation for more details.
"""

import os
import subprocess
import collections

try:
    import osUse

except:
    from . import osUse


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class MistoolLaTeXUseError(ValueError):
    pass

def __RAISE_SOME_ERROR__(
    kind,
    path,
    action
):
    if action == 'access':
        action = 'needs the "Super User\'s rights"'

    elif action == 'create':
        action = "can't be created"

    elif action == 'exist':
        action = "doesn't exist"

    elif action == 'notTeX':
        action = "is not a TeX one"

    else:
        raise Exception('BUG !')

    raise MistoolLaTeXUseError(
        "The following {0} {1}.\n\t''{2}''".format(kind, action, path)
    )


# ---------------- #
# -- FORMATTING -- #
# ---------------- #

# --- START-ESCAPE ---#

# The two following variables were automatically built by one Python script.

# Sources :
#    * The page 7 in "The Comprehensive LATEX Symbol List" of Scott Pakin.
#    * http://www.grappa.univ-lille3.fr/FAQ-LaTeX/21.26.html

CHARACTERS_TO_ESCAPE = {
    'text': "{}_$&%#",
    'math': "{}_$&%#"
}

CHARACTERS_TO_LATEXIFY = {
    'text': {
        '\\': "\\textbackslash{}"
    },
    'math': {
        '\\': "\\backslash{}"
    }
}

# --- END-ESCAPE ---#

def escape(
    text,
    mode = "text"
):
    """
    In LaTeX language, somme character has one meaning. The purpose of this
    function is to escape all this special charcaters so as to see them in the
    output produced by LaTeX.

    For example, ``escape("\OH/ & ...")`` is equal to ``"\textbackslash{}OH/ \&
    ..."``, whereas ``escape(text = "\OH/ & ...", mode = "math")`` is equal to
    ``"\backslash{}OH/ \& ..."``. Just notice the difference coming from the use
    of ``mode = "math"``. In math formulas and in simple text, the backslash is
    indicated by using two different LaTeX commands.

    The function ``escape`` has two variables.

        1) ``text`` is simply the text to change.

        2) ``mode`` indicate which kind of text is analysed. By default, ``mode
        = "text"`` is for one content that will be simply text in the document
        produced by LaTeX. You can also use ``mode = "math"`` to indicate
        something that appears in one formula.

    Two global dictionaries are used : ``CHARACTERS_TO_ESCAPE`` indicates only
    characters that must be escaped whit one backslash ``\``, and
    ``CHARACTERS_TO_LATEXIFY`` is for expressions that needs one LaTeX command.
    """
    if not mode in CHARACTERS_TO_ESCAPE:
        raise MistoolLaTeXUseError(
            "Unknown mode : ''{0}''.".format(mode)
        )

    charToLatexify = CHARACTERS_TO_LATEXIFY[mode].items()
    charToEscape   = CHARACTERS_TO_ESCAPE[mode]

    newText = ''

    iMax = len(text)
    i = 0

    while(i < iMax):
        nothingFound = True

        for oneChar in charToEscape:
            if text[i] == oneChar:
                newText += '\\' + oneChar
                i += 1
                nothingFound = False
                break

        if nothingFound:
            for oneChar, oneCharInLaTeX in charToLatexify:
                if text[i:].startswith(oneChar):
                    newText += oneCharInLaTeX
                    i += len(oneChar)
                    nothingFound = False
                    break

        if nothingFound:
            newText += text[i]
            i += 1

    return newText


# --------------- #
# -- COMPILING -- #
# --------------- #

class Build(object):
    """
    This class defines methods related to the compilation of LaTeX documents.

    The variables used are the following ones.

        1) ``path`` is the path of the LaTeX file to compile.

        2) ``repeat`` indicates ow many times the compilation must be done : if
        the document has one table of content, several compilations are needed.
        By default, ``repeat = 1``.

        3) ``verbose``, which default value is ``True``, is to see or not the
        informations sent by LaTeX when it compiles.
    """
# Source :
#    * http://docs.python.org/py3k/library/subprocess.html
    SUBPROCESS_METHOD = {
# ``subprocess.check_call`` prints informations given during the compilation.
        True : subprocess.check_call ,
# ``subprocess.check_output`` does not print informations given during the
# compilation. Indeed it returns all this stuff in one string.
        False: subprocess.check_output
    }

    def __init__(
        self,
        path,
        repeat  = 1,
        verbose = True
    ):
# Does the file to compile exist ?
        if not osUse.isFile(path):
            __RAISE_SOME_ERROR__(
                kind   = "file",
                path   = path,
                action = "exist"
            )

# Do we have TEX file ?
        if osUse.ext(path) != "tex":
            __RAISE_SOME_ERROR__(
                kind   = "file",
                path   = path,
                action = "notTeX"
            )

# Infos given by the user.
        self.path = path
        self.pathNoExt = path[:-4]

        self.repeat = repeat
        self.verbose = verbose

# General infos about the LaTeX distribution and the OS.
        self.aboutLatex = about()

# The directory where to put the "paper" files.
        self.directory = osUse.parentDir(path)

    def __compile__(
        self,
        arguments,
        repeat
    ):
        """
        This method launches the compilation in verbose mode or not.
        """
        for i in range(1, repeat + 1):
            if self.verbose:
                print(
                    '\t+ Start of compilation Nb.{0} +'.format(i)
                )

            self.SUBPROCESS_METHOD[self.verbose](
# We go in the directory of the file to compile.
                cwd = self.directory,
# We use the terminal actions.
                args = arguments
            )

            if self.verbose:
                print(
                    '\t+ End of compilation Nb.{0} +'.format(i)
                )

    def pdf(
        self,
        repeat = None
    ):
        """
        This method asks LaTeX to build PDF output.
        """
        if repeat == None:
            repeat = self.repeat

# The option "-interaction=nonstopmode" don't stop the terminal even if there is
# an error found by ¨latex.
        self.__compile__(
            arguments = [
                'pdflatex',
                '-interaction=nonstopmode',
                self.path
            ],
            repeat = repeat
        )


# -------------- #
# -- CLEANING -- #
# -------------- #

# --- START-CLEAN ---#

# The two following variables were automatically built by one Python script.

CLASSIFIED_TEMP_EXT = collections.OrderedDict([
    ('debug', ['blg', 'ilg', 'log', 'glg']),
    ('html', ['4ct', '4tc', 'idv', 'lg', 'tmp', 'xref']),
    ('slide', ['nav', 'snm', 'vrb']),
    ('editor', ['synctex.gz', 'synctex.gz(busy)']),
    ('float', ['fff', 'ttt']),
    ('list', ['lof', 'lol', 'lot', 'bcl']),
    ('toc', ['toc', 'maf', 'mlf', 'mlt', 'mtc', 'plf', 'plt', 'ptc']),
    ('ref', ['aux', 'brf', 'out', 'glo', 'ist', 'gls', 'idx', 'ind']),
    ('biblio', ['bbl', 'run.xml']),
    ('theorem', ['thm']),
    ('eledmac', ['¨number']),
    ('eledpar', ['R¨number'])
])

ALL_EXT_TO_CLEAN = [
# debug
#
# ``log`` is produced by latex compilations, ``ilg`` and ``glg`` by
# makeindex compilations, where ``glg`` is related to the package glossary.
# ``blg`` is produced by bibtex compilations.
    'blg', 'ilg', 'log', 'glg',
# html
#
# This extensions are produced by the package ``tex4ht``.
    '4ct', '4tc', 'idv', 'lg', 'tmp', 'xref',
# slide
    'nav', 'snm', 'vrb',
# editor
#
# ``synctex.gz`` is produced by some editors to do synchronization between
# the LaTeX source file and its PDF output.
    'synctex.gz', 'synctex.gz(busy)',
# float
    'fff', 'ttt',
# list
#
# ``bcl`` is produced by the package ``bclogo`` : this gives the list of
# the logos.
# ``f`` is for Figure, ``l`` for Listing (cf. the package ``listings``),
# and ``t`` for Table.
    'lof', 'lol', 'lot', 'bcl',
# toc
#
# The package ``minitoc`` produces all this extensions excepted ``toc``.
    'toc', 'maf', 'mlf', 'mlt', 'mtc', 'plf', 'plt', 'ptc',
# ref
#
# ``out`` is produced by the package ``hyperref`` with the option
# ``bookmarks``, and ``brf`` with the option ``backref``.
# The package `` glossary`` produces ``glo`` and ``gls``, and also ``ist``
# if an additional makeindex compilation is launched.
# ``idx`` and ``ind`` are produced by makeindex compilations.
    'aux', 'brf', 'out', 'glo', 'ist', 'gls', 'idx', 'ind',
# biblio
#
# ``bbl`` is produces by bibtex compilations, and ``run.xml`` by biber
# compilations.
    'bbl', 'run.xml',
# theorem
    'thm',
# eledmac
    '¨number',
# eledpar
    'R¨number'
]

# --- END-CLEAN ---#

def clean(
    main,
    keep    = [],
    discard = ALL_EXT_TO_CLEAN,
    depth   = 0,
    verbose = False
):
    """
    The aim of this method is to remove extra files build during LaTeX
    compilations.

    Here are the arguments available.

        1) ``main`` will be either one instance of the class ``Build`` or the
        path of one file or one directory.

        In the case of one instance of the class ``Build`` or the path of one
        file, only the file having the same name of the file build, or
        indicated, will be removed.

        If ``main`` is the path of one directory, then the function will look
        for all files with path::``tex`` extension, an then the eventual extra
        files associated will be removed. In that case, you can use the two
        arguments ``depth`` and ``verbose``.

        2) ``keep`` is for indicate the list of extensions to keep. This is very
        usefull if you use the default value of the argument ``discard``.

        3) ``discard`` is for indicate the list of extensions to discard. By
        default, the extensions to clean are defined in the constant
        ``ALL_EXT_TO_CLEAN``.

        Remark : to build your own list of extensions you can use the dictionary
        ``CLASSIFIED_TEMP_EXT``.

        4) ``depth`` is the maximal depth for the research of the path::``tex``
        files when ``main`` is the path of one directory. The very special value
        ``(-1)`` indicates that there is no maximum. The default value is ``0``
        which asks to only look for in the direct content of the main directory
        to analyse.

        5) ``verbose`` is a boolean which asks to print in one terminal the
        path::``tex`` files found when one search is made inside one directory.
        This argument is usefull only if ``main`` is the path of one directory.
    """
# One instance of ``Build()``
    if type(main) == Build:
        thePathsNoExt = [main.pathNoExt]

# One file
    elif osUse.isFile(main):
        if osUse.ext(main) != "tex":
            __RAISE_SOME_ERROR__(
                kind   = "file",
                path   = main,
                action = "notTeX"
            )

        thePathsNoExt = [main[:-4]]

# One directory
    elif osUse.isDir(main):
        thePathsNoExt = []

        for oneFile in osUse.nextFile(
            main  = main,
            ext   = {'keep': ["tex"]},
            depth = depth
        ):
            thePathsNoExt.append(oneFile[:-4])

# Nothing existing
    else:
        thePathsNoExt = []

# Clean now !
    for pathNoExt in thePathsNoExt:
        if verbose:
            print('---> "{0}.tex"'.format(pathNoExt))

        for oneExt in [x for x in discard if x not in keep]:
            if '¨number' in oneExt:
                print("TODO  --->  ", oneExt)
                ...

            else:
                osUse.destroy(pathNoExt + '.' + oneExt)


# ---------------------------------- #
# -- ABOUT THE LATEX DISTRIBUTION -- #
# ---------------------------------- #

# Sources :
#    * http://tex.stackexchange.com/questions/68730/terminal-commands-to-have-information-about-latex-distribution
#    * http://tex.stackexchange.com/questions/69483/create-a-local-texmf-tree-in-miktex
#    * https://groups.google.com/forum/?hl=fr&fromgroups=#!topic/fr.comp.text.tex/ivuCnUlW7i8

def __mikeTexLocalDirectory():
    return 'C:/texmf-local'

def __localDirOfTexLive(
    osName = None
):
    if osName == None:
        osName = osUse.system()

# "subprocess.check_output" is a byte string, so we have to use the method
# "decode" so as to obtain an "utf-8" string.
    try:
        if osName == "windows":
            localDir = subprocess.check_output(
                args = ['kpsexpand',"'$TEXMFLOCAL'"]
            ).decode('utf8')

            return os.path.normpath(localDir.strip())

        elif osName in ["linux", "mac"]:
            localDir = subprocess.check_output(
                args = ['kpsewhich','--var-value=TEXMFLOCAL']
            ).decode('utf8')

            return os.path.normpath(localDir.strip())

    except:
        ...

def about():
    """
    The aim of this function is to give critical informations so to use the
    class ``Install`.

    This function returns the following kind of dictionary. This example has
    been obtained with a Mac computer where TeXlive is installed.

    python::
        {
            'osName'    : 'mac',
            'latexName' : 'texlive',
            'latexFound': True,
            'localDir'  : '/usr/local/texlive/texmf-local'
        }

    The key ``'localDir'`` contains the path to use to install special packages.
    """
    osName = osUse.system()

    latexFound = False
    localDir = None

# Windows
    if osName == "windows":
        winPath = osUse.pathEnv()

# Is MiKTeX installed ?
        if '\\miktex\\' in winPath:
            latexFound = True
            latexName = 'miktex'

            if osUse.isDir(__mikeTexLocalDirectory()):
                localDir = __mikeTexLocalDirectory()

# Is TeX Live installed ?
        elif '\\texlive\\' in winPath:
            latexFound = True
            latexName = 'texlive'
            localDir = __localDirOfTexLive(osName)

# Linux and Mac
    elif osName in ["linux", "mac"]:
# Is LaTeX installed ?
        if subprocess.check_output(args = ['which', 'pdftex']).strip():
            latexFound = True
            latexName = 'texlive'
            localDir = __localDirOfTexLive(osName)

# Unknown method...
    else:
        raise MistoolLaTeXUseError(
            "The OS ''{0}'' is not supported. ".format(osName)
        )

# The job has been done...
    return {
        'osName'    : osName,
        'latexName' : latexName,
        'latexFound': latexFound,
        'localDir'  : localDir
    }


# ---------------- #
# -- INSTALLING -- #
# ---------------- #

def __issu(localLatexDir):
    """
    This function tests if the script has been launched by the "Super User".
    """
    path = os.path.join(
        localLatexDir,
        '-p-y-t-o-o-l-t-e-s-t-.t-x-t-x-t'
    )

    try:
        osUse.makeTextFile(path)
        osUse.destroy(path)

    except:
        __RAISE_SOME_ERROR__(
            kind   = "directory",
            path   = localLatexDir,
            action = "access"
        )

def makeMiktexLocalDir():
    """
    This function creates one local directory and add it to the list of
    directories analysed by MiKTeX when it looks for packages.

    The path of the local directory is given by ``__mikeTexLocalDirectory()``
    because this variable is also used by the function ``about`` and
    ``install``.

    Even it is a really bad idea, you can change this path just after the
    importation of the module ``latexUse``.
    """
    aboutLatex = about()

    if aboutLatex['osName'] != "windows" or aboutLatex['latexName'] != "miktex":
        raise MistoolLaTeXUseError(
            'This function can only be used with the OS "window" '
            'and the LaTeX distribution "miktex".'
        )

# Creation of the directory
    try:
        osUse.makeDir(__mikeTexLocalDirectory())

    except:
        __RAISE_SOME_ERROR__(
            kind   = "directory",
            path   = __mikeTexLocalDirectory(),
            action = "create"
        )

# Adding the directory to the ones analysed by MiKTex.
    subprocess.check_output(
        args = [
            'initexmf',
            '--admin',
            '--user-roots=' + __mikeTexLocalDirectory(),
#           '--register-root=' + __mikeTexLocalDirectory(),
            '--verbose'
        ]
    )

    subprocess.check_output(
        args = [
            'initexmf',
            '--update-fndb',
            '--verbose'
            ]
        )

    print(
        '',
        'The following local directory for MiKTeX has been created.',
        '\t"{0}"'.format(__mikeTexLocalDirectory()),
        sep = "\n"
    )

def refresh(
    aboutLatex = None
):
    """
    This function refreshes the list of packages directly known by the LaTeX
    distribution. This always works in verbose mode (LaTeX is not very talkative
    when it refreshes).

    The only variable used is ``aboutLatex`` which is simply the informations
    returned by the function ``about``.
    """
    if aboutLatex == None:
        aboutLatex = about()

# TeX Live
    if aboutLatex['latexName'] == 'texlive':
        subprocess.check_call(args = ['mktexlsr'])

# MiKTex
    elif aboutLatex['latexName'] == 'miktex':
        subprocess.check_call(args = ['initexmf', '--update-fndb', '--verbose'])

# Unkonwn !!!
    else:
        raise MistoolLaTeXUseError(
            'The refresh of the list of LaTeX packages is not supported '
            'with your LaTeX distribution.'
        )

def install(
    listFile,
    name,
    clean = False
):
    """
    This function helps you to easily install unofficial packages to your LaTeX
    distribution.

    This function can only be called by one script launched by the super user.

    The variables used are the following ones.

        1) ``name`` is the name of the LaTeX package to build.

        2) ``listFile`` is the list of the whole paths of the files to copy.
        This variable can be easily build with the function ``listFile`` from
        the module ``osUse`` contained in the ¨python package ``Mistool``. The
        tree structure will be similar to the one corresponding to the files to
        copy.

        3) ``clean`` is boolean variable to delete or not an eventual package
        directory named ``name`` in the local directory of the LaTeX
        distribution. The default value of ``clean`` is ``False``.
    """
# Sources :
#    * http://www.commentcamarche.net/forum/affich-7670740-linux-supprimer-le-contenu-d-un-repertoire
#    * http://stackoverflow.com/questions/185936/delete-folder-contents-in-python

# Directories
    aboutLatex = about()

    localLatexDir = aboutLatex['localDir']

    if localLatexDir == None:
        message = "No local directroy for special packages has been found."

        if aboutLatex['latexName'] == "miktex":
            message = message + "You can use the function ''makeMiktexLocalDir''."

        raise MistoolLaTeXUseError(message)

    localLatexDir = os.sep.join([
        localLatexDir,
        'tex',
        'latex',
        name
    ])

# We must have the super user's rights with the local Latex directory.
    __issu(localLatexDir)

# We must find the smaller directory that contains all the files so as to
# respect the original tree directory structure.
    mainDir = osUse.commonPath(listFile)

# Directories to add
    dirAndTheirFiles = {}

    for oneFile in listFile:
        if not osUse.isFile(oneFile):
            __RAISE_SOME_ERROR__(
                kind   = "file",
                path   = oneFile,
                action = "exist"
            )

        localLatexSubDir = localLatexDir + osUse.parentDir(
            osUse.relativePath(mainDir,oneFile)
        ).strip()

        if localLatexSubDir in dirAndTheirFiles:
            dirAndTheirFiles[localLatexSubDir].append(oneFile)
        else:
            dirAndTheirFiles[localLatexSubDir] = [oneFile]

# Installation on Mac O$, Linux and Windows
    if aboutLatex['osName'] in ['mac', 'linux', 'windows']:
# Actions to do are...
        actionsToDo = []
        decoTab     = '    * '
        decoTabPlus = '        + '
        decoTabMore = '            - '

# Creation of directories and copy of the files
#
#    1) Deletion of a local package named ``name``
        if clean and osUse.isDir(localLatexDir):
            print(
                decoTab + 'Deletion of the old "{0}" '.format(name) \
                + 'package in the local LaTeX directory.'
            )

            osUse.destroy(localLatexDir)

#    2) Creation of the new local package named ``name``
        print(
            decoTab + 'Creation of the new "{0}" '.format(name) \
            + 'package in the local LaTeX directory.'
        )

#    3) Creation of the new directories with their contents
        for oneNewDir in sorted(dirAndTheirFiles.keys()):
            print(
                decoTabPlus + 'Adding new directory --> "{0}" '.format(
                    oneNewDir
                )
            )

            osUse.makeDir(oneNewDir)

            for oneFile in dirAndTheirFiles[oneNewDir]:
                if not osUse.isFile(oneFile):
                    __RAISE_SOME_ERROR__(
                        kind   = "file",
                        path   = oneFile,
                        action = "exist"
                    )

                print(
                    decoTabMore + 'Adding new file  --> "{0}" '.format(
                        osUse.fileName(oneFile) \
                        + "." \
                        + osUse.ext(oneFile)
                    )
                )

                localLatexSubDir = localLatexDir + osUse.parentDir(
                    osUse.relativePath(mainDir, oneFile)
                ).strip()

                osUse.copy(oneFile, localLatexSubDir)

#   We must refresh of the list of LaTeX packages.
        print(
            decoTab + 'Refreshing the list of LaTeX packages.'
        )

        refresh(aboutLatex)

# Unsupported OS
    else:
        raise MistoolLaTeXUseError(
            'The installation of local packages is not yet supported '
            'with the OS "{0}".'.format(aboutLatex['osName'])
        )
