#!/usr/bin/env python3

"""
Name    : osUse
Version : 2012.09
Author  : Christophe BAL
Mail    : projetmbc@gmail.com

This module contains functions to manipulate easily files, directories and also
to have informations about the system.

See the documentation for more details.
"""

import os
import shutil
import platform
import subprocess


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class MistoolOsUseError(ValueError):
    pass


# ----------- #
# -- TESTS -- #
# ----------- #

def isFile(path):
    """
    This function simply tests if the path points to one existing file.
    """
    return os.path.isfile(path)

def isDir(path):
    """
    This function simply tests if the path points to one existing directory.
    """
    return os.path.isdir(path)

def hasExtIn(
    path,
    listOfExt
):
    """
    This function tests if one path finishs by one of the extensions given in
    the list of extensions.
    """
    if '.' in path:
        for oneExtension in listOfExt:
            L = len(oneExtension) + 1

            if path[-L:] == '.' + oneExtension:
                return True

    return False


# ----------------------- #
# -- INFOS ABOUT PATHS -- #
# ----------------------- #

def name(path):
    """
    This function extracts from one path the name of one file with its extension
    or simply the name of one directory.
    """
    i = path.rfind(os.sep)

    return path[i+1:]

def fileName(path):
    """
    This function extracts the name without its extension of one file given by
    its path.
    """
    i = path.rfind(os.sep)

    return os.path.splitext(path[i+1:])[0]

def ext(path):
    """
    This function extracts the extension of one file given by its path.
    """
    return os.path.splitext(path)[1][1:]

def parentDir(path):
    """
    This function returns the path of the directory that contains the file or
    the directory corresponding to path.
    """
    return os.path.dirname(path)

def relativePath(
    main,
    sub,
):
    """
    Suppose that we have the following paths where ``main`` is the path of the
    main directory and ``sub`` the one of a sub directory or a file contained in
    the main directory.

    python::
        main = "/Users/projects/source_dev"
        sub  = "/Users/projects/source_dev/Mistool/osPlus.py"

    The function will return in that case the following string which always
    begin with one slash ``/``.

    python::
        "/Mistool/osPlus.py"
    """
# Special case of the same path !
    if main == sub:
        raise MistoolOsUseError(
            "The main path and the sub-path are equal."
        )

# We clean the path in case of they contain things like ``../``.
    main = os.path.normpath(main)
    sub  = os.path.normpath(sub)

# The main path must finish by one backslash because
#     "python/Mistool/source_dev/Mistool/osPlus.py"
# is not one sub path of
#     "python/Mistool/source".

    if main[-1] != os.sep:
        main += os.sep

# Is the sub path contained in the main one ?
    if not sub.startswith(main):
        raise MistoolOsUseError(
            "The sub-path\n\t+ {0}\nis not contained in the main path\n\t+ {1}\n" \
                .format(sub, main) \
            + "so it is not possible to have one relative path."
        )

# Everything seems ok...
    i = len(main) - 1

    return sub[i:]

def relativeDepth(
    main,
    sub
):
    """
    Suppose that we have the following paths where ``main`` is the path of the
    main directory and ``sub`` the one of a sub directory or a file contained in
    the main directory. Here are some examples.

        1) The function will return ``1`` in the following case. This means that
        the file path::``osPlus.py`` is contained in one simple sub directory of
        the main directory.

        python::
            main = "/Users/projects/source_dev"
            sub  = "/Users/projects/source_dev/Mistool/osPlus.py"

        2) In the following case, the function will return ``2``.

        python::
            main = "/Users/projects/source_dev"
            sub  = "/Users/projects/source_dev/Mistool/os/osPlus.py"

        3) For the last example just after, the value returned will be ``0``.

        python::
            main = "/Users/projects/source_dev"
            sub  = "/Users/projects/source_dev/osPlus.py"
    """
    return relativePath(
        main        = main,
        sub         = sub
    ).count(os.sep) - 1

def commonPath(
    listPath
):
    """
    This function returns the smaller directory that contains the objects having
    the paths ``path1`` and ``path2``.
    """
    if len(listPath) < 2:
        raise MistoolOsUseError(
            "You must give at least two paths."
        )

    if len(listPath) == 2:
        answer = []

        path1 = listPath[0].split(os.sep)
        path2 = listPath[1].split(os.sep)

        for i in range(min(len(path1), len(path2))):
            if path1[i] == path2[i]:
                answer.append(path1[i])
            else:
                break

        return os.sep.join(answer)

    else:
        return commonPath([
            commonPath(listPath[:-1]), listPath[-1]
        ])


# ------------- #
# -- READING -- #
# ------------- #

def readTextFile(
    path,
    encoding = 'utf-8'
):
    """
    This function returns the text like content of one file given by its path.
    """
    with open(
        path,
        mode     = "r",
        encoding = encoding
    ) as f:
        return f.read()


# ------------------------------------- #
# -- OPENING WITH ASSOCIATED PROGRAM -- #
# ------------------------------------- #

def watch(path):
    """
    This function open one directory, or one file. Indeed, files are opened
    within their associated applications, if this last ones exist.
    """
# Nothing to open...
    isOneFile = isFile(path)

    if not isOneFile and not isDir(path):
        raise MistoolOsUseError(
            "The following path does not point to one existing file " \
            "or directory.\n\t''{0}''".format(path)
        )

# Each OS has its own method.
    osName = system()

# Windows
    if osName == "windows":
        if isOneFile:
            os.startfile(path)
        else:
            subprocess.check_call(args = ['explorer', path])

# Mac
    elif osName == "mac":
        subprocess.check_call(args = ['open', path])

# Linux
#
# Source :
#     * http://forum.ubuntu-fr.org/viewtopic.php?pid=3952590#p3952590
    elif osName == "linux":
        subprocess.check_call(args = ['xdg-open', path])

# Unknown method...
    else:
        raise MistoolOsUseError(
            "The opening of file on ''{0}'' OS is not "
            "supported.".format(osName)
        )


# ----------------------------------- #
# -- CREATION, DELETION AND MOVING -- #
# ----------------------------------- #

def makeDir(path):
    """
    This function build the directory with the given path. If one parent
    directory must be build, the function will do the job.
    """
    if not isDir(path):
        os.makedirs(path)

def makeTextFile(
    path,
    text     = '',
    encoding = 'utf-8'
):
    """
    This function build the file with the given path and the text like content.
    """
    makeDir(parentDir(path))

    with open(
        path,
        mode     = "w",
        encoding = encoding
    ) as f:
        f.write(text)

def move(
    source,
    destination
):
    """
    This function moves the file or the directory from ``source`` to
    ``destination``. If the source and the destination has the same parent
    directory, then you just rename the file or the directory.
    """
    if isDir(path):
        os.renames(source, destination)

    elif isFile(source):
        copy(source, destination)

        if isFile(destination):
            destroy(source)

def copy(
    source,
    destination
):
    """
    This function copy the file having the path ``source`` to the destination
    given by the path ``destination``.
    """
    if isFile(source):
        dir = parentDir(destination)

        if not isDir(dir):
            makeDir(dir)

        shutil.copy(source, destination)

    elif isDir(source):
        raise MistoolOsUseError(
            "The copy of one directory is not yet supported."
        )

    else:
        raise MistoolOsUseError(
            "The following path points nowhere.\n\t''{0}''".format(source)
        )

def destroy(path):
    """
    This function removes the directory or the file given by its path.
    """
    if isDir(path):
        shutil.rmtree(path)

    elif isFile(path):
        os.remove(path)

def clean(
    path,
    ext,
    depth = 0
):
    """
    This function removes extra files in one directory regarding to their
    extension.

    The variables are the following ones.

        1) ``path`` is simply the path of the directory to clean.

        2) ``ext`` is the list of the extensions of the files to remove.

        3) ``depth`` is the maximal depth for the research of the files to
        remove. The very special value ``(-1)`` indicates that there is no
        maximum. The default value is ``0`` which asks to only look for in
        the direct content of the main directory to analyse.
    """
    for oneFile in nextFile(
        main  = path,
        ext   = {'keep': ext},
        depth = depth
    ):
        destroy(oneFile)


# ------------------- #
# -- LIST OF FILES -- #
# ------------------- #

def __isSubDepthGood(
    main,
    sub,
    depth
):
    """
    This small function is used by ``nextFile`` so as to know if one directory
    must be inspected or not.
    """
    if depth == -1:
        return True

    elif depth == 0:
        return bool(main == sub)

# ``relativeDepth``  sends logically one error if the two directories are equal,
# so we have to take care of this very special case.
    elif main == sub:
        return True

    else:
# The use of ``... + 1`` in the test comes from the fact we work with files
# contained in the sub directory, and the depth of this files is equal to the
# one of sub directory the plus one.
        return bool(
            relativeDepth(main = main, sub = sub) <= depth - 1
        )

def nextFile(
    main,
    ext      = {},
    depth    = 0,
    sub      = {},
    keepDir  = None
):
    """
    This function is one iterator that sends infos about files that respects
    some constraints. At each iteration, the function returns the whole path of
    the file.

    The variables are the following ones.

        1) ``main`` is simply the path of the main directory to analyse.

        2) ``ext`` is a dictionary of the following kind which is used to keep
        or to discard some kinds of files.

        python::
            {
                'keep'   : list of the extensions of files to look for,
                'discard': list of the extensions of files to discard
            }

        If ``ext`` is an empty dictionary, or if the two lists are empty ones
        then the function will send every kind of files.

        3) ``depth`` is the maximal depth for the research. The very special
        value ``(-1)`` indicates that there is no maximum. The default value is
        ``0`` which asks to only look for in the direct content of the main
        directory to analyse.

        4) Let see with one example how to define the dictionary value of the
        variable ``sub``. We assume that the main directory path::``source_dev``
        has the following simple structure with two sub directories.

        dir::
            + source_dev
                + change_log
                [...]
                + debug
                [...]

        Regarding to the main directory, the two sub directories have the
        relative paths path::``/change_log`` and path::``/debug`` respectively.

        Let also suppose that we only want to keep files having the extension
        path::``txt`` in the sub directory path::``change_log``, and that we
        don't want to look for files in teh directories contained in  the sub
        directory path::``debug``. Here is the value to give to the variable
        ``sub`` so as to achieve this. Here the research specifications for
        the sub directories erase the ones for the main directory.

        python::
            {
                'change_log': {
                    'keep'   : ['txt']
                },
                'debug': {
                    'depth': 0
                }
            }

        In the preceding dictionary, the keys are the relative path of the sub
        directory with or without the leading slash path::``/``.

        You can combine the use of the keys ``'depth'`` and ``'keep'`` for the
        same sub directory, and it is also possible to indicate extensions to
        discard with the key ``'discard'``.

        We finish with one last example. Let suppose here that in the main
        directory we are only looking for files with the extension path::``py``,
        and that we want to look for this kind of files and also the ones
        having the extension path::``txt`` in the sub directory
        path::``change_log``. For such a situation, it suffises to use the key
        ``'inherit'`` in the dictionary that defines the extensions to keep or
        to discard. Here is the value of the variable ``sub`` to use.

        python::
            {
                'change_log': {
                    'inherit': True ,
                    'keep'   : ['txt']
                }
            }

        The value of the key ``'inherit'`` can be ``True``, or ``False`` which
        is the default value, but you can also use couple of boolean values like
        ``(False, True)`` which asks to only inherit the extensions for the
        files to discard.

        5) The variable ``keepDir`` has been added to be used by the class
        ``TreeDir``, but you can play with it if you want. The possible values
        ar the following ones.

            a) The default value ``None`` indicates to unkeep any directory.
            This is what is expected when you call one function named
            ``nextFile``.

            b) ``"minimal"`` will make the function returns only the directories
            that contains at least one file from the ones searched.

            c) ``"all"`` will make the function returns any sub directories even
            if it doesn't contained one file from the ones searched.

            d) You can also use ``"Minimal"`` and ``"All"``, be carefull of the
            first uppercase letters, so as to also have idiomatic paths ended by
            path::"..." to indicate other files that the ones searched which
            also mustn't be discarded.
    """
# Directories to keep ?
    if not keepDir in [None, "minimal", "Minimal", "all", "All"]:
        raise MistoolOsUseError("Illegal value of the variable ''keepDir''.")

    dirMustBeKept = bool(keepDir in ["minimal", "Minimal", "all", "All"])

# Indicate or not the files not matching the search queries ?
    indicateAllFiles = bool(keepDir in ["Minimal", "All"])

# Does the directory exist ?
    if not isDir(main):
        raise MistoolOsUseError(
            "The following path does not point to one existing directory." \
            "\n\t+ {0}".format(main)
        )

# Is there some specific researches regarding to some sub directories ?
    listOfsubPath    = []
    subRelativePaths = {}

    for oneRelativePath in sub:
        realPath = oneRelativePath

        if not realPath.startswith('/'):
            realPath = '/' + realPath

        realPath = main + realPath

        if not isDir(realPath):
            raise MistoolOsUseError(
                "The following path does not point to one existing "
                "sub directory.\n\t+ {0}".format(realPath)
            )

        listOfsubPath.append(realPath)
        subRelativePaths[realPath] = oneRelativePath

    extToKeep    = ext.get('keep', [])
    extToDiscard = ext.get('discard', [])

    for root, dirs, files in os.walk(main):
### CASE OF ONE SUB DIRECTORY WITH SPECIAL CRITERIONS ###
        if root in listOfsubPath:
            subResearch = sub[subRelativePaths[root]]

            inherit         = subResearch.get('inherit', False)
            subExtToKeep    = subResearch.get('keep', [])
            subExtToDiscard = subResearch.get('discard', [])
            subDepth        = subResearch.get('depth', 0)

# Inherit or not inherit ? That is the question...
            if inherit:
                subExtToKeep    += extToKeep
                subExtToDiscard += extToDiscard
                subDepth        = depth

# Let's look for the files in the sub directory...
            for oneFileInsub in nextFile(
                main  = root,
                ext  = {
                    'keep'   : subExtToKeep,
                    'discard': subExtToDiscard
                },
                depth = subDepth,
                keepDir  = keepDir
            ):
                yield oneFileInsub

### CASE OF ONE SUB DIRECTORY WITHOUT SPECIAL CRITERIONS ###
        elif __isSubDepthGood(
            main  = main,
            sub   = root,
            depth = depth
        ):
            noBadFileFound  = True
            noGoodFileFound = True

            for oneFile in files:
                path = root + os.sep + oneFile

                if not hasExtIn(path, extToDiscard):
                    if extToKeep:
# Looking for good files
                        if hasExtIn(path, extToKeep):
# Directory of the first good file found to display ?
                            if noGoodFileFound and dirMustBeKept:
                                noGoodFileFound = False
                                yield root

                            yield path

# One bad file found
                        elif noBadFileFound:
                            noBadFileFound = False

# Looking for any file
                    else:
                        if noGoodFileFound and dirMustBeKept:
                            noGoodFileFound = False
                            yield root

                        yield path

# Directory without any good file
            if noGoodFileFound and keepDir in ["all", "All"]:
                yield root

# Directory without some good files
            if indicateAllFiles and not noBadFileFound:
                yield root + os.sep + "..."

def listFile(*args, **kwargs):
    """
    This function is similar to the function ``nextFile`` and it has the same
    variables that the ones of ``nextFile``, but instead of sending infos about
    the files found one at a time, this function directly sends the whole list
    of the infos found.

    See the documentation of the function ``nextFile`` to have precisions about
    the available variables and the structure of each single info that will be
    in the list returned by ``listFile``.
    """
# The use of ``*args`` and ``**kwargs`` makes it very easy to implement the fact
# that ``listFile`` and ``nextFile`` have the same variables.
    theList = []

    for onePath in nextFile(*args, **kwargs):
        theList.append(onePath)

    return theList

class DirView(object):
    """
    This class displays the tree structure of one directory with the possibility
    to keep only some relevant informations.

    Indeed, this class uses the variables ``main``, ``ext``, ``depth`` and
    ``sub`` and ``keepDir`` which have exactly the same meaning and behavior
    that with the function ``nextFile``, except that here the default value of
    ``keepDir`` is ``"minimal"`` and not ``None``.

    There is also two other variables which control the way the tree structure
    will be displayed.

        1) ``output`` is for the writings of the paths.

            a) The default value ``"all"`` will display the whole paths of the
            files and directories found.

            b) ``"relative"`` will display relative paths comparing to the main
            directory analysed.

            c) ``"short"`` will only display names of directories found, and
            names, with its extensions, of the files found.

        2) ``seeMain`` is one boolean value to display or not the main directory
        which is analyzed. The default value is ``True``.
    """
    ASCII_DECORATION = {
        'directory': "+",
        'file'     : "*",
        'tab'      : " "*4
    }

    def __init__(
        self,
        main,
        ext      = {},
        depth    = 0,
        sub      = {},
        keepDir  = "minimal",
        output   = "all",
        seeMain  = True
    ):
# Directories to keep ?
        if keepDir == None:
            raise MistoolOsUseError("Illegal value of the variable ''keepDir''.")

        self.main    = main
        self.ext     = ext
        self.depth   = depth
        self.sub     = sub
        self.keepDir = keepDir
        self.output  = output
        self.seeMain = seeMain

        self.listView = []
        self.format   = {}

    def build(self):
        """
        This function returns one list of dictionary of the following kind.

        python::
            {
                'kind' : "directory" or "file",
                'depth': the depth level regarding to the main directory,
                'path' : the path of one directory or file found
            }
        """
        self.format   = {}

        for onePath in nextFile(
            main    = self.main,
            ext     = self.ext,
            depth   = self.depth,
            sub     = self.sub,
            keepDir = self.keepDir
        ):
            if onePath == self.main:
                self.listView.append({
                    'kind' : "directory",
                    'depth': -1,
                    'path' : onePath
                })

            else:
# Which kind of object ?
                if self.__otherFiles(onePath) or isFile(onePath):
                    kind = "file"
                else:
                    kind = "directory"

# The depth
                depth = relativeDepth(self.main, onePath)

# Let's store the infos found.
                self.listView.append({
                    'kind' : kind,
                    'depth': depth,
                    'path' : onePath
                })

    def __otherFiles(self, onePath):
        return bool(onePath[-3:] == "...")

    def pathToDisplay(
        self,
        onePath,
        kind
    ):
        if self.__otherFiles(onePath):
            return "..."

        elif self.output == "relative":
            if self.main == onePath:
                return onePath[len(parentDir(onePath))+1:]
            else:
                return relativePath(self.main, onePath)

        elif self.output == "short":
            if kind == "file":
                return onePath[onePath.rfind(os.sep)+1:]

            else:
                return onePath[len(parentDir(onePath))+1:]

        return onePath

    @property
    def ascii(self):
        """
        This methods returns a simple ASCCI tree view of the tree structure.
        """
        if 'ascii' not in self.format:
            text = []

            for oneInfo in self.listView:
                depth = oneInfo["depth"]

# Does the main directory must be displayed ?
                if self.seeMain:
                    depth += 1

                tab = self.ASCII_DECORATION['tab']*depth

                decoKind = self.ASCII_DECORATION[oneInfo["kind"]] + " "

                pathToDisplay = self.pathToDisplay(
                    oneInfo["path"], oneInfo["kind"]
                )

                text.append(
                    "{0}{1}{2}".format(tab, decoKind, pathToDisplay)
                )

            self.format['ascii'] = '\n'.join(text)

        return self.format['ascii']


# ------------------- #
# -- GENERAL INFOS -- #
# ------------------- #

def pathEnv():
    """
    This function simply returns the variable ``PATH`` that contains paths of
    some executables known by your OS.
    """
    return os.getenv('PATH')

def system():
    """
    The purpose of this function is to give the name, in lower case, of the OS
    used. Possible names can be "windows", "mac", "linux" and also "java".
    """
    osName = platform.system()

    if not osName:
        raise MistoolOsUseError(
            "The operating sytem can not be found."
        )

    if osName == 'Darwin':
        return "mac"

    else:
        return osName.lower()
