#!/usr/bin/env python3

"""
Name    : pythonUse
Version : 2012.11
Author  : Christophe BAL
Mail    : projetmbc@gmail.com

This module contains some simple tools about the Python programming language.

See the documentation for more details.
"""

import gc
import sys
import collections


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class MistoolPythonUseError(ValueError):
    pass


# ------------------------ #
# -- NAME(S) OF OBJECTS -- #
# ------------------------ #

# Sources :
#    * http://pythonic.pocoo.org/2009/5/30/finding-objects-names
#    * http://www.developpez.net/forums/d1265263/autres-langages/python-zope/general-python/nom-variable/

def allName(
    obj,
    globalKeys
):
    """
    This technical function tries to retrieve all possible names of an object.
    Here is an example of use. Just notice that the value of the variable
    ``globalKeys`` is always ``globals().keys()`` which gives important
    information about the script that is using ``allName``.

    python::
        from mistool import pythonUse

        myDict = {'a':1, 'b':2}

        print(
            pythonUse.allName(
                obj        = myDict,
                globalKeys = globals().keys()
            )
        )

        anotherDict = myDict

        print(
            pythonUse.allName(
                obj        = anotherDict,
                globalKeys = globals().keys()
            )
        )

    This script launched in one terminal will produce the following output.

    terminal::
        ['myDict']
        ['anotherDict', 'myDict']

    In the second case, there are two names because the logic of ¨python is that
    the dictionary ``anotherDict`` is an alias of ``myDict``.

    warning::
        If the function doesn't find anything, it will raise one
        ``MistoolPythonUseError``.
    """
    frame = sys._getframe()

    for frame in iter(lambda: frame.f_back, None):
        frame.f_locals

    result = []

    for referrer in gc.get_referrers(obj):
        if isinstance(referrer, dict):
            for k, v in referrer.items():
                if v is obj:
                    result.append(k)

    if type(obj) == int:
        result = [x for x in result if x in globalKeys]

    if result:
        if result[0] == 'obj':
            result = result[1:]

        return result

    raise MistoolPythonUseError("No name has been found.")

def name(
    obj,
    globalKeys
):
    """
    This function returns the last name in the list produced by the function
    ``allName`` (see the documentation of this function).
    """
    return allName(obj, globalKeys)[-1]


# ------------- #
# -- QUOTING -- #
# ------------- #

QUOTE_SYMBOLS = ['"', "'"]

def __escape__(data):
    """
    This small function escapes all the characters that must be escaped in one
    python-like string.
    """
    if type(data) == str:
        return data.replace('\\', '\\\\')

    else:
        return data

def quote(
    text,
    symbol = '"'
):
    """
    This function put the content of the string variable ``text`` into quotes
    and escapes the eventual quote symbols in ``text``.

    This function has one optional variable ``symbol`` which indicates the
    prefered quoting symbol which is ``"`` by default.

    This function uses the global constant ``QUOTE_SYMBOLS`` which is defined
    by ``QUOTE_SYMBOLS = ['"', "'"]``. This indicates the list of the possible
    quoting symbols.

    For example, ``quote("one \"small\" example")`` is equal to the text ``'one
    "small" example``, and ``quote('another \'example\' to "see" more")``` is
    equal to ``"another 'example' to \"see\" more"``.
    """
    if all(x in text for x in QUOTE_SYMBOLS):
        text = text.replace(symbol , '\\' + symbol)

    elif symbol in text:
        for x in QUOTE_SYMBOLS:
            if x != symbol:
                symbol = x

    return "{0}{1}{0}".format(symbol, text)


# -------------------------- #
# -- WRITING PYTHON FILES -- #
# -------------------------- #

def __isGoodType__(
    kind,
    objType,
    wantedType
):
    if objType != wantedType:
        messageType = {
            str : 'string',
            list: 'list',
            dict: 'dictionary'
        }

        wantedType = messageType.get(wantedType, str(wantedType))

        raise MistoolPythonUseError(
            'With ''kind = "{0}"'', the object to write '
            'must be one {1}.'.format(kind, wantedType)
        )

def write(
    obj,
    kind  = None,
    depth = -1,
    tab   = " "*4,
    level = 0
):
    """
    This function returns the string representation in ¨python codes of one
    ¨python object.

    There are five variables. See the documentation for more explanations.

        1) ``obj`` is the python object to print in one ¨python code.

        1) ``kind`` is an optional argument to indicate the kind of object.
        The default value is ``None`` which asks to use the type of the object.
        The supported values are the following ones.

            * ``"bool"``, ``"dict"``, ``"float"``, ``"int"``, ``"list"`` and
            ``"str"`` indeed corresponds to ¨python types supported.

            * ``"repr"`` simply asks to use the standard ¨python representation.

            * ``"lambda"`` is for producing lambda expressions only from their
            definitions given via one string.

        1) ``depth`` is an optional argument to indicate how deep to go in the
        printings of dictionaries and lists. The default value is ``(-1)`` which
        indicates to go the deeper is possible.

        1) ``tab`` is an optional argument which is the tabulation used for the
        representation of dictionaries and lists. The default value is
        ``" "*4``.

        1) ``level`` is an optional argument that indicates the current level
        of search. Normally, you don't have to use this variable that is used
        during recursive calls. The default value is ``0``.
    """
# We have to guess the type of the object.
    objType = type(obj)

    if kind == None:
        if objType in [bool, dict, float, int, list, str]:
            kind = str(objType)
            kind = kind[kind.find("'")+1:]
            kind = kind[:kind.find("'")]

        else:
            raise MistoolPythonUseError(
                'Unsupported type "{0}".'.format(objType)
            )

# We do the job.
    if kind == "bool":
        answer = repr(bool(obj))

    elif kind == "str":
        answer = quote(__escape__(obj))

    elif kind == "lambda":
        __isGoodType__(
            kind       = kind,
            objType    = objType,
            wantedType = str
        )

        answer = "lambda " + obj

    elif kind == "list":
        __isGoodType__(
            kind       = kind,
            objType    = objType,
            wantedType = list
        )

        answer = __writeList__(
            oneList = obj,
            level   = level,
            tab     = tab,
            depth   = depth
        )

    elif kind == "dict":
        __isGoodType__(
            kind       = kind,
            objType    = objType,
            wantedType = dict
        )

        answer = __writeDict__(
            oneDict = obj,
            level   = level,
            tab     = tab,
            depth   = depth
        )

    elif kind in ["float", "int", "repr"]:
        answer = repr(obj)

    else:
        raise MistoolPythonUseError('Unknown kind "{0}".'.format(kind))

    return tab*level + answer

def __writeList__(
    oneList,
    level,
    tab,
    depth
):
    """
    This function is for writing lists.
    """
    if depth == 0:
        return repr(oneList)

    text = []

    for elt in oneList:
        elt = write(
            obj     = elt,
            level   = level + 1,
            tab     = tab,
            depth   = depth - 1
        )

        text.append(elt + ",")

    text[-1] = text[-1][:-1]

    text.append(tab*level + ']')
    text = ['['] + text

    return '\n'.join(text)

def __writeDict__(
    oneDict,
    level,
    tab,
    depth
):
    """
    This function is for writing dictionaries.
    """
    try:
        sortedKeys = sorted(oneDict.keys())
    except:
        sortedKeys = oneDict.keys()

    if depth == 0:
        text = ['{']

        for key in sortedKeys:
            value = repr(oneDict[key])
            key   = repr(key)

            text.append("{0}: {1},".format(key, value))

        text[-1] = text[-1][:-1]

        text.append('}')

        return ''.join(text)

    else:
        text = []

        for key in sortedKeys:
            value = write(
                obj     = oneDict[key],
                level   = level + 1,
                tab     = tab,
                depth   = depth - 1
            ).lstrip()

            key = tab*(level+1) +  repr(key)

            text.append("{0}: {1},".format(key, value))

        text[-1] = text[-1][:-1]

        text.append(tab*level + '}')
        text = ['{'] + text

        return '\n'.join(text)
