#!/usr/bin/env python3

# -- Source :
# --    http://docs.python.org/library/re.html

"""
Name    : regexUse
Version : 2013.03
Author  : Christophe BAL
Mail    : projetmbc@gmail.com

This script gathers some testing functions based on regex patterns.

See the documentation for more details.
"""

import re
from decimal import Decimal
from fractions import Fraction

# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class MistoolRegexUseError(ValueError):
    pass


# ------------------------------ #
# -- LEGAL NAME FOR VARIABLES -- #
# ------------------------------ #

PATTERN_VAR_NAME = {
    'standard'      : re.compile(
        "^[a-zA-Z][\d_a-zA-Z]*$"
    ),
    'python'        : re.compile(
        "^[_a-zA-Z][\d_a-zA-Z]*$"
    ),
    'lazzy'         : re.compile(
        "^[\d_a-zA-Z]+$"
    ),
    'pybaLexerBlock': re.compile(
        "^[a-zA-Z][\d_a-zA-Z]*(-[\d_a-zA-Z]+)*(\*)?$"
    )
}

def isVarName(
    text,
    kind = "standard"
):
    """
    This function tests if one text is one legal name for variable for different
    contexts given by the value of ``kind`. The possible values of this
    argument are the following ones.

        1) The default value is ``"standard"`` which is for names starting with
        one ASCII letter followed by possible digits, ASCII letters or the
        underscore ``_``.

        2) ``"python"`` is for Python names for variables which can start with
        one ASCII letter or one underscore ``_`` following by possible digits,
        ASCII letters or the underscore ``_``.

        3) ``"lazzy"`` is for names which use digits, ASCII letters and the
        underscore ``_``.

        4) ``"pybaLexerBlock"`` is used in ``pyLexer`` which is another project
        of the author of ``Mistool``. This kind of variables must start with one
        ASCII letter followed by possible digits, ASCII letters, underscores
        ``_``, isolated single minus signs ``-``, and eventually at the end one
        single star ``*``.

    Indeed, all the patterns used are defined in the following dictionary that
    you can customize by adding for example new kinds.

    python::
        PATTERN_VAR_NAME = {
            'standard'      : re.compile("^[a-zA-Z][\d_a-zA-Z]*$"),
            'python'        : re.compile("^[_a-zA-Z][\d_a-zA-Z]*$"),
            'lazzy'         : re.compile("^[\d_a-zA-Z]+$"),
            'pybaLexerBlock': re.compile(
                "^[\d_a-zA-Z]+(-[\d_a-zA-Z]+)*(\*)?$"
            )
        }
    """
    if not kind in PATTERN_VAR_NAME:
        raise MistoolRegexUseError(
            "Unknown kind ''{0}''.".format(kind)
        )

    return bool(PATTERN_VAR_NAME[kind].search(text))


# ----------------------- #
# -- NATURAL NUMBERS ? -- #
# ----------------------- #

# The following pattern comes from the book "Dive into Python".

PATTERN_ROMAN_NUMERAL = re.compile(
    """
    ^                   # beginning of string
    M{0,4}              # thousands: 0 to 4 M's
    (CM|CD|D?C{0,3})    # hundreds: 900 (CM),
                        #           400 (CD),
                        #           0-300 (0 to 3 C's) or
                        #           500-800 (D, followed by 0 to 3 C's)
    (XC|XL|L?X{0,3})    # tens: 90 (XC), 40 (XL),
                        #       0-30 (0 to 3 X's) or
                        #       50-80 (L, followed by 0 to 3 X's)
    (IX|IV|V?I{0,3})    # ones: 9 (IX), 4 (IV),
                        #       0-3 (0 to 3 I's) or
                        #       5-8 (V, followed by 0 to 3 I's)
    $                   # end of string
    """ ,
    re.VERBOSE
)

def isUpperRoman(text):
    """
    This function tests if one text corresponds to one roman number. The
    pattern used is the following one coming from the book "Dive into Python".

    python::
        PATTERN_ROMAN_NUMERAL = re.compile(
            '''
            ^                   # beginning of string
            M{0,4}              # thousands: 0 to 4 M's
            (CM|CD|D?C{0,3})    # hundreds: 900 (CM),
                                #           400 (CD),
                                #           0-300 (0 to 3 C's) or
                                #           500-800 (D, followed by 0 to 3 C's)
            (XC|XL|L?X{0,3})    # tens: 90 (XC), 40 (XL),
                                #       0-30 (0 to 3 X's) or
                                #       50-80 (L, followed by 0 to 3 X's)
            (IX|IV|V?I{0,3})    # ones: 9 (IX), 4 (IV),
                                #       0-3 (0 to 3 I's) or
                                #       5-8 (V, followed by 0 to 3 I's)
            $                   # end of string
            ''' ,
            re.VERBOSE
        )
    """
    return bool(PATTERN_ROMAN_NUMERAL.search(text))

def isNatural(text):
    """
    This function tests if one text corresponds to one natural number.
    """
    return text.isdigit()

def isInteger(text):
    """
    This function tests if one text corresponds to one integer.
    """
    try:
# We cannot use  isinstance(text, int).
        int(text)
        return True
    except:
        return False

def isDecimal(text):
    """
    This function tests if one text corresponds to one decimal number.
    """
    try:
        Decimal(text)
        return True
    except:
        return False

def isRational(text):
    """
    This function tests if one text corresponds to one rational number.
    """
    try:
        Fraction(text)
        return True
    except:
        return False

