#!/usr/bin/env python3

"""
Name    : logTestUse
Version : 2013.04
Author  : Christophe BAL
Mail    : projetmbc@gmail.com

This modules contains some utilities that can be usefull for unit tests made
with the standard module ``unittest``.

See the documentation for more details.
"""

# Source used :
#    1) http://agiletesting.blogspot.com/2005/01/python-unit-testing-part-1-unittest.html
#    2) http://stackoverflow.com/questions/67631/how-to-import-a-module-given-the-full-path
#    3) http://docs.python.org/3.2/library/unittest.html?highlight=unittest.texttestrunner#basic-example

import imp
import inspect
import unittest
import collections

try:
    import osUse
    import pythonUse
    import stringUse

except:
    from . import osUse
    from . import pythonUse
    from . import stringUse


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class MistoolLogTestUseError(ValueError):
    pass

def __unexpectedError__(text):
    """
    This is for errors not expected during the tests.
    """
    logPrint(
        '    UNEXPECTED ERROR DETECTED DURING THE TEST :\n' \
        + '        ' \
        + '        \n'.join(text.split('\n'))
    )


# ---------------------- #
# -- LOGGING MESSAGES -- #
# ---------------------- #

# This variable is used to define some formatting rules from one path or one
# name of a function, a method or a class.

ASCII_ASSO = {
    '_'  : " ",
    '__' : " - ",
    '___': " ---> ",
    '/'  : " : "
}

def what(
    text,
    isMethod = False
):
    """


    In any case, the function ``what`` transforms some text like
    ``Abreviations__NewAdding_Good`` into ``Abreviations___New_Adding__Good``,
    and finally returns ``Abreviations ---> New Adding - Good`` where the
    replacements correspond to the following dictionary.

    python::
        ASCII_ASSO = {
            '_'  : " ",
            '__' : " - ",
            '___': " ---> ",
            '/'  : " : "
        }


    There are two arguments.

        1) ``text`` is a text of something that is tested.

        2) ``isMethod`` is an optional boolean which is equal to ``False`` by
        default.

        If ``isMethod = True``, then the text must start with ``"test"``. If it
        is the case, this piece of text will be removed (the module ``unittest``
        only use methods with name started by ``"test"``).
    """
    if isMethod:
        if not text.startswith('test'):
            raise MistoolLogTestUseError('The text does not start with "test".')

        text = text[len('test'):]

    text = stringUse.camelTo(text, "title")

    text = stringUse.replace(
        text        = text,
        replacement = ASCII_ASSO
    )

    return text.strip()


def diffDict(
    dict_1,
    dict_2,
    dictName
):
    """
    ??

    This function analyses if the two dictionaries ``dict_1`` and ``dict_2`` are
    equal. If it is the case, an empty string is returned. In the other case, a
    message indicating the differences between the two dictionaries is returned.

    warning::
        The names of the dictionaries are guessed by the tricky function
        ``name`` of the module ``pythonUse``. Unfortunately, this function can
        be impaired. In that case, you can give directly the names of your
        dictionaries by using the 2-uple ``dictName``.
    """
    name_1, name_2 = dictName

    if sorted(dict_1.keys()) != sorted(dict_2.keys()):
        errorText = [
            '{0} and {1} have not all the same keys.'.format(name_1, name_2)
        ]

    else:
        for keyDict_1, valueDict_1 in dict_1.items():
            valueDict_2 = dict_2[keyDict_1]

            if valueDict_1 != valueDict_2:
                nameValue_1 = name_1 + '["{0}"]'.format(keyDict_1)
                nameValue_2 = name_2 + '["{0}"]'.format(keyDict_1)

                typeValueDict_1 = type(valueDict_1)
                typeValueDict_2 = type(valueDict_2)

                errorText = [
                    '',
                    nameValue_1,
                    '',
                    '<>',
                    '',
                    nameValue_2,
                    '',
                    '',
                ]

                if typeValueDict_1 != typeValueDict_2:
                    errorText += [
                        '---- x ---',
                        '',
                        '',
                        'Type of 1st value : {0}'.format(typeValueDict_1),
                        '',
                        'Type of 2nd value : {0}'.format(typeValueDict_2)
                    ]

                elif typeValueDict_1 in [dict, collections.OrderedDict]:
                    errorText += [
                        '=========================================',
                        'Differences of the two value dictonnaries',
                        '=========================================',
                        '',
                        diffDict(
                            valueDict_1,
                            valueDict_2,
                            (nameValue_1, nameValue_2)
                        )
                    ]

                else:
                    errorText += [
                        '---- x ---',
                        '',
                        '',
                        str(valueDict_1),
                        '',
                        '<>',
                        '',
                        str(valueDict_2)
                    ]

                return '\n'.join(errorText)


# ----------- #
# -- PRINT -- #
# ----------- #

def logPrint(text):
    """
    This function is used to display expected messages. You can redefine this
    function to change the way the error are "displayed" for example so as to
    build log files. By default, ``logPrint`` is simply equal to ``print``.
    """
    print(text)

def info(
    text,
    kind  = "Testing",
    start = "    + ",
    end   = "..."
):
    """
    This function is usefull for displaying very short infos in one line. This
    function has four arguments.

        1) ``text`` is the text to display.

        2) ``kind`` is a leading text put just before ``text``. By default,
        ``kind = "Testing"``.

        3) ``start`` and ``end`` are texts put at the start and the end ``kind
        + text``. By default ``start = "    + "``and ``end = "..."``.
    """
    logPrint(
        '{0}{1} "{2}"{3}'.format(start, kind, text, end)
    )

TEST_PROBLEM_FRAME = {
    'rule' : {
        'up'   : "*",
        'down' : "*",
        'left' : "* ---->>",
        'right': "<<---- *"
    },
    'extra': {
        'rule' : {
            'up'   : "*",
            'down' : "*",
            'left' : "*",
            'right': "*"
        }
    }
}

def problem(
    text,
    format = TEST_PROBLEM_FRAME,
    center = True
):
    """
    This function is useful for displaying problems met during one test.

    Indeed this function just displays the text ``stringUse.frame(text, format,
    center)`` where the variables have the default values ``TEST_PROBLEM_FRAME``
    and ``True`` respectively. The variable ``TEST_PROBLEM_FRAME`` has the
    following definition.

    python::
            TEST_PROBLEM_FRAME = {
                'rule' : {
                    'up'   : "*",
                    'down' : "*",
                    'left' : "* ---->>",
                    'right': "<<---- *"
                },
                'extra': {
                    'rule' : {
                        'up'   : "*",
                        'down' : "*",
                        'left' : "*",
                        'right': "*"
                    }
                }
            }
    """
    logPrint(
        stringUse.frame(
            text,
            format = format,
            center = center
        )
    )


# ---------------- #
# -- TEST SUITE -- #
# ---------------- #

def launchTestSuite(
    dir,
    depth     = 0,
    verbosity = 0,
    message   = "",
    sort      = lambda x: x
):
    """
    This function helps a lot to launch simply unit tests via the module
    ``unittest``. All you need to do is to use the following arguments.

        1) ``dir`` is the only obligatory argument. It indicates the directory
        where you have put all your files for testing.

        << Rules to follow : >> you must follow the txo simple rules.

            a) All the files where the unit tests are defined must have a name
            starting with ``test_``.

            b) All the class making the unit tests must have a name
            starting with ``Test``.

        2) ``depth`` is an optional argument with default value ``0``. This is
        to indicate the maximal depth for the research of the files to remove.
        The very special value ``(-1)`` indicates that there is no maximum.
        The default value ``0`` asks to only look for in the direct content of
        the main directory to analyse.

        3) ``verbosity`` is an optional argument with default value ``0``.
        Indeed the meaning of this value is the same at its eponym for the class
        ``unittest.TextTestRunner``. See the documentation of ``unitttest``.

        4) ``message`` is an optional argument which is an empty string by
        default. You can use ``message`` if you want to display, via the
        function ``logPrint``, at the very beginning of the test suite.

        5) ``sort`` is a function used to sort the path of the files doing the
        tests. By default, ``sort = lambda x: x``.
    """
# Let's look for the good Python files.
    pathTestFiles = [
        onePath
        for onePath in osUse.nextFile(
            main  = dir,
            ext   = {'keep': ["py"]},
            depth = depth
        )
        if osUse.name(onePath).startswith('test_')
    ]

# Let's sort the paths.
    pathTestFiles.sort(key = sort)

# Let's look for the testing classes.
    listOfClasses = []

    for onePythonFile in pathTestFiles:
        test = imp.load_source(
            osUse.name(onePythonFile),
            onePythonFile
        )

        for nameClassFound, oneClassFound in inspect.getmembers(
            test,
            inspect.isclass
        ):
            if nameClassFound.startswith('Test'):
                listOfClasses.append(oneClassFound)

# It's time to test.
    if listOfClasses:
        suiteOfTests = unittest.TestSuite()

        for oneClass in listOfClasses:
            suiteOfTests.addTest(unittest.makeSuite(oneClass))

        if message:
            logPrint(
                "\n" + stringUse.frame(
                    text   = message,
                    format = stringUse.PYTHON_FRAME
                ) + "\n"
            )

        unittest \
            .TextTestRunner(verbosity = verbosity) \
            .run(suiteOfTests)

    else:
        logPrint('No test has been found...')

