#!/usr/bin/env python3

"""
Directory : mistool
Name      : logTestUse
Version   : 2013.04
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This modules contains some utilities that can be usefull for unit tests made
with the standard module ``unittest``.

See the documentation for more details.
"""

# Source used :
#    1) http://agiletesting.blogspot.com/2005/01/python-unit-testing-part-1-unittest.html
#    2) http://stackoverflow.com/questions/67631/how-to-import-a-module-given-the-full-path
#    3) http://docs.python.org/3.2/library/unittest.html?highlight=unittest.texttestrunner#basic-example

import imp
import inspect
import unittest
import collections

from mistool import os_use, python_use, string_use


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class LogTestUseError(ValueError):
    pass

def __unexpectedError(text):
    """
This is for errors not expected during the tests.
    """
    logPrint(
        '    UNEXPECTED ERROR DETECTED DURING THE TEST :\n' \
        + '        ' \
        + '        \n'.join(text.split('\n'))
    )


# ---------------------- #
# -- LOGGING MESSAGES -- #
# ---------------------- #

# This variable is used to define some formatting rules from one path or one
# name of a function, a method or a class.

ASCII_ASSO = {
    '_'  : " ",
    '__' : " - ",
    '___': " ---> ",
    '/'  : " : "
}

def what(
    text,
    isMethod = False
):
    """
This function transforms some text like ``Abreviations__NewAdding_Good`` into
``Abreviations ---> New Adding - Good`` where the replacements are defined in
the global constant following dictionary.

python::
    ASCII_ASSO = {
        '_'  : " ",
        '__' : " - ",
        '___': " ---> ",
        '/'  : " : "
    }


There are two arguments.

    1) ``text`` is a text of something that is tested.

    2) ``isMethod`` is an optional boolean which is equal to ``False`` by
    default.

    If ``isMethod = True``, then the text must start with ``"test"``. If it is
    the case, this piece of text will be removed (the module ``unittest`` only
    use methods with name started by ``"test"``).
    """
    if isMethod:
        if not text.startswith('test'):
            raise LogTestUseError('The text does not start with "test".')

        text = text[len('test'):]

    text = string_use.camelTo(text, "title")

    text = string_use.replace(
        text        = text,
        replacement = ASCII_ASSO
    )

    return text.strip()

def diffDict(
    dict_1,
    dict_2
):
    """
This function analyses if the two dictionaries ``dict_1`` and ``dict_2`` are
equal. If it is the case, an empty string is returned. In the other case, a
message indicating briefly the differences between the two dictionaries is
returned.
    """
    if sorted(dict_1.keys()) != sorted(dict_2.keys()):
        text = 'The dictionaries have different keys.'

    else:
        text = ''

        for keyDict_1, valueDict_1 in dict_1.items():
            if valueDict_1 != dict_2[keyDict_1]:
                text = 'The values for the key << {0} >> '.format(
                    python_use.pyRepr(keyDict_1)
                ) + 'are different.'

                break

    return text


# ----------- #
# -- PRINT -- #
# ----------- #

def logPrint(text):
    """
This function is used to display expected messages. You can redefine this
function to change the way the error are "displayed" for example so as to build
log files. By default, ``logPrint`` is simply equal to ``print``.
    """
    print(text)

def info(
    text,
    kind  = "Testing",
    start = "    + ",
    end   = "..."
):
    """
This function is usefull for displaying very short infos in one line. This
function has four arguments.

    1) ``text`` is the text to display.

    2) ``kind`` is a leading text put just before ``text``. By default, ``kind =
    "Testing"``.

    3) ``start`` and ``end`` are texts put at the start and the end ``kind +
    text``. By default ``start = "    + "``and ``end = "..."``.
    """
    logPrint(
        '{0}{1} "{2}"{3}'.format(start, kind, text, end)
    )

TEST_PROBLEM_FRAME = {
    'rule' : {
        'up'   : "*",
        'down' : "*",
        'left' : "* ---->>",
        'right': "<<---- *"
    },
    'extra': {
        'rule' : {
            'up'   : "*",
            'down' : "*",
            'left' : "*",
            'right': "*"
        }
    }
}

def problem(
    text,
    format = TEST_PROBLEM_FRAME,
    center = True
):
    """
This function is useful for displaying problems met during one test.

Indeed this function just displays the text ``string_use.frame(text, format,
center)`` where the variables have the default values ``TEST_PROBLEM_FRAME`` and
``True`` respectively. The variable ``TEST_PROBLEM_FRAME`` has the following
definition.

python::
    TEST_PROBLEM_FRAME = {
        'rule' : {
            'up'   : "*",
            'down' : "*",
            'left' : "* ---->>",
            'right': "<<---- *"
        },
        'extra': {
            'rule' : {
                'up'   : "*",
                'down' : "*",
                'left' : "*",
                'right': "*"
            }
        }
    }
    """
    logPrint(
        string_use.frame(
            text,
            format = format,
            center = center
        )
    )


# ---------------- #
# -- TEST SUITE -- #
# ---------------- #

def launchTestSuite(
    dir,
    depth     = 0,
    verbosity = 0,
    message   = "",
    sort      = lambda x: x
):
    """
This function helps a lot to launch simply unit tests via the module
``unittest``. All you need to do is to use the following arguments.

    1) ``dir`` is the only obligatory argument. It indicates the directory where
    you have put all your files for testing.

    << Rules to follow : >> you must follow the txo simple rules.

        a) All the files where the unit tests are defined must have a name
        starting with ``test_``.

        b) All the class making the unit tests must have a name starting with
        ``Test``.

    2) ``depth`` is an optional argument with default value ``0``. This is to
    indicate the maximal depth for the research of the files to remove. The very
    special value ``(-1)`` indicates that there is no maximum. The default value
    ``0`` asks to only look for in the direct content of the main directory to
    analyse.

    3) ``verbosity`` is an optional argument with default value ``0``. Indeed
    the meaning of this value is the same at its eponym for the class
    ``unittest.TextTestRunner``. See the documentation of ``unitttest``.

    4) ``message`` is an optional argument which is an empty string by default.
    You can use ``message`` if you want to display, via the function
    ``logPrint``, at the very beginning of the test suite.

    5) ``sort`` is a function used to sort the path of the files doing the
    tests. By default, ``sort = lambda x: x``.
    """
# Let's look for the good Python files.
    pathTestFiles = [
        onePath
        for onePath in os_use.nextFile(
            main  = dir,
            ext   = {'keep': ["py"]},
            depth = depth
        )
        if os_use.name(onePath).startswith('test_')
    ]

# Let's sort the paths.
    pathTestFiles.sort(key = sort)

# Let's look for the testing classes.
    listOfClasses = []

    for onePythonFile in pathTestFiles:
        test = imp.load_source(
            os_use.name(onePythonFile),
            onePythonFile
        )

        for nameClassFound, oneClassFound in inspect.getmembers(
            test,
            inspect.isclass
        ):
            if nameClassFound.startswith('Test'):
                listOfClasses.append(oneClassFound)

# It's time to test.
    if listOfClasses:
        suiteOfTests = unittest.TestSuite()

        for oneClass in listOfClasses:
            suiteOfTests.addTest(unittest.makeSuite(oneClass))

        if message:
            logPrint(
                "\n" + string_use.frame(
                    text   = message,
                    format = string_use.PYTHON_FRAME
                ) + "\n"
            )

        unittest \
            .TextTestRunner(verbosity = verbosity) \
            .run(suiteOfTests)

    else:
        logPrint('No test has been found...')

