#!/usr/bin/env python3

"""
Directory : mistool
Name      : pythonUse
Version   : 2013.04
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This module contains some simple tools about the Python programming language.

See the documentation for more details.
"""

import gc
import sys
import collections

# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class PythonUseError(ValueError):
    pass


# ---------------------- #
# -- FOR DICTIONARIES -- #
# ---------------------- #

def dictSingleValues(oneDict):
    """
This function returns a single value list which contains the values stored in
one dictionary.
    """
    return list(set(oneDict.values()))


# ------------- #
# -- QUOTING -- #
# ------------- #

QUOTE_SYMBOLS = ["'", '"']

def __escape__(data):
    """
This small function escapes all the characters that must be escaped in one
python-like string.
    """
    if type(data) == str:
        return data.replace('\\', '\\\\')

    else:
        return data

def quote(
    text,
    symbol = "'"
):
    """
This function put the content of the string variable ``text`` into quotes and
escapes the eventual quote symbols in ``text``.

This function has one optional variable ``symbol`` which indicates the prefered
quoting symbol which is the single quote ``'`` by default.

This function uses the global constant ``QUOTE_SYMBOLS`` which is defined by
``QUOTE_SYMBOLS = ["'", '"']``. This indicates the list of the possible quoting
symbols.

For example, ``quote("one \"small\" example")`` is equal to the text ``'one
"small" example``, and ``quote('another \'example\' to "see" more")``` is equal
to ``'another \'example\' to "see" more'``.
    """
    if all(x in text for x in QUOTE_SYMBOLS):
        text = text.replace(symbol , '\\' + symbol)

    elif symbol in text:
        for x in QUOTE_SYMBOLS:
            if x != symbol:
                symbol = x

    return "{0}{1}{0}".format(symbol, text)


# -------------------------- #
# -- WRITING PYTHON FILES -- #
# -------------------------- #

def lambdify(text):
    """
This function simply produces lambda expressions only from their definitions 
given via one string.
    """
    return "lambda " + text

def __isGoodType(
    kind,
    objType,
    wantedType
):
    if objType != wantedType:
        messageType = {
            str : 'string',
            list: 'list',
            dict: 'dictionary'
        }

        wantedType = messageType.get(wantedType, str(wantedType))

        raise PythonUseError(
            'With ''kind = "{0}"'', the object to repr '
            'must be one {1}.'.format(kind, wantedType)
        )

def pyRepr(
    obj,
    kind   = None,
    depth  = -1,
    tab    = " "*4,
    indent = 0
):
    """
This function returns one string representation, usable in ¨python codes, of one
¨python object.

There are five variables. See the documentation for more explanations.

    1) ``obj`` is the python object to print in one ¨python code.

    2) ``kind`` is an optional argument to indicate the kind of object. The
    default value is ``None`` which asks to use the type of the object. The
    supported values are the following ones.

        * ``"bool"``, ``"dict"``, ``"float"``, ``"int"``, ``"list"`` and
        ``"str"`` indeed corresponds to ¨python types supported.

        * ``"regex"`` is useful for pattern definied using
        ``re.compile("...")``.

        * ``"repr"`` simply asks to use the standard ¨python representation.

    3) ``depth`` is an optional argument to indicate how deep to go in the
    printings of dictionaries and lists. The default value is ``(-1)`` which
    indicates to go the deeper is possible.

    4) ``tab`` is an optional argument which is the tabulation used for the
    representation of dictionaries and lists. The default value is ``" "*4``.

    5) ``indent`` is an optional argument that indicates the current level
    of indentation. The default value is ``0``.
    """
# We have to guess the type of the object.
    objType = type(obj)

    if kind == None:
        if objType in [
            bool,
            float, int,
            list, dict,
            str,
        ]:
            kind = str(objType)
            kind = kind[kind.find("'")+1:]
            kind = kind[:kind.find("'")]

# Very ugly !!!
        elif str(objType) == "<class '_sre.SRE_Pattern'>":
            kind = "regex"

        else:
            raise PythonUseError(
                'Unsupported type "{0}".'.format(objType)
            )

# We do the job.
    if kind == "bool":
        answer = repr(bool(obj))

    elif kind == "str":
        answer = repr(obj)

    elif kind in ["float", "int", "repr"]:
        answer = repr(obj)

    elif kind == "regex":
        answer = '\n'.join([
            're.compile(',
            tab*(indent+1) + '"{0}"'.format(obj.pattern),
            tab*indent + ')'
        ])

    elif kind == "list":
        __isGoodType(
            kind       = kind,
            objType    = objType,
            wantedType = list
        )

        answer = __reprList(
            oneList = obj,
            indent  = indent,
            tab     = tab,
            depth   = depth
        )

    elif kind == "dict":
        __isGoodType(
            kind       = kind,
            objType    = objType,
            wantedType = dict
        )

        answer = __reprDict(
            oneDict = obj,
            indent  = indent,
            tab     = tab,
            depth   = depth
        )


    else:
        raise PythonUseError('Unknown kind "{0}".'.format(kind))

    return tab*indent + answer

def __reprList(
    oneList,
    indent,
    tab,
    depth
):
    """
This function is for writing lists.
    """
    if depth == 0:
        return repr(oneList)

    text = []

    for elt in oneList:
        elt = pyRepr(
            obj    = elt,
            indent = indent + 1,
            tab    = tab,
            depth  = depth - 1
        )

        text.append(elt + ",")

    text[-1] = text[-1][:-1]

    text.append(tab*indent + ']')
    text = ['['] + text

    return '\n'.join(text)

def __reprDict(
    oneDict,
    indent,
    tab,
    depth
):
    """
This function is for writing dictionaries.
    """
    try:
        sortedKeys = sorted(oneDict.keys())
    except:
        sortedKeys = oneDict.keys()

    if depth == 0:
        text = ['{']

        for key in sortedKeys:
            value = repr(oneDict[key])
            key   = repr(key)

            text.append("{0}: {1}, ".format(key, value))

        text[-1] = text[-1][:-2]

        text.append('}')

        return ''.join(text)

    else:
        text = []

        for key in sortedKeys:
            value = pyRepr(
                obj    = oneDict[key],
                indent = indent + 1,
                tab    = tab,
                depth  = depth - 1
            ).lstrip()

            key = tab*(indent+1) +  repr(key)

            text.append("{0}: {1},".format(key, value))

        text[-1] = text[-1][:-1]

        text.append(tab*indent + '}')
        text = ['{'] + text

        return '\n'.join(text)
