#!/usr/bin/env python3

"""
Directory : mistool
Name      : stringUse
Version   : 2013.03
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This script contains some usefull functions for manipulating strings.

See the documentation for more details.
"""


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class StringUseError(ValueError):
    pass


# ------------- #
# -- REPLACE -- #
# ------------- #

def replace(
    text,
    replacement
):
    """
This function replaces in the argument ``text`` using the associations defined
in the dictionary ``replacement``.
    """
    for old in sorted(
        replacement.keys(),
        key = lambda t: -len(t)
    ):
        text = text.replace(old, replacement[old])

    return text


# -------------------- #
# -- JOIN AND SPLIT -- #
# -------------------- #

AND_TEXT = "and"

def joinAnd(
    listText,
    andText = None
):
    """
This function joins texts given in the argument ``listText`` using coma as
separator excepted for the list piece of text which wil be preceded by default
by "and".

There is two ways to change "and".

    1) Local change : the function has on optional argument ``andText`` which is
    the string value  used before the last piece of text.

    2) Global change : ``AND_TEXT`` is the global constant to use so as to
    change the default text used before the last piece of text.
    """
    if len(listText) == 1:
        return listText[0]

    if andText == None:
        andText = AND_TEXT

    return ", ".join(listText[:-1]) + " " + andText + " " + listText[-1]

def beforeAfter(
    text,
    start,
    end,
    keep = False
):
    """
This function needs the three arguments ``text``, ``start`` and ``end`` which
are all strings. There is also one optional argument ``keep`` which has the
default value ``False``.

The function will look for the piece of text between the first texts ``start``
and ``end`` found in ``text``. Then by default the function returns the text
before the text ``start`` and the one after the text ``end`` in a couple.

If ``keep = True`` then the text ``start`` will be added at the end of the first
text returned, and the second text returned will begin with the text ``end``.

You can also use ``keep = (True, False)`` or ``keep = (True, False)`` so as to
use just one half of the features which have just been explained. Indeed ``keep
= True`` and ``keep = False`` are shortcuts for ``keep = (True, True)`` and
``keep = (False, False)``.
    """
    if start == "" and end == "":
        raise StringUseError(
            'The variables ``start`` and ``end`` can not be both empty.'
        )

    if type(keep) == bool:
        keepStart = keep
        keepEnd   = keep

    else:
        keepStart, keepEnd = keep

    if start == "":
        s = 0

    else:
        s = text.find(start)

        if s == -1:
            raise StringUseError(
                'The starting text << {0} >> has not been found.'.format(start)
            )

        s += len(start)

    if end == "":
        e = s

    else:
        e = text.find(end, s)

        if e == -1:
            message = 'The ending text  << {0} >> has not been found after ' \
                    + 'the first text << {1} >>.'

            raise StringUseError(
                message.format(start, end)
            )

        if start == "":
            s = e


    if not keepStart and start:
        s -= len(start)

    if not keepEnd and end:
        e += len(end)


    return (text[:s], text[e:])


# ---------------------------------- #
# -- PLAYING WITH CASE OF LETTERS -- #
# ---------------------------------- #

def case(
    text,
    kind
):
    """
This function produces different case variants of the text contained in the
string variable ``text``.

The possible case variants, given by the value of the string variable ``kind``,
are the following ones.

    1) ``"lower"``, ``"upper"``, ``"sentence"`` and ``"title"`` are for lower
    case, upper case, sentence and title case versions of one text. For example,
    ``case("OnE eXamPLe", "lower")``, ``case("OnE eXamPLe", "upper")`` and
    ``case("OnE eXamPLe", "sentence")`` and ``case("One example", "title")`` are
    respectively equal to "one example", "ONE EXAMPLE", "OnE eXamPLe" and "One
    Example".

    2) ``"firstLast"`` can look weird because ``case("OnE eXamPLe",
    "firstLast")`` is equal to "One examplE". Indeed, this is used in the
    langage of pyBaNaMa which is another project of the author of the package
    ¨mistool.

    3) ``"all"`` gives all the possible case variants. For example, ``case("OnE
    eXamPLe", "all")`` is equal to the following dictionary.

    python::
        {
            'lower'    : 'one example',
            'upper'    : 'ONE EXAMPLE',
            'sentence' : 'One example',
            'title'    : 'One Example',
            'firstLast': 'One examplE'
        }

    4) If for example, you only need to have the lower case and the title case
    versions of one text, then you just have to use ``case("OnE eXamPLe", "lower
    + title")`` which is equal to the following dictionary.

    python::
        {
            'lower': 'ONE EXAMPLE',
            'title': 'One Example'
        }
    """
    kind = kind.strip()

    if kind == 'all':
        answer = {}
        for kind in ['lower', 'upper', 'sentence', 'title', 'firstLast']:
            answer[kind] = case(text, kind)
        return answer

    elif '+' in kind:
        answer = {}
        for kind in kind.split('+'):
            kind = kind.strip()

            if not kind in answer:
                newCaseVariant = case(text, kind)

                if type(newCaseVariant) == dict:
                    for oneKind, oneCaseVariant in newCaseVariant.items():
                        answer[oneKind] = oneCaseVariant

                else:
                    answer[kind] = newCaseVariant

        return answer

    elif kind == 'lower':
        return text.lower()

    elif kind == 'upper':
        return text.upper()

    elif kind == 'sentence':
        return text[0].upper() + text[1:].lower()

    elif kind == 'title':
        return text.title()

    elif kind == 'firstLast':
        return text[0].upper() + text[1:-1].lower() + text[-1].upper()

    else:
        raise StringUseError(
            '"{0}" is not an existing kind of case variant.'.format(kind)
        )

def camelTo(
    text,
    kind
):
    """
This function transforms one text using camel syntax to one of the case variants
proposed by the function ``case``. Indeed, each time one upper letter is met,
one underscore followed by one space is added before it, then the function
``case`` is called, and finally extra spaces are removed. For example,
``camelTo("oneExample", "title")`` is equal to ``"One_Example"``.
    """
    newText = ''

    for oneChar in text:
        if oneChar.isupper():
            newText += '_ '

        newText += oneChar

    newText = case(newText, kind)

    if type(newText) == dict:
        for kind, text in newText.items():
            newText[kind] = text.replace('_ ', '_')

    else:
        newText = newText.replace('_ ', '_')

    return newText


# ------------------- #
# -- DECORATE TEXT -- #
# ------------------- #

PYTHON_FRAME = {
    'rule' : {
        'up'   : "#",
        'down' : "#",
        'left' : "#",
        'right': "#"
    }
}

PYTHON_PRETTY_FRAME = {
    'rule' : {
        'up'   : "-",
        'down' : "-",
        'left' : "--",
        'right': "--"
    },
    'extra': {
        'rule' : {
            'left' : "#",
            'right': "#"
        }
    }
}

C_FRAME = {
    'rule' : {
        'up'   : "*",
        'down' : "*",
        'left' : "*",
        'right': "*"
    },
    'corner': {
        'leftup'   : "/",
        'rightdown': "/"
    }
}

PYBA_NB_TITLE_FRAME = {
    'rule' : {
        'up'   : "=",
        'down' : "=",
    }
}

PYBA_NO_NB_TITLE_FRAME = {
    'rule' : {
        'up'   : "-",
        'down' : "-"
    }
}

PYBA_UNITTEST_FRAME = {
    'rule' : {
        'up'   : "*",
        'down' : "*",
        'left' : "*",
        'right': "*"
    }
}

def __drawHorizontalRule(
    charRule,
    leftCorner,
    rightCorner,
    ruleLenght,
    nbSpaceToAdd
):
    """
This function is used to draw the first and last horizontal rules.
    """
    if charRule:
        return [
            ' '*(nbSpaceToAdd - len(leftCorner))
            + leftCorner
            + charRule*ruleLenght
            + rightCorner
        ]

    elif leftCorner or rightCorner:
        return [
            leftCorner
            + ' '*ruleLenght
            + rightCorner
        ]

    else:
        return []

def frame(
    text,
    format,
    center = True
):
    """
This function makes it possible to put one text into one frame materialized by
ASCII characters. This can be usefull for console outputs or for pretty comments
for listings like the following python comment.

python::
    #############
    # one       #
    # comment   #
    # easily    #
    # formatted #
    #############

This text has been produced using the following lines.

python::
    import Mistool

    textToDecorate = '''one
    comment
    easily
    formatted'''

    print(
        Mistool.stringUse.frame(
            text   = textToDecorate,
            format = Mistool.stringUse.PYTHON_FRAME,
            center = False
        )
    )

By default, ``center`` is equal ``True`` so as to merely center the content of
the frame.

Here we use one predefined frame ``Mistool.stringUse.PYTHON_FRAME`` which is the
following dictionary.

python::
    PYTHON_FRAME = {
        'rule' : {
            'up'   : "#",
            'down' : "#",
            'left' : "#",
            'right': "#"
        }
    }


``Mistool.stringUse.C_FRAME`` is another predefined frame which is the following
dictionary.

python::
    C_FRAME = {
        'rule' : {
            'up'   : "*",
            'down' : "*",
            'left' : "*",
            'right': "*"
        },
        'corner': {
            'leftup'   : "/",
            'rightdown': "/"
        }
    }

This will give the following format of frame.

code_c::
    /***************
     * one         *
     * comment     *
     * with C-like *
     * style       *
     ***************/


The last example of ``Mistool.stringUse.PYTHON_PRETTY_FRAME`` shows the
possibility to us an extra frame so as to obtain a little more elaborated frames
in only one call.

python::
    PYTHON_PRETTY_FRAME = {
        'rule' : {
            'up'   : "-",
            'down' : "-",
            'left' : "--",
            'right': "--"
        },
        'extra': {
            'rule' : {
                'left' : "#,
                'right': "#"
            }
        }
    }

This will give the following format of frame.

python::
    # ------------- #
    # -- one     -- #
    # -- pretty  -- #
    # -- comment -- #
    # ------------- #


Here is the general structure of the dictionnaries used to build frames.

python::
    {
        'rule' : {
            'up'   : "Only one character",
            'down' : "Only one character",
            'left' : "The characters wanted",
            'right': "The characters wanted"
        },
        'corner': {
            'leftup'   : "The characters wanted",
            'leftdown' : "The characters wanted",
            'rightup'  : "The characters wanted",
            'rightdown': "The characters wanted"
        },
        'extra': {
            'rule' : {
                'up'   : "Only one character",
                'down' : "Only one character",
                'left' : "The characters wanted",
                'right': "The characters wanted"
            },
            'corner': {
                'leftup'   : "The characters wanted",
                'leftdown' : "The characters wanted",
                'rightup'  : "The characters wanted",
                'rightdown': "The characters wanted"
            },
        }
    }
    """
# Default values must be choosen if nothing is given.
    for key in ['rule', 'corner']:
        if key not in format:
            format[key] = {}

    for styleKey in ['up', 'down', 'left', 'right']:
        if styleKey not in format['rule']:
            format['rule'][styleKey] = ""

    for styleKey in ['leftup', 'leftdown', 'rightup', 'rightdown']:
        if styleKey not in format['corner']:
            format['corner'][styleKey] = ""

# Horizontal rules can only use one single character.
    for onePosition in ['up', 'down']:
        if len(format['rule'][onePosition]) > 1:
            raise StringUseError(
                "You can only use nothing or one single charcater for rules." \
                "\nSee ''{0}'' for the {1} rule.".format(
                    format['rule'][onePosition],
                    onePosition
                )
            )

# Infos about the lines of the text.
    lines = [oneLine.rstrip() for oneLine in text.splitlines()]
    nbMaxChar = max([len(oneLine) for oneLine in lines])

# Space to add before vertical rules.
    nbSpaceToAdd = max(
        len(format['corner']['leftup']),
        len(format['corner']['leftdown'])
    )

    spaceToAdd = ' '*nbSpaceToAdd

# Text decoration for vertical rules
    if format['rule']['left']:
        leftRuleText = format['rule']['left'] + ' '
    else:
        leftRuleText = ''

    if format['rule']['right']:
        rightRuleText = ' ' + format['rule']['right']
    else:
        rightRuleText = ''

# Length of the rule without the corners
    ruleLenght = nbMaxChar + len(leftRuleText) + len(rightRuleText)

# First line of the frame
    answer = __drawHorizontalRule(
        charRule     = format['rule']['up'],
        leftCorner   = format['corner']['leftup'],
        rightCorner  = format['corner']['rightup'],
        ruleLenght   = ruleLenght,
        nbSpaceToAdd = nbSpaceToAdd
    )

# Management of the lines of the text
    for oneLine in lines:
        nbSpacesMissing = nbMaxChar - len(oneLine)

# Space before and after one line of text.
        if center:
            if nbSpacesMissing % 2 == 1:
                spaceAfter = ' '
            else:
                spaceAfter = ''

            nbSpacesMissing = nbSpacesMissing // 2

            spaceBefore = ' '*nbSpacesMissing
            spaceAfter += spaceBefore

        else:
            spaceBefore = ''
            spaceAfter = ' '*nbSpacesMissing

        answer.append(
            spaceToAdd
            +
            '{0}{1}{2}{3}{4}'.format(
                leftRuleText,
                spaceBefore,
                oneLine,
                spaceAfter,
                rightRuleText
            )
        )

# Last line of the frame
    answer += __drawHorizontalRule(
        charRule     = format['rule']['down'],
        leftCorner   = format['corner']['leftdown'],
        rightCorner  = format['corner']['rightdown'],
        ruleLenght   = ruleLenght,
        nbSpaceToAdd = nbSpaceToAdd
    )

    answer = '\n'.join(answer)

# Does we have an extra frame ?
    if 'extra' in format:
        try:
            answer = frame(
                text   = answer,
                format = format['extra'],
                center = center
            )

        except StringUseError as e:
            raise StringUseError(
                str(e)[:-1] + " in the definition of the extra frame."
            )

# All the job has been done.
    return answer

class step():
    """
This class displays texts for step by step actions. The numbering of the steps is automatically updated and displayed.

There are two optional variables.

    1) ``nb`` is the number of the current step. When the class is instanciated,
    the default value is ``1``.

    2) ``deco`` indicates how to display the numbers. When the class is
    instanciated, the default value is ``""1)""`` where ``1`` symbolises the
    numbers.
    """
    def __init__(
        self,
        nb   = 1,
        deco = "1)"
    ):
        self.nb   = nb
        self.deco = deco.replace('1', '{0}')

    def print(
        self,
        deco,
        text
    ):
        """
This method prints ``deco`` the text of the actual numbering, and the text of
the actual step.

You can redefine this method for more fine customizations.
        """
        print(
            deco,
            text,
            sep = " "
        )

    def display(
        self,
        text
    ):
        """
This method simply calls the method python:: ``self.print`` so as to print the
informations contained in the variable ``text``, and then ``self.nb`` is
augmented by one unit.
        """
        self.print(
            self.deco.format(self.nb),
            text
        )

        self.nb += 1


# ------------------- #
# -- SHORTEN LINES -- #
# ------------------- #

def __addNewWord(
    word,
    line,
    newText,
    width,
    indentation
):
    """
This function is used by the function ``cut``. It manages the lines and it
returns one uplet ``(word, line, newText)`` which contains the new values of
``word``, ``line`` and ``newText``.
    """
    lenWord        = len(word)
    lenIndentation = len(indentation)

    if len(line) + lenWord > width:
        newText.append(line)

        if lenWord + lenIndentation > width:
            newText.append(indentation + word)
            line = ''

        else:
            line = indentation + word

    elif word:
        if line:
            line += ' '

        else:
            line = indentation

        line += word

    word = ''

    return (word, line, newText)

def __indentation(text):
    """
This function is used by the function ``cut``. It returns the leading spaces and
tabulations of one text.
    """
    indentation = ''

    if text.strip():
        i = 0

        while text[i] in "\t ":
            indentation += text[i]
            i+=1

    return indentation

def cut(
    text,
    width = 80
):
    """
This function tries to cut lines of one text so as to have less than `81`
characters. Multiple following spaces are rendered as one single space, and the
final text is tripped so as to start and end with none empty lines. Furthermore,
when lines are cut, their indentations are respected.

There are two arguments.

    1) ``text`` is simply the text to cut.

    2) ``width`` is an integer that gives the maximum of characters wanted to be
    in one line. Its default value is `80`.
    """
    if type(width) != int or width <= 0:
        raise StringUseError(
            "``width``must be a positive integer."
        )

    word        = ''
    line        = ''
    indentation = __indentation(text)
    newText     = []

    for i in range(len(text)):
        oneChar = text[i]

# One space
        if oneChar == ' ':
            word, line, newText = __addNewWord(
                word, line, newText,
                width, indentation
            )

# One back return
        elif oneChar == '\n':
            word, line, newText = __addNewWord(
                word, line, newText,
                width, indentation
            )

            indentation = __indentation(text[i+1:])

            if line:
                newText.append(line)
                line = ''

            else:
                newText.append('')

# The word becomes bigger...
        else:
            word += oneChar

# Let's finish the job !
    word, line, newText = __addNewWord(
        word, line, newText,
        width, indentation
    )

    if word:
        if line:
            line += ' '

        line += word

    if line:
        newText.append(indentation + line)

    return '\n'.join(newText).strip()
