#!/usr/bin/env python3

"""
Directory : mistool
Name      : stringUse
Version   : 2013.05
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This script contains some useful functions for manipulating strings.
"""

from mistool import regex_use


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class StringUseError(ValueError):
    """
-----------------
Small description
-----------------

Base class for errors in the ``string_use`` module of the package ``mistool``.
    """
    pass


# ------------- #
# -- REPLACE -- #
# ------------- #

def replace(
    text,
    replacement
):
    """
-----------------
Small description
-----------------

This function does replacements in the argument ``text`` using the associations
defined in the dictionary ``replacement``.


warning::
    The function does the replacements sequentially from the longer word to the
    shorter one.


Here is a small example.

python::
    from mistool import string_use

    littleExample = string_use.replace(
            text        = "One, two, three,..."
            replacement = {
                'One'  : "1",
                'Two'  : "2",
                'Three': "3"
            }
        )
    )


In that code, ``littleExample`` is equal to ``"1, 2, 3,..."``.


info::
    This function has not been build for texts to be replaced that contains some
    other texts to also replace. If you need this kind of feature, take a look
    at the class ``MultiReplace``.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``text`` is a string argument corresponding to the text where the
    replacements must be done.

    2) ``replacement`` is a dictionary where each couple ``(key, value)`` is of
    the kind ``(text to find, replacement)``.
    """
    longToShortKeys = sorted(
        replacement.keys(),
        key = lambda t: -len(t)
    )

    for old in longToShortKeys:
        text = text.replace(old, replacement[old])

    return text

PATTERN_GROUP_WORD = regex_use.PATTERN_GROUP_WORD

class MultiReplace:
    """
-----------------
Small description
-----------------

The purpose of this class is to replace texts that can contain some other texts
to also be replaced. Here is an example of use.

python::
    form mistool import string_use

    myReplace = string_use.MultiReplace(
        replacement = {
            'W1' : "word #1",
            'W2' : "word #2",
            'W12': "W1 and W2"
        },
        pattern = string_use.PATTERN_GROUP_WORD["var"]
    )

    print(myReplace.replace("W1 and W2 = W12"))


Launched in a terminal, the preceding code will produce the following output.

terminal::
    word #1 and word #2 = word #1 and word #2


The only technical thing is the use of ``string_use.PATTERN_GROUP_WORD["var"]``
which is a regex grouping pattern. You can use directly the following patterns
or use your own grouping pattern.

    1) ``string_use.PATTERN_GROUP_WORD["en"]`` is for words only made of ¨ascii
    letters.

    2) ``string_use.PATTERN_GROUP_WORD["fr"]`` is for words only made of ¨ascii
    letters and the special letters "â", "à", "é", "è", "ê", "ë", "î", "ï", "ô",
    "ù", "ü", and "ç".

    2) ``string_use.PATTERN_GROUP_WORD["var"]`` is for words only starting with
    one ¨ascii letter followed eventually by other ¨ascii letters, digits and
    underscores.


warning::
    Before doing the replacements in a text, the class first build the
    dictionary ``self.directReplacement`` for which the replacements has been changed so to not contain text to be replaced. With the preceeding code,
    this dictionary is equal to the following one.

    python::
        {
            'W1' : "word #1",
            'W2' : "word #2",
            'W12': "word #1 and word #2"
        }


info::
    In the following dictionnary defining replacements, there are cyclic
    definitions. In that case, the class will raise an error.

    python::
        replacement = {
            'WRONG_1': "one small text and  WRONG_2",
            'WRONG_2': "one small text, and then WRONG_3",
            'WRONG_3': "with WRONG_1, there is one problem here"
        }


-------------
The arguments
-------------

The instanciation of this class uses the following variables.

    1) ``replacement`` is a dictionary where each couple ``(key, value)`` is of
    the kind ``(text to find, replacement)``. Here the replacements can contain
    also text to be found.

    2) ``pattern`` is a regex grouping pattern indicating the kind of words to
    be replaced. By default, ``pattern = PATTERN_GROUP_WORD["en"]`` where
    ``PATTERN_GROUP_WORD`` is a renaming of ``regex_use.PATTERN_GROUP_WORD``
    (see the module ``regex_use`` for more informations).
    """

    def __init__(
        self,
        replacement,
        pattern = PATTERN_GROUP_WORD["en"]
    ):
        self.pattern           = pattern
        self.replacement       = replacement
        self.directReplacement = None

        self.__crossReplace = None
        self.__oldWords     = None

        self.__update()

    def __update(self):
        """
-----------------
Small description
-----------------

This method simply launches methods so as to verify that there is no cyclic
replacements, and then to build the direct replacements dictionary
``self.directReplacement``.
        """
        self.__crossReplace = {}

        self.__noCycle()

        self.__replaceInReplacement()

    def __noCycle(self):
        """
-----------------
Small description
-----------------

This method verifies that there is no cyclic replacements.
        """
# Building the crossing replacements.
        self.__oldWords = list(self.replacement.keys())

        for old, new in self.replacement.items():
            self.__crossReplace[old] = [
                x for x in self.pattern.findall(new)
                if x in self.__oldWords
            ]

        self.__viciousCircles()

    def __viciousCircles(
        self,
        visitedWord = [],
        nextWord    = None
    ):
        """
-----------------
Small description
-----------------

This method does all the major job for ``self.__noCycle``.
        """
        if nextWord == None:
            nextWord = self.__oldWords

        for old in nextWord:
            oldInNew = self.__crossReplace[old]

            if old in oldInNew:
                raise StringUseError(
                    "<< {0} >> is used in its associated replacement." \
                        .format(old)
                )

            elif old in visitedWord:
                pos = visitedWord.index(old)

                visitedWord = visitedWord[pos:]
                visitedWord.append(old)
                visitedWord = [""] + visitedWord

                raise StringUseError(
                    "The following viscious circle has been found." \
                    + "\n\t + ".join(visitedWord)
                )

            else:
                for new in oldInNew:
                    self.__viciousCircles(
                        visitedWord = visitedWord + [old],
                        nextWord    = self.__crossReplace[old]
                    )

    def __replaceInReplacement(self):
        """
-----------------
Small description
-----------------

This method builds ``self.directReplacement`` the direct replacements
dictionary.
        """
        self.directReplacement = {}

        for old, new in self.replacement.items():
            self.directReplacement[old] = self.__replace(new)

    def __replace(
        self,
        text
    ):
        """
-----------------
Small description
-----------------

This method does all the major job for ``self.__replaceInReplacement``.
        """
        if not text:
            return text

        newText = ""

        while True:
# Source :
#    http://www.developpez.net/forums/d958712/autres-langages/python-zope/general-python/amelioration-split-evolue/#post5383767
            newText = self.pattern.sub(self.__apply, text)

            if newText == text:
                return text

            else:
                text = newText

    def __apply(
        self,
        match
    ):
        """
-----------------
Small description
-----------------

This method is used to does the replacements corresponding the matching groups.
        """
        return self.replacement.get(match.group(1), match.group(0))

    def replace(
        self,
        text
    ):
        """
This method does the replacements in one text (indeed it simply calls the
function ``replace`` with the attribut ``self.directReplacement``).
        """
        return replace(
            text        = text,
            replacement = self.directReplacement
        )


# ----------------- #
# -- LOOKING FOR -- #
# ----------------- #

# Source : http://www.developpez.net/forums/d921494/autres-langages/python-zope/general-python/naturel-chaine-stockage

class AutoComplete:
    """
-----------------
Small description
-----------------

The aim of this class is to ease auto completions. Here is an example.

python::
    from mistool import string_use

    myAC = string_use.AutoComplete(
        listWord = [
            "article", "artist", "art",
            "when", "who", "whendy",
            "bar", "barbie", "barber", "bar"
        ]
    )

    print(myAC.matching("art"))

    print('---')

    print(myAC.matching(""))


Launched in a terminal, the preceding code will produce something similar to the
following output.

terminal::
    ['article', 'artist']
    ---
    [
        'art', 'article', 'artiste',
        'bar', 'barbie', 'barber',
        'when', 'whendy', 'who'
    ]


The method ``matching`` simply gives all the words starting with the prefix
given. If the prefix is empty, the matching words are all the words defining the
auto completion.

This search indeed uses a special "magical" dictionary which is stored in the
attribut ``dict``. Here ``myAC.dict`` is equal to the dictionary above where the
lists of integers correspond to the good indexes in the list of words.

python::
    {
        'words': [
    # A
            'art', 'article', 'artist',
    # B
            'bar', 'barber', 'barbie',
    # W
            'when', 'whendy', 'who'
        ],
        'completions': {
    # A
            'a'     : [0, 3],
            'ar'    : [0, 3],
            'art'   : [1, 3],
            'arti'  : [1, 3],
            'artic' : [1, 2],
            'artis' : [2, 3],
            'articl': [1, 2],
    # B
            'b'    : [3, 6],
            'ba'   : [3, 6],
            'bar'  : [4, 6],
            'barb' : [4, 6],
            'barbe': [4, 5],
            'barbi': [5, 6],
    # W
            'w'    : [6, 9],
            'wh'   : [6, 9],
            'whe'  : [6, 8],
            'when' : [7, 8],
            'whend': [7, 8]
        }
    }


You can directly give this dictionary like in the following fictive example.
This can be very useful when you always use the same list of words : just ask
one time to the class to build the "magical" dictionary, by giving the fixed
list of words just one time, and then store this dictionary to reuse it later
(you can use the function ``pyRepr``  of the module ``python_use`` to hard store
the dictionary).

python::
    myAC = string_use.AutoComplete(
        dict = myMagicDict
    )


There is two other useful methods (see their docstrings for more informations).

    1) ``build`` simply builds the "magical" dictionary. This method can be used
    for local updating of the list of words used for the auto completion.

    2) ``missing`` simply gives the letters remaining after one prefix in a
    word.


-------------
The arguments
-------------

The instanciation of this class uses the following variables.

    1) ``listWord`` is the list of words to use for the auto completions.

    2) ``depth`` is the minimal size of the prefix used to look for the auto
    completions. By default, ``depth = 0`` which indicates to start the auto
    completion with the first letter.

    infoo:
        `3` seems to be a good value of ``depth`` for ¨gui application.

    3) ``dictAsso`` is a "magical" dictionary that eases the auto completion.
    This dictionary is build by the method ``build`` from the list of words tio
    be used for the auto completions.
    """

    def __init__(
        self,
        listWord = None,
        dict     = None,
        depth    = 0
    ):
        if listWord == None and dictAsso == None:
            raise StringUseError(
                "You must give either the value of ``listWord`` or ``dict``."
            )

        self.listWord = listWord
        self.dict     = dict
        self.depth    = depth

# We have to build the magical dictionary.
        if self.dict == None:
            self.build()

    def build(self):
        """
-----------------
Small description
-----------------

This method builds the "magical" dictionary that will ease the auto completions.
Indeed, if you create a class ``AutoComplete`` without giving a "magical"
dictionary ``dict``, the method ``build`` is automatically called.


info::
    The idea of the magical dictionary comes from cf::``this discusion ;
    http://www.developpez.net/forums/d921494/autres-langages/python-zope/general-python/naturel-chaine-stockage``
        """
# Sorted list of single words.
        shortSortedList = list(set(self.listWord))
        shortSortedList = sorted(shortSortedList)

# Maximum depth.
        depth = self.depth

        if depth == 0:
            for oneWord in shortSortedList:
                depth = max(depth, len(oneWord) - 1)

# Let's build the magical dictionary.
        self.dict = {
            'words'      : shortSortedList,
            'completions': {}
        }

        for nbWord, oneWord in enumerate(shortSortedList):
            maxSize = min(depth, len(oneWord))

            for i in range(maxSize):
                prefix = oneWord[:i+1]

                if prefix != shortSortedList[nbWord]:
                    if prefix in self.dict['completions']:
                        self.dict['completions'][prefix][1] = nbWord + 1

                    else:
                        self.dict['completions'][prefix] = [nbWord, nbWord + 1]

    def matching(
        self,
        prefix
    ):
        """
-----------------
Small description
-----------------

This method looks for words given in the list ``listWord`` that start with the
string variable ``prefix``.
        """
        if prefix.strip() == '':
            return self.dict['words']

        if prefix in self.dict['completions']:
            first, last = self.dict['completions'][prefix]

            return self.dict['words'][first: last]

    def missing(
        self,
        prefix,
        word
    ):
        """
-----------------
Small description
-----------------

Giving a word ``"prefixExample"`` and one prefix ``"pre"``, this method will
return simply ``"fixExample"``.


-------------
The arguments
-------------

This method uses the following variables.

    1) ``prefix`` is a string corresponding to the prefix expected.

    2) ``word`` is a string where the prefix ``prefix`` must be removed.
        """
        if not word.startswith(prefix):
            raise StringUseError(
                "The word << {0} >> does not star with the prefix << {1} >>."\
                    .format(word, prefix)
            )

        return word[len(prefix):]


# -------------------- #
# -- JOIN AND SPLIT -- #
# -------------------- #

AND_TEXT = "and"

def joinAnd(
    listText,
    andText = None
):
    """
-----------------
Small description
-----------------

This function joins texts given in the argument ``listText`` using coma as
separator excepted for the list piece of text which wil be preceded by default
by "and".

Here is a small example.

python::
    from mistool import string_use

    littleExample = string_use.joinAnd(["one", "two", "three"])


In that code, ``littleExample`` is equal to ``one, two and three"``.


You can change the text "and". There is two ways to do that.

    1) **Local change :** the function has on optional argument ``andText``
    which is the string value used before the last piece of text.

    2) **Global change :** ``AND_TEXT`` is the global constant to use so as to
    change the default text used before the last piece of text if the optional
    argument ``andText`` is not used when calling the function.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``listText`` is a list of texts.

    2) ``andText`` is the text used betwen the two last texts. By default,
    ``andText = None`` which indicates to use the global constant ``AND_TEXT``
    which is equal to ``"and"`` by default.
    """
    if len(listText) == 1:
        return listText[0]

    if andText == None:
        andText = AND_TEXT

    return ", ".join(listText[:-1]) + " " + andText + " " + listText[-1]

def split(
    text,
    sep,
    escape = None,
    strip  = False
):
    """
-----------------
Small description
-----------------

This function allows to split a text using a list of separators, and not only a
single separator. Here is an example.

python::
    from mistool import string_use

    splitText = string_use.split(
        text  = "p_1 ; p_2 ; p_3 | r_1 ; r_2 | s",
        sep   = ["|", ";"],
        strip = True
    )

In this code, the variable ``splitText`` is equal to the following list :
``['p_1', 'p_2', 'p_3', 'r_1', 'r_2', 's']``.


You can escape separators like in the following example that also uses the
possibility to give a single string instead of a one value list.

python::
    from mistool import string_use

    splitText = string_use.split(
        text   = "p_1 \; p_2 ; p_3",
        sep    = ";",
        escape = "\\",
        strip  = True
    )

In this code, the variable ``splitText`` is equal to the following list :
``['p_1 \; p_2', 'p_3']``.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``text`` is simply the text to split.

    2) ``sep`` is either a string for a single separator, or a list of
    separators.

    3) ``escape`` is a string used to escape the separators. By default,
    ``escape = None`` which indicates that there is no escaping feature.

    4) ``strip`` is a boolean variable to strip, or not, each pieces of text
    found. By default, ``strip = False``.
"""
# No separator
    if not sep:
        if strip:
            text = text.strip()

        return [text]

# At least one separator
    if isinstance(sep, str):
        sep = [sep]

    if not isinstance(sep, list) \
    and not isinstance(sep, tuple):
        raise StringUseError(
            "The variable << sep >> must be a list or tuple of strings."
            "\n\t{0}".format(sep)
        )

    for oneSep in sep:
        if not isinstance(oneSep, str):
            raise StringUseError(
                "The variable << sep >> must be list or tuple of strings."
                "\n\t{0}".format(sep)
            )

# << Warning ! >> We must sort the opening symbolic tags from the longer one
# to the shorter.
    sorted(sep, key = lambda x: -len(x))

    answer = []
    iMax   = len(text)
    i      = 0
    iLast  = i

    while(i < iMax):
# The bigger is the winner !
        for oneSep in sep:
            if text[i:].startswith(oneSep):
                lastPiece = text[iLast: i]

                if not(escape and lastPiece.endswith(escape)):
                    i    += len(oneSep)
                    iLast = i

                    answer.append(lastPiece)

                    break

        i += 1

    remainingText = text[iLast:]

    if remainingText:
        answer.append(remainingText)

    if strip:
        answer = [x.strip() for x in answer]

# The job has been done !
    return answer

class _SplitKind:
    """
-----------------
Small description
-----------------

This class is simply an object like class used by the method ``__iter__`` of
the class ``MultiSplit``.
    """

    def __init__(
        self,
        type,
        val
    ):
        self.type = type
        self.val  = val

class MultiSplit:
    """
-----------------
Small description
-----------------

The purpose of this class is to split a text at different levels. You can use
either a list view version of the split text, or work with a special iterator.


Here is an example of the list view version of the split text.

python::
    from mistool import string_use

    splitText = string_use.MultiSplit(
        text  = "p_1 , p_2 ; p_3 | r_1 ; r_2 | s",
        sep   = ["|", (";", ",")],
        strip = True
    )

    listView = splitText.listView


In this code, the variable ``listView`` is equal to the following list. There
is as many level of lists that the length of the list ``sep``. The use of
``(";", ",")`` simply indicates that the separators ``;`` and ``,`` have the
same importance.

python::
    [
        ['p_1', 'p_2', 'p_3'],
        ['r_1', 'r_2'],
        ['s']
    ]


Let see with an example how to use the instance as an iterator so as to ease
the walk in the split text.

python::
    from mistool import string_use

    splitText = string_use.MultiSplit(
        text  = "p_1 , p_2 ; p_3 | r_1 ; r_2 | s",
        sep   = ["|", ";", ","],
        strip = True
    )

    for x in splitText:
        print("{0} ---> {1}".format(x.type, x.val))


Launched in a terminal, the code will produce the following output that shows
how to work easily with the iterator of the class ``MultiSplit``. You have to
know that the type is a string, and also that for the "sep" type, the associated
value is equal to the separator used when the instance has been created.

terminal::
    sep ---> |
    sep ---> ;
    sep ---> ,
    list ---> ['p_1', 'p_2']
    sep ---> ;
    sep ---> ,
    list ---> ['p_3']
    sep ---> |
    sep ---> ;
    sep ---> ,
    list ---> ['r_1']
    sep ---> ;
    sep ---> ,
    list ---> ['r_2']
    sep ---> |
    sep ---> ;
    sep ---> ,
    list ---> ['s']


-------------
The arguments
-------------

The instanciation of this class uses the following variables.

    1) ``text`` is simply the text to split.

    2) ``sep`` is either a string for a single separator, or a list from the
    stronger separator to the weaker one. You can use list or tuple to gather
    separators having the same level of priority.

    3) ``escape`` is a string used to escape the separators. By default,
    ``escape = None`` which indicates that there is no escaping feature.

    4) ``strip`` is a boolean variable to strip, or not, each pieces of text
    found. By default, ``strip = False``.
"""

    def __init__(
        self,
        text,
        sep,
        strip  = False,
        escape = None
    ):
        self.text   = text
        self.sep    = sep
        self.strip  = strip
        self.escape = escape

        if isinstance(self.sep, str):
            self.sep = [self.sep]

        self.listView = self.build()

    def build(self):
        """
-----------------
Small description
-----------------

This method builds the list view version of the split text.

Indeed all the job is done by the hidden method ``self.__build``.
        """
        return self.__build(
            text   = self.text,
            sep    = self.sep,
            strip  = self.strip,
            escape = self.escape
        )

    def __build(
        self,
        text,
        sep,
        strip,
        escape
    ):
# No separator
        if not sep:
            if strip:
                text = text.strip()

            answer = [text]

# At least one separator
        else:
            answer = split(
                text   = text,
                sep    = sep[0],
                escape = escape,
                strip  = strip
            )

            remainingSep = sep[1:]

            if remainingSep:
                answer = [
                    self.__build(
                        text   = onePiece,
                        sep    = remainingSep,
                        escape = escape,
                        strip  = strip
                    )
                    for onePiece in answer
                ]

# The job has been done !
        return answer

    def __iter__(self):
        return self.__iter(listView = self.listView)

    def __iter(
        self,
        listView,
        depth = 0
    ):
        if listView:
            if isinstance(listView[0], str):
                yield _SplitKind(
                    type = "sep",
                    val  = self.sep[depth]
                )

                yield _SplitKind(
                    type = "list",
                    val  = listView
                )

            else:
                for x in listView:
                    yield _SplitKind(
                        type = "sep",
                        val  = self.sep[depth]
                    )

                    for y in self.__iter(
                        listView = x,
                        depth    = depth + 1
                    ):
                        yield y

def beforeAfter(
    text,
    start,
    end,
    keep = False
):
    """
-----------------
Small description
-----------------

The function will look for the piece of text between the first texts ``start``
and ``end`` found in ``text``. Then by default the function returns the text
before the text ``start`` and the one after the text ``end`` in a couple.

If ``keep = True`` then the text ``start`` will be added at the end of the first
text returned, and the second text returned will begin with the text ``end``.

You can also use ``keep = (True, False)`` or ``keep = (True, False)`` so as to
use just one half of the features which have just been explained. Indeed ``keep
= True`` and ``keep = False`` are shortcuts for ``keep = (True, True)`` and
``keep = (False, False)`` respectively.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``text`` is a string argument corresponding to the texte where the search
    must be done.

    2) ``start`` and ``end`` are string arguments that indicated the start and
    end pieces of text to find.

    3) ``keep`` is an optional argument which can be either a boolen or a couple
    of boolean. By default, ``keep = False`` which indicates to unkeep the start
    and the end in the piece of text found.
    """
    if start == "" and end == "":
        raise StringUseError(
            'The variables << start >> and  << end >> can not be both empty.'
        )

    if isinstance(keep, bool):
        keepStart = keep
        keepEnd   = keep

    else:
        keepStart, keepEnd = keep

    if start == "":
        s = 0

    else:
        s = text.find(start)

        if s == -1:
            raise StringUseError(
                'The starting text << {0} >> has not been found.'.format(start)
            )

        s += len(start)

    if end == "":
        e = s

    else:
        e = text.find(end, s)

        if e == -1:
            message = 'The ending text  << {1} >> has not been found after ' \
                    + 'the first text << {0} >>.'

            raise StringUseError(
                message.format(start, end)
            )

        if start == "":
            s = e


    if not keepStart and start:
        s -= len(start)

    if not keepEnd and end:
        e += len(end)


    return (text[:s], text[e:])


# ---------------------------------- #
# -- PLAYING WITH CASE OF LETTERS -- #
# ---------------------------------- #

def case(
    text,
    kind
):
    """
-----------------
Small description
-----------------

This function produces different case variants of the text contained in the
string variable ``text``.


For example, ``case("OnE eXamPLe", "lower")``, ``case("OnE eXamPLe", "upper")``
and ``case("OnE eXamPLe", "sentence")`` and ``case("One example", "title")`` are
respectively equal to "one example", "ONE EXAMPLE", "OnE eXamPLe" and "One
Example".


You can alos the weird transformation ``case("OnE eXamPLe", "firstLast")`` which
is equal to "One examplE". This special feature is indeed used by pyBaNaMa which
is another project of the author of the package ¨mistool.


If you need all the possible case variants, juste use  ``case("OnE eXamPLe",
"all")`` which is equal to the following dictionary.

python::
    {
        'lower'    : 'one example',
        'upper'    : 'ONE EXAMPLE',
        'sentence' : 'One example',
        'title'    : 'One Example',
        'firstLast': 'One examplE'
    }


If for example you only need to have the lower case and the title case versions
of one text, then you just have to use ``case("OnE eXamPLe", "lower + title")``
which is equal to the following dictionary.

python::
    {
        'lower': 'ONE EXAMPLE',
        'title': 'One Example'
    }


-------------
The arguments
-------------

This function uses the following variables.

    1) ``text` is a string variable corresponding to the text to modify.

    2) ``kind`` indicates the case transformation wanted. The possible case
    variants are the following ones.

        a) ``"lower"``, ``"upper"``, ``"sentence"`` and ``"title"`` are for
        lower case, upper case, sentence and title case versions of one text.

        b) ``"firstLast"`` is to have a lower case text with its firts and last
        letter in upper case.

        c) ``"all"`` is to have all the possible case variants.

        d) If for example you only need to have the lower case and the title
        case versions of one text, then you just have to use ``kind = "lower +
        title")``.
    """
    kind = kind.strip()

    if kind == 'all':
        answer = {}
        for kind in ['lower', 'upper', 'sentence', 'title', 'firstLast']:
            answer[kind] = case(text, kind)
        return answer

    elif '+' in kind:
        answer = {}
        for kind in kind.split('+'):
            kind = kind.strip()

            if not kind in answer:
                newCaseVariant = case(text, kind)

                if isinstance(newCaseVariant, dict):
                    for oneKind, oneCaseVariant in newCaseVariant.items():
                        answer[oneKind] = oneCaseVariant

                else:
                    answer[kind] = newCaseVariant

        return answer

    elif kind == 'lower':
        return text.lower()

    elif kind == 'upper':
        return text.upper()

    elif kind == 'sentence':
        return text[0].upper() + text[1:].lower()

    elif kind == 'title':
        return text.title()

    elif kind == 'firstLast':
        return text[0].upper() + text[1:-1].lower() + text[-1].upper()

    else:
        raise StringUseError(
            '<< {0} >> is not an existing kind of case variant.'.format(kind)
        )

def camelTo(
    text,
    kind
):
    """
-----------------
Small description
-----------------

This function transforms one text using camel syntax to one of the case variants
proposed by the function ``case``. Indeed, each time one upper letter is met,
one underscore followed by one space is added before it, then the function
``case`` is called, and finally extra spaces are removed. For example,
``camelTo("oneExample", "title")`` is equal to ``"One_Example"``.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``text` is a string variable corresponding to the text to modify.

    2) ``kind`` indicates the case transformation wanted. The possible case
    variants are the same as for the function ``case``.
    """
    newLines = ''

    for oneChar in text:
        if oneChar.isupper():
            newLines += '_ '

        newLines += oneChar

    newLines = case(newLines, kind)

    if isinstance(newLines, dict):
        for kind, text in newLines.items():
            newLines[kind] = text.replace('_ ', '_')

    else:
        newLines = newLines.replace('_ ', '_')

    return newLines


# ------------------- #
# -- SHORTEN LINES -- #
# ------------------- #

def __addNewWord(
    word,
    line,
    newLines,
    width,
    indentation
):
    """
-----------------
Small description
-----------------

This function manages the addition of a new word to the new lines.

It returns ``(line, newLines)`` where ``line`` is the line that must be
completed and ``newLines`` is the list of the already built lines of the cut
text.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``word`` is the actual word to add to the new lines.

    2) ``line`` is the the line being built.

    3) ``newLines`` is the list of the lines cut.

    4) ``width`` is the width expected for the lines.

    5) ``indentation`` is the indentation of the current line analyzed.
    """
    lenWord        = len(word)
    lenIndentation = len(indentation)

    if len(line) + lenWord > width:
        newLines.append(line)

        if lenWord + lenIndentation > width:
            newLines.append(indentation + word)
            line = ''

        else:
            line = indentation + word

    elif word:
        if line:
            line += ' '

        else:
            line = indentation

        line += word

    return (line, newLines)

def __indentation(text):
    """
-----------------
Small description
-----------------

This function, which is used by the function ``cut``, simply returns the leading
spaces and tabulations of one text.


-------------
The arguments
-------------

This function has only one variable ``text`` which is the text to analyse.
    """
    indentation = ''

    if text.strip():
        i = 0

        while text[i] in " \t":
            indentation += text[i]
            i+=1

    return indentation

def cut(
    text,
    width = 80
):
    """
-----------------
Small description
-----------------

This function tries to cut lines of a text so as they have less than `81`
characters.

Multiple following spaces are rendered as one single space, and the final text
is stripped so as to start and end with none empty lines.

One last thing, when lines are cut their indentations are respected.


-------------
The arguments
-------------

There are two arguments.

    1) ``text`` is simply the text to cut.

    2) ``width`` is an integer that gives the maximum of characters wanted to be
    in one line. Its default value is `80`.
    """
    if not isinstance(width, int) or width <= 0:
        raise StringUseError("<< width >> must be a positive integer.")

    word        = ''
    line        = ''
    indentation = __indentation(text)
    newLines    = []

    for i in range(len(text)):
        oneChar = text[i]

# One space
        if oneChar == ' ':
            line, newLines = __addNewWord(
                word, line, newLines,
                width, indentation
            )

            word = ''

# One back return
        elif oneChar == '\n':
            line, newLines = __addNewWord(
                word, line, newLines,
                width, indentation
            )

            word = ''

            indentation = __indentation(text[i+1:])

            if line:
                newLines.append(line)
                line = ''

            else:
                newLines.append('')

# The word becomes bigger...
        else:
            word += oneChar

# Let's finish the job !
    line, newLines = __addNewWord(
        word, line, newLines,
        width, indentation
    )

    if line:
        newLines.append(indentation + line)

    return '\n'.join(newLines).strip()


# ------------------- #
# -- DECORATE TEXT -- #
# ------------------- #

# The default frames
#    * ``C_FRAME``
#    * ``C_PRETTY_FRAME``
#    * ``PYTHON_FRAME``
#    * ``PYTHON_PRETTY_FRAME``
#    * ``PYBA_NB_PART_FRAME``
#    * ``PYBA_NO_NB_PART_FRAME``
#    * ``PYBA_NB_TITLE_FRAME``
#    * ``PYBA_NO_NB_TITLE_FRAME``
#    * ``UNITTEST_FRAME``
#    * ``UNITTEST_PROBLEM_FRAME``

C_FRAME = {
    'rule' : {
        'up'   : "*",
        'down' : "*",
        'left' : "*",
        'right': "*"
    },
    'corner': {
        'leftup'   : "/",
        'rightdown': "/"
    }
}

C_PRETTY_FRAME = {
    'rule' : {
        'up'   : "-",
        'down' : "-",
        'left' : "--",
        'right': "--"
    },
    'extra': {
        'rule' : {
            'left' : "* ",
            'right': " *"
        },
        'corner': {
            'leftup'   : "/",
            'rightdown': "/"
        }
    }
}

PYBA_NB_PART_FRAME = {
    'rule' : {
        'up'   : "*",
        'down' : "*",
    }
}

PYBA_NO_NB_PART_FRAME = {
    'rule' : {
        'up'   : ":",
        'down' : ":"
    }
}

PYBA_NB_TITLE_FRAME = {
    'rule' : {
        'up'   : "=",
        'down' : "=",
    }
}

PYBA_NO_NB_TITLE_FRAME = {
    'rule' : {
        'up'   : "-",
        'down' : "-"
    }
}

PYTHON_FRAME = {
    'rule' : {
        'up'   : "#",
        'down' : "#",
        'left' : "#",
        'right': "#"
    }
}

PYTHON_PRETTY_FRAME = {
    'rule' : {
        'up'   : "-",
        'down' : "-",
        'left' : "--",
        'right': "--"
    },
    'extra': {
        'rule' : {
            'left' : "#",
            'right': "#"
        }
    }
}

UNITTEST_FRAME = {
    'rule' : {
        'up'   : "*",
        'down' : "*",
        'left' : "*",
        'right': "*"
    }
}

UNITTEST_PROBLEM_FRAME = {
    'rule' : {
        'up'   : "*",
        'down' : "*",
        'left' : "* ---->>",
        'right': "<<---- *"
    },
    'extra': {
        'rule' : {
            'up'   : "*",
            'down' : "*",
            'left' : "*",
            'right': "*"
        }
    }
}


def __drawHorizontalRule(
    charRule,
    left,
    right,
    lenght,
    nbSpace
):
    """
-----------------
Small description
-----------------

This function is used to draw the horizontal rules of a frame around one text.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``charRule`` is the character used to draw the rule.

    2) `left`` and ``right`` are the first and last additional texts used around
    the rule.

    3) ``lenght`` is an integer giving the lenght of the rule.

    4) ``nbSpace`` is the number of spaces to add before the first additional
    text (this is for cases when left corners have different lenghts).
    """
    if charRule:
        return [
            ' '*(nbSpace - len(left))
            + left
            + charRule*lenght
            + right
        ]

    elif left or right:
        return [
            left
            + ' '*lenght
            + right
        ]

    else:
        return []

def frame(
    text,
    format,
    center = True
):
    """
-----------------
Small description
-----------------

This function makes it possible to put one text into one frame materialized by
ASCII characters. This can be usefull for console outputs or for pretty comments
for listings like the following python comment.

python::
    #############
    # one       #
    # comment   #
    # easily    #
    # formatted #
    #############

This text has been produced using the following lines.

python::
    import Mistool

    textToDecorate = '''one
    comment
    easily
    formatted'''

    print(
        Mistool.stringUse.frame(
            text   = textToDecorate,
            format = Mistool.stringUse.PYTHON_FRAME,
            center = False
        )
    )

By default, ``center`` is equal ``True`` so as to merely center the content of
the frame. Here we use one predefined frame ``Mistool.stringUse.PYTHON_FRAME``
which is the following dictionary.

python::
    PYTHON_FRAME = {
        'rule' : {
            'up'   : "#",
            'down' : "#",
            'left' : "#",
            'right': "#"
        }
    }


``Mistool.stringUse.C_FRAME`` is another predefined frame which is the following
dictionary.

python::
    C_FRAME = {
        'rule' : {
            'up'   : "*",
            'down' : "*",
            'left' : "*",
            'right': "*"
        },
        'corner': {
            'leftup'   : "/",
            'rightdown': "/"
        }
    }

This will give the following format of frame.

code_c::
    /***************
     * one         *
     * comment     *
     * with C-like *
     * style       *
     ***************/


The last example of ``Mistool.stringUse.PYTHON_PRETTY_FRAME`` shows the
possibility to us an extra frame so as to obtain a little more elaborated frames
in only one call.

python::
    PYTHON_PRETTY_FRAME = {
        'rule' : {
            'up'   : "-",
            'down' : "-",
            'left' : "--",
            'right': "--"
        },
        'extra': {
            'rule' : {
                'left' : "#,
                'right': "#"
            }
        }
    }

This will give the following format of frame.

python::
    # ------------- #
    # -- one     -- #
    # -- pretty  -- #
    # -- comment -- #
    # ------------- #


-------------
The arguments
-------------

This function uses the following variables.

    1) ``text`` is a string value corresponding to the text to put in a frame.

    2) ``center`` is a boolean variable to center or not the text inside the
    frame. By default, ``center = True``.

    3) ``format`` is a dictionary defining the frame. Here are the default
    dictionaries proposed by ``string_use``.

        * ``C_FRAME``
        * ``C_PRETTY_FRAME``
        * ``PYTHON_FRAME``
        * ``PYTHON_PRETTY_FRAME``
        * ``PYBA_NB_PART_FRAME``
        * ``PYBA_NO_NB_PART_FRAME``
        * ``PYBA_NB_TITLE_FRAME``
        * ``PYBA_NO_NB_TITLE_FRAME``
        * ``UNITTEST_FRAME``
        * ``UNITTEST_PROBLEM_FRAME``


    The general structure of this kind of dictionary is the following one.

    python::
        {
            'rule' : {
                'up'   : "Only one character",
                'down' : "Only one character",
                'left' : "The characters wanted",
                'right': "The characters wanted"
            },
            'corner': {
                'leftup'   : "The characters wanted",
                'leftdown' : "The characters wanted",
                'rightup'  : "The characters wanted",
                'rightdown': "The characters wanted"
            },
            'extra': {
                'rule' : {
                    'up'   : "Only one character",
                    'down' : "Only one character",
                    'left' : "The characters wanted",
                    'right': "The characters wanted"
                },
                'corner': {
                    'leftup'   : "The characters wanted",
                    'leftdown' : "The characters wanted",
                    'rightup'  : "The characters wanted",
                    'rightdown': "The characters wanted"
                },
            }
        }
    """
# Default values must be choosen if nothing is given.
    for key in ['rule', 'corner']:
        if key not in format:
            format[key] = {}

    for styleKey in ['up', 'down', 'left', 'right']:
        if styleKey not in format['rule']:
            format['rule'][styleKey] = ""

    for styleKey in ['leftup', 'leftdown', 'rightup', 'rightdown']:
        if styleKey not in format['corner']:
            format['corner'][styleKey] = ""

# Horizontal rules can only use one single character.
    for onePosition in ['up', 'down']:
        if len(format['rule'][onePosition]) > 1:
            message = "You can only use nothing or one single character " \
                    + "for rules.\nSee << {0} >> for the {1} rule."

            raise StringUseError(
                message.format(
                    format['rule'][onePosition],
                    onePosition
                )
            )

# Infos about the lines of the text.
    lines = [oneLine.rstrip() for oneLine in text.splitlines()]
    nbMaxChar = max([len(oneLine) for oneLine in lines])

# Space to add before vertical rules.
    nbSpace = max(
        len(format['corner']['leftup']),
        len(format['corner']['leftdown'])
    )

    spaceToAdd = ' '*nbSpace

# Text decoration for vertical rules
    if format['rule']['left']:
        leftRule = format['rule']['left'] + ' '
    else:
        leftRule = ''

    if format['rule']['right']:
        rightRule = ' ' + format['rule']['right']
    else:
        rightRule = ''

# Length of the rule without the corners
    lenght = nbMaxChar + len(leftRule) + len(rightRule)

# First line of the frame
    answer = __drawHorizontalRule(
        charRule = format['rule']['up'],
        left     = format['corner']['leftup'],
        right    = format['corner']['rightup'],
        lenght   = lenght,
        nbSpace  = nbSpace
    )

# Management of the lines of the text
    for oneLine in lines:
        nbSpacesMissing = nbMaxChar - len(oneLine)

# Space before and after one line of text.
        if center:
            if nbSpacesMissing % 2 == 1:
                spaceAfter = ' '
            else:
                spaceAfter = ''

            nbSpacesMissing = nbSpacesMissing // 2

            spaceBefore = ' '*nbSpacesMissing
            spaceAfter += spaceBefore

        else:
            spaceBefore = ''
            spaceAfter = ' '*nbSpacesMissing

        answer.append(
            spaceToAdd
            +
            '{0}{1}{2}{3}{4}'.format(
                leftRule,
                spaceBefore,
                oneLine,
                spaceAfter,
                rightRule
            )
        )

# Last line of the frame
    answer += __drawHorizontalRule(
        charRule = format['rule']['down'],
        left     = format['corner']['leftdown'],
        right    = format['corner']['rightdown'],
        lenght   = lenght,
        nbSpace  = nbSpace
    )

    answer = '\n'.join(answer)

# Does we have an extra frame ?
    if 'extra' in format:
        try:
            answer = frame(
                text   = answer,
                format = format['extra'],
                center = center
            )

        except StringUseError as e:
            raise StringUseError(
                str(e)[:-1] + " in the definition of the extra frame."
            )

# All the job has been done.
    return answer

class step:
    """
-----------------
Small description
-----------------

This class displays texts for step by step actions. The numbering of the steps
is automatically updated and displayed.


-------------
The arguments
-------------

There are two optional variables.

    1) ``nb`` is the number of the current step. When the class is instanciated,
    the default value is ``1``.

    2) ``deco`` indicates how to display the numbers. When the class is
    instanciated, the default value is ``""1)""`` where ``1`` symbolises the
    numbers.
    """
    def __init__(
        self,
        nb   = 1,
        deco = "1)"
    ):
        self.nb   = nb
        self.deco = deco.replace('1', '{0}')

    def print(
        self,
        text,
        deco
    ):
        """
-----------------
Small description
-----------------

This method simply prints ``deco`` the text of the actual numbering, and then
``text`` the content of the actual step.

You can redefine this method for finer features.


-------------
The arguments
-------------

This method uses the following variables.

    1) ``text`` is simply the text of the actual step.

    2) ``deco`` is a string corresponding to the text indicated what is the
    actual step number.
        """
        print(
            deco,
            text,
            sep = " "
        )

    def display(
        self,
        text
    ):
        """
-----------------
Small description
-----------------

This method simply calls the method ``self.print`` so as to print the
informations contained in the variable ``text``, and then ``self.nb`` is
augmented by one unit.

You can redefine the method ``self.print`` for finer features.


-------------
The arguments
-------------

This method uses one variable ``text`` which is the text of the actual step.
        """
        self.print(
            self.deco.format(self.nb),
            text
        )

        self.nb += 1
