#!/usr/bin/env python3

"""
Directory : mistool
Name      : log_test_use
Version   : 2013.09
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This modules contains some utilities that can be useful for unit tests made
with the standard module ``unittest``.
"""

# Source used :
#    1) http://agiletesting.blogspot.com/2005/01/python-unit-testing-part-1-unittest.html
#    2) http://stackoverflow.com/questions/67631/how-to-import-a-module-given-the-full-path
#    3) http://docs.python.org/3.2/library/unittest.html?highlight=unittest.texttestrunner#basic-example

import imp
import inspect
import unittest
import collections

from mistool import os_use, python_use, string_use


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class LogTestUseError(ValueError):
    """
-----------------
Small description
-----------------

Base class for errors in the ``log_test_use`` module of the package ``mistool``.
    """
    pass

def __unexpectedError(text):
    """
-----------------
Small description
-----------------

This is for errors not expected during the tests.
    """
    message = '    UNEXPECTED ERROR DETECTED DURING THE TEST :\n' \
            + '        ' \
            + '        \n'.join(text.split('\n'))

    logPrint(message)


# ---------------------- #
# -- LOGGING MESSAGES -- #
# ---------------------- #

# This variable is used to define some formatting rules from one path or one
# name of a function, a method or a class.

ASCII_ASSO = {
    '_'  : " ",
    '__' : " - ",
    '___': " ---> ",
    '/'  : " : "
}

def what(
    text,
    isMethod = False
):
    """
-----------------
Small description
-----------------

This function transforms some text like ``Abreviations__NewAdding_Good`` into
``Abreviations ---> New Adding - Good`` where the replacements are defined in
the global constant ``ASCII_ASSO`` which is defined as follows.

python::
    ASCII_ASSO = {
        '_'  : " ",
        '__' : " - ",
        '___': " ---> ",
        '/'  : " : "
    }


There is a usefull optional argument ``isMethod``. If ``isMethod = True``, then
the text must start with ``test``. If it is the case, this piece of text will be
removed. This is useful because the module ``unittest`` only calls the methods
with a name started by ``test``.


-------------
The arguments
-------------

This function uses two arguments.

    1) ``text`` is a text of something that is tested.

    2) ``isMethod`` is an optional boolean variable. By default, ``isMethod = False``.
    """
    if isMethod:
        if not text.startswith('test'):
            raise LogTestUseError('The text does not start with "test".')

        text = text[len('test'):]

    text = string_use.camelTo(text, "title")

    text = string_use.replace(
        text        = text,
        replacement = ASCII_ASSO
    )

    return text.strip()

def __isDict(obj):
    """
-----------------
Small description
-----------------

This function returns ``True`` if the argument ``obj`` is a dictionnary (ordered
or not). If this is not the case, the function returns ``False``.
    """
    return isinstance(obj, (dict, collections.OrderedDict))

def diffDict(
    dict_1,
    dict_2,
    recursive = False
):
    """
-----------------
Small description
-----------------

This function analyses if the two dictionaries ``dict_1`` and ``dict_2`` are
equal. If it is the case, an empty string is returned. In the other case, a
message indicating briefly the differences between the two dictionaries is
returned.


If you want a more precise message for the differences found, you can use the
optional argument ``recursive`` whose default value is ``False``. If ``recursive
= True``, each time that two differents values are found and are also two
dictionaries, then the function ``diffDict`` will be called recursively for a
finer message indicating the differences.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``dict_1`` and `` dict_2`` are the two dictionaries to analyse.

    2) ``recursive`` is a boolean variable to ask to recursively analysed
    different values that are dictionaries. By default, ``recursive = False``.
    """
    if not(dict_1 and dict_2):
        text = 'One of the dictionary is empty but not the other.'

    elif sorted(dict_1.keys()) != sorted(dict_2.keys()):
        text = 'The dictionaries have different keys.'

    else:
        text = ''

        for keyDict_1, valueDict_1 in dict_1.items():
            valueDict_2 = dict_2[keyDict_1]

            if valueDict_1 != valueDict_2:
                text = 'The values for the key << {0} >> '.format(
                    python_use.pyRepr(keyDict_1)
                ) + 'are different.'

                if recursive \
                and __isDict(valueDict_1) and __isDict(valueDict_2):
                    text += '\n\n>>> RECURSIVE DIFFERENCES <<<\n\n' + diffDict(
                        valueDict_1,
                        valueDict_2,
                        recursive = True
                    )

                break

    return text


# ----------- #
# -- PRINT -- #
# ----------- #

def logPrint(text):
    """
-----------------
Small description
-----------------

This function is used to display messages. By default, ``logPrint`` is simply
equal to ``print``.


You can redefine this function to change the way the messages are "displayed"
so as to build log files for example.


-------------
The arguments
-------------

This function has only one variable ``text`` which is a string.
    """
    print(text)

def info(
    text,
    kind  = "Testing",
    start = "    + ",
    end   = "..."
):
    """
-----------------
Small description
-----------------

This function is usefull for displaying very short infos in one line.


-------------
The arguments
-------------

This function has four arguments.

    1) ``text`` is the text to display.

    2) ``kind`` is a leading text put just before ``text``. By default, ``kind =
    "Testing"``.

    3) ``start`` and ``end`` are texts put at the start and the end of the text
    ``kind + text``. By default ``start = "    + "``and ``end = "..."``.
    """
    logPrint(
        '{0}{1} "{2}"{3}'.format(start, kind, text, end)
    )


def problem(
    text,
    format = string_use.UNITTEST_PROBLEM_FRAME,
    center = True
):
    """
-----------------
Small description
-----------------

This function is useful for displaying problems met during one test.

Indeed this function just displays the text ``string_use.frame(text, format,
center)`` (take a look at the documentation of the module ``string_use`` if
you want to use your own format of frame).


-------------
The arguments
-------------

This function uses the following variables.

    1) ``text`` is the text indicating the problem.

    2) ``format`` is the format of the frame following the specifications of the
    function ``string_use.frame``. This an optional argument. By default,
    ``format = string_use.UNITTEST_PROBLEM_FRAME``.

    3) ``center`` is a boolean optional variable which aks to center or not the
    text. By default, ``center = True``.
    """
    logPrint(
        string_use.frame(
            text,
            format = format,
            center = center
        )
    )


# ---------------- #
# -- TEST SUITE -- #
# ---------------- #

def launchTestSuite(
    dir,
    depth     = 0,
    verbosity = 0,
    message   = "",
    sort      = lambda x: x
):
    """
-----------------
Small description
-----------------

This function simplifies a lot the launching of unit tests made via the module
``unittest``.


See the meaning of the arguments so as to know how to use this function.


-------------
The arguments
-------------

This function uses the following variables.

    1) ``dir`` is the only obligatory argument. It indicates the directory where
    you have put all your files for testing.

    warning::
        You must follow the two simple rules beyond.

            a) All the files where the unit tests are defined must have a name
            starting with ``test_``.

            b) All the class making the unit tests must have a name starting
            with ``Test``.

    2) ``depth`` is an optional argument with default value ``0``. This is to
    indicate the maximal depth for the research of the files to remove.

    The very special value ``(-1)`` indicates that there is no maximum. The
    default value ``0`` asks to only look for in the direct content of the main
    directory to analyse.

    3) ``verbosity`` is an optional argument with default value ``0``. Indeed
    the meaning of this value is the same at its eponym for the class
    ``unittest.TextTestRunner``. See the documentation of ``unitttest``.

    4) ``message`` is an optional argument which is an empty string by default.
    You can use ``message`` if you want to display some text, via the function
    ``logPrint``, at the very beginning of the test suite.

    5) ``sort`` is a function used to sort the path of the files doing the
    tests. By default, ``sort = lambda x: x``.
    """
# Let's look for the good Python files.
    pathTestFiles = os_use.listFile(
        main   = dir,
        ext    = "py",
        prefix = "test_",
        depth  = depth
    )

# Let's sort the paths.
    pathTestFiles.sort(key = sort)

# Let's look for the testing classes.
    listOfClasses = []

    for onePythonFile in pathTestFiles:
        test = imp.load_source(
            os_use.name(onePythonFile),
            onePythonFile
        )

        for nameClassFound, oneClassFound in inspect.getmembers(
            test,
            inspect.isclass
        ):
            if nameClassFound.startswith('Test'):
                listOfClasses.append(oneClassFound)

# It's time to test.
    if listOfClasses:
        suiteOfTests = unittest.TestSuite()

        for oneClass in listOfClasses:
            suiteOfTests.addTest(unittest.makeSuite(oneClass))

        if message:
            logPrint(
                "\n" + string_use.frame(
                    text   = message,
                    format = string_use.PYTHON_FRAME
                ) + "\n"
            )

        unittest \
            .TextTestRunner(verbosity = verbosity) \
            .run(suiteOfTests)

    else:
        logPrint('No test has been found...')

