#!/usr/bin/env python3

"""
Directory : mistool
Name      : os_use
Version   : 2013.10
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This module contains functions to manipulate easily files, directories and also
to have informations about the system.
"""

import os
import shutil
import platform
import subprocess


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class OsUseError(ValueError):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Base class for errors in the ``os_use`` module of the package ``mistool``.
    """
    pass


# ----------- #
# -- TESTS -- #
# ----------- #

def isFile(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function simply tests if the path ``path`` points to an existing file.
    """
    return os.path.isfile(path)

def isDir(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function simply tests if the path ``path`` points to an existing directory.
    """
    return os.path.isdir(path)

def hasExtIn(
    path,
    listOfExt
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function tests if the path ``path`` finishes by one of the extensions given
in the list of extensions ``listOfExt``.
    """
    if '.' in path:
        for oneExtension in listOfExt:
            if path.endswith('.' + oneExtension):
                return True

    return False


# ----------------------- #
# -- INFOS ABOUT PATHS -- #
# ----------------------- #

SEP = os.sep

def name(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function extracts from the path ``path`` the name of the file with its
extension or simply the name of one directory.
    """
    i = path.rfind(SEP)

    return path[i+1:]

def fileName(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function extracts from the path ``path`` the name of the file without its
extension.
    """
    return os.path.splitext(name(path))[0]

def ext(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function extracts from the path ``path`` the extension of the file.
    """
    return os.path.splitext(path)[1][1:]

def pathNoExt(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function removes from the path ``path`` the extension of the file.
    """
    i = len(ext(path))

    if i != 0:
        i += 1

        return path[:-i]

    return path

def parentDir(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function returns the path of the directory that contains the file or the
directory corresponding to the path ``path``.
    """
    return os.path.dirname(path)

def relativePath(
    main,
    sub,
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Suppose that we have the following paths where ``main`` is the path of the main
directory and ``sub`` the one of a sub directory or a file contained in the main
directory.

python::
    main = "/Users/projects/source_dev"
    sub  = "/Users/projects/source_dev/misTool/os_use.py"

The function will return in that case the following string which always begin
with one slash ``/``.

python::
    "/misTool/os_use.py"
    """
# Special case of the same path !
    if main == sub:
        raise OsUseError("The main path and the sub-path are equal.")

# We clean the path in case of they contain things like ``../``.
    main = os.path.normpath(main)
    sub  = os.path.normpath(sub)

# The main path must finish by one backslash because
#     "python/misTool/source_dev/misTool/os_use.py"
# is not one sub path of
#     "python/misTool/source".

    if main[-1] != SEP:
        main += SEP

# Is the sub path contained in the main one ?
    if not sub.startswith(main):
        raise OsUseError(
            "The sub-path\n\t+ {0}\nis not contained in the main path\n\t+ {1}\n" \
                .format(sub, main) \
            + "so it is not possible to have one relative path."
        )

# Everything seems ok...
    i = len(main) - 1

    return sub[i:]

def relativeDepth(
    main,
    sub
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Suppose that we have the following paths where ``main`` is the path of the main
directory and ``sub`` the one of a sub directory or a file contained in the main
directory. Here are some examples.

    1) The function will return ``1`` in the following case.

    python::
        main = "/Users/projects/source_dev"
        sub  = "/Users/projects/source_dev/misTool/os_use.py"

    This means that the file path::``os_use.py`` is contained in one simple sub
    directory of the main directory.

    2) In the following case, the function will return ``2``.

    python::
        main = "/Users/projects/source_dev"
        sub  = "/Users/projects/source_dev/misTool/os/os_use.py"

    3) For the last example just after, the value returned will be ``0``.

    python::
        main = "/Users/projects/source_dev"
        sub  = "/Users/projects/source_dev/os_use.py"
    """
    return relativePath(
        main        = main,
        sub         = sub
    ).count(SEP) - 1

def commonPath(paths):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function returns the smaller directory that contains the objects having the
paths given in the list ``paths``.
    """
    if len(paths) < 2:
        raise OsUseError(
            "You must give at least two paths."
        )

    if len(paths) == 2:
        answer = []

        pieces_1 = paths[0].split(SEP)
        pieces_2 = paths[1].split(SEP)

        for i in range(min(len(pieces_1), len(pieces_2))):
            if pieces_1[i] == pieces_2[i]:
                answer.append(pieces_1[i])
            else:
                break

        return SEP.join(answer)

    else:
        return commonPath([
            commonPath(paths[:-1]), paths[-1]
        ])


# ------------------------------------- #
# -- OPENING WITH ASSOCIATED PROGRAM -- #
# ------------------------------------- #

def watch(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function tries to open one directory, or one file. Indeed, files are opened
within their associated applications, if this last ones exist.
    """
# Nothing to open...
    isOneFile = isFile(path)

    if not isOneFile and not isDir(path):
        raise OsUseError(
            "The following path does not point to one existing file " \
            "or directory.\n\t<< {0} >>".format(path)
        )

# Each OS has its own method.
    osName = system()

# Windows
    if osName == "windows":
        if isOneFile:
            os.startfile(path)
        else:
            subprocess.check_call(args = ['explorer', path])

# Mac
    elif osName == "mac":
        subprocess.check_call(args = ['open', path])

# Linux
#
# Source :
#     * http://forum.ubuntu-fr.org/viewtopic.php?pid=3952590#p3952590
    elif osName == "linux":
        subprocess.check_call(args = ['xdg-open', path])

# Unknown method...
    else:
        raise OsUseError(
            "The opening of file on << {0} >> OS is not "
            "supported.".format(osName)
        )


# ------------- #
# -- READING -- #
# ------------- #

def readTextFile(
    path,
    encoding = "utf-8"
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function returns the text like content of one file given by its path.

You can indicate the encoding of the file with the optional argument
``encoding`` whose dfault value is ``"utf-8"``. The available encodings are the
same as the ones for the standard ¨python function ``open``.
    """
    with open(
        path,
        mode     = "r",
        encoding = encoding
    ) as f:
        return f.read()


# ----------------------------------- #
# -- CREATION, DELETION AND MOVING -- #
# ----------------------------------- #

def makeDir(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function build the directory with the given path ``path``. If one parent
directory must be build, the function will do the job.
    """
    if not isDir(path):
        os.makedirs(path)

def makeTextFile(
    path,
    text     = '',
    encoding = 'utf-8'
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function build the file with the given path ``path`` and the text like
content ``text``.

You can indicate the encoding of the file with the optional argument ``encoding``
whose dfault value is ``"utf-8"``. The available encodings are the same as the
ones for the standard ¨python function ``open``.
    """
    makeDir(parentDir(path))

    with open(
        path,
        mode     = "w",
        encoding = encoding
    ) as f:
        f.write(text)

def move(
    source,
    destination
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function moves the file or the directory from the path ``source`` to the
path ``destination``. If the source and the destination have the same parent
directory, then the final result is just a renaming of the file or directory.
    """
    if isDir(path):
        os.renames(source, destination)

    elif isFile(source):
        copy(source, destination)

        if isFile(destination):
            destroy(source)

def copy(
    source,
    destination
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function copy the file having the path ``source`` to the destination path
``destination``.
    """
    if isFile(source):
        dir = parentDir(destination)

        if not isDir(dir):
            makeDir(dir)

        shutil.copy(source, destination)

    elif isDir(source):
        raise OsUseError(
            "The copy of one directory is not yet supported."
        )

    else:
        raise OsUseError(
            "The following path points nowhere.\n\t<< {0} >>".format(source)
        )

def destroy(path):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function removes the directory or the file given by its path ``path``.
    """
    if isDir(path):
        shutil.rmtree(path)

    elif isFile(path):
        os.remove(path)

def clean(
    path,
    ext,
    depth = 0
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function removes files in one directory regarding to their extension.


:::::::::::::
The arguments
:::::::::::::

This functions uses the following variables.

    1) ``path`` is simply the path of the directory to clean.

    2) ``ext`` is the list of the extensions of the files to remove.

    3) ``depth`` is the maximal depth for the research of the files to remove.
    The very special value ``(-1)`` indicates that there is no maximum. The
    default value is ``0`` which asks to only look for in the direct content of
    the main directory to analyse.
    """
    for oneFile in nextFile(
        main  = path,
        ext   = {'keep': ext},
        depth = depth
    ):
        destroy(oneFile)


# ------------------- #
# -- LIST OF FILES -- #
# ------------------- #

def __isSubDepthGood(
    main,
    sub,
    depth
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This small function is used by ``nextFile`` so as to know if one directory must
be inspected or not.
    """
    if depth == -1:
        return True

    elif depth == 0:
        return bool(main == sub)

# ``relativeDepth`` sends logically one error if the two directories are equal,
# so we have to take care of this very special case.
    elif main == sub:
        return True

    else:
# The use of ``... + 1`` in the test comes from the fact we work with files
# contained in the sub directory, and the depth of this file is equal to the
# one of the sub directory plus one.
        return bool(
            relativeDepth(main = main, sub = sub) + 1 <= depth
        )

def __isFileGood(
    path,
    ext,
    prefix
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This small function is used by ``nextFile`` so as to know if the file found
matches the search criteria about extensions and prefixes.
    """
    path = name(path)

    toKeep = True

    if ext['discard'] and hasExtIn(path, ext['discard']):
        toKeep = False

    if toKeep:
        if ext['keep'] and not hasExtIn(path, ext['keep']):
            toKeep = False

    if toKeep:
        for onePrefix in prefix['discard']:
            if path.startswith(onePrefix):
                toKeep = False
                break

    if toKeep and prefix['keep']:
        toKeep = False

        for onePrefix in prefix['keep']:
            if path.startswith(onePrefix):
                toKeep = True
                break

    return toKeep

def __toKeepToDiscard(extOrPref):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This small function is used by ``nextFile`` so as to produce one normalized
dictionary of the variables ``ext`` and ``prefix``.
    """
    if extOrPref == None:
        extOrPref = {'keep': "..."}

    elif isinstance(extOrPref, str) \
    or isinstance(extOrPref, list):
        extOrPref = {'keep': extOrPref}

    toKeep = extOrPref.get('keep', [])

    if isinstance(toKeep, str):
        toKeep = [toKeep]

    toDiscard = extOrPref.get('discard', [])

    if isinstance(toDiscard, str):
        toDiscard = [toDiscard]

    return {
        'keep'   : toKeep,
        'discard': toDiscard
    }

def nextFile(
    main,
    ext     = {},
    prefix  = {},
    depth   = 0,
    keepDir = None
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function is an iterator that sends path of files in a directory with the
possibility to use some criteria of research.


Suppose for example that we have the following directory structure.

directory::
    + mistool
        * __init__.py
        * description.rst
        * latex_use.py
        * os_use.py
        * ...
        + change_log
            + 2012
                * 07.pdf
                * 08.pdf
                * 09.pdf
        + debug
            * debug_latex_use.py
            * debug_os_use.py
            + debug_latex_use
                * latexTest.pdf
                * latexTest.tex
                * latexTestBuilder.py
                * ...
        + toUse
            + latex
                * ...


Let's consider the following code.

python::
    from mistool import os_use

    for onePath in os_use.nextFile(
        main  = "/Users/mistool",
        ext   = "py",
        depth = -1
    ):
        print(onePath)


If we launch the preceding code in a terminal, then we obtain the following
outputs.

terminal::
    /Users/mistool/__init__.py
    /Users/mistool/latex_use.py
    /Users/mistool/os_use.py
    /Users/mistool/debug/debug_latex_use.py
    /Users/mistool/debug/debug_os_use.py
    /Users/mistool/debug/debug_latex_use/latexTestBuilder.py


You can also have files with different extensions like in the following code.

python::
    from mistool import os_use

    for onePath in os_use.nextFile(
        main  = "/Users/mistool",
        ext   = ["tex", "pdf"]
        depth = -1
    ):
        print(onePath)


If we launch the preceding code in a terminal, then we obtain the following
outputs.

terminal::
    /Users/mistool/change_log/2012/07.pdf
    /Users/mistool/change_log/2012/08.pdf
    /Users/mistool/change_log/2012/09.pdf
    /Users/mistool/debug/debug_latex_use/latexTest.pdf
    /Users/mistool/debug/debug_latex_use/latexTest.tex


info::
    You can also exclude extensions and use some other tuning. See the
    presentation of the arguments.


It is also possible to use criteria of search regarding to the prefixes of the
names of the files. For example, the following code will only return the path of
the file which name starts with "latexTest".

python::
    from mistool import os_use

    for onePath in os_use.nextFile(
        main   = "/Users/mistool",
        prefix = "latex",
        depth  = -1
    ):
        print(onePath)


The preceding code will product the following output in a terminal.

terminal::
    /Users/mistool/debug/debug_latex_use/latexTest.pdf
    /Users/mistool/debug/debug_latex_use/latexTest.tex
    /Users/mistool/debug/debug_latex_use/latexTestBuilder.py


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``main`` is simply the path of the main directory to analyse.

    2) ``ext`` gives informations about the extensions of the files wanted. It
    can be one of the following kinds.

        i) ``ext`` can be simply a string giving only one extension to keep.

        ii) ``ext`` can be a list of strings giving the list of extensions to
        keep.

        iii) ``ext`` can be a dictionary of the following kind which is used to
        keep or to discard some kinds of files.

        python::
            {
                'keep'   : the extensions of files to look for,
                'discard': the extensions of files to discard
            }


        The values giving the extensions can be either a single string, or a list
        of strings.

    By default, ``ext`` is an empty dictionary. This asks to send every kind of
    files.

    3) ``prefix`` can be used in the same way that `èxt``must be used. This
    variable indicates files to keep or discard regarding to the prefix of their
    name.

    4) ``depth`` is the maximal depth for the research. The very special value
    ``(-1)`` indicates that there is no maximum.

    The default value is ``0`` which asks to only look for in the direct content
    of the main directory to analyse.

    5) The variable ``keepDir`` has been added so as to be used by the class
    ``TreeDir``, but you can play with it if you want. The possible values are
    the following ones.

        a) The default value ``None`` indicates to unkeep any directory. This is
        what is expected when you call one function named ``nextFile``.

        b) ``"minimal"`` will make the function returns only the directories
        that contains at least one file from the ones searched.

        c) ``"all"`` will make the function returns any sub directories even if
        it doesn't contained one file from the ones searched.

        d) You can also use ``"Minimal"`` and ``"All"``, be carefull of the
        first uppercase letters, so as to also have idiomatic paths ended by
        path::"..." to indicate other files that the ones searched which also
        mustn't be discarded.
    """
# Directories to keep ?
    if not keepDir in [None, "minimal", "Minimal", "all", "All"]:
        raise OsUseError("Illegal value of the variable << keepDir >>.")

    dirMustBeKept = bool(keepDir in ["minimal", "Minimal", "all", "All"])

# Indicate or not the files not matching the search queries ?
    indicateAllFiles = bool(keepDir in ["Minimal", "All"])

# Does the directory exist ?
    if not isDir(main):
        raise OsUseError(
            "The following path does not point to one existing directory." \
            "\n\t+ {0}".format(main)
        )

# Extensions and prefixes
    ext    = __toKeepToDiscard(ext)
    prefix = __toKeepToDiscard(prefix)


# It's time to walk in the directory...
    for root, dirs, files in os.walk(main):
        if __isSubDepthGood(
            main  = main,
            sub   = root,
            depth = depth
        ):
# The following boolean are used for the directory views !
            noBadFileFound  = True
            noGoodFileFound = True

            for oneFile in files:
                path = root + SEP + oneFile

# Looking for good files
                if __isFileGood(path, ext, prefix):
# Directory of the first good file found to display ?
                    if noGoodFileFound and dirMustBeKept:
                        noGoodFileFound = False
                        yield root

                    yield path

# One bad file found
                elif noBadFileFound:
                    noBadFileFound = False

# Directory without any good file
            if noGoodFileFound and keepDir in ["all", "All"]:
                yield root

# Directory without some good files
            if indicateAllFiles and not noBadFileFound:
                yield root + SEP + "..."

def listFile(*args, **kwargs):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function is similar to the function ``nextFile``, and it has the same
variables that the ones of ``nextFile``, but instead of sending infos about the
files found one at a time, this function directly sends the whole list of the
infos found.

See the documentation of the function ``nextFile`` to have precisions about the
available variables and the structure of each single info that will be in the
list returned by ``listFile``.
    """
# The use of ``*args`` and ``**kwargs`` makes it very easy to implement the fact
# that ``listFile`` and ``nextFile`` have the same variables.
    return [p for p in nextFile(*args, **kwargs)]


# ----------------------- #
# -- ABOUT DIRECTORIES -- #
# ----------------------- #

def nextDir(
    main,
    depth = 0
):
    """"
:::::::::::::::::
Small description
:::::::::::::::::

This function is an iterator that sends path of directories contained in a
directory.


Suppose for example that we have the following directory structure.

directory::
    + mistool
        * __init__.py
        * description.rst
        * latex_use.py
        * os_use.py
        * ...
        + change_log
            + 2012
                * 07.pdf
                * 08.pdf
                * 09.pdf
        + debug
            * debug_latex_use.py
            * debug_os_use.py
            + debug_latex_use
                * latexTest.pdf
                * latexTest.tex
                * latexTestBuilder.py
                * ...
        + toUse
            + latex
                * ...


Let's consider the following code.

python::
    from mistool import os_use

    for onePath in os_use.nextFile(
        main  = "/Users/mistool",
        depth = -1
    ):
        print(onePath)


If we launch the preceding code in a terminal, then we obtain the following
outputs.

terminal::
    /Users/mistool/change_log
    /Users/mistool/change_log/2012
    /Users/mistool/debug
    /Users/mistool/debug/debug_latex_use
    /Users/mistool/toUse
    /Users/mistool/toUse/latex


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``main`` is simply the path of the main directory to analyse.

    2) ``depth`` is the maximal depth for the research. The very special value
    ``(-1)`` indicates that there is no maximum.

    The default value is ``0`` which asks to only look for in the direct content
    of the main directory to analyse. In other word, ``depth = 0`` asks to find
    the directories directly contained in the main directory analysed.
    """
    if depth == -1:
        subDepth = - 1

    else:
        if depth == 0:
            subDepth = None

        else:
            subDepth = depth - 1

        depth   += 1

    for pathDir in os.listdir(main):
        pathDir = main + SEP + pathDir

        if isDir(pathDir) \
        and __isSubDepthGood(
            main  = main,
            sub   = pathDir,
            depth = depth
        ):
            yield pathDir

            if subDepth != None:
                for subPathDir in nextDir(
                    pathDir,
                    depth = subDepth
                ):
                    if __isSubDepthGood(
                        main  = main,
                        sub   = subPathDir,
                        depth = depth
                    ):
                        yield subPathDir

def listDir(*args, **kwargs):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function is similar to the function ``nextDir``, and it has the same
variables that the ones of ``nextDir``, but instead of sending infos about the
directories found one at a time, this function directly sends the whole list of
the infos found.

See the documentation of the function ``nextDir`` to have precisions about the
available variables and the structure of each single info that will be in the
list returned by ``listDir``.
    """
# The use of ``*args`` and ``**kwargs`` makes it very easy to implement the fact
# that ``listFile`` and ``nextFile`` have the same variables.
    return [p for p in nextDir(*args, **kwargs)]

class DirView:
    """
:::::::::::::::::
Small description
:::::::::::::::::

This class displays the tree structure of one directory with the possibility to
keep only some relevant informations like in the following example.

code::
    + misTool
        * __init__.py
        * latexUse.py
        * os_use.py
        * ...
        + change_log
            + 2012
                * 07.pdf
                * 08.pdf
                * 09.pdf
        + debug
            * debug_latexUse.py
            * debug_os_use.py
            + debug_latexUse
                * latexTest.pdf
                * latexTest.tex
                * latexTestBuilder.py
                * ...
        + toUse
            + latex
                * ...


This output has been obtained with the following ¨python code.

python::
    from mistool import os_use

    dirView = os_use.dirView(
        main    = "/Users/misTool",
        ext     = {'keep': ["py", "txt", "tex", "pdf"]},
        depth   = -1,
        keepDir = "All",
        output  = "short",
        seeMain = True
    )

    print(dirView.ascii)


:::::::::::::
The arguments
:::::::::::::

This class uses the following variables.

    1) The variables ``main``, ``ext``, ``depth`` and ``sub`` and ``keepDir``
    have exactly the same meaning and behavior that they have with the function
    ``nextFile``, except that here the default value of ``keepDir`` is
    ``"minimal"`` and not ``None``.

    2) ``output`` is for the writings of the paths.

        a) The default value ``"all"`` asks to display the whole paths of the
        files and directories found.

        b) ``"relative"`` asks to display relative paths comparing to the main
        directory analysed.

        c) ``"short"`` asks to only display names of directories found, and
        names, with its extensions, of the files found.

    3) ``seeMain`` is one boolean value to display or not the main directory
    which is analyzed. The default value is ``True``.
    """
    ASCII_DECORATION = {
        'directory': "+",
        'file'     : "*",
        'tab'      : " "*4
    }

    def __init__(
        self,
        main,
        ext     = {},
        prefix  = {},
        depth   = 0,
        keepDir = "minimal",
        output  = "all",
        seeMain = True
    ):
# Directories to keep ?
        if keepDir == None:
            raise OsUseError(
                "Illegal value of the variable << keepDir >>."
            )

        self.main    = main
        self.ext     = ext
        self.prefix  = prefix
        self.depth   = depth
        self.keepDir = keepDir
        self.output  = output
        self.seeMain = seeMain

        self.build()

    def build(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds one list of dictionaries of the following kind.

python::
    {
        'kind' : "directory" or "file",
        'depth': the depth level regarding to the main directory,
        'path' : the path of one directory or file found
    }
        """
        self.listView = []
        self.format   = {}

        for onePath in nextFile(
            main    = self.main,
            ext     = self.ext,
            prefix  = self.prefix,
            depth   = self.depth,
            keepDir = self.keepDir
        ):
            if onePath == self.main:
                self.listView.append({
                    'kind' : "directory",
                    'depth': -1,
                    'path' : onePath
                })

            else:
# Which kind of object ?
                if self.__otherFiles(onePath) or isFile(onePath):
                    kind = "file"
                else:
                    kind = "directory"

# The depth
                depth = relativeDepth(self.main, onePath)

# << WARNING ! >> We must take care of directories without any file which
# have the same depth of previous files.
                if kind == "directory" and self.listView:
                    if depth >= self.listView[-1]['depth']:
                        lastParentDir = commonPath([
                            self.listView[-1]['path'],
                            onePath
                        ])

                        i = len(lastParentDir) + 1

                        listSubDirName = onePath[i:].split(SEP)
                        lastDepth      = depth - len(listSubDirName)

                        for oneSubDirName in listSubDirName[:-1]:
                            lastParentDir += SEP + oneSubDirName
                            lastDepth     += 1

                            self.listView.append({
                                'kind' : "directory",
                                'depth': lastDepth,
                                'path' : lastParentDir
                            })

# Let's store the infos found.
                self.listView.append({
                    'kind' : kind,
                    'depth': depth,
                    'path' : onePath
                })

    def __otherFiles(self, onePath):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method displays text indicated that other files are in the directory.
        """
        return onePath.endswith("...")

    def pathToDisplay(
        self,
        onePath,
        kind
    ):
        if self.__otherFiles(onePath):
            return "..."

        elif self.output == "relative":
            if self.main == onePath:
                return onePath[len(parentDir(onePath)) + 1:]
            else:
                return relativePath(self.main, onePath)

        elif self.output == "short":
            if kind == "file":
                return onePath[onePath.rfind(SEP)+1:]

            else:
                return onePath[len(parentDir(onePath))+1:]

        return onePath

    @property
    def ascii(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This property like method returns a simple ASCCI tree view of the tree structure.
        """
        if 'ascii' not in self.format:
            text = []

            for oneInfo in self.listView:
                depth = oneInfo["depth"]

# Does the main directory must be displayed ?
                if self.seeMain:
                    depth += 1

                tab = self.ASCII_DECORATION['tab']*depth

                decoKind = self.ASCII_DECORATION[oneInfo["kind"]] + " "

                pathToDisplay = self.pathToDisplay(
                    oneInfo["path"], oneInfo["kind"]
                )

                text.append(
                    "{0}{1}{2}".format(tab, decoKind, pathToDisplay)
                )

            self.format['ascii'] = '\n'.join(text)

        return self.format['ascii']


# ------------------- #
# -- GENERAL INFOS -- #
# ------------------- #

def pathEnv():
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function simply returns the variable ``PATH`` that contains paths of some
executables known by your OS.
    """
    return os.getenv('PATH')

def system():
    """
:::::::::::::::::
Small description
:::::::::::::::::

The purpose of this function is to give the name, in lower case, of the OS used.
Possible names can be "windows", "mac", "linux" and also "java".
    """
    osName = platform.system()

    if not osName:
        raise OsUseError(
            "The operating sytem can not be found."
        )

    if osName == 'Darwin':
        return "mac"

    else:
        return osName.lower()
