#!/usr/bin/env python3

"""
Directory : mistool
Name      : python_use
Version   : 2013.10
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This module contains some simple tools about the Python programming language.
"""

import gc
import sys
import collections


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class PythonUseError(ValueError):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Base class for errors in the ``python_use`` module of the package ``mistool``.
    """
    pass


# ---------------------- #
# -- FOR DICTIONARIES -- #
# ---------------------- #

def dictSingleValues(obj):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function returns a single value list which contains all the values stored
in one dictionary.
    """
    return list(set(obj.values()))


# ------------- #
# -- QUOTING -- #
# ------------- #

QUOTE_SYMBOLS = ["'", '"']

def __escape__(data):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This small function escapes all the characters that must be escaped in one
python-like string.
    """
    if isinstance(data, str):
        return data.replace('\\', '\\\\')

    else:
        return data

def quote(
    text,
    symbol = "'"
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function put the content of the string variable ``text`` into quotes and
escapes the eventual quote symbols in ``text``.

This function has one optional variable ``symbol`` which indicates the prefered
quoting symbol which is the single quote ``'`` by default.


This function uses the global constant ``QUOTE_SYMBOLS`` which is defined by
``QUOTE_SYMBOLS = ["'", '"']``. This indicates the list of the possible quoting
symbols.

For example, ``quote("one \"small\" example")`` is equal to the text ``'one
"small" example``, and ``quote('another \'example\' to "see" more")``` is equal
to ``'another \'example\' to "see" more'``.
    """
    if all(x in text for x in QUOTE_SYMBOLS):
        text = text.replace(symbol , '\\' + symbol)

    elif symbol in text:
        for x in QUOTE_SYMBOLS:
            if x != symbol:
                symbol = x

    return "{0}{1}{0}".format(symbol, text)


# -------------------------- #
# -- WRITING PYTHON FILES -- #
# -------------------------- #

def lambdify(text):
    return "lambda " + text

def pyRepr(
    obj,
    kind   = None,
    depth  = -1,
    tab    = " "*4,
    indent = 0
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function returns one string representation, usable in ¨python codes, of one
¨python object.


:::::::::::::
The arguments
:::::::::::::

This function uses five variables.

    1) ``obj`` is the ¨python object to print in one ¨python code.

    2) ``kind`` is an optional argument to indicate the kind of object. The
    default value is ``None`` which asks to use the type of the object. The
    supported values are the following ones.

        * ``"bool"``, ``"float"``, ``"int"``, ``"dict"``, ``"orderedDict"``,
        ``"tuple"``, ``"list"``, ``"set"`` and ``"str"`` indeed corresponds to
        ¨python types supported.

        * ``"regex"`` is useful for pattern definied via ``re.compile("...")``.

        * ``"repr"`` simply asks to use the standard ¨python representation.

    3) ``depth`` is an optional argument to indicate how deep to go in the
    printings of dictionaries and lists. The default value is ``(-1)`` which
    indicates to go the deeper is possible.

    4) ``tab`` is an optional argument which is the tabulation used for the
    representation of dictionaries and lists. The default value is ``" "*4``.

    5) ``indent`` is an optional argument that indicates the current level
    of indentation. The default value is ``0``.
    """
# -- INNER FUNCTIONS -- #

    def __isGoodType(wantedType):
        if objType != wantedType:
            messageType = {
                str                    : 'string',
                list                   : 'list',
                tuple                  : 'tuple',
                set                    : 'set',
                dict                   : 'dictionary',
                collections.OrderedDict: 'ordered dictionary'
            }

            wantedType = messageType.get(wantedType, str(wantedType))

            message = 'With << kind = "{0}" >>, the object to represent ' \
                    + 'must be one << {1} >>.'

            raise PythonUseError(
                message.format(kind, wantedType)
            )

    def __reprListLike():
        if depth == 0:
            return repr(obj)

        if kind == "list":
            start, end = "[", "]"

        elif kind == "tuple":
            start, end = "(", ")"

        elif kind == "set":
            start, end = "{", "}"

        if not obj:
            return start + end

        text = []

        for elt in obj:
            elt = pyRepr(
                obj    = elt,
                indent = indent + 1,
                tab    = tab,
                depth  = depth - 1
            )

            text.append(elt + ",")

        text[-1] = text[-1][:-1]

        text.append(tab*indent + end)
        text = [start] + text

        return '\n'.join(text)

    def __reprDict():
        if not obj:
            return "{}"

        try:
            sortedKeys = sorted(obj.keys())

        except:
            sortedKeys = obj.keys()

        if depth == 0:
            text = ['{']

            for key in sortedKeys:
                value = pyRepr(
                    obj   = obj[key],
                    depth = 0
                )

                key = pyRepr(
                    obj    = key,
                    depth  = 0
                )

                text.append("{0}: {1}, ".format(key, value))

            text[-1] = text[-1][:-2]

            text.append('}')

            return ''.join(text)

        else:
            text = []

            for key in sortedKeys:
                value = pyRepr(
                    obj    = obj[key],
                    indent = indent + 1,
                    tab    = tab,
                    depth  = depth - 1
                ).lstrip()

                key = tab*(indent+1) + pyRepr(
                    obj    = key,
                    depth  = 0
                )

                text.append("{0}: {1},".format(key, value))

            text[-1] = text[-1][:-1]

            text.append(tab*indent + '}')
            text = ['{'] + text

            return '\n'.join(text)

    def __reprOrderedDict():
        if not obj:
            return "collections.OrderedDict()"

        if depth == 0:
            text = ['collections.OrderedDict([']

            for key, value in obj.items():
                value = pyRepr(
                    obj   = value,
                    depth = 0
                )

                key = pyRepr(
                    obj    = key,
                    depth  = 0
                )

                text.append("({0}, {1}), ".format(key, value))

            text[-1] = text[-1][:-2]

            text.append('])')

            return ''.join(text)

        else:
            text = []

            for key, value in obj.items():
                text.append(
                    pyRepr(
                        obj    = (key, value),
                        indent = indent + 1,
                        tab    = tab,
                        depth  = depth - 1
                    ) + ","
                )

            text[-1] = text[-1][:-1]

            text.append(tab*indent + '])')
            text = ['collections.OrderedDict(['] + text

            return '\n'.join(text)

# -- ACTIONS -- #

# Special cae of the None class.
    if obj == None:
        return repr(None)

# We have to guess the type of the object.
    objType = type(obj)

    if kind == None:
        if objType in [
            bool,
            float, int,
            list, tuple, set,
            dict, collections.OrderedDict,
            str,
        ]:
            kind = str(objType)
            kind = kind[kind.find("'")+1:]
            kind = kind[:kind.find("'")]

# Very ugly !!!
        elif str(objType) == "<class '_sre.SRE_Pattern'>":
            kind = "regex"

        else:
            raise PythonUseError(
                'Unsupported type "{0}".'.format(objType)
            )

    elif kind == "orderedDict":
        kind = "collections.OrderedDict"

# We do the job.
    if kind == "bool":
        answer = repr(bool(obj))

    elif kind == "str":
        answer = repr(obj)

    elif kind in ["float", "int", "repr"]:
        answer = repr(obj)

    elif kind == "regex":
        if depth == 0:
            answer = 're.compile("{0}")'.format(obj.pattern)

        else:
            answer = '\n'.join([
                're.compile(',
                tab*(indent+1) + '"{0}"'.format(obj.pattern),
                tab*indent + ')'
            ])

    elif kind == "list":
        __isGoodType(list)
        answer = __reprListLike()

    elif kind == "tuple":
        __isGoodType(tuple)
        answer = __reprListLike()

    elif kind == "set":
        __isGoodType(set)
        answer = __reprListLike()

    elif kind == "dict":
        __isGoodType(dict)
        answer = __reprDict()

    elif kind == "collections.OrderedDict":
        __isGoodType(collections.OrderedDict)

        answer = __reprOrderedDict()

    else:
        raise PythonUseError('Unknown kind "{0}".'.format(kind))

    return tab*indent + answer
