#!/usr/bin/env python3

"""
Directory : mistool
Name      : string_use
Version   : 2013.10
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This script contains some useful functions for manipulating strings.
"""

import collections

from mistool import config, regex_use


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class StringUseError(ValueError):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Base class for errors in the ``string_use`` module of the package ``mistool``.
    """
    pass


# ------------- #
# -- REPLACE -- #
# ------------- #

def replace(
    text,
    replacement
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function does replacements in the argument ``text`` using the associations
defined in the dictionary ``replacement``.


warning::
    The function does the replacements sequentially from the longer word to the
    shorter one.


Here is a small example.

python::
    from mistool import string_use

    littleExample = string_use.replace(
            text        = "one, two, three,..."
            replacement = {
                'one'  : "1",
                'two'  : "2",
                'three': "3"
            }
        )
    )


In that code, ``littleExample`` is equal to ``"1, 2, 3,..."``.


info::
    This function has not been build for texts to be replaced that contains some
    other texts to also replace. If you need this kind of feature, take a look
    at the class ``MultiReplace``.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text`` is a string argument corresponding to the text where the
    replacements must be done.

    2) ``replacement`` is a dictionary where each couple ``(key, value)`` is of
    the kind ``(text to find, replacement)``.
    """
    longToShortKeys = sorted(
        replacement.keys(),
        key = lambda t: -len(t)
    )

    for old in longToShortKeys:
        text = text.replace(old, replacement[old])

    return text

PATTERN_GROUP_WORD = regex_use.PATTERN_GROUP_WORD

class MultiReplace:
    """
:::::::::::::::::
Small description
:::::::::::::::::

The purpose of this class is to replace texts that can contain some other texts
to also be replaced. Here is an example of use.

python::
    form mistool import string_use

    myReplace = string_use.MultiReplace(
        replacement = {
            'W1' : "word #1",
            'W2' : "word #2",
            'W12': "W1 and W2"
        },
        pattern = string_use.PATTERN_GROUP_WORD["var"]
    )

    print(myReplace.replace("W1 and W2 = W12"))


Launched in a terminal, the preceding code will produce the following output.

terminal::
    word #1 and word #2 = word #1 and word #2


The only technical thing is the use of ``string_use.PATTERN_GROUP_WORD["var"]``
which is a regex grouping pattern. You can use directly the following patterns
or use your own grouping pattern.

    1) ``string_use.PATTERN_GROUP_WORD["en"]`` is for words only made of ¨ascii
    letters.

    2) ``string_use.PATTERN_GROUP_WORD["fr"]`` is for words only made of ¨ascii
    letters and the special letters "â", "à", "é", "è", "ê", "ë", "î", "ï", "ô",
    "ù", "ü", and "ç".

    3) ``string_use.PATTERN_GROUP_WORD["var"]`` is for words only starting with
    one ¨ascii letter followed eventually by other ¨ascii letters, digits and
    underscores.


warning::
    Before doing the replacements in a text, the class first build the
    dictionary ``self.directReplacement`` for which the replacements has been
    changed so to not contain text to be replaced. With the preceeding code,
    this dictionary is equal to the following one.

    python::
        {
            'W1' : "word #1",
            'W2' : "word #2",
            'W12': "word #1 and word #2"
        }


info::
    In the following dictionnary defining replacements, there are cyclic
    definitions. In that case, the class will raise an error.

    python::
        replacement = {
            'WRONG_1': "one small text and  WRONG_2",
            'WRONG_2': "one small text, and then WRONG_3",
            'WRONG_3': "with WRONG_1, there is one problem here"
        }


:::::::::::::
The arguments
:::::::::::::

The instanciation of this class uses the following variables.

    1) ``replacement`` is a dictionary where each couple ``(key, value)`` is of
    the kind ``(text to find, replacement)``. Here the replacements can contain
    also text to be found.

    2) ``pattern`` is a regex grouping pattern indicating the kind of words to
    be replaced. By default, ``pattern = PATTERN_GROUP_WORD["en"]`` where
    ``PATTERN_GROUP_WORD`` is a renaming of ``regex_use.PATTERN_GROUP_WORD``
    (see the module ``regex_use`` for more informations).
    """

    def __init__(
        self,
        replacement,
        pattern = PATTERN_GROUP_WORD["en"]
    ):
        self.pattern           = pattern
        self.replacement       = replacement
        self.directReplacement = None

        self.__crossReplace = None
        self.__oldWords     = None

        self.__update()

    def __update(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method simply launches methods so as to verify that there is no cyclic
replacements, and then to build the direct replacements dictionary
``self.directReplacement``.
        """
        self.__crossReplace = {}

        self.__noCycle()

        self.__replaceInReplacement()

    def __noCycle(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method verifies that there is no cyclic replacements.

Indeed all the job is done by the hidden method ``self.__viciousCircles``.
        """
# Building the crossing replacements.
        self.__oldWords = list(self.replacement.keys())

        for old, new in self.replacement.items():
            self.__crossReplace[old] = [
                x for x in self.pattern.findall(new)
                if x in self.__oldWords
            ]

        self.__viciousCircles()

    def __viciousCircles(
        self,
        visitedWord = [],
        nextWord    = None
    ):
        if nextWord == None:
            nextWord = self.__oldWords

        for old in nextWord:
            oldInNew = self.__crossReplace[old]

            if old in oldInNew:
                raise StringUseError(
                    "<< {0} >> is used in its associated replacement." \
                        .format(old)
                )

            elif old in visitedWord:
                pos = visitedWord.index(old)

                visitedWord = visitedWord[pos:]
                visitedWord.append(old)
                visitedWord = [""] + visitedWord

                raise StringUseError(
                    "The following viscious circle has been found." \
                    + "\n\t + ".join(visitedWord)
                )

            else:
                for new in oldInNew:
                    self.__viciousCircles(
                        visitedWord = visitedWord + [old],
                        nextWord    = self.__crossReplace[old]
                    )

    def __replaceInReplacement(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds ``self.directReplacement`` the direct replacements dictionary.

Indeed all the job is done by the hidden method ``self.__replace``.
        """
        self.directReplacement = {}

        for old, new in self.replacement.items():
            self.directReplacement[old] = self.__replace(new)

    def __replace(
        self,
        text
    ):
        if not text:
            return text

        newText = ""

        while True:
# Source :
#    http://www.developpez.net/forums/d958712/autres-langages/python-zope/general-python/amelioration-split-evolue/#post5383767
            newText = self.pattern.sub(self.__apply, text)

            if newText == text:
                return text

            else:
                text = newText

    def __apply(
        self,
        match
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method is used to does the replacements corresponding the matching groups.
        """
        return self.replacement.get(match.group(1), match.group(0))

    def replace(
        self,
        text
    ):
        """
This method does the replacements in one text (indeed it simply calls the
function ``replace`` with the attribut ``self.directReplacement``).
        """
        return replace(
            text        = text,
            replacement = self.directReplacement
        )


# ------------------------------------- #
# -- ASCII TRANSLATION OF AN UTF TEXT-- #
# ------------------------------------- #

REPLACEMENT_ASCII = config.ascii.REPLACEMENT_ASCII
ASCII_CHAR        = config.ascii.ASCII_CHAR

def ascii(
    text,
    strict = True
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

The aim of this function is to give an ¨ascii translation of a text. The typical
use is for avoiding strange names of files. For example, the following lines of
code launched in a terminal will display the text ``Viva Espana!``.

python::
    from mistool import string_use

    print(
        string_use.ascii("¡Viva España!")
    )


You can change the constant ``REPLACEMENT_ASCII`` like in the following lines so
as to simply obtain ``Viva Espana`` rather than ``Viva Espana!``.

python::
    from mistool import string_use

    string_use.REPLACEMENT_ASCII['!'] = ""

    print(
        string_use.ascii("¡Viva España!")
    )


It's easy to improve the special characters managed by default.

    1) The first method can be used for very special tunings like in the
    preceding code : just add keys with their ¨ascii value to the dictionary
    ``string_use.REPLACEMENT_ASCII``.

    2) If you think that your tuning is general enough, just send to the author
    of this module a text file of the following kind wich contains the special
    characters with their ¨ascii translation (don't use ``code::``).

    code::
        a::
            à ä â

        empty::
            ¡ ¿


    This code uses the ¨peuf specifications of the package ¨orpyste. Here we
    indicate two things.

        1) The characters ``à``, ``ä`` and ``â`` must be translated to ``a``.

        2) The characters ``¡`` and ``¿`` must be removed.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text`` is simply the text to be translated.

    2) ``strict`` is an optional argument to use if you don't want the function
    to raise an error when the translation has been partial.

    By default, ``strict = True`` indicates to raise an error.
    """
    asciiText = replace(
        text        = text,
        replacement = REPLACEMENT_ASCII
    )

    if strict:
        extraChar = set(asciiText) - ASCII_CHAR

        if extraChar:
            if len(extraChar) == 1:
                plurial = ""

            else:
                plurial = "s"

            extraChar = " , ".join([
                "<< {0} >>".format(x) for x in extraChar
            ])

            raise StringUseError(
                "The following character{0} can't be translated : {1} ."\
                    .format(plurial, extraChar)
            )

    return asciiText


# ----------------- #
# -- LOOKING FOR -- #
# ----------------- #

# Source : http://www.developpez.net/forums/d921494/autres-langages/python-zope/general-python/naturel-chaine-stockage

class AutoComplete:
    """
:::::::::::::::::
Small description
:::::::::::::::::

The aim of this class is to ease auto completions. Here is an example.

python::
    from mistool import string_use

    myAC = string_use.AutoComplete(
        words = [
            "article", "artist", "art",
            "when", "who", "whendy",
            "bar", "barbie", "barber", "bar"
        ]
    )

    print(myAC.matching("art"))

    print('---')

    print(myAC.matching(""))


Launched in a terminal, the preceding code will produce something similar to the
following output.

terminal::
    ['article', 'artist']
    ---
    [
        'art', 'article', 'artiste',
        'bar', 'barbie', 'barber',
        'when', 'whendy', 'who'
    ]


The method ``matching`` simply gives all the words starting with the prefix
given. If the prefix is empty, the matching words are all the words defining the
auto completion.


The search indeed uses a special "magical" dictionary which is stored in the
attribut ``dict``. With the preceding example, ``myAC.dict`` is equal to the
dictionary above where the lists of integers correspond to the good indexes in
the ordered list of words.

python::
    {
        'words': [
    # A
            'art', 'article', 'artist',
    # B
            'bar', 'barber', 'barbie',
    # W
            'when', 'whendy', 'who'
        ],
        'completions': {
    # A
            'a'     : [0, 3],
            'ar'    : [0, 3],
            'art'   : [1, 3],
            'arti'  : [1, 3],
            'artic' : [1, 2],
            'artis' : [2, 3],
            'articl': [1, 2],
    # B
            'b'    : [3, 6],
            'ba'   : [3, 6],
            'bar'  : [4, 6],
            'barb' : [4, 6],
            'barbe': [4, 5],
            'barbi': [5, 6],
    # W
            'w'    : [6, 9],
            'wh'   : [6, 9],
            'whe'  : [6, 8],
            'when' : [7, 8],
            'whend': [7, 8]
        }
    }


You can directly give this dictionary like in the following fictive example.
This can be very useful when you always use the same list of words : just ask
one time to the class to build the "magical" dictionary, by giving the fixed
list of words just one time, and then store this dictionary to reuse it later
(you can use the function ``pyRepr``  of the module ``python_use`` to hard store
the dictionary).

python::
    myAC = string_use.AutoComplete(
        dictAsso = myMagicDict  # Previously build and stored somewhere.
    )


There is two other useful methods (see their docstrings for more informations).

    1) ``build`` simply builds the "magical" dictionary. This method can be used
    for local updating of the list of words used for the auto completion.

    2) ``missing`` simply gives the letters remaining after one prefix in a
    word.


:::::::::::::
The arguments
:::::::::::::

The instanciation of this class uses the following variables.

    1) ``words`` is the list of words to use for the auto completions.

    2) ``depth`` is the minimal size of the prefix used to look for the auto
    completions. By default, ``depth = 0`` which indicates to start the auto
    completion with the first letter.

    infoo:
        `3` seems to be a good value of ``depth`` for ¨gui application.

    3) ``dictAsso`` is a "magical" dictionary that eases the auto completion.
    This dictionary is build by the method ``build`` from the list of words to
    be used for the auto completions.
    """

    def __init__(
        self,
        words    = None,
        dictAsso = None,
        depth    = 0
    ):
        if words == None and dictAsso == None:
            raise StringUseError(
                "You must give either the value of ``words`` or ``dictAsso``."
            )

        self.words = words
        self.dict  = dictAsso
        self.depth = depth

# We have to build the magical dictionary.
        if self.dict == None:
            self.build()

    def build(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds the "magical" dictionary that will ease the auto completions.
Indeed, if you create a class ``AutoComplete`` without giving a "magical"
dictionary ``dict``, the method ``build`` is automatically called.


info::
    The idea of the magical dictionary comes from cf::``this discusion ;
    http://www.developpez.net/forums/d921494/autres-langages/python-zope/general-python/naturel-chaine-stockage``
        """
# Sorted list of single words.
        shortSortedList = list(set(self.words))
        shortSortedList = sorted(shortSortedList)

# Maximum depth.
        depth = self.depth

        if depth == 0:
            for oneWord in shortSortedList:
                depth = max(depth, len(oneWord) - 1)

# Let's build the magical dictionary.
        self.dict = {
            'words'      : shortSortedList,
            'completions': {}
        }

        for nbWord, oneWord in enumerate(shortSortedList):
            maxSize = min(depth, len(oneWord))

            for i in range(maxSize):
                prefix = oneWord[:i+1]

                if prefix != shortSortedList[nbWord]:
                    if prefix in self.dict['completions']:
                        self.dict['completions'][prefix][1] = nbWord + 1

                    else:
                        self.dict['completions'][prefix] = [nbWord, nbWord + 1]

    def matching(
        self,
        prefix
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method looks for words given in the list ``words`` that start with the
string variable ``prefix``.
        """
        if prefix.strip() == '':
            return self.dict['words']

        if prefix in self.dict['completions']:
            first, last = self.dict['completions'][prefix]

            return self.dict['words'][first: last]

    def missing(
        self,
        prefix,
        word
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

Giving a word ``"prefixExample"`` and one prefix ``"pre"``, this method will
return simply ``"fixExample"``.


:::::::::::::
The arguments
:::::::::::::

This method uses the following variables.

    1) ``prefix`` is a string corresponding to the prefix expected.

    2) ``word`` is a string where the prefix ``prefix`` must be removed.
        """
        if not word.startswith(prefix):
            raise StringUseError(
                "The word << {0} >> does not star with the prefix << {1} >>."\
                    .format(word, prefix)
            )

        return word[len(prefix):]


# -------------------- #
# -- JOIN AND SPLIT -- #
# -------------------- #

AND_TEXT = "and"

def joinAnd(
    texts,
    andText = None
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function joins texts given in the argument ``texts`` using coma as
separator excepted for the list piece of text which wil be preceded by default
by "and".

Here is a small example.

python::
    from mistool import string_use

    littleExample = string_use.joinAnd(["one", "two", "three"])


In that code, ``littleExample`` is equal to ``one, two and three"``.


You can change the text "and". There is two ways to do that.

    1) **Local change :** the function has on optional argument ``andText``
    which is the string value used before the last piece of text.

    2) **Global change :** ``AND_TEXT`` is the global constant to use so as to
    change the default text used before the last piece of text if the optional
    argument ``andText`` is not used when calling the function.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``texts`` is a list of texts.

    2) ``andText`` is the text used betwen the two last texts. By default,
    ``andText = None`` which indicates to use the global constant ``AND_TEXT``
    which is equal to ``"and"`` by default.
    """
    if len(texts) == 1:
        return texts[0]

    if andText == None:
        andText = AND_TEXT

    return ", ".join(texts[:-1]) + " " + andText + " " + texts[-1]

def split(
    text,
    sep,
    escape = "",
    strip  = False
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function allows to split a text using a list of separators, and not only a
single separator. Here is an example.

python::
    from mistool import string_use

    splitText = string_use.split(
        text  = "p_1 ; p_2 ; p_3 | r_1 ; r_2 | s",
        sep   = ["|", ";"],
        strip = True
    )

In this code, the variable ``splitText`` is equal to the following list :
``['p_1', 'p_2', 'p_3', 'r_1', 'r_2', 's']``.


You can escape separators like in the following example that also uses the
possibility to give a single string instead of a one value list.

python::
    from mistool import string_use

    splitText = string_use.split(
        text   = "p_1 \; p_2 ; p_3",
        sep    = ";",
        escape = "\\",
        strip  = True
    )

In this code, the variable ``splitText`` is equal to the following list :
``['p_1 \; p_2', 'p_3']``.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text`` is simply the text to split.

    2) ``sep`` is either a string for a single separator, or a list of
    separators.

    3) ``escape`` is a string used to escape the separators. By default,
    ``escape = None`` which indicates that there is no escaping feature.

    4) ``strip`` is a boolean variable to strip, or not, each pieces of text
    found. By default, ``strip = False``.
"""
# No separator
    if not sep:
        if strip:
            text = text.strip()

        return [text]

# At least one separator
    if isinstance(sep, str):
        sep = [sep]

    if not isinstance(sep, list) \
    and not isinstance(sep, tuple):
        raise StringUseError(
            "The variable << sep >> must be a list or tuple of strings."
            "\n\t{0}".format(sep)
        )

    for oneSep in sep:
        if not isinstance(oneSep, str):
            raise StringUseError(
                "The variable << sep >> must be list or tuple of strings."
                "\n\t{0}".format(sep)
            )

# << Warning ! >> We must sort the opening symbolic tags from the longer one
# to the shorter.
    sorted(sep, key = lambda x: -len(x))

    answer = []
    iMax   = len(text)
    i      = 0
    iLast  = i

    while(i < iMax):
# The bigger is the winner !
        for oneSep in sep:
            if text[i:].startswith(oneSep):
                lastPiece = text[iLast: i]

                if not(escape and lastPiece.endswith(escape)):
                    i    += len(oneSep)
                    iLast = i

                    answer.append(lastPiece)

                    break

        i += 1

    remainingText = text[iLast:]

    if remainingText:
        answer.append(remainingText)

    if strip:
        answer = [x.strip() for x in answer]

# The job has been done !
    return answer

class _SplitKind:
    """
:::::::::::::::::
Small description
:::::::::::::::::

This class is simply an object like class used by the method ``__iter__`` of
the class ``MultiSplit``.
    """

    def __init__(
        self,
        type,
        val
    ):
        self.type = type
        self.val  = val

class MultiSplit:
    """
:::::::::::::::::
Small description
:::::::::::::::::

The purpose of this class is to split a text at different levels. You can use
either a list view version of the split text, or work with a special iterator.


Here is an example of the list view version of the split text.

python::
    from mistool import string_use

    SplitText = string_use.MultiSplit(
        text = "p_1 , p_2 ; p_3 | r_1 ; r_2 | s",
        sep  = [
            "|",
            (";", ",")
        ],
        strip = True
    )

    listView = SplitText.listView


In this code, the variable ``listView`` is equal to the following list. There
is as many level of lists that the length of the list ``sep``. The use of
``(";", ",")`` simply indicates that the separators ``;`` and ``,`` have the
same importance.

python::
    [
        ['p_1', 'p_2', 'p_3'],
        ['r_1', 'r_2'],
        ['s']
    ]


Let see with an example how to use the instance as an iterator so as to ease
the walk in the split text.

python::
    from mistool import string_use

    SplitText = string_use.MultiSplit(
        text  = "p_1 , p_2 ; p_3 | r_1 ; r_2 | s",
        sep   = ["|", ";", ","],
        strip = True
    )

    for x in SplitText:
        print("{0} ---> {1}".format(x.type, x.val))


Launched in a terminal, the code will produce the following output that shows
how to work easily with the iterator of the class ``MultiSplit``. You have to
know that the type is a string, and also that for the "sep" type, the associated
value is equal to the separator used when the instance has been created.

terminal::
    sep ---> |
    sep ---> ;
    sep ---> ,
    list ---> ['p_1', 'p_2']
    sep ---> ;
    sep ---> ,
    list ---> ['p_3']
    sep ---> |
    sep ---> ;
    sep ---> ,
    list ---> ['r_1']
    sep ---> ;
    sep ---> ,
    list ---> ['r_2']
    sep ---> |
    sep ---> ;
    sep ---> ,
    list ---> ['s']


:::::::::::::
The arguments
:::::::::::::

The instanciation of this class uses the following variables.

    1) ``text`` is simply the text to split.

    2) ``sep`` is either a string for a single separator, or a list from the
    stronger separator to the weaker one. You can use list or tuple to gather
    separators having the same level of priority.

    3) ``escape`` is a string used to escape the separators. By default,
    ``escape = None`` which indicates that there is no escaping feature.

    4) ``strip`` is a boolean variable to strip, or not, each pieces of text
    found. By default, ``strip = False``.
"""

    def __init__(
        self,
        text,
        sep,
        escape = "",
        strip  = False
    ):
        self.text   = text
        self.sep    = sep
        self.strip  = strip
        self.escape = escape

        if isinstance(self.sep, str):
            self.sep = [self.sep]

        self.listView = self.build()

    def build(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds the list view version of the split text.

Indeed all the job is done by the hidden method ``self.__build``.
        """
        return self.__build(
            text   = self.text,
            sep    = self.sep,
            escape = self.escape,
            strip  = self.strip
        )

    def __build(
        self,
        text,
        sep,
        escape,
        strip
    ):
# No separator
        if not sep:
            if strip:
                text = text.strip()

            answer = [text]

# At least one separator
        else:
            answer = split(
                text   = text,
                sep    = sep[0],
                escape = escape,
                strip  = strip
            )

            remainingSep = sep[1:]

            if remainingSep:
                answer = [
                    self.__build(
                        text   = onePiece,
                        sep    = remainingSep,
                        escape = escape,
                        strip  = strip
                    )
                    for onePiece in answer
                ]

# The job has been done !
        return answer

    def __iter__(self):
        return self.__iter(listView = self.listView)

    def __iter(
        self,
        listView,
        depth = 0
    ):
        if listView:
            if isinstance(listView[0], str):
                yield _SplitKind(
                    type = "sep",
                    val  = self.sep[depth]
                )

                yield _SplitKind(
                    type = "list",
                    val  = listView
                )

            else:
                for x in listView:
                    yield _SplitKind(
                        type = "sep",
                        val  = self.sep[depth]
                    )

                    for y in self.__iter(
                        listView = x,
                        depth    = depth + 1
                    ):
                        yield y

def beforeAfter(
    text,
    start,
    end,
    keep = False
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

The function will look for the piece of text between the first texts ``start``
and ``end`` found in ``text``. Then by default the function returns the text
before the text ``start`` and the one after the text ``end`` in a couple.

If ``keep = True`` then the text ``start`` will be added at the end of the first
text returned, and the second text returned will begin with the text ``end``.

You can also use ``keep = (True, False)`` or ``keep = (True, False)`` so as to
use just one half of the features which have just been explained. Indeed ``keep
= True`` and ``keep = False`` are shortcuts for ``keep = (True, True)`` and
``keep = (False, False)`` respectively.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text`` is a string argument corresponding to the texte where the search
    must be done.

    2) ``start`` and ``end`` are string arguments that indicated the start and
    end pieces of text to find.

    3) ``keep`` is an optional argument which can be either a boolen or a couple
    of boolean. By default, ``keep = False`` which indicates to unkeep the start
    and the end in the piece of text found.
    """
    if start == "" and end == "":
        raise StringUseError(
            'The variables << start >> and  << end >> can not be both empty.'
        )

    if isinstance(keep, bool):
        keepStart = keep
        keepEnd   = keep

    else:
        keepStart, keepEnd = keep

    if start == "":
        s = 0

    else:
        s = text.find(start)

        if s == -1:
            raise StringUseError(
                'The starting text << {0} >> has not been found.'.format(start)
            )

        s += len(start)

    if end == "":
        e = s

    else:
        e = text.find(end, s)

        if e == -1:
            message = 'The ending text  << {1} >> has not been found after ' \
                    + 'the first text << {0} >>.'

            raise StringUseError(
                message.format(start, end)
            )

        if start == "":
            s = e


    if not keepStart and start:
        s -= len(start)

    if not keepEnd and end:
        e += len(end)


    return (text[:s], text[e:])


# ------------------------ #
# -- LOOKING FOR GROUPS -- #
# ------------------------ #

class FindGroups:
    """
:::::::::::::::::
Small description
:::::::::::::::::

----------------------------------------
Short overview trough one simple example
----------------------------------------

Let's consider the text ``"A(B)C[D(E)F]"`` where there are well balanced groups
using parenthesis and hooks. The class ``FindGroups`` allows to have easily the
following ascii tree representing the groups of braces.

terminal::
    + A
    + ( ... )
        + B
    + C
    + [ ... ]
        + D
        + ( ... )
            + E
        + F
    + G


This tree has been obtained using the following code where the argument
``groups`` is the list of groups defined by giving an tuple of the kind
``(opening delimiter, closing delimiter)``.

python::
    from mistool import string_use

    myAnalyze = string_use.FindGroups(
        text   = "A(B)C[D(E)F]",
        groups =  [
            ("(", ")"),
            ("[", "]")
        ]
    )

    print(myAnalyze.ascii)


There is an efficient way to walk in the tree. The recursive function ``walkin``
above shows how to do that kind of thing. In this example, we build by hand
an ascii tree representing the groups found.

python::
    from mistool import string_use

    myAnalyze = string_use.FindGroups(
        text    = 'A(B)C[D(E)F]',
        groups  = [("[", "]"), ("(", ")")]
    )

    def walkin(groupView, depth = 0):
        deco = " "*depth + "* "

        for kind, delim, content in groupView:
            if kind == 'text':
    # In that case, ``delim = []``.
                print(deco + content)

            elif kind == 'group':
                open, close = delim
                print(deco + "{0} <---> {1}".format(open, close))

    # Let's use a recursive call.
                walkin(groupView = content, depth = depth + 4)

    walkin(myAnalyze.groupView)


Launched in a terminal, the code will print the following tree.

terminal::
    * A
    * ( <---> )
        * B
    * C
    * [ <---> ]
        * D
        * ( <---> )
            * E
        * F


Indeed, the class ``FindGroups`` always build first a list view which is stored
in the attribut ``listView``. For example, by changing ``myAnalyze.ascii``
to ``myAnalyze.listView`` in the first python code, we obtain the following
output.

terminal::
    [
        ('text', 'A', 0),
        ('open', '(', 0),
        ('text', 'B', 1),
        ('close', ')', 0),
        ('text', 'C', 0),
        ('open', '[', 0),
        ('text', 'D', 1),
        ('open', '(', 1),
        ('text', 'E', 2),
        ('close', ')', 1),
        ('text', 'F', 1),
        ('close', ']', 0)
    ]


In this list, each tuple uses the specification ``(kind, text, depth)``.

    1) ``kind`` can be ``"text"`` indicating a simple text, ``"open"`` for an
    opening delimiter, or ``"close"`` for a closing delimiter.

    2) ``text`` is simply a content or a string corresponding to a delimiter.

    3) ``depth`` indicates the depth regarding to the groups : ``0`` is for
    something out of any group, ``1`` indicates something directly inside one
    group and so on...



-------------------------
Defining one single group
-------------------------

In this well braced expression ``A(B)C(D(E)F)G``, we want only to look for one
kind of groups so as to obtain the following tree.

terminal::
    + A
    + ( ... )
        + B
    + C
    + ( ... )
        + D
        + ( ... )
            + E
        + F
    + G


Instead of using ``groups = [...]``, when you have only one kind of groups, you
can directly give it like in the code above used to obtain the preceding tree.

python::
    from mistool import string_use

    myAnalyze = string_use.FindGroups(
        text   = "A(B)C(D(E)F)G",
        groups = ("(", ")")
    )

    print(myAnalyze.ascii)


warning::
    To indicate quickly one single group with different delimiters, you **must
    use a tuple** and not a list because ``groups`` is indeed in general a list
    of groups.



---------------------------------------------------
Groups with the same opening and closing delimiters
---------------------------------------------------

In the following code, we use a short version for indicating groups delimited by
quotes. Instead of ``('"', '"')``, we can simply use ``'"'`` (here we also use
the short version for defining one single group without using a list).

python::
    from mistool import python_use, string_use

    myAnalyze = string_use.FindGroups(
        text   = 'A:"BCD" et "F"',
        groups = '"'
    )

    print(myAnalyze.ascii)


Launched in a terminal, the code will produce the expected output.

terminal::
    + A:
    + " ... "
        + BCD
    +  et
    + " ... "
        + F



-------------------
Escaping delimiters
-------------------

It can be useful to have the possibility to escape delimiters. The code above allows to use escaped quotes so as to not see them as delimiters.

python::
    from mistool import string_use

    myAnalyze = string_use.FindGroups(
        text   = 'A:"BCD\\" et \\"F"',
        groups = '"',
        escape = '\\'
    )

    print(myAnalyze.ascii)


The associated ascii tree is this one.

terminal::
    + A:
    + " ... "
        + BCD" et "F


You can choose which groups can be escaped. Let's consider for example the text
``A\(B\)C\"D\"E:"F"`` where we only want to escape the quotes. This corresponds
to the following ascii tree.

terminal::
    + A\
    + ( ... )
        + B\
    + C"D"E:
    + " ... "
        + F


To obtain this kind of feature, you have to use a code like the one above where
instead of using the `2`-tuple ``("(", ")")``, we use the `3`-tuple `` ("(", ")",
False)`` where the additional third boolean value indicate to not escape the
delimiters ``(`` and ``)``.

python::
    from mistool import string_use

    myAnalyze = string_use.FindGroups(
        text   = 'A\\(B\\)C\\"D\\"E:"F"',
        groups = [
            '"',
            ("(", ")", False)
        ],
        escape = '\\'
    )

    print(myAnalyze.ascii)



-----------------------------
Ignoring groups inside others
-----------------------------

In the text ``A("B")"C(D(E)F)"G``, we want that each characters between quotes
are meaningless. More precisely, we want to seek for parenthesis and quotes, but
between quotes the parenthesis must be ignored. This can be done using the lines
of code above.

python::
    from mistool import string_use

    myAnalyze = string_use.FindGroups(
        text   = 'A("B")"C(D(E)F)"G',
        groups = [
            '"',
            ("(", ")")
        ],
        ignore = {'"': "("}
    )

    print(myAnalyze.ascii)


The corresponding output is the following one.

python::
    + A
    + ( ... )
        + " ... "
            + B
    + " ... "
        + C(D(E)F)
    + G


In the assignation ``ignore = {'"': "("}``, we use one dictionary with the
following rules to respect.

    1) The keys are any kind of supported definitions of one group. For example,
    we can use ``ignore = {("(", ")"): ...}`` or even ``ignore = {("(", ")",
    False): ...}``.

    2) The values can be either a single string indicating an opening delimiter,
    or any kind of supported definitions of one group like ``("(", ")")``, or
    ``("(", ")", False)``, or better a list of the value of one of the two
    preceding kinds.


The flexibility in the definitions of the dictionary ``ignore`` allows to use
efficient codes like the following one.

python::
    from mistool import string_use

    quoteGroup = '"'
    parGroup   = ("(", ")")

    myAnalyze = string_use.FindGroups(
        text   = 'A("B")"C(D(E)F)"G',
        groups = [quoteGroup, parGroup],
        ignore = {quoteGroup: parGroup}
    )

    print(myAnalyze.ascii)


info::
    You can ask to ignore for example parenthesis inside parenthesis so as to
    only allow one depth parenthesis.


You can easily define more complex rules. Suppose for example that in the text
``A(B[C"D"E]F)G[H(I"J")]K"L[M(N)]"O``, we want to ignore each kind of groups,
parenthesis, hooks and quotes, in any other kind of groups. This can be done
using dictionary comprehension like in the code above.

python::
    from mistool import string_use

    quoteGroup = '"'
    parGroup   = ("(", ")")
    hookGroup  = ("[", "]")

    allGroups = [quoteGroup, parGroup, hookGroup]

    myAnalyze = string_use.FindGroups(
        text   = 'A(B[C"D"E]F)G[H(I"J")]K"L[M(N)]"O',
        groups = allGroups,
        ignore = {x: allGroups for x in allGroups}
    )

    print(myAnalyze.ascii)


This produces the ascii tree expected.

terminal::
    + A
    + ( ... )
        + B[C"D"E]F
    + G
    + [ ... ]
        + H(I"J")
    + K
    + " ... "
        + L[M(N)]
    + O



------------------------------
Excluding groups inside others
------------------------------

In the text ``A("B")"C(D(E)F)"G``, we do not want to allow the use of parenthesis
between quotes. You can do by using the argument ``exclude`` which follows the
same rules for its value that the argument ``ignore``. The following code will
raise one exception ``StringUseError`` indicating the illegal use of parenthesis
inside quotes.

python::
    from mistool import string_use

    myAnalyze = string_use.FindGroups(
        text    = 'A("B")"C(D(E)F)"G',
        groups  = ['"', ("(", ")")],
        exclude = {'"': "("}
    )

    print(myAnalyze.ascii)

This code will raise the following error message where the symbol ``^`` indicates
where the problem has been detected.

terminal::
    The group << ( ... ) >> is not allowed inside the group << " ... " >>.
    A("B")"C(D(E)F)"G
            ^


--------------------
The method ``build``
--------------------

You can each time you want the attribut ``text`` which stores the text to
analyze. If you do that, you will have to explicitly call the method ``build``
to force the analysis of the text.

python::
    from mistool import string_use

    myAnalyze = string_use.FindGroups(
        text    = 'A("B")"C"D',
        groups  = ['"', ("(", ")")]
    )

    print(myAnalyze.ascii)

    print('-------------')

    myAnalyze.text = 'A("B")'
    myAnalyze.build()

    print(myAnalyze.ascii)


This will produces the following two ascii trees.

terminal::
    + ( ... )
        + " ... "
            + B
    + " ... "
        + C
    + D
    ---
    + A
    + ( ... )
        + " ... "
            + B


info::
    The next section gives a better way to achieve the analysis of several texts
    using the same kinds of groups.



--------------------------------------------
Reusing directly the internal specifications
--------------------------------------------

When you instantiate the class ``FindGroups``, you use relatively easy to define
specifications via the arguments ``groups``, ``escape``, ``ignore`` and
``exclude``, but internally the class uses the ugly dictionary ``spec`` which is
an attribut of the class.

You can directly give this dictionary like in the following fictive example. This
can be very useful when you always use the same groups : just ask one time to the
class to build the ugly dictionary, by giving an empty string and the arguments
``groups``, ``escape``, ``ignore`` and ``exclude`` just one time, and then store
the dictionary to reuse it later (you can use the function ``pyRepr`` of the
module ``python_use`` to hard store the dictionary).

python::
    MyGroups      = string_use.FindGroups
    MyGroups.spec = mySpec  # Previously build and stored somewhere.

    oneAnalyze = MyGroups(text = '...')

    print(oneAnalyze.ascii)



:::::::::::::
The arguments
:::::::::::::

The instanciation of this class uses the following variables.

    1) ``text`` is simply the text to analyze.

    2) ``groups`` is an optional argument which can be either a single value, or
    a list of values of the following types.

        a) A single string, like ``'"'``, indicates a groups escapable having the
        same opening and closing delimiters.

        b) A 2-tuple ``(open, close)`` where ``open`` and ``close`` are strings
        indicating the opening and closing delimiters. In that case, the
        delimiters are escapable.

        c) Either ``('"', False)``, or ``(open, close, False)`` allows to
        indicates that some delimiters are not escapable.

    By default, ``groups = []``.

    3) ``escape`` is an optional string used to escape delimiters. The default
    value is ``""`` which indicates that there is not escaping sequence of
    characters.

    4) ``ignore`` is an optional dictionary indicating which groups must be
    ignored in others groups. The keys of this dictionary can be any supported
    kind of definition for one single group. The values can be either one single
    group, or a list of groups.

    The meaning of ``ignore`` is the following one.

    python::
        {
            one single group G : the groups to ignore inside G.
        }

    By default, ``ignore = {}``.

    5) ``exclude`` is an optional dictionary following exactly the same rules as
    the dictionary ``ignore``. The meaning of ``exclude`` is the following one.

    python::
        {
            one single group G : the groups forbidden inside G.
        }

    By default, ``exclude = {}``.
    """
    spec = None

    def __init__(
        self,
        text,
        groups  = [],
        escape  = "",
        ignore  = {},
        exclude = {},
    ):
# Arguments
        self.text    = text
        self.groups  = groups
        self.escape  = escape
        self.ignore  = ignore
        self.exclude = exclude

# Customisable constants.
        self.asciiTreeTab  = " "*4
        self.asciiTreeNode = "+"

# Internal constants.
        self.__lenText   = len(self.text)
        self.__lenEscape = len(self.escape)

        self.__listView  = None
        self.__asciiTree = None
        self.__groupView = None

        self.__delimStack = []

# Building of the internal infos about the groups.
        self.build()


# -- INTERNAL SPECIFICATIONS -- #

    def __groupToList(
        self,
        groups
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method simply build, if it is necessary, a list version of the argument
``groups``.
        """
        if not isinstance(groups, list):
            return  [groups]

        __groups = []

        for oneGroup in groups:
            if isinstance(oneGroup, str):
                oneGroup = [oneGroup]

            __groups.append(oneGroup)

        return __groups

    def __normalizeGroupInfo(
        self,
        groupInfo
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method allows to use different formats for indicating one single group by
normalizing all the different formats supported.
        """
        delim    = []
        escapeIt = []

        for oneInfo in groupInfo:
            if isinstance(oneInfo, str):
                delim.append(oneInfo)

            elif isinstance(oneInfo, bool):
                escapeIt.append(oneInfo)

            else:
                raise StringUseError(
                    "Illegal value for the argument ``groups``:"
                    "\n<< {0} >>".format(groupInfo)
                )

        if not(0 < len(delim) < 3) \
        or len(escapeIt) > 1:
            raise StringUseError(
                "Illegal value for the argument ``groups``:"
                "\n<< {0} >>".format(groupInfo)
            )

        if len(delim) == 1:
            delim.append(delim[0])

        if len(escapeIt) == 0:
            escapeIt = True

        else:
            escapeIt = escapeIt[0]

        return [tuple(delim), escapeIt]

    def __buildExcludeIgnoreDict(
        self,
        oneDict,
        kind
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds the internal versions of the infos given in the arguments ``exclude`` and ``ignore`` used during the creation of the class.
        """
        answer = {}

        for oneGroup, groupToExlude in oneDict.items():
            delimOpen = self.__normalizeGroupInfo(oneGroup)[0][0]

            if not delimOpen in self.spec['open']:
                raise StringUseError(
                    "Illegal value for a key in the argument ``{0}``:"
                    "\n<< {1} >>".format(kind, oneGroup)
                )

            for oneGroupToExlude in self.__groupToList(groupToExlude):
                if isinstance(oneGroupToExlude, str):
                    openToExlude = oneGroupToExlude
                    closeToExlude = self.spec['open'][openToExlude]

                else:
                    openToExlude, closeToExlude \
                    = self.__normalizeGroupInfo(oneGroupToExlude)[0]

                ignoreList = answer.get(delimOpen, [])

                ignoreList.append(openToExlude)
                ignoreList.append(closeToExlude)

                if isinstance(delimOpen, tuple):
                    delimOpen = delimOpen[0]

                answer[delimOpen] = ignoreList

        return answer

    def build(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method build the internal representations of the informations given in the
arguments used during the creation of the class.
        """
# We have to erase the last views.
        self.__listView  = None
        self.__asciiTree = None
        self.__groupView = None

# Nothing has to be done.
        if self.spec != None:
            return None

# We have to work...
        self.spec = {
            'open'   : {},
            'close'  : {},
            'all'    : [],
            'escape' : {},
            'ignore' : {},
            'exclude': {}
        }

        self.groups = self.__groupToList(self.groups)

# The infos given in the argument ``groups``.
        for oneGroup in self.groups:
            (open, close), escapeIt = self.__normalizeGroupInfo(oneGroup)

            self.spec['open'][open]   = close
            self.spec['close'][close] = open

            self.spec['escape'][open]  = escapeIt
            self.spec['escape'][close] = escapeIt

        self.spec['all'] = list(self.spec['open'].keys())

        for x in self.spec['close'].keys():
            if not x in self.spec['all']:
                self.spec['all'].append(x)

        self.spec['all'].sort(key = lambda x : -len(x))

# The infos given in the arguments ``exclude`` and ``ignore``.
        self.spec['exclude'] = self.__buildExcludeIgnoreDict(
            oneDict = self.exclude,
            kind    = "exclude"
        )

        self.spec['ignore']  = self.__buildExcludeIgnoreDict(
            oneDict = self.ignore,
            kind    = "ignore"
        )


# -- LIST VIEW -- #

    def __ignoreDelimiter(
        self,
        oneDelim
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method simply returns a boolean to know if one delimiter must be ignored
or not regarding to the infos given in the argument ``ignore``.
        """
        if self.__delimStack:
            lastOpen = self.__delimStack[-1]

            if oneDelim in self.spec['close'] \
            and lastOpen == self.spec['close'][oneDelim]:
                return False

            if lastOpen in self.spec['ignore'] \
            and oneDelim in self.spec['ignore'][lastOpen]:
                return True

        return False

    def __excludeDelimiter(
        self,
        oneDelim
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method returns ``False`` or a tuple ``(openIllegalDelim, closeIllegalDelim,
lastOpen, lastCloseDelim)`` so as to indicate an illegal group in another.
        """
        if self.__delimStack:
# We have a group immediately closed.
            lastOpen = self.__delimStack[-1]

            if oneDelim in self.spec['close'] \
            and self.spec['close'][oneDelim] == lastOpen:
                return False

# Let's see the others opening delimiters found.
            for openFound in self.__delimStack:
                if openFound in self.spec['exclude'] \
                and oneDelim in self.spec['exclude'][openFound]:
                    closeDelim = self.spec['open'][openFound]

                    if oneDelim in self.spec['open']:
                        openIllegalDelim  = oneDelim
                        closeIllegalDelim = self.spec['open'][oneDelim]

                    else:
                        openIllegalDelim  = self.spec['close'][oneDelim]
                        closeIllegalDelim = oneDelim

                    return (
                        openIllegalDelim,
                        closeIllegalDelim,
                        oneDelim,
                        closeDelim
                    )

# Everything looks "ok".
        return False

    def __find(
        self,
        oneDelim,
        position
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method looks for one delimiter ``oneDelim`` unescaped if it is necessary.
The search because a the given positioN
        """
        if self.__ignoreDelimiter(oneDelim):
            return -1

        if not self.spec['escape'][oneDelim] \
        or self.__lenEscape == 0:
            return self.text.find(oneDelim, position)

        escaped = self.escape + oneDelim

        i = position

        while(True):
            j = self.text.find(escaped, i)
            i = self.text.find(oneDelim, i)

            if i == - 1 or i != j + self.__lenEscape:
                return i

            i = max(i, j) + 1

    def __cleanText(
        self,
        text
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method replaces each active escaping character.
        """
        if self.__lenEscape != 0:
            for oneDelim in self.spec['all']:
                text = text.replace(self.escape + oneDelim, oneDelim)

        return text

    @property
    def listView(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This property like method produces a list of tuples which can be one of the
following kinds where the value of ``level`` corresponds to the depth in the
tree version of the groups found.

    1) The tuples can be ``(kind, delimiter, level)`` with ``delimiter`` one
    delimiter found and ``kind`` equal to ``"open"" or ``"close"``.

    2) The tuple can also be ``('text', text, level)``.
        """
        if self.__listView != None:
            return self.__listView

        self.__listView   = []
        self.__delimStack = []

        position = 0
        depth    = 0
        lenText  = len(self.text) + 1

        while(True):
            iFound     = lenText
            delimFound = None

            for oneDelim in self.spec['all']:
                i = self.__find(oneDelim, position)

                if i != -1 and i < iFound:
                    iFound     = i
                    delimFound = oneDelim

            if delimFound:
# Does we must exclude the delimiter ?
                toExclude = self.__excludeDelimiter(delimFound)

                if toExclude != False:
                    openIllegal, closeIllegal, lastOpen, lastClose = toExclude

                    raise StringUseError(
                        "The group << {0} ... {1} >> is not allowed ".format(
                            openIllegal,
                            closeIllegal
                        ) + "inside the group << {0} ... {1} >>.".format(
                            lastOpen,
                            lastClose
                        ) + "\n\n{0}\n{1}^".format(self.text, " "*iFound)
                    )

# Open/close delimiter like in ``"..."``.
                if delimFound in self.spec['open']\
                and delimFound in self.spec['close']:
                    if not self.__delimStack \
                    or self.__delimStack[-1] != delimFound:
                        self.__delimStack.append(delimFound)

                        kind  = 'open'
                        delim = delimFound

                    else:
                        kind  = 'close'
                        delim = delimFound

                        self.__delimStack.pop(-1)

# Open delimiter.
                elif delimFound in self.spec['open']:
                    self.__delimStack.append(delimFound)

                    kind  = 'open'
                    delim = delimFound

# Close delimiter.
                else:
                    if not self.__delimStack \
                    or self.__delimStack[-1] != \
                    self.spec['close'][delimFound]:
                        raise StringUseError(
                            "Not opened group << ... {0} >>.".format(
                                delimFound
                            ) + "\n\n{0}\n{1}^".format(self.text, " "*iFound)
                        )

                    kind  = 'close'
                    delim = delimFound

                    self.__delimStack.pop(-1)

# Let's store the infos and continue.
                textBefore = self.text[position: iFound]

                if textBefore:
                    self.__listView.append((
                        'text',
                        self.__cleanText(textBefore),
                        depth
                    ))

                if kind == "close":
                    depth -= 1
                    self.__listView.append((kind, delim, depth))

                elif kind == "open":
                    self.__listView.append((kind, delim, depth))
                    depth += 1

                position = iFound + len(delimFound)

            else:
                break





# Some groups have not been closed...
        if self.__delimStack:
            firstUnclosedGroup = self.__delimStack[0]
            p = 0

            raise StringUseError(
                "Not closed group << {0} ... >>.".format(
                    firstUnclosedGroup
                ) + "\n\n{0}\n{1}^".format(self.text, " "*p)
            )

# Last piece of text
        textEnd = self.text[position:]

        if textEnd:
            self.__listView.append((
                'text',
                self.__cleanText(textEnd),
                depth
            ))

        return self.__listView


# -- ASCII VIEW -- #

    @property
    def ascii(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This property like method produces an ascii version of the tree found.
        """
        if self.__asciiTree != None:
            return self.__asciiTree

        self.__asciiTree = []

        for kind, text, level in self.listView:
            if kind == 'close':
                text = None

            elif kind == 'open':
                text = "{0} ... {1}".format(
                    text,
                    self.spec[kind][text][0]
                )


            if text != None:
                self.__asciiTree.append(
                    "{0}{1} {2}".format(
                        self.asciiTreeTab * level,
                        self.asciiTreeNode,
                        text
                    )
                )


        self.__asciiTree = '\n'.join(self.__asciiTree)

        return self.__asciiTree


# -- GROUP VIEW -- #

    def __buildGroupView(
        self,
        oneListView
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method produces a nested list which can be used for easy walk in the
groups found.
        """
        groupView = []

        i    = 0
        iMax = len(oneListView)

        while(i < iMax):
            kind, text, depth = oneListView[i]

            if kind == 'open':
                closeDelim = self.spec['open'][text]
                iClose     = i + oneListView[i:].index(
                    ('close', closeDelim, depth)
                )

                groupView.append((
                    'group',
                    (text, closeDelim),
                    self.__buildGroupView(oneListView[i+1: iClose])
                ))

                i = iClose

            else:
                groupView.append((kind, [], text))

            i += 1

        return groupView

    @property
    def groupView(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This property like method produces a group view version which is easy to walk in.
Indeed, all the job is done by the method ``self.__buildGroupView``.
        """
        if self.__groupView == None:
            self.__groupView = self.__buildGroupView(self.listView)

        return self.__groupView


# --------------------- #
# -- CASE FORMATTING -- #
# --------------------- #

__CASE_VARIANTS = ['lower', 'upper', 'sentence', 'title', 'firstLast']

def case(
    text,
    kind
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function produces different case variants of the text contained in the
string variable ``text``.


For example, ``case("onE eXamPLe", "lower")``, ``case("onE eXamPLe", "upper")``
and ``case("onE eXamPLe", "sentence")`` and ``case("onE example", "title")`` are
respectively equal to "one example", "ONE EXAMPLE", "OnE eXamPLe" and "One
Example".


You can also the weird transformation ``case("onE eXamPLe", "firstLast")`` which
is equal to "One examplE". This special feature is indeed used by pyBaNaMa which
is another project of the author of the package ¨mistool.


If you need all the possible case variants, juste use ``case("onE eXamPLe",
"all")`` which is equal to the following dictionary.

python::
    {
        'lower'    : 'one example',
        'upper'    : 'ONE EXAMPLE',
        'sentence' : 'One example',
        'title'    : 'One Example',
        'firstLast': 'One examplE'
    }


If for example you only need to have the lower case and the title case versions
of one text, then you just have to use ``case("OnE eXamPLe", "lower + title")``
which is equal to the following dictionary.

python::
    {
        'lower': 'ONE EXAMPLE',
        'title': 'One Example'
    }


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text` is a string variable corresponding to the text to modify.

    2) ``kind`` indicates the case transformation wanted. The possible case
    variants are the following ones.

        a) ``"lower"``, ``"upper"``, ``"sentence"`` and ``"title"`` are for
        lower case, upper case, sentence and title case versions of one text.

        b) ``"firstLast"`` is to have a lower case text with its firts and last
        letter in upper case.

        c) ``"all"`` is to have all the possible case variants.

        d) If for example you only need to have the lower case and the title
        case versions of one text, then you just have to use ``kind = "lower +
        title")``.
    """
    kind = kind.strip()

    if kind == 'all':
        answer = {}
        for kind in __CASE_VARIANTS:
            answer[kind] = case(text, kind)
        return answer

    elif '+' in kind:
        answer = {}
        for kind in kind.split('+'):
            kind = kind.strip()

            if not kind in answer:
                newCaseVariant = case(text, kind)

                if isinstance(newCaseVariant, dict):
                    for oneKind, oneCaseVariant in newCaseVariant.items():
                        answer[oneKind] = oneCaseVariant

                else:
                    answer[kind] = newCaseVariant

        return answer

    elif kind == 'lower':
        return text.lower()

    elif kind == 'upper':
        return text.upper()

    elif kind == 'sentence':
        return text[0].upper() + text[1:].lower()

    elif kind == 'title':
        return text.title()

    elif kind == 'firstLast':
        return text[0].upper() + text[1:-1].lower() + text[-1].upper()

    else:
        raise StringUseError(
            '<< {0} >> is not an existing kind of case variant formatting.' \
                .format(kind)
        )

def camelTo(
    text,
    kind
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function transforms one text using camel syntax to one of the case variants
proposed by the function ``case``. Indeed, each time one upper letter is met,
one underscore followed by one space is added before it, then the function
``case`` is called, and finally extra spaces are removed. For example,
``camelTo("oneExample", "title")`` is equal to ``"One_Example"``.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text` is a string variable corresponding to the text to modify.

    2) ``kind`` indicates the case transformation wanted. The possible case
    variants are the same as for the function ``case``.
    """
    newLines = ''

    for oneChar in text:
        if oneChar.isupper():
            newLines += '_ '

        newLines += oneChar

    newLines = case(newLines, kind)

    if isinstance(newLines, dict):
        for kind, text in newLines.items():
            newLines[kind] = text.replace('_ ', '_')

    else:
        newLines = newLines.replace('_ ', '_')

    return newLines


# ------------------ #
# -- CASE TESTING -- #
# ------------------ #

def isCase(
    text,
    kind
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

If ``kind`` is one of the strings ``"lower"``, ``"upper"``, ``"sentence"``,
``"title"`` or ``"firstLast"`` the function will simply return the result of
the test ``text == case(text, kind)``. See the documentation of the function
``case`` for more informations about this case formats.

If ``kind = "mix"``, the answer returned will be ``True`` only if the text is
not all lower, not all upper and not in sentence-case format.


warning::
    Something like ``One examplE`` has both the case ``"mix"`` and
    ``"sentence"``.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text`` is simply a string representing the text to be tested.

    2) ``kind`` is a string whose values can only be ``"lower"``, ``"upper"``,
    ``"sentence"``, ``"title"``, ``"firstLast"`` or ``"mix"``.
    """
    if kind == "mix":
        return not text.islower() \
        and not text.isupper() \
        and text != case(text, "sentence")

    elif kind not in __CASE_VARIANTS:
        raise StringUseError(
            '<< {0} >> is not an existing kind of case variant testing.' \
                .format(kind)
        )

    elif kind == "lower":
        return text.islower()

    elif kind == "upper":
        return text.isupper()

    else:
        return text == case(text, kind)


# ------------------- #
# -- SHORTEN LINES -- #
# ------------------- #

def __addNewWord(
    word,
    line,
    newLines,
    width,
    indent,
    lenIndent
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function manages the addition of a new word to the new lines.

It returns ``(line, newLines)`` where ``line`` is the line that must be
completed and ``newLines`` is the list of the already built lines of the cut
text.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``word`` is the actual word to add to the new lines.

    2) ``line`` is the the line being built.

    3) ``newLines`` is the list of the lines cut.

    4) ``width`` is the width expected for the lines.

    5) ``indent`` is the indentation of the current line analyzed.

    6) ``lenIndent`` is the number of spaces of the indentation of the current
    line analyzed.
    """
    lenWord = len(word)

# The new word is too big.
    if len(line) + lenWord >= width:
        newLines.append(line)

        if lenWord + lenIndent > width:
            newLines.append(indent + word)
            line = ''

        else:
            line = indent + word

# The new word is small.
    elif word:
        if line:
            line += ' '

        else:
            line = indent

        line += word

    return (line, newLines)

def __indent(
    text,
    spaceTab
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function, which is used by the function ``cut``, simply returns the leading
spaces and tabulations of one text. Each tabulation is converted to spaces using
the variable ``spaceTab``.


:::::::::::::
The arguments
:::::::::::::

This function uses the following arguments.

    1) ``text`` is the text to analyse.

    2) ``spaceTab`` indicates the string of spaces which replaces each
    tabulation.
    """
    indent = ''

    if text.strip():
        i = 0

        while text[i] in " \t":
            indent += text[i]
            i+=1

    indent = indent.replace('\t', spaceTab)

    return indent

def cut(
    text,
    width    = 80,
    spaceTab = ' '*4
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function tries to cut lines of a text so as they have less than `81`
characters (by default but you can change this number).

Multiple following spaces are rendered as only one single space, and the final
text is stripped so as to start and end with none empty lines.

One last thing, when lines are cut their indentations are respected but each
leading tabulation will be replaced by spaces (see the description of the
variable ``spaceTab``).


:::::::::::::
The arguments
:::::::::::::

This function uses two arguments.

    1) ``text`` is simply the text to cut.

    2) ``width`` is an optional integer that gives the maximum of characters
    wanted to be in one line.

    By default, ``width = 80``.

    3) ``spaceTab`` indicates the string of spaces which replaces each
    tabulation.
    """
    if not isinstance(width, int) or width <= 0:
        raise StringUseError("<< width >> must be a positive integer.")

    newLines = []

# Let's work line by line.
    for oldLine in text.split('\n'):
        indent = __indent(
            text     = oldLine,
            spaceTab = spaceTab
        )

        lenIndent = len(indent)

        oldLine = oldLine.strip()
        word    = ''
        line    = ''

# A not empty line.
        if oldLine:
            for oneChar in oldLine:
# One space found...
                if oneChar == ' ':
                    line, newLines = __addNewWord(
                        word,
                        line, newLines,
                        width,
                        indent, lenIndent
                    )

                    word = ''

# The word becomes bigger...
                else:
                    word += oneChar

# The end of the line
            if word:
                line, newLines = __addNewWord(
                    word,
                    line, newLines,
                    width,
                    indent, lenIndent
                )

                if line:
                    newLines.append(line)
                    line = ''

                else:
                    newLines.append('')

# An empty line
        else:
            newLines.append('')

# All the job has been done !
    return '\n'.join(newLines)


# ------------------- #
# -- DECORATE TEXT -- #
# ------------------- #

FRAME_FORMATS = config.frame.FRAME_FORMATS
DEFAULT_FRAME = FRAME_FORMATS['python_basic']

_ABREV_FRAME = config.frame._ABREV_FRAME
_KEY_FRAME   = config.frame._KEY_FRAME

def __drawHorizontalRule(
    charRule,
    left,
    right,
    lenght,
    nbSpace
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function is used to draw the horizontal rules of a frame around one text.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``charRule`` is the character used to draw the rule.

    2) `left`` and ``right`` are the first and last additional texts used around
    the rule.

    3) ``lenght`` is an integer giving the lenght of the rule.

    4) ``nbSpace`` is the number of spaces to add before the first additional
    text (this is for cases when left corners have different lenghts).
    """
    if charRule:
        return [
            ' '*(nbSpace - len(left))
            + left
            + charRule*lenght
            + right
        ]

    elif left:
        return [
            left
            + ' '*lenght
            + right
        ]

    elif right:
        return [
            ' '*(nbSpace + lenght)
            + right
        ]

    else:
        return []

def frame(
    text,
    format = DEFAULT_FRAME,
    center = True
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

--------------
Default frames
--------------

This function makes it possible to put one text into one frame materialized by
ASCII characters. This can be usefull for console outputs or for pretty comments
in listings like the following python comment.

python::
    #############
    # one       #
    # comment   #
    # easily    #
    # formatted #
    #############


This text has been produced using the following lines.

python::
    from mistool import string_use

    oneText = '''one
    comment
    easily
    formatted'''

    print(
        string_use.frame(
            text   = oneText,
            center = False
        )
    )


By default, ``center`` is equal ``True`` which asks to merely center the content
of the frame. Here we use the default frame ``DEFAULT_FRAME`` which is equal to
``FRAME_FORMATS['python_basic']``. The dictionary ``FRAME_FORMATS`` contains all
the default formats. For example, in the following code we use another default
formats.

python::
    from mistool import string_use

    oneText = '''one
    comment
    with C-like
    style'''

    print(
        string_use.frame(
            text   = oneText,
            format = string_use.FRAME_FORMATS['c_basic'],
            center = False
        )
    )


This will give the following output.

code_c::
    /***************
     * one         *
     * comment     *
     * with C-like *
     * style       *
     ***************/


---------------
Homemade frames
---------------

The following frame can be obtained by using the default format
``string_use.FRAME_FORMATS['python_pretty']``.

python::
    # ------------- #
    # -- one     -- #
    # -- pretty  -- #
    # -- comment -- #
    # ------------- #


Let see the definition ``FRAME_FORMATS['python_pretty']``.

python::
    {
        'rule': {
            'down': '-',
            'left': '--',
            'right': '--',
            'up': '-'
        },
        'extra': {
            'rule': {
                'left': '#',
                'right': '#'
            }
        }
    }


In this dictionary, we define a frame and then an extra frame. Indeed, you can
use a dictionary looking like the one above. A missing key is a shortcut to
indicate an empty string.

python::
    {
        'rule' : {
            'up'   : "Single character",
            'down' : "Single character",
            'left' : "Some characters",
            'right': "Some characters"
        },
        'corner': {
            'leftup'   : "Some characters",
            'leftdown' : "Some characters",
            'rightup'  : "Some characters",
            'rightdown': "Some characters"
        },
        'extra': {
            'rule' : {
                'up'   : "Single character",
                'down' : "Single character",
                'left' : "Some characters",
                'right': "Some characters"
            },
            'corner': {
                'leftup'   : "Some characters",
                'leftdown' : "Some characters",
                'rightup'  : "Some characters",
                'rightdown': "Some characters"
            },
        }
    }


You can use the following abreviations for the positional keys.

    * ``u``, ``d``, ``l`` and ``r`` are abreviations for ``up``, ``down``,
    ``left`` and ``right`` respectively.

    * ``lu``, ``ld``, ``ru`` and ``rd`` are abreviations for ``leftup``,
    ``leftdown``, ``rightup`` and ``rightdown`` respectively.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text`` is a string value corresponding to the text to put in a frame.

    2) ``center`` is a boolean variable to center or not the text inside the
    frame. By default, ``center = True``.

    3) ``format`` is an optional dictionary defining the frame. By default,
    ``format = DEFAULT_FRAME`` which is equal to
    ``FRAME_FORMATS['python_basic']``.

    info::
        All the default formats are in the dictionary ``FRAME_FORMATS``.


    The general structure of a dictionary to use with ``format`` is the following
    one.
    """
# Default values must be chosen if nothing is given.
    if not set(format.keys()) <= {'rule', 'corner', 'extra'}:
        raise StringUseError("Illegal key for the dictionary << format >>.")

    for kind in ['rule', 'corner', 'extra']:
        if kind not in format:
            format[kind] = {}


    for kind in ['rule', 'corner']:
        if not set(format[kind].keys()) <= _KEY_FRAME[kind]:
            raise StringUseError(
                "Illegal key for the dictionary << format >>. "
                "See the kind << {0} >>.".format(kind)
            )

        for key, abrev in _ABREV_FRAME[kind].items():
            if abrev in format[kind] and key in format[kind]:
                message = "Use of the key << {0} >> and its abreviation " \
                        + "<< {1} >> for the dictionary << format >>."

                raise StringUseError(message.format(key, abrev))

            if abrev in format[kind]:
                format[kind][key] = format[kind][abrev]

            elif key not in format[kind]:
                format[kind][key] = ""

# Horizontal rules can only use one single character.
    for onePosition in ['up', 'down']:
        if len(format['rule'][onePosition]) > 1:
            message = "You can only use nothing or one single character " \
                    + "for rules.\nSee << {0} >> for the {1} rule."

            raise StringUseError(
                message.format(
                    format['rule'][onePosition],
                    onePosition
                )
            )

# Infos about the lines of the text.
    lines = [oneLine.rstrip() for oneLine in text.splitlines()]
    nbMaxChar = max([len(oneLine) for oneLine in lines])

# Space to add before vertical rules.
    nbSpace = max(
        len(format['corner']['leftup']),
        len(format['corner']['leftdown'])
    )

    spaceToAdd = ' '*nbSpace

# Text decoration for vertical rules
    if format['rule']['left']:
        leftRule = format['rule']['left'] + ' '
    else:
        leftRule = ''

    if format['rule']['right']:
        rightRule = ' ' + format['rule']['right']
    else:
        rightRule = ''

# Length of the rule without the corners
    lenght = nbMaxChar + len(leftRule) + len(rightRule)

# First line of the frame
    answer = __drawHorizontalRule(
        charRule = format['rule']['up'],
        left     = format['corner']['leftup'],
        right    = format['corner']['rightup'],
        lenght   = lenght,
        nbSpace  = nbSpace
    )

# Management of the lines of the text
    for oneLine in lines:
        nbSpacesMissing = nbMaxChar - len(oneLine)

# Space before and after one line of text.
        if center:
            if nbSpacesMissing % 2 == 1:
                spaceAfter = ' '
            else:
                spaceAfter = ''

            nbSpacesMissing = nbSpacesMissing // 2

            spaceBefore = ' '*nbSpacesMissing
            spaceAfter += spaceBefore

        else:
            spaceBefore = ''
            spaceAfter = ' '*nbSpacesMissing

        answer.append(
            spaceToAdd
            +
            '{0}{1}{2}{3}{4}'.format(
                leftRule,
                spaceBefore,
                oneLine,
                spaceAfter,
                rightRule
            )
        )

# Last line of the frame
    answer += __drawHorizontalRule(
        charRule = format['rule']['down'],
        left     = format['corner']['leftdown'],
        right    = format['corner']['rightdown'],
        lenght   = lenght,
        nbSpace  = nbSpace
    )

    answer = '\n'.join([x.rstrip() for x in answer])

# Does we have an extra frame ?
    if format['extra']:
        try:
            answer = frame(
                text   = answer,
                format = format['extra'],
                center = center
            )

        except StringUseError as e:
            raise StringUseError(
                str(e)[:-1] + " in the definition of the extra frame."
            )

# All the job has been done.
    return answer


# ------------------ #
# -- STEP BY STEP -- #
# ------------------ #

class Step:
    """
:::::::::::::::::
Small description
:::::::::::::::::

This class displays texts for step by step actions. The numbering of the steps
is automatically updated and displayed.


:::::::::::::
The arguments
:::::::::::::

There are two optional variables.

    1) ``nb`` is the number of the current step. When the class is instanciated,
    the default value is ``1``.

    2) ``deco`` indicates how to display the numbers. When the class is
    instanciated, the default value is ``""1)""`` where ``1`` symbolises the
    numbers.
    """
    def __init__(
        self,
        nb   = 1,
        deco = "1)"
    ):
        self.nb   = nb
        self.deco = deco.replace('1', '{0}')

    def print(
        self,
        text,
        deco
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method simply prints ``deco`` the text of the actual numbering, and then
``text`` the content of the actual step.

You can redefine this method for finer features.


:::::::::::::
The arguments
:::::::::::::

This method uses the following variables.

    1) ``text`` is simply the text of the actual step.

    2) ``deco`` is a string corresponding to the text indicated what is the
    actual step number.
        """
        print(
            deco,
            text,
            sep = " "
        )

    def display(
        self,
        text
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method simply calls the method ``self.print`` so as to print the
informations contained in the variable ``text``, and then ``self.nb`` is
augmented by one unit.

You can redefine the method ``self.print`` for finer features.


:::::::::::::
The arguments
:::::::::::::

This method uses one variable ``text`` which is the text of the actual step.
        """
        self.print(
            text = text,
            deco = self.deco.format(self.nb)
        )

        self.nb += 1
