#!/usr/bin/env python3

"""
Directory : mistool/parse_use
Name      : parse_use
Version   : 2013.10
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This module contains some tools for tokenizing a text.
"""


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class ParseUseError(ValueError):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Base class for errors in the ``regex_use`` module of the package ``mistool``.
    """
    pass


# --------------------- #
# -- DEFINING GROUPS -- #
# --------------------- #

class Groups:
    """
:::::::::::::::::
Small description
:::::::::::::::::

------------------------
What this class is for ?
------------------------

When you want to use the class ``FindGroups`` so as to find groups, you have to
define the groups and some rules they follow. It can be also necessary to define
groups when you use the class ``Tokens``, from the module ``parse_use.token`` so
as to tokenize a text. In this two situations, you will have to use the class
``Groups`` for defining your groups.


----------------
Groups in groups
----------------

Let's suppose that we want to define groups ``( ... )`` and ``[ ... ]`` that can
be inside each others. Here is how to define this.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups =  [
            ("(", ")"),
            ("[", "]")
        ]
    )


We just have to give a list of tuples of the kind ``(opening delimiter, closing
delimiter)``. The class build the following dictionary that is stored in the
attribut ``rules``. You can notice some automatic settings. We will see later
how to set this particular features.

python::
    {
        'all': [
            '(', '[', ']', ')'
        ],
        'open': {
            '(': ')',
            '[': ']'
        },
        'close': {
            ')': '(',
            ']': '['
        },
        'escape': {
            '(': False,
            ')': False,
            '[': False,
            ']': False
        },
        'exclude': {},
        'ignore': {},
        'untoken': [],
        'verbatim': []
    }


-------------------------
Defining one single group
-------------------------

If we only want to define groups ``( ... )``, we can directly give it.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups = ("(", ")")
    )


warning::
    To indicate quickly one single group with different delimiters, you **must
    use a tuple** and not a list because ``groups`` is indeed seen in general as
    a list of groups.


---------------------------------------------------
Groups with the same opening and closing delimiters
---------------------------------------------------

In the following code, we use a short version for indicating groups delimited by
quotes. Instead of the tuple ``('"', '"')``, we can simply use ``'"'``.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(groups = '"')


-------------------
Escaping delimiters
-------------------

It can be useful to have the possibility to escape delimiters. The code above
allows to use escaped quotes so as to not see them as delimiters.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups = ('"', True),
        escape = '\\'
    )


Let's suppose know that we also have groups ``( ... )`` that can be escaped. In
that case, we have to use a code like the one above where instead of using the
`2`-tuple ``("(", ")")``, we use the `3`-tuple `` ("(", ")", False)`` where the
additional third boolean value indicate to not escape the delimiters ``(`` and
``)``.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups = [
            ('"', True),
            ("(", ")")
        ],
        escape = '\\'
    )


info::
    By default, all the groups can not be escaped.


-----------------------------
Ignoring groups inside others
-----------------------------

Suppose now that we have groups ``" ... "`` and ``( ... )`` where the parenthesis
have to be ignored inside quotes. This can be obtained by the following code.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups = [
            '"',
            ("(", ")")
        ],
        ignore = {'"': "("}
    )


In the assignation ``ignore = {'"': "("}``, we use one dictionary with the
following rules to respect.

    1) The keys are any kind of supported definitions of one group. For example,
    we can use ``ignore = {("(", ")"): ...}`` or even ``ignore = {("(", ")",
    False): ...}``.

    2) The values can be either a single string indicating an opening delimiter,
    or any kind of supported definitions of one group like ``("(", ")")``, or
    ``("(", ")", False)``, or better a list of the value of one of the two
    preceding kinds.


The flexibility in the definitions of the dictionary ``ignore`` allows to use
efficient codes like the following one.

python::
    from mistool import parse_use

    quoteGroup = '"'
    parGroup   = ("(", ")")

    myGroups = parse_use.group.Groups(
        groups = [quoteGroup, parGroup],
        ignore = {quoteGroup: parGroup}
    )


info::
    You can ask to ignore for example parenthesis inside parenthesis so as to
    only allow only one level of parenthesis.


You can easily define more complex rules. Suppose for example that we want to
ignore parenthesis, hooks and quotes, in any other kind of groups. This can be
done using dictionary comprehension like in the code above.

python::
    from mistool import parse_use

    quoteGroup = '"'
    parGroup   = ("(", ")")
    hookGroup  = ("[", "]")

    allGroups = [quoteGroup, parGroup, hookGroup]

    myGroups = parse_use.group.Groups(
        groups = allGroups,
        ignore = {x: allGroups for x in allGroups}
    )


The following section gives a mor efficient way to do that.


----------------------
Verbatim is our friend
----------------------

In most langage, ``"..."`` indicates strings where no character have a special
meaning. Rather than indicating that all the groups must be ignored inside the
group ``"..."``, it suffises to do as in the following example.

python::
    from mistool import parse_use

    quoteGroup = '"'
    parGroup   = ("(", ")")
    hookGroup  = ("[", "]")

    allGroups = [quoteGroup, parGroup, hookGroup]

    myGroups = parse_use.group.Groups(
        groups   = allGroups,
        verbatim = quoteGroup
    )


------------------------------
Excluding groups inside others
------------------------------

Let's suppose now that we do not want to allow the use of parenthesis between
quotes. You can do by using the argument ``exclude`` which follows the same rules
for its values that the argument ``ignore``.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups  = ['"', ("(", ")")],
        exclude = {'"': "("}
    )


----------------------
Groups to not tokenize
----------------------

The class ``Tokens`` allows to tokenize a text following some simple rules. In
some circumstances, it is needed to not tokenize some groups. You can indicate
the list of the groups to not tokenize in the list argument ``untoken``.


warning::
    The use of ``verbatim`` and ``untoken`` are not equivalent because
    ``verbatim`` is used for looking for group, whereas ``untoken`` is used
    during the tokenization.


:::::::::::::
The arguments
:::::::::::::

The instanciation of this class uses the following variables.

    1) ``groups`` is an optional argument which can be either a single value, or
    a list of values of the following types.

        a) A single string, like ``'"'``, indicates a groups escapable having the
        same opening and closing delimiters.

        b) A 2-tuple ``(open, close)`` where ``open`` and ``close`` are strings
        indicating the opening and closing delimiters. In that case, the
        delimiters are escapable.

        c) Either ``('"', False)``, or ``(open, close, False)`` allows to
        indicates that some delimiters are not escapable.

    By default, ``groups = []``.

    2) ``escape`` is an optional string used to escape delimiters. The default
    value is ``""`` which indicates that there is not escaping sequence of
    characters.

    3) ``ignore`` is an optional dictionary indicating which groups must be
    ignored in others groups. The keys of this dictionary can be any supported
    kind of definition for one single group. The values can be either one single
    group, or a list of groups.

    The meaning of ``ignore`` is the following one.

    python::
        {
            one single group G : the groups to ignore inside G.
        }

    By default, ``ignore = {}``.

    4) ``exclude`` is an optional dictionary following exactly the same rules as
    the dictionary ``ignore``. The meaning of ``exclude`` is the following one.

    python::
        {
            one single group G : the groups forbidden inside G.
        }

    By default, ``exclude = {}``.

    5) ``untoken`` is an optional list for the groups to not tokenize. By
    default, ``untoken = []``.
    """
    def __init__(
        self,
        groups   = [],
        escape   = "",
        ignore   = {},
        exclude  = {},
        verbatim = [],
        untoken  = []
    ):
        self.groups   = groups
        self.escape   = escape
        self.ignore   = ignore
        self.exclude  = exclude
        self.verbatim = verbatim
        self.untoken  = untoken

        self.rules = {
            'open'    : {},
            'close'   : {},
            'all'     : [],
            'escape'  : {},
            'ignore'  : {},
            'exclude' : {},
            'verbatim': [],
            'untoken' : []
        }

        self.build()


# -- MAIN METHOD -- #

    def build(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds the dictionary of rules that will be used by other classes.
        """
        self.groups = self.__listify(self.groups)

        for oneGroup in self.groups:
            (open, close), escapeIt = self.__normalize(oneGroup)

            self.rules['open'][open]   = close
            self.rules['close'][close] = open

            self.rules['escape'][open]  = escapeIt
            self.rules['escape'][close] = escapeIt

        self.rules['all'] = list(self.rules['open'].keys())

        for x in self.rules['close'].keys():
            if not x in self.rules['all']:
                self.rules['all'].append(x)

        self.rules['all'].sort(key = lambda x : -len(x))

# The infos given in the argument ``verbatim``.
        self.rules['verbatim'] = self.__verbatimList()

# The infos given in the arguments ``exclude`` and ``ignore``.
        self.rules['exclude'] = self.__excludeLikeDict(
            oneDict = self.exclude,
            kind    = "exclude"
        )

        self.rules['ignore']  = self.__excludeLikeDict(
            oneDict = self.ignore,
            kind    = "ignore"
        )

# The infos given in the argument ``untoken``.
        self.rules['untoken'] = self.__untokenList()


# -- NORMALIZATION -- #

    def __listify(
        self,
        groups
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method simply builds, if it is necessary, a list version of the argument
``groups``.
        """
        if not isinstance(groups, list):
            return  [groups]

        __groups = []

        for oneGroup in groups:
            if isinstance(oneGroup, str):
                oneGroup = [oneGroup]

            __groups.append(oneGroup)

        return __groups

    def __normalize(
        self,
        groupInfo
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method allows to use different formats for indicating one single group by
normalizing all the different formats supported.
        """
        delim    = []
        escapeIt = []

        for oneInfo in groupInfo:
            if isinstance(oneInfo, str):
                delim.append(oneInfo)

            elif isinstance(oneInfo, bool):
                escapeIt.append(oneInfo)

            else:
                raise ParseUseError(
                    "Illegal informations for the argument ``groups``:"
                    "\n<< {0} >>".format(groupInfo)
                )

        if len(escapeIt) > 1:
            raise ParseUseError(
                "Only one string can indicate an escaping sequence."
            )

        if not(0 < len(delim) < 3):
            raise ParseUseError(
                "Illegal  vvv value for the argument ``groups``:"
                "\n<< {0} >>".format(delim)
            )

        if len(delim) == 1:
            delim.append(delim[0])

        if len(escapeIt) == 0:
            escapeIt = False

        else:
            escapeIt = escapeIt[0]

        return [tuple(delim), escapeIt]


# -- INTERNAL VERSIONS OF THE RULES -- #

    def __excludeLikeDict(
        self,
        oneDict,
        kind
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds the internal version of the infos given in the arguments ``exclude`` and ``ignore`` used during the creation of the class.
        """
        answer = {}

        for oneGroup, groupToExlude in oneDict.items():
            open = self.__normalize(oneGroup)[0][0]

            if not open in self.rules['open']:
                raise ParseUseError(
                    "Illegal value for a key in the argument ``{0}``:"
                    "\n<< {1} >>".format(kind, oneGroup)
                )

            for oneToExlude in self.__listify(groupToExlude):
                if isinstance(oneToExlude, str):
                    openToExlude  = oneToExlude
                    closeToExlude = self.rules['open'][openToExlude]

                else:
                    openToExlude, closeToExlude \
                    = self.__normalize(oneToExlude)[0]

                ignoreList = answer.get(open, [])

                ignoreList.append(openToExlude)
                ignoreList.append(closeToExlude)

                if isinstance(open, tuple):
                    open = open[0]

                answer[open] = ignoreList

        return answer

    def __untokenList(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds the internal version of the infos given in the argument
``untoken`` used during the creation of the class.
        """
        self.untoken = self.__listify(self.untoken)

        answer = []

        for oneGroup in self.untoken:
            open = self.__normalize(oneGroup)[0][0]

            if not open in self.rules['open']:
                raise ParseUseError(
                    "Illegal value for a key in the argument ``untoken``:"
                    "\n<< {1} >>".format(kind, oneGroup)
                )

            answer.append(open)

        self.untoken = answer

    def __verbatimList(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method builds the internal version of the infos given in the argument
``verbatim`` used during the creation of the class.
        """
        self.verbatim = self.__listify(self.verbatim)

        verbatim = []

        for oneGroup in self.verbatim:
            open = self.__normalize(oneGroup)[0][0]

            if not open in self.rules['open']:
                raise ParseUseError(
                    "Illegal value for a key in the argument ``verbatim``:"
                    "\n<< {} >>".format(oneGroup)
                )

            verbatim.append(open)

        return verbatim


# ------------------------ #
# -- LOOKING FOR GROUPS -- #
# ------------------------ #

class FindGroups:
    """
:::::::::::::::::
Small description
:::::::::::::::::

-------------
To read first
-------------

See the documentation of ``Groups`` so as to learn how to define groups. That's
a necesserary-to-know documentation.


----------------------------------------
Short overview trough one simple example
----------------------------------------

Let's consider the text ``"A(B)C[D(E)F]"`` where there are well balanced groups
using parenthesis and hooks. The class ``FindGroups`` allows to have easily the
following ascii tree representing the groups of braces.

terminal::
    + A
    + ( ... )
        + B
    + C
    + [ ... ]
        + D
        + ( ... )
            + E
        + F
    + G


This tree has been obtained using the following code.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups =[ ("(", ")"), ("[", "]") ]
    )

    myAnalyze = parse_use.FindGroups(
        text   = "A(B)C[D(E)F]",
        groups = myGroups
    )

    print(myAnalyze)


info::
    The ascii representation is stored in the attribut ``ascii``.


There is an efficient way to walk in the tree. The recursive function ``walkin``
above shows how to do that kind of thing. In this example, we build by hand
an ascii tree representing the groups found.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups =[ ("(", ")"), ("[", "]") ]
    )

    myAnalyze = parse_use.FindGroups(
        text   = "A(B)C[D(E)F]",
        groups = myGroups
    )

    def walkin(groupView, depth = 0):
        deco = " "*depth + "* "

        for kind, delim, content in groupView:
            if kind == 'text':
    # In that case, ``delim = []``.
                print(deco + content)

            elif kind == 'group':
                open, close = delim
                print(deco + "{0} <---> {1}".format(open, close))

    # Let's use a recursive call.
                walkin(groupView = content, depth = depth + 4)

    walkin(myAnalyze.groupView)


Launched in a terminal, the code will print the following tree.

terminal::
    * A
    * ( <---> )
        * B
    * C
    * [ <---> ]
        * D
        * ( <---> )
            * E
        * F


Indeed, the class ``FindGroups`` always build first a list view which is stored
in the attribut ``listView``. For example, by changing ``myAnalyze.ascii``
to ``myAnalyze.listView`` in the first python code, we obtain the following
output.

terminal::
    [
        ('text', 'A', 0),
        ('open', '(', 0),
        ('text', 'B', 1),
        ('close', ')', 0),
        ('text', 'C', 0),
        ('open', '[', 0),
        ('text', 'D', 1),
        ('open', '(', 1),
        ('text', 'E', 2),
        ('close', ')', 1),
        ('text', 'F', 1),
        ('close', ']', 0)
    ]


In this list, each tuple uses the specification ``(kind, text, depth)``.

    1) ``kind`` can be ``"text"`` indicating a simple text, ``"open"`` for an
    opening delimiter, or ``"close"`` for a closing delimiter.

    2) ``text`` is simply a content or a string corresponding to a delimiter.

    3) ``depth`` indicates the depth regarding to the groups : ``0`` is for
    something out of any group, ``1`` indicates something directly inside one
    group and so on...


-------------------------
Defining one single group
-------------------------

In this well braced expression ``A(B)C(D(E)F)G``, we want only to look for one
kind of groups so as to obtain the following tree.

terminal::
    + A
    + ( ... )
        + B
    + C
    + ( ... )
        + D
        + ( ... )
            + E
        + F
    + G


This can be obtained using the following code.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups = ("(", ")")
    )

    myAnalyze = parse_use.FindGroups(
        text   = "A(B)C(D(E)F)G",
        groups = myGroups
    )

    print(myAnalyze)


------------------------------
Excluding groups inside others
------------------------------

In the text ``A("B")"C(D(E)F)"G``, we do not want to allow the use of parenthesis
between quotes. The following code will raise one exception ``ParseUseError``
indicating the illegal use of parenthesis inside quotes.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups  = ['"', ("(", ")")],
        exclude = {'"': "("}
    )

    myAnalyze = parse_use.FindGroups(
        text   = 'A("B")"C(D(E)F)"G',
        groups = myGroups
    )

    print(myAnalyze)

This code will raise the following error message where the symbol ``^`` indicates
where the problem has been detected.

terminal::
    The group << ( ... ) >> is not allowed inside the group << " ... " >>.
    A("B")"C(D(E)F)"G
            ^


---------------------
Analyse several texts
---------------------

You can change the attribut ``text``  each time you want like in the following
code.

python::
    from mistool import parse_use

    myGroups = parse_use.group.Groups(
        groups  = ['"', ("(", ")")]
    )

    myAnalyze = parse_use.FindGroups(
        text   = 'A("B")"C"D',
        groups = myGroups
    )

    print(myAnalyze)

    print('-------------')

    myAnalyze.text = 'X("Y")'

    print(myAnalyze)


This will produces the following two ascii trees.

terminal::
    + ( ... )
        + " ... "
            + B
    + " ... "
        + C
    + D
    -------------
    + X
    + ( ... )
        + " ... "
            + Y


:::::::::::::
The arguments
:::::::::::::

The instanciation of this class uses the following variables.

    1) ``text`` is simply the text to analyze.

    2) ``groups`` is an optional argument whose default value is ``Groups()``.
    This argument must be an instance of the class ``Groups()`` which define the
    rules followed by the groups.
    """
    def __init__(
        self,
        text,
        groups = Groups(),
    ):
# Arguments
        self.text       = text
        self.__lastText = None

        if not isinstance(groups, Groups):
            raise ParseUseError("``groups`` must be an instance of ``Groups``.")

        self.rules    = groups.rules
        self.escape   = groups.escape

# Customisable constants.
        self.asciiTreeTab  = " "*4
        self.asciiTreeNode = "+"

# Internal constants.
        self.__lenText  = len(self.text)
        self.__iTextMax = self.__lenText - 1

        self.__escape_2 = self.escape*2

        self.__listView  = []
        self.__asciiTree = ""
        self.__groupView = []

        self.__delimStack = []


# -- TOOLS -- #

    def __isNewText(self):
        if self.text != self.__lastText:
            self.__lastText = self.text

            self.__listView  = []
            self.__asciiTree = ""
            self.__groupView = []

            return True

        return False

    def __ignoreDelimiter(
        self,
        oneDelim
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method simply returns a boolean to know if one delimiter must be ignored
or not regarding to the infos given in the argument ``ignore``.
        """
        if self.__delimStack:
            lastOpen = self.__delimStack[-1]

            if oneDelim in self.rules['close'] \
            and lastOpen == self.rules['close'][oneDelim]:
                return False

            if lastOpen in self.rules['ignore'] \
            and oneDelim in self.rules['ignore'][lastOpen]:
                return True

        return False

    def __excludeDelimiter(
        self,
        oneDelim
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method returns ``False`` or a tuple ``(openIllegalDelim, closeIllegalDelim,
lastOpen, lastCloseDelim)`` so as to indicate an illegal group in another.
        """
        if self.__delimStack:
# We have a group immediately closed.
            lastOpen = self.__delimStack[-1]

            if oneDelim in self.rules['close'] \
            and self.rules['close'][oneDelim] == lastOpen:
                return False

# Let's see the others opening delimiters found.
            for openFound in self.__delimStack:
                if openFound in self.rules['exclude'] \
                and oneDelim in self.rules['exclude'][openFound]:
                    closeDelim = self.rules['open'][openFound]

                    if oneDelim in self.rules['open']:
                        openIllegalDelim  = oneDelim
                        closeIllegalDelim = self.rules['open'][oneDelim]

                    else:
                        openIllegalDelim  = self.rules['close'][oneDelim]
                        closeIllegalDelim = oneDelim

                    return (
                        openIllegalDelim,
                        closeIllegalDelim,
                        oneDelim,
                        closeDelim
                    )

# Everything looks "ok".
        return False

    def __find(
        self,
        oneDelim,
        position
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method looks for one delimiter ``oneDelim`` unescaped if it is necessary.
The search because a the given positioN
        """
        if self.__ignoreDelimiter(oneDelim):
            return -1

        if not self.rules['escape'][oneDelim] \
        or not self.escape:
            return self.text.find(oneDelim, position)

        escaped   = self.escape + oneDelim
        escaped_2 = self.__escape_2 + oneDelim

        i = position

        while(True):
            i = self.text.find(oneDelim, i)

            if i in [- 1, self.__iTextMax] \
            or self.text[:i+1].endswith(escaped_2) \
            or not self.text[:i+1].endswith(escaped):
                return i

            i += 1

    def __cleanText(
        self,
        text
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method replaces each active escaping character.
        """
        if self.escape:
            for oneDelim in self.rules['all']:
                text = text.replace(self.escape + oneDelim, oneDelim)

            text = text.replace(self.__escape_2, self.escape)

        return text


# -- ERROR AND LOCALIZATION -- #   TOUT REVOIR !!!!!

    def __raise(
        self,
        message,
        iFound
    ):
        raise ParseUseError(message)

# ???????
# ???????
# ???????
# ???????
# ???????
        iBis = len(self.text) - iFound

        if iFound <= 100:
            if len(self.text) > 100:
                text = self.text[:100] + " [[...TRONCATED]]"

            message += "\n\n{0}\n{1}^".format(self.text, " "*iFound)

        elif iBis <= 100:
            if len(self.text) > 100:
                text = "[[TRONCATED...]] " + self.text[-100:]

            message += "\n\n{0}\n{1}^".format(text, " "*(len(self.text)-iBis))

        else:
            message += " The error found can't be precisely indicated."

        print('---',self.text,'---',sep="\n")
        raise ParseUseError(message)


# -- LIST VIEW -- #

    @property
    def listView(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This property like method produces a list of tuples which can be one of the
following kinds where the value of ``level`` corresponds to the depth in the
tree version of the groups found.

    1) The tuples can be ``(kind, delimiter, level)`` with ``delimiter`` one
    delimiter found and ``kind`` equal to ``"open"" or ``"close"``.

    2) The tuple can also be ``('text', text, level)``.
        """
        if (not self.__isNewText()) and self.__listView:
            return self.__listView

        self.__listView   = []
        self.__delimStack = []

        position = 0
        depth    = 0
        lenText  = len(self.text) + 1

        while(True):
            iFound     = lenText
            delimFound = None

            for oneDelim in self.rules['all']:
                i = self.__find(oneDelim, position)

                if i != -1 and i < iFound:
                    iFound     = i
                    delimFound = oneDelim

            if delimFound:
# Does we must exclude the delimiter ?
                toExclude = self.__excludeDelimiter(delimFound)

                if toExclude != False:
                    openIllegal, closeIllegal, lastOpen, lastClose = toExclude

                    message = "The group << {0} ... {1} >> is not allowed ".format(
                        openIllegal, closeIllegal
                    ) + "inside the group << {0} ... {1} >>.".format(
                        lastOpen, lastClose
                    )

                    self.__raise(message, iFound)

# Verbatim ?
                if delimFound in self.rules['verbatim']:
                    kind = "verbatim"

                    close = self.rules['open'][delimFound]
                    iBis  = self.__find(close, iFound + 1)

                    if iBis == -1:
                        message = "Not closed group << {0} ... >>.".format(
                            delimFound
                        )

                        self.__raise(message, iFound)

                    textBefore = self.text[position: iFound]

                    if textBefore:
                        self.__listView.append((
                            'text',
                            self.__cleanText(textBefore),
                            depth
                        ))

                    self.__listView.append(("open", delimFound, depth))
                    self.__listView.append(
                        ("text", self.text[iFound+1:iBis], depth + 1)
                    )
                    self.__listView.append(("close", close, depth))

                    position = iBis + len(close)

# Open/close delimiter like in ``"..."``.
                elif delimFound in self.rules['open']\
                and delimFound in self.rules['close']:
                    if not self.__delimStack \
                    or self.__delimStack[-1] != delimFound:
                        kind  = 'open'
                        self.__delimStack.append(delimFound)

                    else:
                        kind  = 'close'
                        self.__delimStack.pop(-1)

# Open delimiter.
                elif delimFound in self.rules['open']:
                    kind = 'open'
                    self.__delimStack.append(delimFound)

# Close delimiter.
                else:
                    if not self.__delimStack \
                    or self.__delimStack[-1] != self.rules['close'][delimFound]:
                        message = "Not opened group << ... {0} >>.".format(
                            delimFound
                        )

                        self.__raise(message, iFound)

                    kind  = 'close'
                    delim = delimFound

                    self.__delimStack.pop(-1)

# Let's store the infos and continue.
                if kind != "verbatim":
                    textBefore = self.text[position: iFound]

                    if textBefore:
                        self.__listView.append((
                            'text',
                            self.__cleanText(textBefore),
                            depth
                        ))

                    if kind == "close":
                        depth -= 1
                        self.__listView.append((kind, delimFound, depth))

                    elif kind == "open":
                        self.__listView.append((kind, delimFound, depth))
                        depth += 1

                    position = iFound + len(delimFound)

            else:
                break

# Some groups have not been closed...
        if self.__delimStack:
            firstUnclosedGroup = self.__delimStack[0]
            p = 0

            message = "Not closed group << {0} ... >>.".format(
                firstUnclosedGroup
            )

            self.__raise(message, p)

# Last piece of text
        textEnd = self.text[position:]

        if textEnd:
            self.__listView.append((
                'text',
                self.__cleanText(textEnd),
                depth
            ))

        return self.__listView


# -- ASCII VIEW -- #

    def __str__(self):
        return self.ascii

    @property
    def ascii(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This property like method produces an ascii version of the tree found.
        """
        if (not self.__isNewText()) and self.__asciiTree:
            return self.__asciiTree

        self.__asciiTree = []

        for kind, text, level in self.listView:
            if kind == 'close':
                text = None

            elif kind == 'open':
                text = "{0} ... {1}".format(
                    text,
                    self.rules[kind][text][0]
                )


            if text != None:
                self.__asciiTree.append(
                    "{0}{1} {2}".format(
                        self.asciiTreeTab * level,
                        self.asciiTreeNode,
                        text
                    )
                )


        self.__asciiTree = '\n'.join(self.__asciiTree)

        return self.__asciiTree


# -- GROUP VIEW -- #

    def __buildGroupView(
        self,
        oneListView
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method produces a nested list which can be used for easy walk in the
groups found.
        """
        groupView = []

        i    = 0
        iMax = len(oneListView)

        while(i < iMax):
            kind, text, depth = oneListView[i]

            if kind == 'open':
                closeDelim = self.rules['open'][text]
                iClose     = i + oneListView[i:].index(
                    ('close', closeDelim, depth)
                )

                groupView.append((
                    'group',
                    (text, closeDelim),
                    self.__buildGroupView(oneListView[i+1: iClose])
                ))

                i = iClose

            else:
                groupView.append((kind, [], text))

            i += 1

        return groupView

    @property
    def groupView(self):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This property like method produces a group view version which is easy to walk in.
Indeed, all the job is done by the method ``self.__buildGroupView``.
        """
        if (not self.__isNewText()) and self.__groupView:
            return self.__groupView

        return self.__buildGroupView(self.listView)


# --------------------- #
# -- INDENTED BLOCKS -- #
# --------------------- #

def findIndent(
    text,
    spaceTab = None
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function simply returns the leading spaces and tabulations of one text. Each
tabulation is converted to spaces using the variable ``spaceTab`` if ``spaceTab
!= None`` where ``None`` is the default value.


:::::::::::::
The arguments
:::::::::::::

This function uses the following arguments.

    1) ``text`` is the text to analyse.

    2) ``spaceTab`` indicates the string of spaces which replaces each
    tabulation.
    """
    indent = ''

    if text.strip():
        i = 0

        while text[i] in " \t":
            indent += text[i]
            i+=1

    if spaceTab != None:
        indent = indent.replace('\t', spaceTab)

    return indent
