#!/usr/bin/env python3

"""
Directory : mistool
Name      : date_use
Version   : 2013.09
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This script contains only one function so as to translate dates.
"""

from mistool.config import date_name


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class DateUseError(ValueError):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Base class for errors in the ``date_use`` module of the package ``mistool``.
    """
    pass


# ----------------- #
# -- TRANSLATING -- #
# ----------------- #

LANG = 'en_GB'

__ALL_LANGS           = date_name.__ALL_LANGS
__POINTERS            = date_name.__POINTERS
__FORMAT_TRANSLATIONS = date_name.__FORMAT_TRANSLATIONS

def translate(
    date,
    format,
    lang = LANG
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

The aim of this function is to avoid the use of something like in the following
code even if the documentation of the standard package ``locale`` avoid to do
that kind of things.

python::
    import locale
    import datetime

    locale.setlocale(locale.LC_ALL, 'fr_FR')
    print (datetime.date(2013, 9, 21).strftime("%A %d %B %Y"))

This code prints the text terminal::``Samedi 29 septembre 2013`` in a terminal.
This can be achieved using the function ``translate`` like in the following code.

python::
    import datetime

    from mistool import date_use

    print(
        date_use.translate(
            date   = datetime.date(2013, 9, 21),
            format = "%A %d %B %Y",
            lang   = "fr_FR"
        )
    )


If you always want to use the same language, you can do it like this.

python::
    import datetime

    from mistool import date_use

    date_use.LANG = 'fr_FR'

    print(
        date_use.translate(
            date   = datetime.date(2013, 9, 21),
            format = "%A %d %B %Y"
        )
    )


info::
    The mechanism used in backstage is very primitive : it never calls the
    standard package ``locale`` !


:::::::::::::
The arguments
:::::::::::::

This function uses three variables.

    1) ``date`` is a date defined using the class ``datetime.date`` from the
    standard package ``datetime``.

    2) ``format`` is a string that uses the special formatters available with
    the method ``strftime`` of the class ``datetime.date``.

    3) ``lang`` is a language respecting the convention needed for the use of
    the function ``locale.setlocale`` of the standard package ``locale``.

    This optional variable used the default value ``LANG`` where ``LANG`` is a
    module constat defined by ``LANG = "en_GB"``.
"""
    if lang not in __ALL_LANGS:
        raise DateUseError(
            'Illegal value << {0} >> for the argument ``lang``.'.format(lang)
        )

    nbDay   = date.weekday()
    nbMonth = date.month - 1

    for oneSpeFormat in ['%a', '%A']:
        if oneSpeFormat in format:
            dayName = __POINTERS[
                __FORMAT_TRANSLATIONS[oneSpeFormat][lang]
            ][nbDay]

            format = format.replace(oneSpeFormat, dayName)

    for oneSpeFormat in ['%b', '%B']:
        if oneSpeFormat in format:
            monthName = __POINTERS[
                __FORMAT_TRANSLATIONS[oneSpeFormat][lang]
            ][nbMonth]

            format = format.replace(oneSpeFormat, monthName)

    return date.strftime(format)
