#!/usr/bin/env python3

"""
Directory : mistool
Name      : latex_use
Version   : 2012.09
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This script gives some useful functions in relation with the powerful LaTeX
langage for typing scientific documents.
"""

from os.path import normpath, join
from subprocess import check_call, check_output

from mistool import os_use
from mistool.config import latex


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class LaTeXUseError(ValueError):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Base class for errors in the ``latex_use`` module of the package ``mistool``.
    """
    pass

def __raiseSomeError(
    kind,
    path,
    action
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This method simply eases the raising of some specific errors.
    """
    if action == 'access':
        action = 'needs the "Super User\'s rights"'

    elif action == 'create':
        action = "can't be created"

    elif action == 'exist':
        action = "doesn't exist"

    elif action == 'notTeX':
        action = "is not a TeX one"

    else:
        raise Exception('BUG !')

    raise LaTeXUseError(
        "The following {0} {1}.\n\t<< {2} >>".format(kind, action, path)
    )


# ---------------- #
# -- FORMATTING -- #
# ---------------- #

CHAR_TO_ESCAPE   = latex.CHAR_TO_ESCAPE
CHAR_TO_LATEXIFY = latex.CHAR_TO_LATEXIFY

def escape(
    text,
    mode = "text"
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

In LaTeX language, somme character has one meaning. The purpose of this function
is to escape all this special charcaters so as to see them in the output
produced by LaTeX.

For example, ``escape("\OH/ & ...")`` is equal to ``"\textbackslash{}OH/ \&
..."``, whereas ``escape(text = "\OH/ & ...", mode = "math")`` is equal to
``"\backslash{}OH/ \& ..."``. Just notice the difference coming from the use of
``mode = "math"``. In math formulas and in simple text, the backslash is
indicated by using two different LaTeX commands.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``text`` is simply the text to change.

    2) ``mode`` indicate which kind of text is analysed. By default, ``mode =
    "text"`` is for one content that will be simply text in the document
    produced by LaTeX. You can also use ``mode = "math"`` to indicate something
    that appears in one formula.


info::
    Two global dictionaries are used : ``CHAR_TO_ESCAPE`` indicates only
    characters that must be escaped whit one backslash ``\``, and
    ``CHAR_TO_LATEXIFY`` is for expressions that needs one LaTeX command.
    """
    if not mode in CHAR_TO_ESCAPE:
        raise LaTeXUseError(
            "Unknown mode : << {0} >>.".format(mode)
        )

    charToLatexify = CHAR_TO_LATEXIFY[mode].items()
    charToEscape   = CHAR_TO_ESCAPE[mode]

    newText = ''

    iMax = len(text)
    i = 0

    while(i < iMax):
        nothingFound = True

        for oneChar in charToEscape:
            if text[i] == oneChar:
                newText += '\\' + oneChar
                i += 1
                nothingFound = False
                break

        if nothingFound:
            for oneChar, oneCharInLaTeX in charToLatexify:
                if text[i:].startswith(oneChar):
                    newText += oneCharInLaTeX
                    i += len(oneChar)
                    nothingFound = False
                    break

        if nothingFound:
            newText += text[i]
            i += 1

    return newText


# --------------- #
# -- COMPILING -- #
# --------------- #

class Build:
    """
:::::::::::::::::
Small description
:::::::::::::::::

This class defines methods related to the compilation of LaTeX documents.


:::::::::::::
The arguments
:::::::::::::

This class uses the following variables.

    1) ``path`` is the path of the LaTeX file to compile.

    2) ``repeat`` indicates how many compilations must be done (for example, if
    the document has one table of content, several compilations are needed). By
    default, ``repeat = 1``.

    3) ``verbose`` is a boolean with default value ``True``. Use ``verbose`` if
    you want to see, or not to see, the informations sent by LaTeX when it
    does the compilation.
    """
# Source :
#    * http://docs.python.org/py3k/library/subprocess.html
    SUBPROCESS_METHOD = {
# ``check_call`` prints informations given during the compilation.
        True : check_call ,
# ``check_output`` does not print informations given during the
# compilation. Indeed it returns all this stuff in one string.
        False: check_output
    }

    def __init__(
        self,
        path,
        repeat  = 1,
        verbose = True
    ):
# Does the file to compile exist ?
        if not os_use.isFile(path):
            __raiseSomeError(
                kind   = "file",
                path   = path,
                action = "exist"
            )

# Do we have TEX file ?
        if os_use.ext(path) != "tex":
            __raiseSomeError(
                kind   = "file",
                path   = path,
                action = "notTeX"
            )

# Infos given by the user.
        self.path      = path
        self.pathNoExt = os_use.fileName(path)

        self.repeat  = repeat
        self.verbose = verbose

# General infos about the LaTeX distribution and the OS.
        self.aboutLatex = about()

# The directory where to put the "paper" files.
        self.directory = os_use.parentDir(path)

    def __compile(
        self,
        arguments,
        repeat
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method launches the compilation in verbose mode or not.
        """
        for i in range(1, repeat + 1):
            if self.verbose:
                print(
                    '\t+ Start of compilation Nb.{0} +'.format(i)
                )

            self.SUBPROCESS_METHOD[self.verbose](
# We go in the directory of the file to compile.
                cwd = self.directory,
# We use the terminal actions.
                args = arguments
            )

            if self.verbose:
                print(
                    '\t+ End of compilation Nb.{0} +'.format(i)
                )

    def pdf(
        self,
        repeat = None
    ):
        """
:::::::::::::::::
Small description
:::::::::::::::::

This method calls LaTeX so as to build the PDF output.
        """
        if repeat == None:
            repeat = self.repeat

# The option "-interaction=nonstopmode" don't stop the terminal even if there is
# an error found by ¨latex.
        self.__compile(
            arguments = [
                'pdflatex',
                '-interaction=nonstopmode',
                self.path
            ],
            repeat = repeat
        )


# -------------- #
# -- CLEANING -- #
# -------------- #

CLASSIFIED_TEMP_EXT = latex.CLASSIFIED_TEMP_EXT
ALL_EXT_TO_CLEAN    = latex.ALL_EXT_TO_CLEAN

def clean(
    main,
    keep    = [],
    discard = ALL_EXT_TO_CLEAN,
    depth   = 0,
    verbose = False
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function removes extra files build during a LaTeX compilation.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``main`` will be either one instance of the class ``Build`` or the path
    of one file or of one directory.

    In the case of one instance of the class ``Build`` or of one file, only the
    file having the same name of the file build, or indicated, will be
    removed.

    If ``main`` is the path of one directory, then the function will look for
    all files with path::``tex`` extension, and then the eventual extra files
    associated will be removed. In that case, you can use the two arguments
    ``depth`` and ``verbose``.

    2) ``keep`` is for indicate the list of extensions to keep. This can be very
    usefull if you use the default value of the argument ``discard``.

    3) ``discard`` is for indicate the list of extensions to discard. By
    default, the extensions to clean are defined in the constant
    ``ALL_EXT_TO_CLEAN``.

    info::
        If you want to build your own list of extensions you can use the
        dictionary ``CLASSIFIED_TEMP_EXT``.

    4) ``depth`` is the maximal depth for the research of the path::``tex``
    files when ``main`` is the path of one directory. The very special value
    ``(-1)`` indicates that there is no maximum. The default value is ``0``
    which asks to only look for in the direct content of the main directory to
    analyse.

    5) ``verbose`` is a boolean which asks to print in one terminal the
    path::``tex`` files found when one search is made inside one directory. This
    argument is usefull only if ``main`` is the path of one directory.
    """
# One instance of ``Build()``
    if isinstance(main, Build):
        thePathsNoExt = [main.pathNoExt]

# One file
    elif os_use.isFile(main):
        if os_use.ext(main) != "tex":
            __raiseSomeError(
                kind   = "file",
                path   = main,
                action = "notTeX"
            )

        thePathsNoExt = [main[:-4]]

# One directory
    elif os_use.isDir(main):
        thePathsNoExt = [
            oneFile[:-4]
            for oneFile in os_use.nextFile(
                main  = main,
                ext   = {'keep': ["tex"]},
                depth = depth
            )
        ]

# Nothing existing
    else:
        thePathsNoExt = []

# Clean now !
    for pathNoExt in thePathsNoExt:
        if verbose:
            print('---> "{0}.tex"'.format(pathNoExt))

        for oneExt in [x for x in discard if x not in keep]:
            if '¨' in oneExt:
                print("TODO  --->  ", oneExt)
                ...

            else:
                os_use.destroy(pathNoExt + '.' + oneExt)


# ---------------------------------- #
# -- ABOUT THE LATEX DISTRIBUTION -- #
# ---------------------------------- #

# Sources :
#    * http://tex.stackexchange.com/questions/68730/terminal-commands-to-have-information-about-latex-distribution
#    * http://tex.stackexchange.com/questions/69483/create-a-local-texmf-tree-in-miktex
#    * https://groups.google.com/forum/?hl=fr&fromgroups=#!topic/fr.comp.text.tex/ivuCnUlW7i8

def mikeTexLocalDirectory():
    """
:::::::::::::::::
Small description
:::::::::::::::::

You can redefine this function to choose another path than
path::``C:/texmf-local`` for the directory where we'll put special LaTeX
packages.


warning::
    Customize this function only if you know what you are doing !
    """
    return 'C:/texmf-local'

def __localDirOfTexLive(
    osName = None
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function returns the path of the TeX Live directory where we'll put special
LaTeX packages.
    """
    if osName == None:
        osName = os_use.system()

# "check_output" is a byte string, so we have to use the method
# "decode" so as to obtain an "utf-8" string.
    try:
        if osName == "windows":
            localDir = check_output(
                args = ['kpsexpand',"'$TEXMFLOCAL'"]
            ).decode('utf8')

            return normpath(localDir.strip())

        elif osName in ["linux", "mac"]:
            localDir = check_output(
                args = ['kpsewhich','--var-value=TEXMFLOCAL']
            ).decode('utf8')

            return normpath(localDir.strip())

    except:
        ...

def about():
    """
:::::::::::::::::
Small description
:::::::::::::::::

The aim of this function is to give critical informations so to use the function
``install`.

This function returns the following kind of dictionary. This example has been
obtained with a Mac computer where TeXlive is installed.

python::
    {
        'osName'    : 'mac',
        'latexName' : 'texlive',
        'latexFound': True,
        'localDir'  : '/usr/local/texlive/texmf-local'
    }

The key ``'localDir'`` contains the path to use to install special packages.
    """
    osName = os_use.system()

    latexFound = False
    localDir = None

# Windows
    if osName == "windows":
        winPath = os_use.pathEnv()

# Is MiKTeX installed ?
        if '\\miktex\\' in winPath:
            latexFound = True
            latexName = 'miktex'

            if os_use.isDir(mikeTexLocalDirectory()):
                localDir = mikeTexLocalDirectory()

# Is TeX Live installed ?
        elif '\\texlive\\' in winPath:
            latexFound = True
            latexName = 'texlive'
            localDir = __localDirOfTexLive(osName)

# Linux and Mac
    elif osName in ["linux", "mac"]:
# Is LaTeX installed ?
        if check_output(args = ['which', 'pdftex']).strip():
            latexFound = True
            latexName = 'texlive'
            localDir = __localDirOfTexLive(osName)

# Unknown method...
    else:
        raise LaTeXUseError(
            "The OS << {0} >> is not supported. ".format(osName)
        )

# The job has been done...
    return {
        'osName'    : osName,
        'latexName' : latexName,
        'latexFound': latexFound,
        'localDir'  : localDir
    }


# ---------------- #
# -- INSTALLING -- #
# ---------------- #

def __issu(localLatexDir):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function tests if the script has been launched by the "Super User".
    """
    path = join(
        localLatexDir,
        '-p-y-t-o-o-l-t-e-s-t-.t-x-t-x-t'
    )

    try:
        os_use.makeTextFile(path)
        os_use.destroy(path)

    except:
        __raiseSomeError(
            kind   = "directory",
            path   = localLatexDir,
            action = "access"
        )

def makeMiktexLocalDir():
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function creates one local directory and add it to the list of directories
analysed by MiKTeX when it looks for packages.

The path of the local directory is given by ``mikeTexLocalDirectory()``
because this variable is also used by the function ``about`` and ``install``.

Even it is a really bad idea, you can change this path just after the
importation of the module ``latexUse``.
    """
    aboutLatex = about()

    if aboutLatex['osName'] != "windows" or aboutLatex['latexName'] != "miktex":
        raise LaTeXUseError(
            'This function can only be used with the OS "window" '
            'and the LaTeX distribution "miktex".'
        )

# Creation of the directory
    try:
        os_use.makeDir(mikeTexLocalDirectory())

    except:
        __raiseSomeError(
            kind   = "directory",
            path   = mikeTexLocalDirectory(),
            action = "create"
        )

# Adding the directory to the ones analysed by MiKTex.
    check_output(
        args = [
            'initexmf',
            '--admin',
            '--user-roots=' + mikeTexLocalDirectory(),
#           '--register-root=' + mikeTexLocalDirectory(),
            '--verbose'
        ]
    )

    check_output(
        args = [
            'initexmf',
            '--update-fndb',
            '--verbose'
            ]
        )

    print(
        '',
        'The following local directory for MiKTeX has been created.',
        '\t<< {0} >>'.format(mikeTexLocalDirectory()),
        sep = "\n"
    )

def refresh(
    aboutLatex = None
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function refreshes the list of packages directly known by the LaTeX
distribution. This always works in verbose mode (LaTeX is not very talkative
when it refreshes).

The only variable used is ``aboutLatex`` which is simply the informations
returned by the function ``about``.
    """
    if aboutLatex == None:
        aboutLatex = about()

# TeX Live
    if aboutLatex['latexName'] == 'texlive':
        check_call(args = ['mktexlsr'])

# MiKTex
    elif aboutLatex['latexName'] == 'miktex':
        check_call(args = ['initexmf', '--update-fndb', '--verbose'])

# Unkonwn !!!
    else:
        raise LaTeXUseError(
            'The refresh of the list of LaTeX packages is not supported '
            'with your LaTeX distribution.'
        )

def install(
    listFile,
    name,
    clean = False
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function helps you to easily install unofficial packages to your LaTeX
distribution.


warning::
    This function can only be called by one script launched by the super user.


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``name`` is the name of the LaTeX package to build.

    2) ``listFile`` is the list of the whole paths of the files to copy. This
    variable can be easily build with the function ``listFile`` from the module
    ``os_use`` contained in the ¨python package ``Mistool``. The tree structure
    will be similar to the one corresponding to the files to copy.

    3) ``clean`` is boolean variable to delete or not an eventual package
    directory named ``name`` in the local directory of the LaTeX distribution.
    The default value of ``clean`` is ``False``.
    """
# Sources :
#    * http://www.commentcamarche.net/forum/affich-7670740-linux-supprimer-le-contenu-d-un-repertoire
#    * http://stackoverflow.com/questions/185936/delete-folder-contents-in-python

# Directories
    aboutLatex = about()

    localLatexDir = aboutLatex['localDir']

    if localLatexDir == None:
        message = "No local directroy for special packages has been found."

        if aboutLatex['latexName'] == "miktex":
            message += "You can use the function << makeMiktexLocalDir >>."

        raise LaTeXUseError(message)

    localLatexDir = os_use.SEP.join([
        localLatexDir,
        'tex',
        'latex',
        name
    ])

# We must have the super user's rights with the local Latex directory.
    __issu(localLatexDir)

# We must find the smaller directory that contains all the files so as to
# respect the original tree directory structure.
    mainDir = os_use.commonPath(listFile)

# Directories to add
    dirAndTheirFiles = {}

    for oneFile in listFile:
        if not os_use.isFile(oneFile):
            __raiseSomeError(
                kind   = "file",
                path   = oneFile,
                action = "exist"
            )

        localLatexSubDir = localLatexDir + os_use.parentDir(
            os_use.relativePath(mainDir,oneFile)
        ).strip()

        if localLatexSubDir in dirAndTheirFiles:
            dirAndTheirFiles[localLatexSubDir].append(oneFile)
        else:
            dirAndTheirFiles[localLatexSubDir] = [oneFile]

# Installation on Mac O$, Linux and Windows
    if aboutLatex['osName'] in ['mac', 'linux', 'windows']:
# Actions to do are...
        actionsToDo = []
        decoTab     = '    * '
        decoTabPlus = '        + '
        decoTabMore = '            - '

# Creation of directories and copy of the files
#
#    1) Deletion of a local package named ``name``
        if clean and os_use.isDir(localLatexDir):
            print(
                decoTab + 'Deletion of the old << {0} >> '.format(name) \
                + 'package in the local LaTeX directory.'
            )

            os_use.destroy(localLatexDir)

#    2) Creation of the new local package named ``name``
        print(
            decoTab + 'Creation of the new << {0} >> '.format(name) \
            + 'package in the local LaTeX directory.'
        )

#    3) Creation of the new directories with their contents
        for oneNewDir in sorted(dirAndTheirFiles.keys()):
            print(
                decoTabPlus + 'Adding new directory --> << {0} >> '.format(
                    oneNewDir
                )
            )

            os_use.makeDir(oneNewDir)

            for oneFile in dirAndTheirFiles[oneNewDir]:
                if not os_use.isFile(oneFile):
                    __raiseSomeError(
                        kind   = "file",
                        path   = oneFile,
                        action = "exist"
                    )

                print(
                    decoTabMore + 'Adding new file  --> << {0} >> '.format(
                        os_use.fileName(oneFile) \
                        + "." \
                        + os_use.ext(oneFile)
                    )
                )

                localLatexSubDir = localLatexDir + os_use.parentDir(
                    os_use.relativePath(mainDir, oneFile)
                ).strip()

                os_use.copy(oneFile, localLatexSubDir)

#   We must refresh of the list of LaTeX packages.
        print(
            decoTab + 'Refreshing the list of LaTeX packages.'
        )

        refresh(aboutLatex)

# Unsupported OS
    else:
        raise LaTeXUseError(
            'The installation of local packages is not yet supported '
            'with the OS << {0} >>.'.format(aboutLatex['osName'])
        )
