#!/usr/bin/env python3

"""
Directory : mistool
Name      : python_use
Version   : 2013.10
Author    : Christophe BAL
Mail      : projetmbc@gmail.com

This module contains some simple tools about the Python programming language.
"""

from collections import OrderedDict

from mistool.config.group import GROUPS
from mistool.config.token import SEPARATORS
from mistool.parse_use.token import Groups, Tokens
from mistool.string_use import wrap


# ------------------------- #
# -- FOR ERRORS TO RAISE -- #
# ------------------------- #

class PythonUseError(ValueError):
    """
:::::::::::::::::
Small description
:::::::::::::::::

Base class for errors in the ``python_use`` module of the package ``mistool``.
    """
    pass


# ---------------------- #
# -- FOR DICTIONARIES -- #
# ---------------------- #

def dictSingleValues(obj):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function returns a single value list which contains all the values stored
in one dictionary.
    """
    return list(set(obj.values()))


# ------------- #
# -- QUOTING -- #
# ------------- #

QUOTE_SYMBOLS = ["'", '"']

def __escape__(data):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This small function escapes all the characters that must be escaped in one
python-like string.
    """
    if isinstance(data, str):
        return data.replace('\\', '\\\\')

    else:
        return data

def quote(
    text,
    symbol = "'"
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function put the content of the string variable ``text`` into quotes and
escapes the eventual quote symbols in ``text``.

This function has one optional variable ``symbol`` which indicates the prefered
quoting symbol which is the single quote ``'`` by default.


This function uses the global constant ``QUOTE_SYMBOLS`` which is defined by
``QUOTE_SYMBOLS = ["'", '"']``. This indicates the list of the possible quoting
symbols.

For example, ``quote("one \"small\" example")`` is equal to the text ``'one
"small" example``, and ``quote('another \'example\' to "see" more")``` is equal
to ``'another \'example\' to "see" more'``.
    """
    if all(x in text for x in QUOTE_SYMBOLS):
        text = text.replace(symbol , '\\' + symbol)

    elif symbol in text:
        for x in QUOTE_SYMBOLS:
            if x != symbol:
                symbol = x

    return "{0}{1}{0}".format(symbol, text)


# -------------------------- #
# -- WRITING PYTHON FILES -- #
# -------------------------- #

PY_PROTECT_TOKENS = Tokens(
    groups = GROUPS["python"],
    sep    = SEPARATORS["python"]
)

TEXT_WIDTH = 80

def lambdify(text):
    return "lambda " + text

def pyRepr(
    obj,
    name   = "",
    kind   = None,
    depth  = -1,
    tab    = " "*4,
    indent = 0,
    width  = TEXT_WIDTH,
    format = set(),
):
    """
:::::::::::::::::
Small description
:::::::::::::::::

This function returns one string representation, usable in ¨python codes, of one
¨python object. Here is an example of use.

python::
    from mistool import python_use

    myDict = {
    "key_1":["a","very","long","list","no","no","no","no","no","no","no","no",
    "no","no","no","no","no","no","no"],
    "key_2":{4:"four",1:"one",3:"three",2:"two"}
    }

    print(
        python_use.pyRepr(
            obj    = myDict,
            name   = "thiIsADict",
            depth  = 1,
            width  = 50,
            format = "wrap",
        )
    )


Launched in a terminal, the script will print the following wrapped text. You
can notice some problems to fix like the coma alone at the start of a line, and
the double point alone at the end of a line. This will be fixed later.

terminal::
    thisIsADict = {
        'key_1': ['a', 'very', 'long', 'list', 'no',
        'no', 'no', 'no', 'no', 'no', 'no', 'no', 'no'
        , 'no', 'no', 'no', 'no', 'no', 'no'],
        'key_2': {1: 'one', 2: 'two', 3: 'three', 4:
        'four'}
    }



For more informations, just take a look at the documentation of each arguments
that is just below .


:::::::::::::
The arguments
:::::::::::::

This function uses the following variables.

    1) ``obj`` is the ¨python object to print in one ¨python code.


    2) ``name`` is simply the name of the object. By default, this optional
    argument is an empty string. In that case, the function will simply return a
    representation without adding ``name =``.

    3) ``kind`` is an optional argument to indicate the kind of object. The
    default value is ``None`` which asks to use the type of the object. The
    supported values are the following ones.

        * ``"bool"``, ``"float"``, ``"int"``, ``"dict"``, ``"orderedDict"``,
        ``"tuple"``, ``"list"``, ``"set"`` and ``"str"`` indeed corresponds to
        ¨python types supported.

        * ``"regex"`` is useful for pattern definied via ``re.compile("...")``.

        * ``"repr"`` simply asks to use the standard ¨python representation.

    4) ``depth`` is an optional argument to indicate how deep to go in the
    printings of dictionaries and lists. The default value is ``(-1)`` which
    indicates to go the deeper is possible.

    5) ``tab`` is an optional argument which is the tabulation used for the
    representation of dictionaries and lists. The default value is ``" "*4``.

    6) ``indent`` is an optional argument that indicates the current level
    of indentation. The default value is ``0``.

    7) ``format`` is a set to customize the output. By default, ``format`` is
    the empty set. You can use the following strings.

        a) ``"wrap"`` indicates to use a kind of wrap mode for long content
        displayed at a zero depth. The width of the text is given by the module
        constant ``python_use.TEXT_WIDTH`` which is equal to ``80``.

        b) ``"short"`` is for using the short text ``OrderedDict`` instead of ``collections.OrderedDict`` for ordered dictionaries.

        c) ``"key"`` is for having a dictionary defining key by key using the
        syntax ``myDict[oneKey] = oneValue`` rather that directly defining the
        whole dictionary.

        warning::
            If you use very long dictionary, you must have to use the key by key
            definition so as to avoid Python to send an error.

        d) ``"empty"`` is to indicate to add or not an empty line before each new
        definition in the case of a key by key definition.

    info::
        If you only need one special formatting, you can just use a string.
    """

# -- NORMALIZATION AND VALIDATION -- #

    if not isinstance(name, str):
        raise PythonUseError('``name`` must be a string.')

    if isinstance(format, str):
        format = set([format])

    elif not isinstance(format, set):
        raise PythonUseError('``format`` must be a set.')

    if not format <= {"short", "key", "empty", "wrap"}:
        raise PythonUseError(
            "Unknown key(s) in the set ``format``.\n" \
            + ' , '.join([
                '"{}"'.format(x)
                for x in format - {"short", "key", "empty", "wrap"}
            ])
        )

    if ("key" in format) and (not name):
        raise PythonUseError(
            "You can't use the key-format for a dictionary without a name."
        )

    before = tab*indent

# -- INNER FUNCTIONS -- #

    def __isGoodType(wantedType):
        if objType != wantedType:
            messageType = {
                str        : 'string',
                list       : 'list',
                tuple      : 'tuple',
                set        : 'set',
                dict       : 'dictionary',
                OrderedDict: 'ordered dictionary'
            }

            wantedType = messageType.get(wantedType, str(wantedType))

            message = 'With << kind = "{0}" >>, the object to represent ' \
                    + 'must be one << {1} >>.'

            raise PythonUseError(
                message.format(kind, wantedType)
            )

    def __glueDelimContent(start, end, text):
        text = "\n".join(text)

        return "{0}\n{2}\n{1}".format(start, before + end, text)

    def __reprListLike():
        if kind == "list":
            start, end = "[", "]"

        elif kind == "tuple":
            start, end = "(", ")"

        elif kind == "set":
            start, end = "{", "}"

        if not obj:
            return start + end

        if depth == 0:
            return repr(obj)

        text = []

        for elt in obj:
            elt = pyRepr(
                obj    = elt,
                indent = indent + 1,
                tab    = tab,
                depth  = depth - 1,
                width  = width,
                format = format - {"wrap", "key", "empty"}
            )

            text.append(elt + ",")

        text[-1] = text[-1][:-1]

        return __glueDelimContent(start, end, text)

    def __reprDict():
        if not obj:
            return "{}"

        try:
            sortedKeys = sorted(obj.keys())

        except:
            sortedKeys = obj.keys()

        if "key" in format:
            return "\n".join([
                "{}",
                __reprDictKeyByKey(sortedKeys)
            ])

        start, end = "{", "}"
        text       = []

# Taking care of the order of the keys even in a inline representation !
        if depth == 0:
            for key in sortedKeys:
                value = pyRepr(
                    obj    = obj[key],
                    depth  = 0,
                    width  = width,
                    format = format - {"wrap", "key", "empty"}
                )

                key = pyRepr(
                    obj    = key,
                    depth  = 0,
                    width  = width,
                    format = format - {"wrap", "key", "empty"}
                )

                text.append("{0}: {1}, ".format(key, value))

            text[-1] = text[-1][:-2]

            return start + "".join(text) + end

        else:
            for key in sortedKeys:
                value = pyRepr(
                    obj    = obj[key],
                    indent = indent + 1,
                    tab    = tab,
                    depth  = depth - 1,
                    width  = width,
                    format = format - {"wrap", "key", "empty"}
                ).lstrip()

                key = tab*(indent+1) + pyRepr(
                    obj    = key,
                    depth  = 0,
                    width  = width,
                    format = format - {"wrap", "key", "empty"}
                )

                text.append("{0}: {1},".format(key, value))

            text[-1] = text[-1][:-1]

            return __glueDelimContent(start, end, text)

    def __reprOrderedDict():
        if "short" in format:
            start = "OrderedDict"

        else:
            start = "collections.OrderedDict"

        if not obj:
            return start + "()"

        if "key" in format:
            return "\n".join([
                start + "()",
                __reprDictKeyByKey(obj.keys())
            ])

        start, end = start + "([", "])"
        text       = []

# Taking care of the order of the keys even in a inline representation !
        if depth == 0:
            for key, value in obj.items():
                value = pyRepr(
                    obj    = value,
                    depth  = 0,
                    width  = width,
                    format = format - {"wrap", "key", "empty"}
                )

                key = pyRepr(
                    obj    = key,
                    depth  = 0,
                    width  = width,
                    format = format - {"wrap", "key", "empty"}
                )

                text.append("({0}, {1}), ".format(key, value))

            text[-1] = text[-1][:-2]

            return start + "".join(text) + end

        else:
            for key, value in obj.items():
                text.append(
                    pyRepr(
                        obj    = (key, value),
                        indent = indent + 1,
                        tab    = tab,
                        depth  = depth - 1,
                        width  = width,
                        format = format - {"wrap", "key", "empty"}
                    ) + ","
                )

            text[-1] = text[-1][:-1]

            return __glueDelimContent(start, end, text)

    def __reprDictKeyByKey(sortedKeys):
        text = []

        if "empty" in format:
            subBefore = "\n"
        else:
            subBefore = ""

        subBefore += before

        if depth == 0:
            subDepth = 0
        else:
            subDepth = depth - 1

        for key in sortedKeys:
            value = pyRepr(
                obj    = obj[key],
                indent = indent,
                tab    = tab,
                depth  = subDepth,
                width  = width,
                format = format - {"key", "empty"}
            ).lstrip()

            key = pyRepr(
                obj    = key,
                depth  = 0,
                width  = width,
                format = format - {"key", "empty"}
            )

            text.append('{0}{1}[{2}] = {3}'.format(subBefore, name, key, value))

        return '\n'.join(text)

# -- ACTIONS -- #

    whatToDo = {
        "list"       : (list, __reprListLike),
        "tuple"      : (tuple, __reprListLike),
        "set"        : (set, __reprListLike),
        "dict"       : (dict, __reprDict),
        "orderedDict": (OrderedDict, __reprOrderedDict)
    }

    internalKind = {y[0]: x for x, y in whatToDo.items()}

# Special case of the None class.
    if obj == None:
        return repr(None)

# We have to guess the type of the object. Very ugly !!!
    objType = type(obj)

    if kind == None:
        if objType in [
            bool,
            float, int,
            list, tuple, set,
            dict,
            str,
        ]:
            kind = str(objType)
            kind = kind[kind.find("'")+1:]
            kind = kind[:kind.find("'")]

        elif objType == OrderedDict:
            kind = "orderedDict"

        elif str(objType) == "<class '_sre.SRE_Pattern'>":
            kind = "regex"

        else:
            raise PythonUseError(
                'Unsupported type "{0}".'.format(objType)
            )

# We do the job.
    if kind == "bool":
        answer = repr(bool(obj))

    elif kind in ["str", "float", "int", "repr"]:
        answer = repr(obj)

    elif kind == "regex":
        if depth == 0:
            answer = 're.compile("{0}")'.format(obj.pattern)

        else:
            answer = '\n'.join([
                're.compile(',
                before + tab + '"{0}"'.format(obj.pattern),
                before + ')'
            ])

    elif kind in whatToDo:
        typeToTest, reprFunc = whatToDo[kind]

        __isGoodType(typeToTest)
        answer = reprFunc()

    else:
        raise PythonUseError('Unknown kind "{0}".'.format(kind))

    if name:
        name += " = "

    answer = "{0}{1}{2}".format(before, name, answer)

    if "wrap" in format:
        answer = wrap(
            text   = answer,
            width  = width,
            tokens = PY_PROTECT_TOKENS
        )

    return answer
