# Copyright (C) 1998-2014 by the Free Software Foundation, Inc.
#
# This file is part of mmci.
#
# mmci is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# mmci is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# mmci.  If not, see <http://www.gnu.org/licenses/>.

import os
import subprocess
import abc
import nose2
import exceptions
from mmci import version
from ConfigParser import ConfigParser
from ConfigParser import NoSectionError, NoOptionError
from twisted.internet import reactor
from twisted.internet.error import ReactorNotRestartable
from mmci.client.base.config import Config
from mmci.client.base.errors import CommandFailure, ProjectNotFound, ConfigNotFound, ConfigError, ConnectionError
from mmci.client.base.logger import Log
from mmci.client.base.client import Client, Factory

__metaclass__ = type
__all__ = ['Command']


class Command(object):
    """ Initializes the command"""

    @staticmethod
    def load_config(client_path):
        """ Load a configuration file
        :param client_path: The path of the client from which command is called
        :type client_path: str
        :return: Returns the ConfigParser object after reading the config file
        :rtype: ConfigParser object
        
        """
        config_path = "%s/config/mmci.cfg" % client_path
        config = ConfigParser()
        config.read(config_path)
        return config

    @staticmethod
    def parse_config(client_path, config):
        """ Parse a configuration file and returns the dictionary of projects
        :param client_path: The path of client from which command is called
        :type client_path: str
        :param config: The ConfigParser object after reading the configuration file
        :type config: ConfigParser object
        :return: Dictionary of projects having further dictionaries for version control, source, path, remote(in git) and branch(in git)
        :rtype: dict
        """
        projects = []
        try:
            config_items = config.items('projects')
        except NoSectionError:
            raise ConfigNotFound("Configation file not found at %s" % client_path)
        for item in config_items:
            if item[1] == '1':
                projects.append(item[0])
        projects_dict = {}
        for project in projects:
            try:
                project_module = __import__(project)
            except ImportError:
                Log("%s is not installed, checking for local configuration" % project, client_path, level=30)
                parent_path = os.path.abspath(os.path.join(client_path, os.pardir))
                try:
                    path = config.get(project, 'path')
                    src = config.get(project, 'src')
                    vcs = config.get(project, 'vcs')
                    if vcs == 'git':
                        remote = config.get(project, 'remote')
                        branch = config.get(project, 'branch')
                    else:
                        remote = None
                        branch = None
                except NoSectionError:
                    raise ConfigError("""Project not installed and not found in config, either install the project using pip install %s
                                      or enter the values in config file at %s/config/mmci.cfg""" % (project, client_path))
                except NoOptionError:
                    raise ConfigError("Values for either path, src, vcs, remote and branch not found in %s/config/mmci.cfg" % client_path)
                Log("Reading configuraion entries for %s" % project, client_path, level=20)
                projects_dict['%s' % project] = {'vcs' : vcs,
                                             'path' : '%s/%s' % (parent_path, path),
                                             'src' : '%s/%s' %(parent_path, src),
                                             'remote' : remote, 'branch': branch}
                continue
            
            src = project_module.__path__[0]
            os.chdir(src)
            while((os.path.isdir('.bzr') == False) and (os.path.isdir('.git') == False)):
                os.chdir('..')
            path = os.getcwd()
            if os.path.isdir('.bzr'):
                vcs = 'bzr'
                remote = None
                branch = None
            elif os.path.isdir('.git'):
                vcs = 'git'
                remote = config.get(project_module.__name__, 'remote')
                branch = config.get(project_module.__name__, 'branch')

            projects_dict['%s' % project_module.__name__] = {'vcs' : vcs, 'path' : path,
                                                      'src' : src, 'remote' : remote,
                                                      'branch' : branch}
        return projects_dict
         

    def pull(self, client_path):
        """ Pulls the projects defined in config/ci.cfg of the created project
        :param client_path: The path of client from which command is called
        :type client_path: str
        :return: None
        """
        config = self.load_config(client_path)
        projects_dict = self.parse_config(client_path, config)
        for project, values in projects_dict.items():
            Log("Updating project %s with source %s" % (project, values['src']), client_path, level=20)
            os.chdir(values['src'])
            if values['vcs'] == 'git':
                remote = config.get(project, 'remote')
                branch = config.get(project, 'branch')
                os.system('%s pull %s %s' % (values['vcs'], remote, branch))
            else:
                os.system('%s pull' % values['vcs'])
        
    def test(self, client_path):
        """ Tests all the projects listed in config/ci.cfg of the created project
        :param client_path: The path of client from which command is called
        :type client_path: str
        :return: None
        """
        config = self.load_config(client_path)
        projects_dict = self.parse_config(client_path, config)
        for project, values in projects_dict.items():
            Log("Testing project %s" % project, client_path, level=20)
            os.chdir(values['src'])
            try:
                nose2.discover(argv=['-v'])
            except:
                pass

    def try_(self, client_path):
        config = self.load_config(client_path)
        projects_dict = self.parse_config(client_path, config)
        for project, values in projects_dict.items():
            os.chdir(values['path'])
            os.system('%s diff > %s/patches/%s.patch' % (values['vcs'], client_path, project))
            Log("%s.patch created" % project, client_path, level=20)
            server_url = config.get('others', 'server_url')
            server_port = int(config.get('others', 'server_port'))
            try_ = Factory()
            reactor.connectTCP(server_url, server_port, try_)
            Log("Connecting Twisted reactor" , client_path, level=20)
            try:
                reactor.run()
            except ReactorNotRestartable:
                raise ConnectionError("Error in connecting to the server")
