#
# Copyright 2011 Alexander Sviridenko
#

"""Utilities for module control."""

import inspect, sys
from types import *

class ModuleError(Exception):
    """The base class for the exceptions related to module."""
    pass

def caller():
    """Return the context of the current function call. Return a tuple 
    (function name, module name, file name, line)."""
    function_name = inspect.getouterframes(inspect.currentframe())[2][3]
    mod_name = inspect.getmodule(inspect.stack()[2][0]).__name__
    file_name = inspect.getmodule(inspect.stack()[2][0]).__file__
    line = inspect.getouterframes(inspect.currentframe())[2][2]
    return (function_name, mod_name, file_name, line)

def get_module_classes(mod_name=None):
    """Return all the classes of a module mod_name in a list of (name, value) 
    pairs. The list does not include imported or built-in classes. If mod_name 
    does not provided the caller's module will be used."""
    if not mod_name:
	mod_name = caller()[1]
    mod = module(mod_name)
    classes = {}
    for name, obj in inspect.getmembers(mod):
	if inspect.isclass(obj) and (obj.__module__ == mod_name):
	    classes[name] = obj
    return classes

def module(mod_name=None):
    """Return module object of a module mod_name. If mod_name does not provided 
    the caller's module will be used."""
    if type(mod_name) is ModuleType:
	return mod_name
    elif type(mod_name) is NoneType:
	mod_name = caller()[1]
    elif type(mod_name) is not StringType:
	raise TypeError
    if not sys.modules.has_key(mod_name):
	raise ModuleError, "Module '%s' does not exist" % mod_name
    return sys.modules[mod_name]

def get_module_name(default=None, mod=None):
    """Return module name. If the name attribute does not exist, default is 
    returned. If mod does not provided the caller's module will be used."""
    if not mod:
	mod_name = caller()[1]
	return mod_name
    mod = module(mod)
    return getattr(mod, '__name__', default)

def get_module_file(mod_name=None):
    """Return the name of the file in which the module mod_name was defined. If 
    mod_name does not provided the caller's module will be used."""
    if not mod_name:
	mod_name = caller()[1]
    mod = module(mod_name)
    return inspect.getfile(mod)

def get_module_version(default=None, mod_name=None):
    """Return module version. If the version attribute does not exist, default is 
    returned. If mod_name does not provided the caller's module will be used."""
    if not mod_name:
	mod_name = caller()[1]
    mod = module(mod_name)
    return getattr(mod, '__version__', default)

def set_module_version(value, mod_name=None):
    """Set module version. If mod_name does not provided the caller's module will 
    be used."""
    if not mod_name:
	mod_name = caller()[1]
    mod = module(mod_name)
    setattr(mod, '__version__', value)

def get_module_copyright(default=None, mod_name=None):
    """Return a string containing the copyright pertaining to the module mod_name.
    If the copyright attribute does not exist, default is returned. If mod_name 
    does not provided the caller's module will be used."""
    if not mod_name:
	mod_name = caller()[1]
    mod = module(mod_name)
    return getattr(mod, '__copyright__', default)

def set_module_copyright(value, mod_name=None):
    """Set module copyright. If mod_name does not provided the caller's module 
    will be used."""
    if not mod_name:
	mod_name = caller()[1]
    mod = module(mod_name)
    setattr(mod, '__copyright__', value)

def get_module_doc(default=None, mod_name=None):
    """Return the documentation string for a module mod_name. If mod_name 
    does not provided the caller's module will be used."""
    if not mod_name:
	mod_name = caller()[1]
    mod = module(mod_name)
    return inspect.getdoc(mod) or default

class Module(object):
    def __init__(self, mod=None):
	if not mod:
	    mod = inspect.getmodule(self)
	self.mod = module(mod)

    def get_name(self, default=None):
	return get_module_name(default, self.mod)

    def get_file(self):
	return get_module_file(self.mod)
    
    def get_version(self, default=None):
	return get_module_version(default, self.mod)

    def set_version(self, value):
	set_module_version(value, self.mod)

    def get_copyright(self, default=None):
	return get_module_copyright(default, self.mod)

    def set_copyright(self, value):
	set_module_copyright(value, self.mod)

    def get_module_doc(self, default=None):
	return get_module_doc(default, self.mod)

set_module_version('0.1.1')
set_module_copyright('Copyright 2011 Alexander Sviridenko')

if __name__ == '__main__':
    print "Module version:", get_module_version()
    print "Module copyright:", get_module_copyright()


