#-*- coding: utf8 -*-

import datetime
import i18n


class TimeDelta(object):
    __slots__ = ("ago", "days", "hours", "minutes", "seconds", "total_seconds")

    def __init__(self, datetime_a, datetime_b):
        if datetime_a > datetime_b:
            self.ago = True
        else:
            self.ago = False
            datetime_a, datetime_b = datetime_b, datetime_a

        delta = datetime_a - datetime_b
        total_seconds = delta.days * 86400 + delta.seconds
        minutes, seconds = divmod(total_seconds, 60)
        hours, minutes = divmod(minutes, 60)
        days, hours = divmod(hours, 24)
        self.total_seconds, self.days, self.hours, self.minutes, self.seconds = \
            total_seconds, days, hours, minutes, seconds

    def __str__(self):
        return "Delta(%s day(s), %s hour(s), %s minute(s), %s second(s))" % (
            self.days, self.hours, self.minutes, self.seconds)


_locales = {}


class LocaleLoadError(ImportError):
    def __init__(self, locale, ex):
        self.message = "Cannot load locale module \"i18n.%s\". Exception: %s." % (locale, ex)

    def __str__(self):
        return self.message


class InputArgumentError(StandardError):
    def __init__(self, argument, ex):
        self.message = "Invalid value for argument \"%s\". Exception: %s." % (argument, ex)

    def __str__(self):
        return self.message


class LocaleContext(object):
    __slots__ = ("locale",)

    def __init__(self, locale="en"):
        self.locale = None
        self.set_locale(locale)

    def set_locale(self, locale):
        try:
            self.locale = _locales[locale]
        except KeyError:
            try:
                locale = getattr(__import__("i18n.%s" % locale, globals=globals()), locale).Locale()
                if not isinstance(locale, i18n.BaseLocale):
                    raise ValueError()
                self.locale = locale
            except Exception, ex:
                raise LocaleLoadError(locale, ex)

    def from_now(self, value, precision=1, from_utc=False):
        if isinstance(value, TimeDelta):
            delta = value
        else:
            if isinstance(value, datetime.datetime):
                datetime_obj = value
            else:
                try:
                    if from_utc:
                        datetime_obj = datetime.datetime.utcfromtimestamp(value)
                    else:
                        datetime_obj = datetime.datetime.fromtimestamp(value)
                except (TypeError, ValueError), ex:
                    raise InputArgumentError("value", ex)

            now = datetime.datetime.now()
            delta = TimeDelta(now, datetime_obj)

        parts = []
        if precision and delta.days > 0:
            precision -= 1
            parts.append(self.locale.relative(i18n.DAY, delta.days))
        if precision and delta.hours > 0:
            precision -= 1
            parts.append(self.locale.relative(i18n.HOUR, delta.hours))
        if precision and delta.minutes > 0:
            precision -= 1
            parts.append(self.locale.relative(i18n.MINUTE, delta.minutes))
        if precision:
            parts.append(self.locale.relative(i18n.SECOND, delta.seconds))

        return (self.locale.past if delta.ago else self.locale.future)(", ".join(parts))


_locale = LocaleContext()


def set_locale(locale):
    _locale.set_locale(locale)


def from_now(datetime_or_timestamp, precision=1, from_utc=False):
    return _locale.from_now(datetime_or_timestamp, precision=precision, from_utc=from_utc)