"""Assorted functions for statistical calculations."""
from __future__ import division
import numpy as np
import scipy as sp
from scipy import stats
import pandas as pd
import statsmodels.api as sm
from sklearn.metrics import r2_score
from sklearn.cross_validation import (cross_val_score,
                                      LeaveOneOut, LeaveOneLabelOut)


def bootstrap(*args, **kwargs):
    """Resample one or more arrays and call a function on each sample.

    Positional arguments are a sequence of arrays to bootrap
    along the first axis and pass to a summary function.

    Keyword arguments:
        n_boot : int
            number of iterations
        axis : int
            will pass axis to ``func``
        smooth : bool
            if True, performs a smoothed bootstrap
            (draws samples from a kernel destiny estimate)
            only works for one-dimensional inputs
        func : callable
            function to call on the args that are passed in

    Returns
    -------
    boot_dist: array
        array of bootstrapped statistic values

    """
    # Ensure list of arrays are same length
    if len(np.unique(map(len, args))) > 1:
        raise ValueError("All input arrays must have the same length")
    n = len(args[0])

    # Default keyword arguments
    n_boot = kwargs.get("n_boot", 10000)
    func = kwargs.get("func", np.mean)
    axis = kwargs.get("axis", None)
    smooth = kwargs.get("smooth", False)
    if axis is None:
        func_kwargs = dict()
    else:
        func_kwargs = dict(axis=axis)

    # Do the bootstrap
    boot_dist = []
    if smooth:
        kde = [stats.gaussian_kde(np.transpose(a)) for a in args]
    for i in xrange(int(n_boot)):
        if smooth:
            sample = [a.resample(n).T for a in kde]
        else:
            resampler = np.random.randint(0, n, n)
            sample = [a[resampler] for a in args]
        boot_dist.append(func(*sample, **func_kwargs))
    return np.array(boot_dist)


def percentiles(a, pcts, axis=None):
    """Like scoreatpercentile but can take and return array of percentiles.

    Parameters
    ----------
    a : array
        data
    pcts : sequence of percentile values
        percentile or percentiles to find score at
    axis : int or None
        if not None, computes scores over this axis

    Returns
    -------
    scores: array
        array of scores at requested percentiles
        first dimension is length of object passed to ``pcts``

    """
    scores = []
    try:
        n = len(pcts)
    except TypeError:
        pcts = [pcts]
        n = 0
    for i, p in enumerate(pcts):
        if axis is None:
            score = stats.scoreatpercentile(a.ravel(), p)
        else:
            score = np.apply_along_axis(stats.scoreatpercentile, axis, a, p)
        scores.append(score)
    scores = np.asarray(scores)
    if not n:
        scores = scores.squeeze()
    return scores


def vector_reject(x, y):
    """Remove y from x using vector rejection."""
    x = np.asarray(x).astype(float).reshape((-1, 1))
    y = np.asarray(y).astype(float).reshape((-1, 1))
    x_ = x - np.dot(x.T, y).T * (y / np.dot(y.T, y))
    return x_.ravel()


def ci(a, which=95, axis=None):
    """Return a percentile range from an array of values."""
    p = 50 - which / 2, 50 + which / 2
    return percentiles(a, p, axis)


def add_constant(a):
    """Add a constant term to a design matrix.

    Parameters
    ----------
    a : array
        original design matrix

    Returns
    -------
    a : array
        design matrix with constant as final column

    """
    return np.column_stack((a, np.ones(len(a))))


def randomize_onesample(a, n_iter=10000, h_0=0, corrected=True,
                        random_seed=None, return_dist=False):
    """Nonparametric one-sample T test through randomization.

    On each iteration, randomly flip the signs of the values in ``a``
    and test the mean against 0.

    If ``a`` is two-dimensional, it is assumed to be shaped as
    (n_observations, n_tests), and a max-statistic based approach
    is used to correct the p values for multiple comparisons over tests.

    Parameters
    ----------
    a : array-like
        input data to test
    n_iter : int
        number of randomization iterations
    h_0 : float, broadcastable to tests in a
        null hypothesis for the group mean
    corrected : bool
        correct the p values in the case of multiple tests
    random_seed : int or None
        seed to use for random number generator
    return_dist : bool
        if True, return the null distribution of t statistics

    Returns
    -------
    obs_t : float or array of floats
        group mean T statistic(s) corresponding to tests in input
    obs_p : float or array of floats
        one-tailed p value that the population mean is greater than h_0
        (1 - percentile under the null)
    dist : ndarray, optional
        if return_dist is True, the null distribution of t statistics

    """
    a = np.asarray(a)
    if a.ndim < 2:
        a = a.reshape(-1, 1)
    n_samp, n_test = a.shape

    a -= h_0

    rs = np.random.RandomState(random_seed)
    flipper = (rs.uniform(size=(n_samp, n_iter)) > 0.5) * 2 - 1
    flipper = (flipper.reshape(n_samp, 1, n_iter) *
               np.ones((n_samp, n_test, n_iter), int))
    rand_dist = a[:, :, None] * flipper

    err_denom = np.sqrt(n_samp - 1)
    std_err = rand_dist.std(axis=0) / err_denom
    t_dist = rand_dist.mean(axis=0) / std_err

    obs_t = a.mean(axis=0) / (a.std(axis=0) / err_denom)
    if corrected:
        cdf = sm.distributions.ECDF(t_dist.max(axis=0))
        obs_p = 1 - cdf(obs_t)
    else:
        obs_p = []
        for obs_i, null_i in zip(obs_t, t_dist):
            cdf = sm.distributions.ECDF(null_i)
            obs_p.append(1 - cdf(obs_i))
        obs_p = np.array(obs_p)

    if a.shape[1] == 1:
        obs_t = np.asscalar(obs_t)
        obs_p = np.asscalar(obs_p)
        t_dist = t_dist.squeeze()

    if return_dist:
        return obs_t, obs_p, t_dist
    return obs_t, obs_p


def randomize_corrmat(a, tail="both", corrected=True, n_iter=1000,
                      random_seed=None, return_dist=False):
    """Test the significance of set of correlations with permutations.

    By default this corrects for multiple comparisons across one side
    of the matrix.

    Parameters
    ----------
    a : n_vars x n_obs array
        array with variables as rows
    tail : both | upper | lower
        whether test should be two-tailed, or which tail to integrate over
    corrected : boolean
        if True reports p values with respect to the max stat distribution
    n_iter : int
        number of permutation iterations
    random_seed : int or None
        seed for RNG
    return_dist : bool
        if True, return n_vars x n_vars x n_iter

    Returns
    -------
    p_mat : float
        array of probabilites for actual correlation from null CDF

    """
    if tail not in ["upper", "lower", "both"]:
        raise ValueError("'tail' must be 'upper', 'lower', or 'both'")

    rs = np.random.RandomState(random_seed)

    a = np.asarray(a)
    flat_a = a.ravel()
    n_vars, n_obs = a.shape

    # Do the permutations to establish a null distribution
    null_dist = np.empty((n_vars, n_vars, n_iter))
    for i_i in xrange(n_iter):
        perm_i = np.concatenate([rs.permutation(n_obs) + (v * n_obs)
                                 for v in range(n_vars)])
        a_i = flat_a[perm_i].reshape(n_vars, n_obs)
        null_dist[..., i_i] = np.corrcoef(a_i)

    # Get the observed correlation values
    real_corr = np.corrcoef(a)

    # Figure out p values based on the permutation distribution
    p_mat = np.zeros((n_vars, n_vars))
    upper_tri = np.triu_indices(n_vars, 1)

    if corrected:
        if tail == "both":
            max_dist = np.abs(null_dist[upper_tri]).max(axis=0)
        elif tail == "lower":
            max_dist = null_dist[upper_tri].min(axis=0)
        elif tail == "upper":
            max_dist = null_dist[upper_tri].max(axis=0)

        cdf = sm.distributions.ECDF(max_dist)

        for i, j in zip(*upper_tri):
            observed = real_corr[i, j]
            if tail == "both":
                p_ij = 1 - cdf(abs(observed))
            elif tail == "lower":
                p_ij = cdf(observed)
            elif tail == "upper":
                p_ij = 1 - cdf(observed)
            p_mat[i, j] = p_ij

    else:
        for i, j in zip(*upper_tri):

            null_corrs = null_dist[i, j]
            cdf = sm.distributions.ECDF(null_corrs)

            observed = real_corr[i, j]
            if tail == "both":
                p_ij = 2 * (1 - cdf(abs(observed)))
            elif tail == "lower":
                p_ij = cdf(observed)
            elif tail == "upper":
                p_ij = 1 - cdf(observed)
            p_mat[i, j] = p_ij

    # Make p matrix symettrical with nans on the diagonal
    p_mat += p_mat.T
    p_mat[np.diag_indices(n_vars)] = np.nan

    if return_dist:
        return p_mat, null_dist
    return p_mat


def randomize_classifier(data, model, n_iter=1000, cv_method="run",
                         random_seed=None, return_dist=False, dv=None):
    """Randomly shuffle class labels to build a null distribution of accuracy.

    Randimization can be distributed over an IPython cluster using the ``dv``
    argument. Otherwise, it runs in serial.

    Parameters
    ----------
    data : dict
        single-subject dataset dictionary
    model : scikit-learn estimator
        model object to fit
    n_iter : int
        number of permutation iterations
    cv_method : run | sample | cv arg for cross_val_score
        cross validate over runs, over samples (leave-one-out)
        or otherwise something that can be provided to the cv
        argument for sklearn.cross_val_score
    random_state : int
        seed for random state to obtain stable permutations
    return_dist : bool
        if True, return null distribution
    dv : IPython direct view
        view onto IPython cluster for parallel execution over iterations

    Returns
    -------
    p_vals : n_tp array
        array of one-sided p values for observed classification scores
        against the empirical null distribution
    null_dist : n_iter x n_tp array
        array of null model scores, only if asked for it

    """
    if dv is None:
        import __builtin__
        _map = __builtin__.map
    else:
        _map = dv.map_sync

    # Set up the data properly
    X = data["X"]
    y = data["y"]
    runs = data["runs"]
    if cv_method == "run":
        cv = LeaveOneLabelOut(runs)
    elif cv_method == "sample":
        cv = LeaveOneOut(len(y))
    else:
        cv = cv_method
    if X.ndim < 3:
        X = [X]

    def _perm_decode(model, X, y, cv, perm):
        """Internal func for parallel purposes."""
        y_perm = y[perm]
        perm_acc = cross_val_score(model, X, y_perm, cv=cv).mean()
        return perm_acc

    # Make lists to send into map()
    model_p = [model for i in range(n_iter)]
    y_p = [y for i in range(n_iter)]
    cv_p = [cv for i in range(n_iter)]

    # Permute within run
    rs = np.random.RandomState(random_seed)
    perms = []
    for i in range(n_iter):
        perm_i = []
        for run in np.unique(runs):
            perm_r = rs.permutation(np.sum(runs == run))
            perm_r += np.sum(runs == run - 1)
            perm_i.append(perm_r)
        perms.append(np.concatenate(perm_i))

    # Actually do the permutations, possibly in parallel
    null_dist = []
    for X_i in X:
        X_p = [X_i for i in range(n_iter)]
        tr_scores = _map(_perm_decode, model_p, X_p, y_p, cv_p, perms)
        null_dist.append(tr_scores)
    null_dist = np.array(null_dist).T

    # Calculate a p value for each TR
    p_vals = []
    for i, dist_i in enumerate(null_dist.T):
        acc_i = cross_val_score(model, X[i], y, cv=cv).mean()
        cdf_i = sm.distributions.ECDF(dist_i)
        p_vals.append(1 - cdf_i(acc_i))
    p_vals = np.array(p_vals)

    if return_dist:
        return p_vals, null_dist
    return p_vals


def transition_probabilities(sched):
    """Return probability of moving from row trial to col trial.

    Parameters
    ----------
    sched : array or pandas Series
        event schedule

    Returns
    -------
    trans_probs : pandas DataFrame

    """
    sched = np.asarray(sched)
    trial_types = np.sort(np.unique(sched))
    n_types = len(trial_types)
    trans_probs = pd.DataFrame(columns=trial_types, index=trial_types)

    for type in trial_types:
        type_probs = pd.Series(np.zeros(n_types), index=trial_types)
        idx, = np.nonzero(sched[:-1] == type)
        idx += 1
        type_probs.update(pd.value_counts(sched[idx]))
        trans_probs[type] = type_probs

    trans_probs = trans_probs.divide(pd.value_counts(sched[:-1]))
    return trans_probs.T


class GammaHRF(object):
    """Fit and predict a single gamma pdf function from timecourse data."""
    def __init__(self, shape=7, loc=0, scale=.9,
                 coef=None, baseline=None, bounds=None):
        """Provide starting values for the optimization and optional bounds.

        Defaults are basically a single Glover HRF.

        If `bounds` is not None, contstrained optimization is performed
        using leastsqbound.

        Parameters
        ----------
        shape, loc, scale : floats
            parameters for scipy.stats.gamma
        coef : float
            height scaling parameter. calculated automatically if None
        baseline : None or float
            value of function at loc_0. uses data min if None
        bounds : dict or None
            keys are parameter names and values are (min, max)
            bounds for each parameter

        """
        self.starting_vals = [shape, loc, scale, coef, baseline]

        if bounds is None:
            self.bounds = None
        else:
            params = ["shape", "loc", "scale", "coef", "baseline"]
            cols = ["min", "max"]
            bdf = pd.DataFrame(index=params, columns=cols)
            bdf[:] = None
            bdf.update(pd.DataFrame(bounds, index=cols).T)
            self.bounds = map(tuple, np.array(bdf).tolist())

    def fit(self, x, y, maxfev=0):
        """Optimize a fit to data using least squares.

        Parameters
        ----------
        x : 1d array
            timepoints (in seconds)
        y : 1d array
            observed data values
        maxfev : int, optional
            maximum function evals (0 sets automatic default)

        Returns
        -------
        self : reference to self

        """
        def _objective(vals):

            shape, loc, scale, coef, baseline = vals
            pdf = stats.gamma(shape, loc, scale).pdf(x)
            pdf *= coef
            pdf += baseline
            return y - pdf

        shape, loc, scale, coef, baseline = self.starting_vals
        if baseline is None:
            baseline = np.min(y)

        if coef is None:
            y_range = np.max(y) - baseline
            starting_mode = (shape - 1) * scale + loc
            gamma_rv = stats.gamma(shape, loc, scale)
            starting_peak = gamma_rv.pdf(starting_mode)
            coef = y_range / starting_peak

        starting_vals = [shape, loc, scale, coef, baseline]
        if self.bounds is None:
            optim_vals, _ = sp.optimize.leastsq(_objective,
                                                starting_vals,
                                                maxfev=maxfev)
        else:
            from moss import leastsqbound
            optim_vals, _ = leastsqbound.leastsqbound(_objective,
                                                      starting_vals,
                                                      bounds=self.bounds,
                                                      maxfev=maxfev)

        shape, loc, scale, coef, baseline = optim_vals
        self.shape_ = shape
        self.loc_ = loc
        self.scale_ = scale
        self.coef_ = coef
        self.baseline_ = baseline
        self.fit_r2_ = self.r2_score(x, y)
        return self

    def predict(self, x):
        """Using fit values, predict heights for new timepoints.

        Parameters
        ----------
        x : array
            timepoints (in seconds)

        Returns
        -------
        hrf : array
            predicted heights at each timepoint

        """
        hrf = stats.gamma(self.shape_, self.loc_, self.scale_).pdf(x)
        hrf *= self.coef_
        hrf += self.baseline_
        return hrf

    def r2_score(self, x, y):
        """Predict new values at x and measure fit with y."""
        hrf = self.predict(x)
        return r2_score(hrf, y)

    @property
    def peak_time_(self):
        """Use fit parameters to predict time to peak."""
        if self.shape_ > 1:
            return (self.shape_ - 1) * self.scale_ + self.loc_
        return self.loc_
