import collections, os, posixpath, copy
import sys, time, datetime, json

import boto, boto.s3.key, boto.s3.bucket

import aws

import logging
log = logging.getLogger('motorboto.s3')

FIVE_MB = 1024 * 1024 * 5
FIVE_GB = FIVE_MB * 1024

MAX_UPLOAD_CHUNK_SIZE = FIVE_MB

from util import property_lazy

class Connection(aws.Connection):
    """Top-level s3 connection, from which you can get buckets
    
    >>> import motorboto
    >>> s3_connection = motorboto.AWS().s3
    >>> s3_connection
    <motorboto.s3.Connection object at ...>
    """
    @property_lazy
    def boto(self):
        return boto.connect_s3(**self.aws._auth_params)

    @property
    def buckets(self):
        c = self
        class BucketsDict(dict):
            def __getitem__(self, key):
                return Bucket(c, key)
                
            def __setitem__(self, key, val):
                raise ValueError
                
        return BucketsDict({ bb.name: Bucket(c, bb)
                        for bb in self.boto.get_all_buckets() })

    def refresh(self):
        try:
            del self.boto
        except AttributeError:
            pass
            


class Bucket(collections.Mapping):
    """Top-level s3 connection, from which you can get buckets
    
    >>> import motorboto
    >>> s3 = motorboto.AWS().s3
    >>> b = s3.buckets['motorboto-testonly-bucket3']
    >>> b
    <motorboto.s3.Bucket object at ...>
    >>> b.delete()
    >>> b.exists
    False
    
    Your bucket is only created when you put something in it.
    
    You can put_string to save strings, and put_file to save file-like objects.
    
    >>> b['test-key'].put_string('test')
    <motorboto.s3.S3SaveResult instance at ...>
    >>> b.exists
    True
    >>> 'test-key' in b
    True
    
    Delete a bucket with the delete method.
    
    >>> b2 = s3.buckets['motorboto-testonly-bucket3']
    >>> b2.delete()
    >>> b2.exists
    False
    
    Warning: Separate bucket objects won't keep up to date with actions from another.
    
    >>> b.exists
    True
    
    ^ Wrong! Out of date!
    
    """
    
    def __init__(self, 
                 connection=None, 
                 bucket=None):
        self.connection = connection

        if isinstance(bucket, boto.s3.bucket.Bucket):
            self.name = bucket.name
            self.boto = bucket
        else:
            self.name = bucket

    @property_lazy
    def exists(self):
        log.info('checking existence')
        bb = self._get_boto_bucket()
        if bb:
            self.boto = bb
            return True
        return False

    def _get_boto_bucket(self, create_if_nonexistent=False):
        b = self.connection.boto.lookup(self.name)
        
        if b:
            return b
            
        if not create_if_nonexistent:
            return None
        
        b = self.connection.boto.create_bucket(self.name)
        self.exists = True
        return b

    @property_lazy
    def boto(self):
        try:
            return self._get_boto_bucket(create_if_nonexistent=True)
        except boto.exception.S3ResponseError as e:
            raise ValueError('error: bucket name already taken perhaps?')
    
    def find(self, prefix='', delimiter=''):
        return self._find_keys(prefix,delimiter)

    def _find_keys(self, prefix='', delimiter=''):
        return { k.name: Key(self, k) for k in self.boto.list(prefix='', delimiter='') }

    @property
    def by_checksum(self):
        return { k.server_checksum: (k.bucket.name, k.name)
                                    for k in self.values() }

    def delete(self):
        if not self.exists:
            try:
                del self.boto
            except AttributeError:
                pass
            return
            
        try:
            for keyname, k in self.items():
                log.info("deleting %s" % keyname)
                k.delete()
    
            log.info('keys deleted, deleting bucket itself')
        except boto.exception.S3ResponseError:
            log.info('404 on bucket lookup')
        
        try:
            self.boto.delete()
        except boto.exception.S3ResponseError:
            log.info('404 on bucket lookup')
            
        self.exists = False
        
        try:
            del self.boto
        except AttributeError:
            pass
        
        
    def __getitem__(self, keyname):
	    return Key(self, keyname)

    def __contains__(self, keyname):
        return Key(self, keyname).exists
        
    def __len__(self):
        raise ValueError

    def __iter__(self):
        return iter(self.keys())
        
    def keys(self):
        return self.find()
    
    def values(self):
        return [Key(self, k) for k in self.keys() ]
    
    def items(self):
        return [(k, Key(self, k)) for k in self.keys()]
	        
    def status_multipart(self):
        return [mp for mp in self.boto.list_multipart_uploads()]

    def cleanup_multipart(self):
        for mp in self.status_multipart():
            mp.cancel_upload()
    
    def refresh(self):
        try:
            del self.boto
        except AttributeError:
            pass
        self.connection.refresh()
    
    

def local_checksum_tuple(f):
    from boto.s3 import key
    return key.Key().compute_md5(f)

def local_checksum(f):
    return local_checksum_tuple(f)[0]

def local_size(f):
    import io
    f.seek(0, io.SEEK_END)
    size = f.tell()
    f.seek(0, io.SEEK_SET)
    return size


    
class Key(object):
    """s3.buckets is a dictionary. Just index into it to grab your bucket. You can even grab non-existent buckets, just get the bucket and start saving to it and it will be created for you.
    
    >>> import motorboto
    >>> s3 = motorboto.AWS().s3
    >>> b = s3.buckets['motorboto-testonly-bucketx']
    >>> k = b['your/key/here']
    >>> k
    <motorboto.s3.Key object at ...>
    >>> k.delete()  # for demo purposes, make sure this key doesn't exist
    >>> result = k.put_string('key contents, whatever you like')
    >>> result.changed
    True
    >>> result = k.put_string('key contents, whatever you like')
    >>> result.changed
    False
    
    You can also add and change the metadata that gets saved with each key.
    
    >>> k.metadata['Cache-Control'] = 'no-cache'
    >>> k.metadata['feeling'] = 'so meta right now'
    >>> result = k.put_string('key contents, whatever you like')
    >>> result.metadata_changed
    True
    >>> 'Cache-Control' in k.metadata
    True
    >>> 'feeling' in k.metadata
    True
    
    >>> b['your/key/here'].get_string()
    'key contents, whatever you like'
    >>> k.exists
    True
    >>> k.delete()
    >>> k.exists
    False
    
    Multipart upload will split your data into chunks and upload them in parallel, nice.
    
    >>> s3.MAX_UPLOAD_CHUNK_SIZE = 5 * 1024
    >>> string_6k = '123456' * 1024
    >>> import hashlib
    >>> hashlib.md5(string_6k).hexdigest()
    '4beea4950fa03d15089d7249220a9791'
    >>> k = b['multipart_key']
    >>> k.delete() # for demo purposes, make sure this key doesn't exist
    >>> k.put_string(string_6k)
    <motorboto.s3.S3SaveResult instance at ...>
    >>> k.size
    6144
    >>> k.server_checksum
    '4beea4950fa03d15089d7249220a9791'
    >>> existing = b.by_checksum
    >>> '4beea4950fa03d15089d7249220a9791' in existing
    True
    >>> k2 = b['another_key']
    >>> k2.delete() # assume this key aint here already
    >>> result = k2.put_string(string_6k, copy_from=existing, public=True)
    >>> result.saved_from_copy
    True
    
    Add async=True to save in parallel. Wait for your parallel uploads to complete with motorboto.wait() (motorboto will also do this automatically before you exit)
    
    """
    
    def __init__(self, 
                 bucket, 
                 keyname_or_boto_key, 
                 fname=None):
        self.bucket = bucket
        self.fname = fname
        
        if isinstance(keyname_or_boto_key, boto.s3.key.Key):
            self.boto = keyname_or_boto_key
            self.name = keyname_or_boto_key.name
        else:
            tokens = keyname_or_boto_key.split(os.sep)
            self.name = posixpath.join(*tokens)
        self._metadata_from_boto = False
        
    def __eq__(self, other):
        return (isinstance(other, self.__class__)
            and self.name == other.name)
        
    def __ne__(self, other):
        return not self.__eq__(other)
    
    def _threadsafe_copy(self):
        k = copy.deepcopy(self)
        k.refresh()
        return k
    
    @property
    def size(self):
        return self.boto.size
        
    @property
    def endpoint(self):
        return self.boto.get_website_endpoint()
        
    def get_string(self):
        return self.boto.get_contents_as_string()
    
    def put_string(self, 
                s,
                public=False,
                async=False,
                copy_from=None):
        import StringIO
        f = StringIO.StringIO()
        f.write(s)
        f.seek(0)
        return self.save(f, public=public, async=async, copy_from=copy_from)
    
    def get_file(self, f):
        return self.boto.get_contents_to_file(f)
    
    def put_file(self, 
                f_or_fname,
                public=False,
                async=False,
                copy_from=None):
        f = f_or_fname
        if isinstance(f, basestring):
            f = open(f, 'rb')
        
        return self.save(f, public=public, async=async, copy_from=copy_from)
    
    
    
    def save(self, f, public, async, copy_from):
        if not async:
            return self._save(f, public, copy_from)
        else:
            import _threadpool
            k = self._threadsafe_copy()
            
            a = _threadpool.executor.submit(k._save, f, public, copy_from)
            _threadpool.futureslist.append(a)
        
    
    
    def _save(self, 
         f,
         public=False,
         copy_from=None):
        MAX_RETRIES = 5
        for attempt in range(MAX_RETRIES):
            f.seek(0)
            try:
                k = self
                class S3SaveResult:
                    def __init__(self):
                        self.key = k
                        self.changed = False
                        self.metadata_changed = False
                        self.multipart = False
                        self.saved_from_copy = False
                result = S3SaveResult()
                
                metadata = self.metadata
                
                if metadata != self._server_metadata:
                    result.metadata_changed = True
                    
                log.info('SAVING TO S3: %s', self.name)
                
                checksum_tuple = local_checksum_tuple(f)
                local_checksum = checksum_tuple[0]
                if local_checksum != self.server_checksum:
                    log.info('UPLOADING: %s', self.name)
                    result.changed = True
                    
                    if copy_from:
                        from_key = copy_from.get(local_checksum)
                        if from_key:
                            from_b, from_k = from_key
                            try:
                                self._copy_from_keyname(from_b, from_k, metadata=metadata)
                                result.saved_from_copy = True
                            except ValueError:
                                log.info('could not do a copy, reuploading...')
                                
                    if not result.saved_from_copy:
                        if local_size(f) > MAX_UPLOAD_CHUNK_SIZE:
                            result.multipart = True
                            log.info('DOING MULTIPART UPLOAD')
                            
                            mp = self.bucket.boto.initiate_multipart_upload(self.name,
                                                                            metadata=metadata)
                            from helpful.fparts import file_parts
                            fps = file_parts(f, MAX_UPLOAD_CHUNK_SIZE)
                            
                            parts = {
                                _threadpool.executor.submit(mp.upload_part_from_file,
                                                                 fp, index+1): index+1
                                                                 for index, fp in enumerate(fps) }
                            for t in parts:
                                _threadpool.futureslist.append(t)
                            for future in futures.as_completed(parts):
                                index = parts[future]
                                log.info('SAVED PART: %s (part #%s)' % (self.name, index))
                            mp.complete_upload()
                        else:
                            for k, v in self.metadata.items():
                                self.boto.set_metadata(k, v)
                            self.boto.set_contents_from_file(f, md5=checksum_tuple)
                        
                else:
                    log.info('UNCHANGED: %s', self.name)
                    
                    if result.metadata_changed:
                        log.info('UPDATING METADATA: %s', self.name)
                        self.bucket.boto.copy_key(self.name, 
                                                  self.bucket.name,
                                                  self.name,
                                                  metadata=metadata)
                                                  
                if public:
                    self.make_public()
                    
                if 'metadata' in self.__dict__:
                    del self.metadata                                  
                self.refresh()
                log.info('SAVED TO S3: %s', self.name)
                return result
    
            except Exception as e:
                import traceback
                log.warning('save error, retrying')
                log.warning(traceback.format_exc())
                time.sleep(attempt * 5)
                if not self.bucket.exists:
                    log.warning('bucket does not exist yet...')
                    self.refresh()
        raise
                
                
                
    
    @property_lazy
    def boto(self):
        _boto = self.bucket.boto.get_key(self.name)
        
        if not _boto:
            _boto = boto.s3.key.Key(self.bucket.boto)
            _boto.key = self.name
        else:
            #get_key returned a full object so the metadata is populated
            self._metadata_from_boto = True
        return _boto
    
    @property
    def exists(self):
        return self.boto.exists()

    @property
    def server_checksum(self):
        if not self.boto.etag:
            return None
        etag = self.boto.etag.strip('"')
        if '-' in etag:
            return self.metadata.get('md5')
        return etag


    @property_lazy
    def metadata(self):
        def add_special(m):
            """
            Disgusting hack so that these two headers are in the
            metadata dictionary with all the other metadata.
            
            Must be a better way to solve this?
            """
            if self.boto.cache_control:
                m['Cache-Control'] = \
                        self.boto.cache_control
            if self.boto.content_disposition:
                m['Content-Disposition'] = \
                        self.boto.content_disposition
        
        if 'boto' in self.__dict__ and not self._metadata_from_boto:
            del self.boto
        
        _metadata = self.boto.metadata.copy()
        add_special(_metadata)
        self._server_metadata = _metadata.copy()
        return _metadata
        

    def refresh(self):
        if 'boto' in self.__dict__:
            del self.boto
        self.bucket.refresh()

    def delete(self):
        self.boto.delete()
        self.refresh()
    
    def copy_from_key(self, from_key, metadata=None):
        self._copy_from_keyname(from_key, metadata)
        self.refresh()                        
                                
    def _copy_from_key(self, from_key, metadata=None):
        if not from_key.exists:
            raise ValueError("trying to copy from a key that doesn't exist")
        self._copy_from_keyname(from_key.bucket.name,
                                from_key.name,
                                metadata,
                                size=from_key.size)                            

    def _copy_from_keyname(self, 
                            from_bucket_name, 
                            from_key_name, 
                            metadata=None,
                            size=None):
        if not size:
            from_bucket = self.bucket.connection.buckets[from_bucket_name]
            size = from_bucket[from_key_name].size
        ksize = size

        if not ksize:
            raise ValueError("there doesn't seem to be a non-zero sized key here")

        if ksize <= FIVE_GB:
            self.bucket.boto.copy_key(self.name, 
                                      from_bucket_name,
                                      from_key_name,
                                      metadata)
        else:
            mp = self.bucket.boto.initiate_multipart_upload(self.name,
                                                metadata=metadata)

            ranges = [(start, min(start+FIVE_GB, ksize)) 
                    for start in range(0,ksize,FIVE_GB)]
            
            with futures.ThreadPoolExecutor(max_workers=128) as executor:
                future_to_key = { _threadpool.executor.submit(mp.copy_part_from_key,
                                                 from_bucket_name,
                                                 from_key_name,
                                                 index+1,
                                                 r[0], r[1]-1,
                                                 ): index+1
                                 for index, r in enumerate(ranges) }
            for t in parts:
                _threadpool.futureslist.append(t)             
            for future in futures.as_completed(future_to_key):
                index = future_to_key[future]
                log.info('COPIED: %s (chunk #%s)' % (self.name, index))
            mp.complete_upload()

    def make_public(self):
        self.boto.make_public()
    




