import os

def paths(path, relative_to='.', recursive=True, filter_method=None):
    """Returns a list of all valid/existing subpaths of the specified path
    >>> from motorboto.helpful.paths import paths

    >>> [p for p in paths('testfolder')]
    ['testfolder/a.txt', 'testfolder/b.txt', 'testfolder/c', 'testfolder/c/c.txt', 'testfolder/c/d.txt']
    
    >>> [p for p in paths('testfolder', recursive=False)]
    ['testfolder/a.txt', 'testfolder/b.txt', 'testfolder/c']
    
    >>> [p for p in paths('testfolder', relative_to='testfolder/c')]
    ['../a.txt', '../b.txt', '.', 'c.txt', 'd.txt']
    
    >>> [p for p in paths('testfolder', relative_to='testfolder', recursive=False)]
    ['a.txt', 'b.txt', 'c']
    
    >>> [p for p in paths('nonexistent')]
    Traceback (most recent call last):
        ...
    ValueError: this path does not exist
    
    """
    
    if not os.path.exists(path):
        raise ValueError("this path does not exist")
    
    for root, dirs, files in os.walk(path):
        files.sort()
        dirs.sort()
    
        for f in (files + dirs):
            p = os.path.join(root, f)
            
            if not filter_method or filter_method(p):
                yield os.path.normpath(os.path.relpath(p, relative_to))
        if not recursive: break


def dirs(*args, **kwargs):
    """Same as paths, but filtered by directories/folders only.
    >>> from motorboto.helpful.paths import dirs
    >>> [p for p in dirs('testfolder', relative_to='testfolder/c')]
    ['.']
    """

    kwargs['filter_method'] = os.path.isdir 
    return paths(*args, **kwargs)


def files(*args, **kwargs):
    """Same as paths, but filtered by files only.
    >>> from motorboto.helpful.paths import files
    >>> [p for p in files('testfolder', relative_to='testfolder/c')]
    ['../a.txt', '../b.txt', 'c.txt', 'd.txt']
    """

    kwargs['filter_method'] = os.path.isfile 
    return paths(*args, **kwargs)


def split(path):
    """Same as paths, but filtered by files only.
    >>> from motorboto.helpful.paths import split
    >>> [part for part in split('a/b/c.txt')]
    ['a', 'b', 'c.txt']
    
    """
    def gen(path):
        while True:
            if not path: break
            head, tail = os.path.split(path)
            path = head
            yield tail
    
    return reversed(list(gen(path)))

def name(path, extension=True):
    _, filename = os.path.split(path)
    if extension: return filename
    return os.path.splitext(filename)[0]
    
