
import io
import os
import threading
lock = threading.Lock()
tlock = threading.Lock()
seeklock = threading.Lock()

SEEK_SET = getattr(io, 'SEEK_SET', 0)
SEEK_CUR = getattr(io, 'SEEK_CUR', 1)
SEEK_END = getattr(io, 'SEEK_END', 2)

def file_parts(f, part_size):
    """
    Get all the file part objects for a particular file-like object
    
    >>> import StringIO
    >>> from motorboto.helpful.fparts import file_parts
    >>> f = StringIO.StringIO()
    >>> f.write('aabbccd')
    >>> parts = file_parts(f, 2)
    >>> len(list(parts))
    4
    
    """
    offset = 0
    while True:
        try:
            yield FilePart(f, offset, part_size)
        except ValueError:
            break
        offset += part_size

class FilePart(io.RawIOBase):
    """
    A file-part is a file-like object that reads only part of a file.
    
    >>> import StringIO, io
    >>> from motorboto.helpful.fparts import FilePart
    >>> f = StringIO.StringIO()
    >>> f.write('aabbccd')
    >>> f0 = FilePart(f, 0, 2)
    >>> f1 = FilePart(f, 2, 2)
    >>> f2 = FilePart(f, 4, 2)
    >>> f3 = FilePart(f, 6, 2)
    >>> f2.tell()
    0
    >>> f0.read()
    'aa'
    >>> f1.read()
    'bb'
    >>> f2.read()
    'cc'
    >>> f3.read()
    'd'
    >>> f2.tell()
    2
    >>> f3.tell()
    1
    >>> f2.seek(0, io.SEEK_SET)
    >>> f2.tell()
    0
    >>> f2.seek(0, io.SEEK_END)
    >>> f2.tell()
    2
    >>> f2.seek(-1, io.SEEK_CUR)
    >>> f2.tell()
    1
    >>> f2.read()
    'c'
    """
    
    def __init__(self, f, offset=0, bytes=None,
        *args, **kwargs):
        """
        Open a file chunk. The mode can only be 'r' for reading. Offset
        is the amount of bytes that the chunks starts after the real file's
        first byte. Bytes defines the amount of bytes the chunk has, which you
        can set to None to include the last byte of the real file.
        """
        
        self.f = f
        self.offset = offset
        
        self.f.seek(0, SEEK_END)
        length = self.f.tell()
        self.f.seek(self.offset, SEEK_SET)
        
        if not bytes:
            bytes = length
        
        remaining_bytes = length - self.offset
        self.bytes = min([bytes, remaining_bytes])
        if self.bytes <= 0:
            raise ValueError
        self.current = self.offset
        

    def seek(self, offset, whence=SEEK_SET):
        if whence == SEEK_SET:
            self.current = self.offset + offset
        elif whence == SEEK_CUR:
            self.current = self.current + offset
        elif whence == SEEK_END:
            self.current = self.offset + self.bytes + offset

    def tell(self):
        """
        Current file position.
        """
        return self.current - self.offset
        
    def read(self, n=-1):
        """
        Read and return at most n bytes.
        """
        max_n = (self.offset + self.bytes) - self.current
        if n < 0:
            n = max_n
        n = min([n, max_n])
        lock.acquire()
        self.f.seek(self.current, SEEK_SET)
        s = self.f.read(n)
        lock.release()
        
        self.current += len(s)
        return s
            





