# -*- coding: utf-8 -*-

from email.header import Header
import mimetypes
import os
import uuid

from .utils import exclusive_overlap

__all__ = ['MultipartIterator', 'Field', 'Multipart']


class State:
    boundary, headers, fields, finished = range(4)


class MultipartIterator(object):
    def __init__(self, multipart, chunksize=4096):
        self.buffer = ''
        self.chunksize = chunksize
        self.index = 0
        self.state = State.boundary
        self.multipart = multipart

    def __iter__(self):
        return self

    def __next__(self):
        boundary = self.multipart.get_boundary()

        while len(self.buffer) < self.chunksize:
            if self.state == State.finished:
                # Seems like we're finished, but if there's data left in the
                # buffer, we'll have to send it.
                if self.buffer:
                    break

                raise StopIteration

            field = self.multipart.fields[self.index]

            if self.state == State.fields:
                chunk = field.value

                # If we got data from the value-attribute, we've got all the
                # data there is. Therefore, we'll set the eof flag to move onto
                # the next field.
                if not chunk is None:
                    eof = True

                # But if we didn't get a chunk of data, we'll try to read from
                # the fileobj instead.
                else:
                    chunk = field.fileobj.read(self.chunksize)

                    # Check if the chunk ends with some of the boundary, and if
                    # it does; read the remainder of the boundary length to
                    # see if the whole boundary really occurs in the data.
                    offset = exclusive_overlap(chunk, boundary)

                    if 0 < offset < len(boundary):
                        chunk += field.fileobj.read(len(boundary) - offset)

                    # We've reached eof if the length of the chunk that was
                    # returned is shorter than what we asked for.
                    eof = len(chunk) < self.chunksize

                if boundary in chunk:
                    raise ValueError('Boundary was found in field data')

                self.buffer += chunk

                # AKA: If we've finished reading the current field
                if eof:
                    self.index += 1
                    self.state = State.boundary

                    # If this was the last part, we'll write the ending
                    # boundary and stop iteration.
                    if self.index >= len(self.multipart.fields):
                        self.buffer += '\r\n--{0}--\r\n'.format(boundary)
                        self.state = State.finished
                        break

                    self.buffer += '\r\n'

                    # Make sure that we try to send our data before going to
                    # the boundary state.
                    break

            if self.state == State.headers:
                self.buffer += field.format_headers()
                self.state = State.fields

            if self.state == State.boundary:
                self.buffer += '--' + boundary + '\r\n'
                self.state = State.headers

        # Grab a chunk out of the buffer and yield it as the result!
        chunk = self.buffer[:self.chunksize]
        self.buffer = self.buffer[self.chunksize:]

        return chunk
    next = __next__


class Field(object):
    def __init__(self, name, value=None, fileobj=None, filename=None,
            filesize=0, content_type=None):

        if all((value, fileobj)):
            raise ValueError('You cannot set both value and fileobj')

        self.name = name
        self.value = value
        self.fileobj = fileobj
        self.filename = filename or fileobj and fileobj.name
        self.filesize = fileobj and self.get_filesize(fileobj) or filesize
        self.content_type = content_type

    def __str__(self):
        return self.format_headers() + (self.value or self.fileobj.read())

    def get_content_disposition(self):
        context = {'name': Header(self.name)}
        template = 'form-data; name="{name}"'

        if self.filename:
            context['filename'] = Header(self.filename)
            template += '; filename="{filename}"'

        return template.format(**context)

    def get_content_type(self):
        content_type = ''

        if not self.content_type:
            if self.value:
                content_type = 'text/plain; charset=utf-8'
            else:
                content_type, content_encoding = mimetypes.guess_type(
                    self.filename or ''
                )

        return self.content_type or content_type or 'application/octet-stream'

    def get_filesize(self, fileobj):
        try:
            return os.fstat(fileobj.fileno()).st_size

        except (OSError, AttributeError):
            try:
                fileobj.seek(0, os.SEEK_END)
                filesize = fileobj.tell()
                fileobj.seek(0)

                return filesize

            except:
                raise ValueError('Could not get filesize')

    def get_headers(self):
        return {
            'Content-Disposition': self.get_content_disposition(),
            'Content-Type': self.get_content_type()
        }

    def format_headers(self):
        headers = []
        for name, value in self.get_headers().iteritems():
            headers.append(name + ': ' + value)

        # A blank line is required between the headers and the body
        headers += ['', '']
        return '\r\n'.join(headers)

    def get_content_length(self):
        return len(self.format_headers()) + \
            (len(self.value or '') or self.filesize)


class Multipart(object):
    def __init__(self, fields, boundary=None, chunksize=4096):
        self.boundary = boundary
        self.chunksize = chunksize
        self.fields = fields

    def __iter__(self):
        return MultipartIterator(self, chunksize=self.chunksize)

    def __str__(self):
        return ''.join(self)

    def get_boundary(self):
        if not self.boundary:
            self.boundary = uuid.uuid4().hex

        return self.boundary

    def get_content_length(self):
        content_length = 0

        # Plus 4 because of the 2 preceding dashes and a CRLF
        boundary_length = 4 + len(self.get_boundary())

        for field in self.fields:
            content_length += boundary_length
            content_length += field.get_content_length()
            content_length += 2 # CRLF, end of field

        # Plus 2, for the two ending dashes of the last boundary
        return content_length + boundary_length + 2

    def get_headers(self):
        return {
            'Content-Type': 'multipart/form-data; boundary=' + \
                self.get_boundary(),
            'Content-Length': str(self.get_content_length())
        }
